import { DateTime } from "./DateTime";
import { Enum } from "./Enum";
import List from "./List";
import { StiString } from "./StiObject";
export default class StiJson {
    static { this.prettyPrint = true; }
    static dateToJsonDate(date) {
        let offset = date.date.getTimezoneOffset() * -1;
        let hoursOffset = Math.abs(Math.floor(offset / 60)).toString();
        let minutesOffset = Math.abs(offset % 60).toString();
        if (hoursOffset.length == 1)
            hoursOffset = "0" + hoursOffset;
        if (minutesOffset.length == 1)
            minutesOffset = "0" + minutesOffset;
        return (
        /* "/Date(" +*/ Date.parse(date.date.toString()).toString() +
            (offset >= 0 ? "+" : "-") +
            hoursOffset +
            minutesOffset /*+ ")/"*/);
    }
    static jsonDateFormatToDate(jsonDate) {
        let dateStr = jsonDate.split("/Date(").join("").split(")/").join("");
        let date = new DateTime(new Date(parseInt(dateStr)));
        let startOffsetIndex = dateStr.indexOf("+");
        if (startOffsetIndex == -1)
            startOffsetIndex = dateStr.indexOf("-");
        let localOffset = date.date.getTimezoneOffset();
        if (startOffsetIndex != -1)
            localOffset = 0;
        let localHoursOffset = Math.floor(localOffset / 60);
        let localMinutesOffset = localOffset % 60;
        let localHoursOffsetString = (Math.abs(localHoursOffset) < 10 ? "0" : "") + Math.abs(localHoursOffset).toString();
        let localMinutesOffsetString = (Math.abs(localMinutesOffset) < 10 ? "0" : "") + Math.abs(localMinutesOffset).toString();
        let offset = (localOffset >= 0 ? "+" : "-") + localHoursOffsetString + localMinutesOffsetString;
        let multiple = 1;
        if (startOffsetIndex != -1)
            offset = dateStr.substring(startOffsetIndex);
        if (offset.substr(0, 1) == "-")
            multiple = -1;
        date.addHours(localHoursOffset + parseInt(offset.substr(1, 2)) * multiple);
        date.addMinutes(localMinutesOffset + parseInt(offset.substr(3, 2)) * multiple);
        return date;
    }
    properties() {
        if (!this.isProperty) {
            if (this.value instanceof List)
                return this.value;
            else
                return this.value.value;
        }
        return new List();
    }
    removeProperty(propertyName) {
        if (propertyName != null || propertyName != "") {
            let elements = this.value.value.filter((x) => x.name == propertyName);
            if (elements.length > 0) {
                this.value.value.remove(elements[0]);
            }
        }
    }
    addPropertyNumber(propertyName, value) {
        if (this.value == null)
            this.value = new StiJson("", new List());
        this.removeProperty(propertyName);
        this.value.value.push(new StiJson(propertyName, value, true));
    }
    addPropertyNumberNotNull(propertyName, value) {
        if (value) {
            this.addPropertyNumber(propertyName, value);
        }
    }
    addPropertyNumberNoDefaultValue(propertyName, value) {
        if (this.value == null)
            this.value = new StiJson("", new List());
        this.removeProperty(propertyName);
        this.value.value.push(new StiJson(propertyName, value, true));
    }
    addPropertyJObject(propertyName, value) {
        if (this.value == null)
            this.value = new StiJson("", new List());
        this.removeProperty(propertyName);
        if (value == null || value.count == 0)
            return;
        this.value.value.push(new StiJson(propertyName, value));
    }
    addPropertyArray(propertyName, elements) {
        if (this.value == null)
            this.value = new StiJson("", new List());
        this.removeProperty(propertyName);
        if (elements == null)
            return;
        let values = [];
        elements.forEach((e) => values.push(e?.saveToJsonObject != null
            ? e.saveToJsonObject(this.options)
            : typeof e == "string"
                ? `"${e
                    .replace(/[\\]/g, "\\\\")
                    .replace(/[\"]/g, '\\"')
                    // .replace(/[\/]/g, '\\/')
                    .replace(/[\b]/g, "\\b")
                    .replace(/[\f]/g, "\\f")
                    .replace(/\r\n/g, "\n")
                    .replace(/[\r]/g, "\n")
                    .replace(/[\r\n]/g, "\\r\\n")
                    .replace(/[\t]/g, "\\t")
                    // eslint-disable-next-line no-control-regex
                    .replace(/[\u0001-\u001F]/g, (chr) => "\\u" + ("0000" + chr.charCodeAt(0).toString(16)).substr(-4))}"`
                : e));
        let json = new StiJson(propertyName, values);
        json.isArray = true;
        this.value.value.push(json);
    }
    addPropertyIdent(propertyName, value) {
        if (this.value == null)
            this.value = new StiJson("", new List());
        this.value.value.push(new StiJson(propertyName, value, true));
    }
    addPropertyBool(propertyName, value) {
        if (this.value == null)
            this.value = new StiJson("", new List());
        this.removeProperty(propertyName);
        this.value.value.push(new StiJson(propertyName, value, true));
    }
    /*addPropertyDateTime(propertyName: string, value: DateTime) {
        if (this.value == null) this.value = new StiJson("", new List());
        this.removeProperty(propertyName);
  
        this.value.value.push(new StiJson(propertyName, StiJson.dateToJsonDate(value), true));
    }*/
    addPropertyEnum(propertyName, enumType, value, defaultValue = null) {
        if (this.value == null)
            this.value = new StiJson("", new List());
        this.removeProperty(propertyName);
        if (typeof defaultValue != "undefined" && value == defaultValue)
            return;
        this.value.value.push(new StiJson(propertyName, Enum.getName(enumType, value), true));
    }
    addPropertyColor(propertyName, color, defColor = null) {
        if (color == null || color.equals(defColor))
            return;
        let value = color.isNamedColor ? color.name.replace(" ", "") : `${color.a},${color.r},${color.g},${color.b}`;
        this.addPropertyString(propertyName, value);
    }
    addPropertyString(propertyName, value, defaultValue) {
        if (this.value == null)
            this.value = new StiJson("", new List());
        this.removeProperty(propertyName);
        if (value == defaultValue)
            return;
        this.value.value.push(new StiJson(propertyName, value, true));
    }
    addPropertyStringNullOrEmpty(propertyName, value) {
        if (this.value == null)
            this.value = new StiJson("", new List());
        this.removeProperty(propertyName);
        if (StiString.isNullOrEmpty(value))
            return;
        this.value.value.push(new StiJson(propertyName, value, true));
    }
    get count() {
        let count = 0;
        if (!this.isProperty && this.value != null) {
            // eslint-disable-next-line @typescript-eslint/no-unused-vars
            for (let propertyName of this.value.value) {
                count++;
            }
        }
        return count;
    }
    getArray() {
        let result = [];
        this.value.forEach((e) => result.push(e.value));
        return result;
    }
    serialize(indent = 0) {
        let jsonString = "";
        let tabs = StiJson.prettyPrint ? Array(indent + 1).join(" ") : "";
        for (let property of this.value.properties()) {
            let valueString = "";
            if (property.isProperty) {
                if (property.value != null && typeof property.value == "string")
                    valueString =
                        '"' +
                            property.value
                                .replace(/[\\]/g, "\\\\")
                                .replace(/[\"]/g, '\\"')
                                // .replace(/[\/]/g, '\\/')
                                .replace(/[\b]/g, "\\b")
                                .replace(/[\f]/g, "\\f")
                                .replace(/\r\n/g, "\n")
                                .replace(/[\r]/g, "\n")
                                .replace(/[\r\n]/g, "\\r\\n")
                                .replace(/[\t]/g, "\\t")
                                // eslint-disable-next-line no-control-regex
                                .replace(/[\u0001-\u001F]/g, (chr) => "\\u" + ("0000" + chr.charCodeAt(0).toString(16)).substr(-4)) +
                            '"';
                // "\"" + property.value.replace(/\"/gi, "\\\"").replace(/\r\n/gi, "\n").replace(/\r/gi, "\n").replace(/\n/gi, "\\r\\n").replace(/\\/gi, "\\\\") + "\"";
                else
                    valueString = property.value;
            }
            else if (property.isArray) {
                if (property.value.length == 0) {
                    valueString += "[";
                }
                for (let element of property.value) {
                    if (valueString)
                        valueString += ",";
                    else
                        valueString += "[";
                    valueString += element?.serialize != null ? element.serialize(indent + 4) : element;
                }
                valueString += "]";
            }
            else {
                valueString = property.value.serialize(indent + 2);
            }
            if (jsonString)
                jsonString += ",";
            if (StiJson.prettyPrint)
                jsonString += "\n" + tabs + "  ";
            jsonString += '"' + property.name + '":';
            if (StiJson.prettyPrint)
                jsonString += " ";
            jsonString += valueString;
        }
        if (!jsonString)
            return "";
        if (StiJson.prettyPrint)
            return "{" + jsonString + "\n" + tabs + "}";
        return "{" + jsonString + "}";
    }
    deserialize(text) {
        if (typeof text == "string") {
            let pos = 0;
            let start = -1;
            while (pos < text.length) {
                if (text[pos] == '"' &&
                    (pos > 0 && text[pos - 1] != "\\"
                        ? true
                        : pos > 1 && text[pos - 2] == "\\" && pos > 2 && text[pos - 3] != "\\")) {
                    if (start > -1) {
                        let subText = text.substring(start, pos);
                        let modText = subText
                            .replace(/\\\\/gi, "#!!#")
                            .replace(/\\t/gi, "#!t#")
                            .replace(/\\\"/gi, "#!q#")
                            .replace(/\\r\\n/gi, "#!rn#")
                            .replace(/\\r/gi, "#!rn#")
                            .replace(/\\n/gi, "#!rn#");
                        if (modText.length != subText.length) {
                            // eslint-disable-next-line @typescript-eslint/restrict-plus-operands
                            text = text.substring(0, start) + modText + text.substring(pos);
                            pos += modText.length - subText.length;
                        }
                        start = -1;
                    }
                    else
                        start = pos + 1;
                }
                pos++;
            }
        }
        let reportObject = typeof text == "object" ? text : JSON.parse(text);
        this.deserializeFromObject(reportObject);
        this.value = this.value.value;
    }
    deserializeFromObject(child) {
        for (let propertyName in child) {
            if (typeof child[propertyName] == "object") {
                let jObject = new StiJson(propertyName);
                if (this.value == null)
                    this.value = new StiJson(null, new List());
                this.value.value.push(jObject);
                jObject.deserializeFromObject(child[propertyName]);
            }
            else if (typeof child[propertyName] != "function") {
                let jObject = new StiJson(propertyName, child[propertyName], true);
                if (this.value == null)
                    this.value = new StiJson(null, new List());
                this.value.value.push(jObject);
            }
        }
        if (child && Array.isArray(child) && child.length == 0) {
            this.value = new StiJson(null, new List());
        }
    }
    toString() {
        return this.name;
    }
    constructor(name = null, value = null, isProperty = false, options = {}) {
        this.isProperty = false;
        this.isArray = false;
        this.name = name;
        if (value != null && typeof value == "string")
            value = value.replace(/#!rn#/gi, "\r\n").replace(/#!q#/gi, '"').replace(/#!t#/gi, "\t").replace(/#!!#/gi, "\\");
        this.value = value;
        this.isProperty = isProperty;
        this.options = options;
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiU3RpSnNvbi5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uLy4uLy4uL3Byb2plY3RzL3N0aW11bHNvZnQtZm9ybXMvc3JjL2xpYi9zeXN0ZW0vU3RpSnNvbi50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFDQSxPQUFPLEVBQUUsUUFBUSxFQUFFLE1BQU0sWUFBWSxDQUFDO0FBQ3RDLE9BQU8sRUFBRSxJQUFJLEVBQUUsTUFBTSxRQUFRLENBQUM7QUFDOUIsT0FBTyxJQUFJLE1BQU0sUUFBUSxDQUFDO0FBQzFCLE9BQU8sRUFBRSxTQUFTLEVBQUUsTUFBTSxhQUFhLENBQUM7QUFFeEMsTUFBTSxDQUFDLE9BQU8sT0FBTyxPQUFPO2FBQ25CLGdCQUFXLEdBQUcsSUFBSSxBQUFQLENBQVE7SUFHMUIsTUFBTSxDQUFDLGNBQWMsQ0FBQyxJQUFjO1FBQ2xDLElBQUksTUFBTSxHQUFXLElBQUksQ0FBQyxJQUFJLENBQUMsaUJBQWlCLEVBQUUsR0FBRyxDQUFDLENBQUMsQ0FBQztRQUN4RCxJQUFJLFdBQVcsR0FBVyxJQUFJLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxHQUFHLEVBQUUsQ0FBQyxDQUFDLENBQUMsUUFBUSxFQUFFLENBQUM7UUFDdkUsSUFBSSxhQUFhLEdBQVcsSUFBSSxDQUFDLEdBQUcsQ0FBQyxNQUFNLEdBQUcsRUFBRSxDQUFDLENBQUMsUUFBUSxFQUFFLENBQUM7UUFDN0QsSUFBSSxXQUFXLENBQUMsTUFBTSxJQUFJLENBQUM7WUFBRSxXQUFXLEdBQUcsR0FBRyxHQUFHLFdBQVcsQ0FBQztRQUM3RCxJQUFJLGFBQWEsQ0FBQyxNQUFNLElBQUksQ0FBQztZQUFFLGFBQWEsR0FBRyxHQUFHLEdBQUcsYUFBYSxDQUFDO1FBRW5FLE9BQU87UUFDTCxlQUFlLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFFBQVEsRUFBRSxDQUFDLENBQUMsUUFBUSxFQUFFO1lBQzNELENBQUMsTUFBTSxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUM7WUFDekIsV0FBVztZQUNYLGFBQWEsQ0FBQyxVQUFVLENBQ3pCLENBQUM7SUFDSixDQUFDO0lBRUQsTUFBTSxDQUFDLG9CQUFvQixDQUFDLFFBQWdCO1FBQzFDLElBQUksT0FBTyxHQUFHLFFBQVEsQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUM7UUFDckUsSUFBSSxJQUFJLEdBQUcsSUFBSSxRQUFRLENBQUMsSUFBSSxJQUFJLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUVyRCxJQUFJLGdCQUFnQixHQUFXLE9BQU8sQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLENBQUM7UUFDcEQsSUFBSSxnQkFBZ0IsSUFBSSxDQUFDLENBQUM7WUFBRSxnQkFBZ0IsR0FBRyxPQUFPLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBRXBFLElBQUksV0FBVyxHQUFXLElBQUksQ0FBQyxJQUFJLENBQUMsaUJBQWlCLEVBQUUsQ0FBQztRQUN4RCxJQUFJLGdCQUFnQixJQUFJLENBQUMsQ0FBQztZQUFFLFdBQVcsR0FBRyxDQUFDLENBQUM7UUFDNUMsSUFBSSxnQkFBZ0IsR0FBVyxJQUFJLENBQUMsS0FBSyxDQUFDLFdBQVcsR0FBRyxFQUFFLENBQUMsQ0FBQztRQUM1RCxJQUFJLGtCQUFrQixHQUFXLFdBQVcsR0FBRyxFQUFFLENBQUM7UUFDbEQsSUFBSSxzQkFBc0IsR0FDeEIsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLGdCQUFnQixDQUFDLEdBQUcsRUFBRSxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxHQUFHLElBQUksQ0FBQyxHQUFHLENBQUMsZ0JBQWdCLENBQUMsQ0FBQyxRQUFRLEVBQUUsQ0FBQztRQUN2RixJQUFJLHdCQUF3QixHQUMxQixDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsa0JBQWtCLENBQUMsR0FBRyxFQUFFLENBQUMsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLEdBQUcsSUFBSSxDQUFDLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDLFFBQVEsRUFBRSxDQUFDO1FBRTNGLElBQUksTUFBTSxHQUFXLENBQUMsV0FBVyxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUMsR0FBRyxzQkFBc0IsR0FBRyx3QkFBd0IsQ0FBQztRQUN4RyxJQUFJLFFBQVEsR0FBRyxDQUFDLENBQUM7UUFFakIsSUFBSSxnQkFBZ0IsSUFBSSxDQUFDLENBQUM7WUFBRSxNQUFNLEdBQUcsT0FBTyxDQUFDLFNBQVMsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO1FBQ3pFLElBQUksTUFBTSxDQUFDLE1BQU0sQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLElBQUksR0FBRztZQUFFLFFBQVEsR0FBRyxDQUFDLENBQUMsQ0FBQztRQUU5QyxJQUFJLENBQUMsUUFBUSxDQUFDLGdCQUFnQixHQUFHLFFBQVEsQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxHQUFHLFFBQVEsQ0FBQyxDQUFDO1FBQzNFLElBQUksQ0FBQyxVQUFVLENBQUMsa0JBQWtCLEdBQUcsUUFBUSxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLEdBQUcsUUFBUSxDQUFDLENBQUM7UUFFL0UsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBT0QsVUFBVTtRQUNSLElBQUksQ0FBQyxJQUFJLENBQUMsVUFBVSxFQUFFLENBQUM7WUFDckIsSUFBSSxJQUFJLENBQUMsS0FBSyxZQUFZLElBQUk7Z0JBQUUsT0FBTyxJQUFJLENBQUMsS0FBSyxDQUFDOztnQkFDN0MsT0FBTyxJQUFJLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQztRQUMvQixDQUFDO1FBQ0QsT0FBTyxJQUFJLElBQUksRUFBVyxDQUFDO0lBQzdCLENBQUM7SUFFRCxjQUFjLENBQUMsWUFBb0I7UUFDakMsSUFBSSxZQUFZLElBQUksSUFBSSxJQUFJLFlBQVksSUFBSSxFQUFFLEVBQUUsQ0FBQztZQUMvQyxJQUFJLFFBQVEsR0FBYyxJQUFJLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQyxJQUFJLElBQUksWUFBWSxDQUFDLENBQUM7WUFDakYsSUFBSSxRQUFRLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRSxDQUFDO2dCQUN4QixJQUFJLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDdkMsQ0FBQztRQUNILENBQUM7SUFDSCxDQUFDO0lBRUQsaUJBQWlCLENBQUMsWUFBb0IsRUFBRSxLQUFhO1FBQ25ELElBQUksSUFBSSxDQUFDLEtBQUssSUFBSSxJQUFJO1lBQUUsSUFBSSxDQUFDLEtBQUssR0FBRyxJQUFJLE9BQU8sQ0FBQyxFQUFFLEVBQUUsSUFBSSxJQUFJLEVBQUUsQ0FBQyxDQUFDO1FBQ2pFLElBQUksQ0FBQyxjQUFjLENBQUMsWUFBWSxDQUFDLENBQUM7UUFFbEMsSUFBSSxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLElBQUksT0FBTyxDQUFDLFlBQVksRUFBRSxLQUFLLEVBQUUsSUFBSSxDQUFDLENBQUMsQ0FBQztJQUNoRSxDQUFDO0lBRUQsd0JBQXdCLENBQUMsWUFBb0IsRUFBRSxLQUFhO1FBQzFELElBQUksS0FBSyxFQUFFLENBQUM7WUFDVixJQUFJLENBQUMsaUJBQWlCLENBQUMsWUFBWSxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBQzlDLENBQUM7SUFDSCxDQUFDO0lBRUQsK0JBQStCLENBQUMsWUFBb0IsRUFBRSxLQUFhO1FBQ2pFLElBQUksSUFBSSxDQUFDLEtBQUssSUFBSSxJQUFJO1lBQUUsSUFBSSxDQUFDLEtBQUssR0FBRyxJQUFJLE9BQU8sQ0FBQyxFQUFFLEVBQUUsSUFBSSxJQUFJLEVBQUUsQ0FBQyxDQUFDO1FBQ2pFLElBQUksQ0FBQyxjQUFjLENBQUMsWUFBWSxDQUFDLENBQUM7UUFFbEMsSUFBSSxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLElBQUksT0FBTyxDQUFDLFlBQVksRUFBRSxLQUFLLEVBQUUsSUFBSSxDQUFDLENBQUMsQ0FBQztJQUNoRSxDQUFDO0lBRUQsa0JBQWtCLENBQUMsWUFBb0IsRUFBRSxLQUFjO1FBQ3JELElBQUksSUFBSSxDQUFDLEtBQUssSUFBSSxJQUFJO1lBQUUsSUFBSSxDQUFDLEtBQUssR0FBRyxJQUFJLE9BQU8sQ0FBQyxFQUFFLEVBQUUsSUFBSSxJQUFJLEVBQUUsQ0FBQyxDQUFDO1FBQ2pFLElBQUksQ0FBQyxjQUFjLENBQUMsWUFBWSxDQUFDLENBQUM7UUFFbEMsSUFBSSxLQUFLLElBQUksSUFBSSxJQUFJLEtBQUssQ0FBQyxLQUFLLElBQUksQ0FBQztZQUFFLE9BQU87UUFFOUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLElBQUksT0FBTyxDQUFDLFlBQVksRUFBRSxLQUFLLENBQUMsQ0FBQyxDQUFDO0lBQzFELENBQUM7SUFFRCxnQkFBZ0IsQ0FBQyxZQUFvQixFQUFFLFFBQWU7UUFDcEQsSUFBSSxJQUFJLENBQUMsS0FBSyxJQUFJLElBQUk7WUFBRSxJQUFJLENBQUMsS0FBSyxHQUFHLElBQUksT0FBTyxDQUFDLEVBQUUsRUFBRSxJQUFJLElBQUksRUFBRSxDQUFDLENBQUM7UUFDakUsSUFBSSxDQUFDLGNBQWMsQ0FBQyxZQUFZLENBQUMsQ0FBQztRQUVsQyxJQUFJLFFBQVEsSUFBSSxJQUFJO1lBQUUsT0FBTztRQUM3QixJQUFJLE1BQU0sR0FBRyxFQUFFLENBQUM7UUFFaEIsUUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsRUFBRSxFQUFFLENBQ3JCLE1BQU0sQ0FBQyxJQUFJLENBQ1QsQ0FBQyxFQUFFLGdCQUFnQixJQUFJLElBQUk7WUFDekIsQ0FBQyxDQUFDLENBQUMsQ0FBQyxnQkFBZ0IsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDO1lBQ2xDLENBQUMsQ0FBQyxPQUFPLENBQUMsSUFBSSxRQUFRO2dCQUN0QixDQUFDLENBQUMsSUFBSSxDQUFDO3FCQUNGLE9BQU8sQ0FBQyxPQUFPLEVBQUUsTUFBTSxDQUFDO3FCQUN4QixPQUFPLENBQUMsT0FBTyxFQUFFLEtBQUssQ0FBQztvQkFDeEIsMkJBQTJCO3FCQUMxQixPQUFPLENBQUMsT0FBTyxFQUFFLEtBQUssQ0FBQztxQkFDdkIsT0FBTyxDQUFDLE9BQU8sRUFBRSxLQUFLLENBQUM7cUJBQ3ZCLE9BQU8sQ0FBQyxPQUFPLEVBQUUsSUFBSSxDQUFDO3FCQUN0QixPQUFPLENBQUMsT0FBTyxFQUFFLElBQUksQ0FBQztxQkFDdEIsT0FBTyxDQUFDLFNBQVMsRUFBRSxRQUFRLENBQUM7cUJBQzVCLE9BQU8sQ0FBQyxPQUFPLEVBQUUsS0FBSyxDQUFDO29CQUN4Qiw0Q0FBNEM7cUJBQzNDLE9BQU8sQ0FBQyxrQkFBa0IsRUFBRSxDQUFDLEdBQUcsRUFBRSxFQUFFLENBQUMsS0FBSyxHQUFHLENBQUMsTUFBTSxHQUFHLEdBQUcsQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLENBQUMsUUFBUSxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsR0FBRztnQkFDMUcsQ0FBQyxDQUFDLENBQUMsQ0FDTixDQUNGLENBQUM7UUFFRixJQUFJLElBQUksR0FBWSxJQUFJLE9BQU8sQ0FBQyxZQUFZLEVBQUUsTUFBTSxDQUFDLENBQUM7UUFDdEQsSUFBSSxDQUFDLE9BQU8sR0FBRyxJQUFJLENBQUM7UUFDcEIsSUFBSSxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQzlCLENBQUM7SUFFRCxnQkFBZ0IsQ0FBQyxZQUFvQixFQUFFLEtBQWE7UUFDbEQsSUFBSSxJQUFJLENBQUMsS0FBSyxJQUFJLElBQUk7WUFBRSxJQUFJLENBQUMsS0FBSyxHQUFHLElBQUksT0FBTyxDQUFDLEVBQUUsRUFBRSxJQUFJLElBQUksRUFBRSxDQUFDLENBQUM7UUFDakUsSUFBSSxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLElBQUksT0FBTyxDQUFDLFlBQVksRUFBRSxLQUFLLEVBQUUsSUFBSSxDQUFDLENBQUMsQ0FBQztJQUNoRSxDQUFDO0lBRUQsZUFBZSxDQUFDLFlBQW9CLEVBQUUsS0FBYztRQUNsRCxJQUFJLElBQUksQ0FBQyxLQUFLLElBQUksSUFBSTtZQUFFLElBQUksQ0FBQyxLQUFLLEdBQUcsSUFBSSxPQUFPLENBQUMsRUFBRSxFQUFFLElBQUksSUFBSSxFQUFFLENBQUMsQ0FBQztRQUNqRSxJQUFJLENBQUMsY0FBYyxDQUFDLFlBQVksQ0FBQyxDQUFDO1FBRWxDLElBQUksQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxJQUFJLE9BQU8sQ0FBQyxZQUFZLEVBQUUsS0FBSyxFQUFFLElBQUksQ0FBQyxDQUFDLENBQUM7SUFDaEUsQ0FBQztJQUVEOzs7OztPQUtHO0lBRUgsZUFBZSxDQUFDLFlBQW9CLEVBQUUsUUFBYSxFQUFFLEtBQVUsRUFBRSxlQUFvQixJQUFJO1FBQ3ZGLElBQUksSUFBSSxDQUFDLEtBQUssSUFBSSxJQUFJO1lBQUUsSUFBSSxDQUFDLEtBQUssR0FBRyxJQUFJLE9BQU8sQ0FBQyxFQUFFLEVBQUUsSUFBSSxJQUFJLEVBQUUsQ0FBQyxDQUFDO1FBQ2pFLElBQUksQ0FBQyxjQUFjLENBQUMsWUFBWSxDQUFDLENBQUM7UUFFbEMsSUFBSSxPQUFPLFlBQVksSUFBSSxXQUFXLElBQUksS0FBSyxJQUFJLFlBQVk7WUFBRSxPQUFPO1FBRXhFLElBQUksQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxJQUFJLE9BQU8sQ0FBQyxZQUFZLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxRQUFRLEVBQUUsS0FBSyxDQUFDLEVBQUUsSUFBSSxDQUFDLENBQUMsQ0FBQztJQUN4RixDQUFDO0lBRUQsZ0JBQWdCLENBQUMsWUFBb0IsRUFBRSxLQUFZLEVBQUUsV0FBa0IsSUFBSTtRQUN6RSxJQUFJLEtBQUssSUFBSSxJQUFJLElBQUksS0FBSyxDQUFDLE1BQU0sQ0FBQyxRQUFRLENBQUM7WUFBRSxPQUFPO1FBRXBELElBQUksS0FBSyxHQUFHLEtBQUssQ0FBQyxZQUFZLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLEdBQUcsRUFBRSxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsR0FBRyxLQUFLLENBQUMsQ0FBQyxJQUFJLEtBQUssQ0FBQyxDQUFDLElBQUksS0FBSyxDQUFDLENBQUMsSUFBSSxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUM7UUFFN0csSUFBSSxDQUFDLGlCQUFpQixDQUFDLFlBQVksRUFBRSxLQUFLLENBQUMsQ0FBQztJQUM5QyxDQUFDO0lBRUQsaUJBQWlCLENBQUMsWUFBb0IsRUFBRSxLQUFhLEVBQUUsWUFBcUI7UUFDMUUsSUFBSSxJQUFJLENBQUMsS0FBSyxJQUFJLElBQUk7WUFBRSxJQUFJLENBQUMsS0FBSyxHQUFHLElBQUksT0FBTyxDQUFDLEVBQUUsRUFBRSxJQUFJLElBQUksRUFBRSxDQUFDLENBQUM7UUFDakUsSUFBSSxDQUFDLGNBQWMsQ0FBQyxZQUFZLENBQUMsQ0FBQztRQUVsQyxJQUFJLEtBQUssSUFBSSxZQUFZO1lBQUUsT0FBTztRQUVsQyxJQUFJLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSxPQUFPLENBQUMsWUFBWSxFQUFFLEtBQUssRUFBRSxJQUFJLENBQUMsQ0FBQyxDQUFDO0lBQ2hFLENBQUM7SUFFRCw0QkFBNEIsQ0FBQyxZQUFvQixFQUFFLEtBQWE7UUFDOUQsSUFBSSxJQUFJLENBQUMsS0FBSyxJQUFJLElBQUk7WUFBRSxJQUFJLENBQUMsS0FBSyxHQUFHLElBQUksT0FBTyxDQUFDLEVBQUUsRUFBRSxJQUFJLElBQUksRUFBRSxDQUFDLENBQUM7UUFDakUsSUFBSSxDQUFDLGNBQWMsQ0FBQyxZQUFZLENBQUMsQ0FBQztRQUVsQyxJQUFJLFNBQVMsQ0FBQyxhQUFhLENBQUMsS0FBSyxDQUFDO1lBQUUsT0FBTztRQUUzQyxJQUFJLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSxPQUFPLENBQUMsWUFBWSxFQUFFLEtBQUssRUFBRSxJQUFJLENBQUMsQ0FBQyxDQUFDO0lBQ2hFLENBQUM7SUFFRCxJQUFJLEtBQUs7UUFDUCxJQUFJLEtBQUssR0FBRyxDQUFDLENBQUM7UUFDZCxJQUFJLENBQUMsSUFBSSxDQUFDLFVBQVUsSUFBSSxJQUFJLENBQUMsS0FBSyxJQUFJLElBQUksRUFBRSxDQUFDO1lBQzNDLDZEQUE2RDtZQUM3RCxLQUFLLElBQUksWUFBWSxJQUFJLElBQUksQ0FBQyxLQUFLLENBQUMsS0FBSyxFQUFFLENBQUM7Z0JBQzFDLEtBQUssRUFBRSxDQUFDO1lBQ1YsQ0FBQztRQUNILENBQUM7UUFFRCxPQUFPLEtBQUssQ0FBQztJQUNmLENBQUM7SUFFRCxRQUFRO1FBQ04sSUFBSSxNQUFNLEdBQUcsRUFBRSxDQUFDO1FBQ2hCLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDO1FBQ2hELE9BQU8sTUFBTSxDQUFDO0lBQ2hCLENBQUM7SUFFRCxTQUFTLENBQUMsTUFBTSxHQUFHLENBQUM7UUFDbEIsSUFBSSxVQUFVLEdBQUcsRUFBRSxDQUFDO1FBQ3BCLElBQUksSUFBSSxHQUFHLE9BQU8sQ0FBQyxXQUFXLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7UUFDbEUsS0FBSyxJQUFJLFFBQVEsSUFBSSxJQUFJLENBQUMsS0FBSyxDQUFDLFVBQVUsRUFBRSxFQUFFLENBQUM7WUFDN0MsSUFBSSxXQUFXLEdBQUcsRUFBRSxDQUFDO1lBQ3JCLElBQUksUUFBUSxDQUFDLFVBQVUsRUFBRSxDQUFDO2dCQUN4QixJQUFJLFFBQVEsQ0FBQyxLQUFLLElBQUksSUFBSSxJQUFJLE9BQU8sUUFBUSxDQUFDLEtBQUssSUFBSSxRQUFRO29CQUM3RCxXQUFXO3dCQUNULEdBQUc7NEJBQ00sUUFBUSxDQUFDLEtBQU07aUNBQ3JCLE9BQU8sQ0FBQyxPQUFPLEVBQUUsTUFBTSxDQUFDO2lDQUN4QixPQUFPLENBQUMsT0FBTyxFQUFFLEtBQUssQ0FBQztnQ0FDeEIsMkJBQTJCO2lDQUMxQixPQUFPLENBQUMsT0FBTyxFQUFFLEtBQUssQ0FBQztpQ0FDdkIsT0FBTyxDQUFDLE9BQU8sRUFBRSxLQUFLLENBQUM7aUNBQ3ZCLE9BQU8sQ0FBQyxPQUFPLEVBQUUsSUFBSSxDQUFDO2lDQUN0QixPQUFPLENBQUMsT0FBTyxFQUFFLElBQUksQ0FBQztpQ0FDdEIsT0FBTyxDQUFDLFNBQVMsRUFBRSxRQUFRLENBQUM7aUNBQzVCLE9BQU8sQ0FBQyxPQUFPLEVBQUUsS0FBSyxDQUFDO2dDQUN4Qiw0Q0FBNEM7aUNBQzNDLE9BQU8sQ0FBQyxrQkFBa0IsRUFBRSxDQUFDLEdBQUcsRUFBRSxFQUFFLENBQUMsS0FBSyxHQUFHLENBQUMsTUFBTSxHQUFHLEdBQUcsQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLENBQUMsUUFBUSxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7NEJBQ3JHLEdBQUcsQ0FBQztnQkFDUix3SkFBd0o7O29CQUNuSixXQUFXLEdBQUcsUUFBUSxDQUFDLEtBQUssQ0FBQztZQUNwQyxDQUFDO2lCQUFNLElBQUksUUFBUSxDQUFDLE9BQU8sRUFBRSxDQUFDO2dCQUM1QixJQUFJLFFBQVEsQ0FBQyxLQUFLLENBQUMsTUFBTSxJQUFJLENBQUMsRUFBRSxDQUFDO29CQUMvQixXQUFXLElBQUksR0FBRyxDQUFDO2dCQUNyQixDQUFDO2dCQUNELEtBQUssSUFBSSxPQUFPLElBQUksUUFBUSxDQUFDLEtBQUssRUFBRSxDQUFDO29CQUNuQyxJQUFJLFdBQVc7d0JBQUUsV0FBVyxJQUFJLEdBQUcsQ0FBQzs7d0JBQy9CLFdBQVcsSUFBSSxHQUFHLENBQUM7b0JBQ3hCLFdBQVcsSUFBSSxPQUFPLEVBQUUsU0FBUyxJQUFJLElBQUksQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQztnQkFDdEYsQ0FBQztnQkFDRCxXQUFXLElBQUksR0FBRyxDQUFDO1lBQ3JCLENBQUM7aUJBQU0sQ0FBQztnQkFDTixXQUFXLEdBQUcsUUFBUSxDQUFDLEtBQUssQ0FBQyxTQUFTLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQyxDQUFDO1lBQ3JELENBQUM7WUFFRCxJQUFJLFVBQVU7Z0JBQUUsVUFBVSxJQUFJLEdBQUcsQ0FBQztZQUNsQyxJQUFJLE9BQU8sQ0FBQyxXQUFXO2dCQUFFLFVBQVUsSUFBSSxJQUFJLEdBQUcsSUFBSSxHQUFHLElBQUksQ0FBQztZQUMxRCxVQUFVLElBQUksR0FBRyxHQUFXLFFBQVEsQ0FBQyxJQUFJLEdBQUcsSUFBSSxDQUFDO1lBQ2pELElBQUksT0FBTyxDQUFDLFdBQVc7Z0JBQUUsVUFBVSxJQUFJLEdBQUcsQ0FBQztZQUMzQyxVQUFVLElBQUksV0FBVyxDQUFDO1FBQzVCLENBQUM7UUFFRCxJQUFJLENBQUMsVUFBVTtZQUFFLE9BQU8sRUFBRSxDQUFDO1FBQzNCLElBQUksT0FBTyxDQUFDLFdBQVc7WUFBRSxPQUFPLEdBQUcsR0FBRyxVQUFVLEdBQUcsSUFBSSxHQUFHLElBQUksR0FBRyxHQUFHLENBQUM7UUFDckUsT0FBTyxHQUFHLEdBQUcsVUFBVSxHQUFHLEdBQUcsQ0FBQztJQUNoQyxDQUFDO0lBRUQsV0FBVyxDQUFDLElBQVM7UUFDbkIsSUFBSSxPQUFPLElBQUksSUFBSSxRQUFRLEVBQUUsQ0FBQztZQUM1QixJQUFJLEdBQUcsR0FBRyxDQUFDLENBQUM7WUFDWixJQUFJLEtBQUssR0FBRyxDQUFDLENBQUMsQ0FBQztZQUNmLE9BQU8sR0FBRyxHQUFHLElBQUksQ0FBQyxNQUFNLEVBQUUsQ0FBQztnQkFDekIsSUFDRSxJQUFJLENBQUMsR0FBRyxDQUFDLElBQUksR0FBRztvQkFDaEIsQ0FBQyxHQUFHLEdBQUcsQ0FBQyxJQUFJLElBQUksQ0FBQyxHQUFHLEdBQUcsQ0FBQyxDQUFDLElBQUksSUFBSTt3QkFDL0IsQ0FBQyxDQUFDLElBQUk7d0JBQ04sQ0FBQyxDQUFDLEdBQUcsR0FBRyxDQUFDLElBQUksSUFBSSxDQUFDLEdBQUcsR0FBRyxDQUFDLENBQUMsSUFBSSxJQUFJLElBQUksR0FBRyxHQUFHLENBQUMsSUFBSSxJQUFJLENBQUMsR0FBRyxHQUFHLENBQUMsQ0FBQyxJQUFJLElBQUksQ0FBQyxFQUN6RSxDQUFDO29CQUNELElBQUksS0FBSyxHQUFHLENBQUMsQ0FBQyxFQUFFLENBQUM7d0JBQ2YsSUFBSSxPQUFPLEdBQVcsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLEVBQUUsR0FBRyxDQUFDLENBQUM7d0JBQ2pELElBQUksT0FBTyxHQUFXLE9BQU87NkJBQzFCLE9BQU8sQ0FBQyxRQUFRLEVBQUUsTUFBTSxDQUFDOzZCQUN6QixPQUFPLENBQUMsT0FBTyxFQUFFLE1BQU0sQ0FBQzs2QkFDeEIsT0FBTyxDQUFDLFFBQVEsRUFBRSxNQUFNLENBQUM7NkJBQ3pCLE9BQU8sQ0FBQyxVQUFVLEVBQUUsT0FBTyxDQUFDOzZCQUM1QixPQUFPLENBQUMsT0FBTyxFQUFFLE9BQU8sQ0FBQzs2QkFDekIsT0FBTyxDQUFDLE9BQU8sRUFBRSxPQUFPLENBQUMsQ0FBQzt3QkFDN0IsSUFBSSxPQUFPLENBQUMsTUFBTSxJQUFJLE9BQU8sQ0FBQyxNQUFNLEVBQUUsQ0FBQzs0QkFDckMscUVBQXFFOzRCQUNyRSxJQUFJLEdBQUcsSUFBSSxDQUFDLFNBQVMsQ0FBQyxDQUFDLEVBQUUsS0FBSyxDQUFDLEdBQUcsT0FBTyxHQUFHLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxDQUFDLENBQUM7NEJBQ2hFLEdBQUcsSUFBSSxPQUFPLENBQUMsTUFBTSxHQUFHLE9BQU8sQ0FBQyxNQUFNLENBQUM7d0JBQ3pDLENBQUM7d0JBQ0QsS0FBSyxHQUFHLENBQUMsQ0FBQyxDQUFDO29CQUNiLENBQUM7O3dCQUFNLEtBQUssR0FBRyxHQUFHLEdBQUcsQ0FBQyxDQUFDO2dCQUN6QixDQUFDO2dCQUNELEdBQUcsRUFBRSxDQUFDO1lBQ1IsQ0FBQztRQUNILENBQUM7UUFDRCxJQUFJLFlBQVksR0FBRyxPQUFPLElBQUksSUFBSSxRQUFRLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUNyRSxJQUFJLENBQUMscUJBQXFCLENBQUMsWUFBWSxDQUFDLENBQUM7UUFDekMsSUFBSSxDQUFDLEtBQUssR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQztJQUNoQyxDQUFDO0lBRU8scUJBQXFCLENBQUMsS0FBVTtRQUN0QyxLQUFLLElBQUksWUFBWSxJQUFJLEtBQUssRUFBRSxDQUFDO1lBQy9CLElBQUksT0FBTyxLQUFLLENBQUMsWUFBWSxDQUFDLElBQUksUUFBUSxFQUFFLENBQUM7Z0JBQzNDLElBQUksT0FBTyxHQUFZLElBQUksT0FBTyxDQUFDLFlBQVksQ0FBQyxDQUFDO2dCQUNqRCxJQUFJLElBQUksQ0FBQyxLQUFLLElBQUksSUFBSTtvQkFBRSxJQUFJLENBQUMsS0FBSyxHQUFHLElBQUksT0FBTyxDQUFDLElBQUksRUFBRSxJQUFJLElBQUksRUFBRSxDQUFDLENBQUM7Z0JBQ25FLElBQUksQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsQ0FBQztnQkFDL0IsT0FBTyxDQUFDLHFCQUFxQixDQUFDLEtBQUssQ0FBQyxZQUFZLENBQUMsQ0FBQyxDQUFDO1lBQ3JELENBQUM7aUJBQU0sSUFBSSxPQUFPLEtBQUssQ0FBQyxZQUFZLENBQUMsSUFBSSxVQUFVLEVBQUUsQ0FBQztnQkFDcEQsSUFBSSxPQUFPLEdBQVksSUFBSSxPQUFPLENBQUMsWUFBWSxFQUFFLEtBQUssQ0FBQyxZQUFZLENBQUMsRUFBRSxJQUFJLENBQUMsQ0FBQztnQkFDNUUsSUFBSSxJQUFJLENBQUMsS0FBSyxJQUFJLElBQUk7b0JBQUUsSUFBSSxDQUFDLEtBQUssR0FBRyxJQUFJLE9BQU8sQ0FBQyxJQUFJLEVBQUUsSUFBSSxJQUFJLEVBQUUsQ0FBQyxDQUFDO2dCQUNuRSxJQUFJLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLENBQUM7WUFDakMsQ0FBQztRQUNILENBQUM7UUFDRCxJQUFJLEtBQUssSUFBSSxLQUFLLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxJQUFJLEtBQUssQ0FBQyxNQUFNLElBQUksQ0FBQyxFQUFFLENBQUM7WUFDdkQsSUFBSSxDQUFDLEtBQUssR0FBRyxJQUFJLE9BQU8sQ0FBQyxJQUFJLEVBQUUsSUFBSSxJQUFJLEVBQUUsQ0FBQyxDQUFDO1FBQzdDLENBQUM7SUFDSCxDQUFDO0lBRUQsUUFBUTtRQUNOLE9BQU8sSUFBSSxDQUFDLElBQUksQ0FBQztJQUNuQixDQUFDO0lBRUQsWUFBWSxPQUFlLElBQUksRUFBRSxRQUFhLElBQUksRUFBRSxVQUFVLEdBQUcsS0FBSyxFQUFFLFVBQWUsRUFBRTtRQXRRakYsZUFBVSxHQUFHLEtBQUssQ0FBQztRQUNuQixZQUFPLEdBQUcsS0FBSyxDQUFDO1FBc1F0QixJQUFJLENBQUMsSUFBSSxHQUFHLElBQUksQ0FBQztRQUNqQixJQUFJLEtBQUssSUFBSSxJQUFJLElBQUksT0FBTyxLQUFLLElBQUksUUFBUTtZQUMzQyxLQUFLLEdBQUcsS0FBSyxDQUFDLE9BQU8sQ0FBQyxTQUFTLEVBQUUsTUFBTSxDQUFDLENBQUMsT0FBTyxDQUFDLFFBQVEsRUFBRSxHQUFHLENBQUMsQ0FBQyxPQUFPLENBQUMsUUFBUSxFQUFFLElBQUksQ0FBQyxDQUFDLE9BQU8sQ0FBQyxRQUFRLEVBQUUsSUFBSSxDQUFDLENBQUM7UUFDbEgsSUFBSSxDQUFDLEtBQUssR0FBRyxLQUFLLENBQUM7UUFDbkIsSUFBSSxDQUFDLFVBQVUsR0FBRyxVQUFVLENBQUM7UUFDN0IsSUFBSSxDQUFDLE9BQU8sR0FBRyxPQUFPLENBQUM7SUFDekIsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCBDb2xvciBmcm9tIFwiLi9Db2xvclwiO1xyXG5pbXBvcnQgeyBEYXRlVGltZSB9IGZyb20gXCIuL0RhdGVUaW1lXCI7XHJcbmltcG9ydCB7IEVudW0gfSBmcm9tIFwiLi9FbnVtXCI7XHJcbmltcG9ydCBMaXN0IGZyb20gXCIuL0xpc3RcIjtcclxuaW1wb3J0IHsgU3RpU3RyaW5nIH0gZnJvbSBcIi4vU3RpT2JqZWN0XCI7XHJcblxyXG5leHBvcnQgZGVmYXVsdCBjbGFzcyBTdGlKc29uIHtcclxuICBzdGF0aWMgcHJldHR5UHJpbnQgPSB0cnVlO1xyXG4gIG9wdGlvbnM6IGFueTtcclxuXHJcbiAgc3RhdGljIGRhdGVUb0pzb25EYXRlKGRhdGU6IERhdGVUaW1lKTogc3RyaW5nIHtcclxuICAgIGxldCBvZmZzZXQ6IG51bWJlciA9IGRhdGUuZGF0ZS5nZXRUaW1lem9uZU9mZnNldCgpICogLTE7XHJcbiAgICBsZXQgaG91cnNPZmZzZXQ6IHN0cmluZyA9IE1hdGguYWJzKE1hdGguZmxvb3Iob2Zmc2V0IC8gNjApKS50b1N0cmluZygpO1xyXG4gICAgbGV0IG1pbnV0ZXNPZmZzZXQ6IHN0cmluZyA9IE1hdGguYWJzKG9mZnNldCAlIDYwKS50b1N0cmluZygpO1xyXG4gICAgaWYgKGhvdXJzT2Zmc2V0Lmxlbmd0aCA9PSAxKSBob3Vyc09mZnNldCA9IFwiMFwiICsgaG91cnNPZmZzZXQ7XHJcbiAgICBpZiAobWludXRlc09mZnNldC5sZW5ndGggPT0gMSkgbWludXRlc09mZnNldCA9IFwiMFwiICsgbWludXRlc09mZnNldDtcclxuXHJcbiAgICByZXR1cm4gKFxyXG4gICAgICAvKiBcIi9EYXRlKFwiICsqLyBEYXRlLnBhcnNlKGRhdGUuZGF0ZS50b1N0cmluZygpKS50b1N0cmluZygpICtcclxuICAgICAgKG9mZnNldCA+PSAwID8gXCIrXCIgOiBcIi1cIikgK1xyXG4gICAgICBob3Vyc09mZnNldCArXHJcbiAgICAgIG1pbnV0ZXNPZmZzZXQgLyorIFwiKS9cIiovXHJcbiAgICApO1xyXG4gIH1cclxuXHJcbiAgc3RhdGljIGpzb25EYXRlRm9ybWF0VG9EYXRlKGpzb25EYXRlOiBzdHJpbmcpOiBEYXRlVGltZSB7XHJcbiAgICBsZXQgZGF0ZVN0ciA9IGpzb25EYXRlLnNwbGl0KFwiL0RhdGUoXCIpLmpvaW4oXCJcIikuc3BsaXQoXCIpL1wiKS5qb2luKFwiXCIpO1xyXG4gICAgbGV0IGRhdGUgPSBuZXcgRGF0ZVRpbWUobmV3IERhdGUocGFyc2VJbnQoZGF0ZVN0cikpKTtcclxuXHJcbiAgICBsZXQgc3RhcnRPZmZzZXRJbmRleDogbnVtYmVyID0gZGF0ZVN0ci5pbmRleE9mKFwiK1wiKTtcclxuICAgIGlmIChzdGFydE9mZnNldEluZGV4ID09IC0xKSBzdGFydE9mZnNldEluZGV4ID0gZGF0ZVN0ci5pbmRleE9mKFwiLVwiKTtcclxuXHJcbiAgICBsZXQgbG9jYWxPZmZzZXQ6IG51bWJlciA9IGRhdGUuZGF0ZS5nZXRUaW1lem9uZU9mZnNldCgpO1xyXG4gICAgaWYgKHN0YXJ0T2Zmc2V0SW5kZXggIT0gLTEpIGxvY2FsT2Zmc2V0ID0gMDtcclxuICAgIGxldCBsb2NhbEhvdXJzT2Zmc2V0OiBudW1iZXIgPSBNYXRoLmZsb29yKGxvY2FsT2Zmc2V0IC8gNjApO1xyXG4gICAgbGV0IGxvY2FsTWludXRlc09mZnNldDogbnVtYmVyID0gbG9jYWxPZmZzZXQgJSA2MDtcclxuICAgIGxldCBsb2NhbEhvdXJzT2Zmc2V0U3RyaW5nOiBzdHJpbmcgPVxyXG4gICAgICAoTWF0aC5hYnMobG9jYWxIb3Vyc09mZnNldCkgPCAxMCA/IFwiMFwiIDogXCJcIikgKyBNYXRoLmFicyhsb2NhbEhvdXJzT2Zmc2V0KS50b1N0cmluZygpO1xyXG4gICAgbGV0IGxvY2FsTWludXRlc09mZnNldFN0cmluZzogc3RyaW5nID1cclxuICAgICAgKE1hdGguYWJzKGxvY2FsTWludXRlc09mZnNldCkgPCAxMCA/IFwiMFwiIDogXCJcIikgKyBNYXRoLmFicyhsb2NhbE1pbnV0ZXNPZmZzZXQpLnRvU3RyaW5nKCk7XHJcblxyXG4gICAgbGV0IG9mZnNldDogc3RyaW5nID0gKGxvY2FsT2Zmc2V0ID49IDAgPyBcIitcIiA6IFwiLVwiKSArIGxvY2FsSG91cnNPZmZzZXRTdHJpbmcgKyBsb2NhbE1pbnV0ZXNPZmZzZXRTdHJpbmc7XHJcbiAgICBsZXQgbXVsdGlwbGUgPSAxO1xyXG5cclxuICAgIGlmIChzdGFydE9mZnNldEluZGV4ICE9IC0xKSBvZmZzZXQgPSBkYXRlU3RyLnN1YnN0cmluZyhzdGFydE9mZnNldEluZGV4KTtcclxuICAgIGlmIChvZmZzZXQuc3Vic3RyKDAsIDEpID09IFwiLVwiKSBtdWx0aXBsZSA9IC0xO1xyXG5cclxuICAgIGRhdGUuYWRkSG91cnMobG9jYWxIb3Vyc09mZnNldCArIHBhcnNlSW50KG9mZnNldC5zdWJzdHIoMSwgMikpICogbXVsdGlwbGUpO1xyXG4gICAgZGF0ZS5hZGRNaW51dGVzKGxvY2FsTWludXRlc09mZnNldCArIHBhcnNlSW50KG9mZnNldC5zdWJzdHIoMywgMikpICogbXVsdGlwbGUpO1xyXG5cclxuICAgIHJldHVybiBkYXRlO1xyXG4gIH1cclxuXHJcbiAgcHVibGljIG5hbWU6IHN0cmluZztcclxuICB2YWx1ZTogYW55O1xyXG4gIHByaXZhdGUgaXNQcm9wZXJ0eSA9IGZhbHNlO1xyXG4gIHByaXZhdGUgaXNBcnJheSA9IGZhbHNlO1xyXG5cclxuICBwcm9wZXJ0aWVzKCk6IExpc3Q8U3RpSnNvbj4ge1xyXG4gICAgaWYgKCF0aGlzLmlzUHJvcGVydHkpIHtcclxuICAgICAgaWYgKHRoaXMudmFsdWUgaW5zdGFuY2VvZiBMaXN0KSByZXR1cm4gdGhpcy52YWx1ZTtcclxuICAgICAgZWxzZSByZXR1cm4gdGhpcy52YWx1ZS52YWx1ZTtcclxuICAgIH1cclxuICAgIHJldHVybiBuZXcgTGlzdDxTdGlKc29uPigpO1xyXG4gIH1cclxuXHJcbiAgcmVtb3ZlUHJvcGVydHkocHJvcGVydHlOYW1lOiBzdHJpbmcpIHtcclxuICAgIGlmIChwcm9wZXJ0eU5hbWUgIT0gbnVsbCB8fCBwcm9wZXJ0eU5hbWUgIT0gXCJcIikge1xyXG4gICAgICBsZXQgZWxlbWVudHM6IFN0aUpzb25bXSA9IHRoaXMudmFsdWUudmFsdWUuZmlsdGVyKCh4KSA9PiB4Lm5hbWUgPT0gcHJvcGVydHlOYW1lKTtcclxuICAgICAgaWYgKGVsZW1lbnRzLmxlbmd0aCA+IDApIHtcclxuICAgICAgICB0aGlzLnZhbHVlLnZhbHVlLnJlbW92ZShlbGVtZW50c1swXSk7XHJcbiAgICAgIH1cclxuICAgIH1cclxuICB9XHJcblxyXG4gIGFkZFByb3BlcnR5TnVtYmVyKHByb3BlcnR5TmFtZTogc3RyaW5nLCB2YWx1ZTogbnVtYmVyKSB7XHJcbiAgICBpZiAodGhpcy52YWx1ZSA9PSBudWxsKSB0aGlzLnZhbHVlID0gbmV3IFN0aUpzb24oXCJcIiwgbmV3IExpc3QoKSk7XHJcbiAgICB0aGlzLnJlbW92ZVByb3BlcnR5KHByb3BlcnR5TmFtZSk7XHJcblxyXG4gICAgdGhpcy52YWx1ZS52YWx1ZS5wdXNoKG5ldyBTdGlKc29uKHByb3BlcnR5TmFtZSwgdmFsdWUsIHRydWUpKTtcclxuICB9XHJcblxyXG4gIGFkZFByb3BlcnR5TnVtYmVyTm90TnVsbChwcm9wZXJ0eU5hbWU6IHN0cmluZywgdmFsdWU6IG51bWJlcikge1xyXG4gICAgaWYgKHZhbHVlKSB7XHJcbiAgICAgIHRoaXMuYWRkUHJvcGVydHlOdW1iZXIocHJvcGVydHlOYW1lLCB2YWx1ZSk7XHJcbiAgICB9XHJcbiAgfVxyXG5cclxuICBhZGRQcm9wZXJ0eU51bWJlck5vRGVmYXVsdFZhbHVlKHByb3BlcnR5TmFtZTogc3RyaW5nLCB2YWx1ZTogbnVtYmVyKSB7XHJcbiAgICBpZiAodGhpcy52YWx1ZSA9PSBudWxsKSB0aGlzLnZhbHVlID0gbmV3IFN0aUpzb24oXCJcIiwgbmV3IExpc3QoKSk7XHJcbiAgICB0aGlzLnJlbW92ZVByb3BlcnR5KHByb3BlcnR5TmFtZSk7XHJcblxyXG4gICAgdGhpcy52YWx1ZS52YWx1ZS5wdXNoKG5ldyBTdGlKc29uKHByb3BlcnR5TmFtZSwgdmFsdWUsIHRydWUpKTtcclxuICB9XHJcblxyXG4gIGFkZFByb3BlcnR5Sk9iamVjdChwcm9wZXJ0eU5hbWU6IHN0cmluZywgdmFsdWU6IFN0aUpzb24pIHtcclxuICAgIGlmICh0aGlzLnZhbHVlID09IG51bGwpIHRoaXMudmFsdWUgPSBuZXcgU3RpSnNvbihcIlwiLCBuZXcgTGlzdCgpKTtcclxuICAgIHRoaXMucmVtb3ZlUHJvcGVydHkocHJvcGVydHlOYW1lKTtcclxuXHJcbiAgICBpZiAodmFsdWUgPT0gbnVsbCB8fCB2YWx1ZS5jb3VudCA9PSAwKSByZXR1cm47XHJcblxyXG4gICAgdGhpcy52YWx1ZS52YWx1ZS5wdXNoKG5ldyBTdGlKc29uKHByb3BlcnR5TmFtZSwgdmFsdWUpKTtcclxuICB9XHJcblxyXG4gIGFkZFByb3BlcnR5QXJyYXkocHJvcGVydHlOYW1lOiBzdHJpbmcsIGVsZW1lbnRzOiBhbnlbXSkge1xyXG4gICAgaWYgKHRoaXMudmFsdWUgPT0gbnVsbCkgdGhpcy52YWx1ZSA9IG5ldyBTdGlKc29uKFwiXCIsIG5ldyBMaXN0KCkpO1xyXG4gICAgdGhpcy5yZW1vdmVQcm9wZXJ0eShwcm9wZXJ0eU5hbWUpO1xyXG5cclxuICAgIGlmIChlbGVtZW50cyA9PSBudWxsKSByZXR1cm47XHJcbiAgICBsZXQgdmFsdWVzID0gW107XHJcblxyXG4gICAgZWxlbWVudHMuZm9yRWFjaCgoZSkgPT5cclxuICAgICAgdmFsdWVzLnB1c2goXHJcbiAgICAgICAgZT8uc2F2ZVRvSnNvbk9iamVjdCAhPSBudWxsXHJcbiAgICAgICAgICA/IGUuc2F2ZVRvSnNvbk9iamVjdCh0aGlzLm9wdGlvbnMpXHJcbiAgICAgICAgICA6IHR5cGVvZiBlID09IFwic3RyaW5nXCJcclxuICAgICAgICAgID8gYFwiJHtlXHJcbiAgICAgICAgICAgICAgLnJlcGxhY2UoL1tcXFxcXS9nLCBcIlxcXFxcXFxcXCIpXHJcbiAgICAgICAgICAgICAgLnJlcGxhY2UoL1tcXFwiXS9nLCAnXFxcXFwiJylcclxuICAgICAgICAgICAgICAvLyAucmVwbGFjZSgvW1xcL10vZywgJ1xcXFwvJylcclxuICAgICAgICAgICAgICAucmVwbGFjZSgvW1xcYl0vZywgXCJcXFxcYlwiKVxyXG4gICAgICAgICAgICAgIC5yZXBsYWNlKC9bXFxmXS9nLCBcIlxcXFxmXCIpXHJcbiAgICAgICAgICAgICAgLnJlcGxhY2UoL1xcclxcbi9nLCBcIlxcblwiKVxyXG4gICAgICAgICAgICAgIC5yZXBsYWNlKC9bXFxyXS9nLCBcIlxcblwiKVxyXG4gICAgICAgICAgICAgIC5yZXBsYWNlKC9bXFxyXFxuXS9nLCBcIlxcXFxyXFxcXG5cIilcclxuICAgICAgICAgICAgICAucmVwbGFjZSgvW1xcdF0vZywgXCJcXFxcdFwiKVxyXG4gICAgICAgICAgICAgIC8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBuby1jb250cm9sLXJlZ2V4XHJcbiAgICAgICAgICAgICAgLnJlcGxhY2UoL1tcXHUwMDAxLVxcdTAwMUZdL2csIChjaHIpID0+IFwiXFxcXHVcIiArIChcIjAwMDBcIiArIGNoci5jaGFyQ29kZUF0KDApLnRvU3RyaW5nKDE2KSkuc3Vic3RyKC00KSl9XCJgXHJcbiAgICAgICAgICA6IGVcclxuICAgICAgKVxyXG4gICAgKTtcclxuXHJcbiAgICBsZXQganNvbjogU3RpSnNvbiA9IG5ldyBTdGlKc29uKHByb3BlcnR5TmFtZSwgdmFsdWVzKTtcclxuICAgIGpzb24uaXNBcnJheSA9IHRydWU7XHJcbiAgICB0aGlzLnZhbHVlLnZhbHVlLnB1c2goanNvbik7XHJcbiAgfVxyXG5cclxuICBhZGRQcm9wZXJ0eUlkZW50KHByb3BlcnR5TmFtZTogc3RyaW5nLCB2YWx1ZTogc3RyaW5nKSB7XHJcbiAgICBpZiAodGhpcy52YWx1ZSA9PSBudWxsKSB0aGlzLnZhbHVlID0gbmV3IFN0aUpzb24oXCJcIiwgbmV3IExpc3QoKSk7XHJcbiAgICB0aGlzLnZhbHVlLnZhbHVlLnB1c2gobmV3IFN0aUpzb24ocHJvcGVydHlOYW1lLCB2YWx1ZSwgdHJ1ZSkpO1xyXG4gIH1cclxuXHJcbiAgYWRkUHJvcGVydHlCb29sKHByb3BlcnR5TmFtZTogc3RyaW5nLCB2YWx1ZTogYm9vbGVhbikge1xyXG4gICAgaWYgKHRoaXMudmFsdWUgPT0gbnVsbCkgdGhpcy52YWx1ZSA9IG5ldyBTdGlKc29uKFwiXCIsIG5ldyBMaXN0KCkpO1xyXG4gICAgdGhpcy5yZW1vdmVQcm9wZXJ0eShwcm9wZXJ0eU5hbWUpO1xyXG5cclxuICAgIHRoaXMudmFsdWUudmFsdWUucHVzaChuZXcgU3RpSnNvbihwcm9wZXJ0eU5hbWUsIHZhbHVlLCB0cnVlKSk7XHJcbiAgfVxyXG5cclxuICAvKmFkZFByb3BlcnR5RGF0ZVRpbWUocHJvcGVydHlOYW1lOiBzdHJpbmcsIHZhbHVlOiBEYXRlVGltZSkge1xyXG4gICAgICBpZiAodGhpcy52YWx1ZSA9PSBudWxsKSB0aGlzLnZhbHVlID0gbmV3IFN0aUpzb24oXCJcIiwgbmV3IExpc3QoKSk7XHJcbiAgICAgIHRoaXMucmVtb3ZlUHJvcGVydHkocHJvcGVydHlOYW1lKTtcclxuXHJcbiAgICAgIHRoaXMudmFsdWUudmFsdWUucHVzaChuZXcgU3RpSnNvbihwcm9wZXJ0eU5hbWUsIFN0aUpzb24uZGF0ZVRvSnNvbkRhdGUodmFsdWUpLCB0cnVlKSk7XHJcbiAgfSovXHJcblxyXG4gIGFkZFByb3BlcnR5RW51bShwcm9wZXJ0eU5hbWU6IHN0cmluZywgZW51bVR5cGU6IGFueSwgdmFsdWU6IGFueSwgZGVmYXVsdFZhbHVlOiBhbnkgPSBudWxsKSB7XHJcbiAgICBpZiAodGhpcy52YWx1ZSA9PSBudWxsKSB0aGlzLnZhbHVlID0gbmV3IFN0aUpzb24oXCJcIiwgbmV3IExpc3QoKSk7XHJcbiAgICB0aGlzLnJlbW92ZVByb3BlcnR5KHByb3BlcnR5TmFtZSk7XHJcblxyXG4gICAgaWYgKHR5cGVvZiBkZWZhdWx0VmFsdWUgIT0gXCJ1bmRlZmluZWRcIiAmJiB2YWx1ZSA9PSBkZWZhdWx0VmFsdWUpIHJldHVybjtcclxuXHJcbiAgICB0aGlzLnZhbHVlLnZhbHVlLnB1c2gobmV3IFN0aUpzb24ocHJvcGVydHlOYW1lLCBFbnVtLmdldE5hbWUoZW51bVR5cGUsIHZhbHVlKSwgdHJ1ZSkpO1xyXG4gIH1cclxuXHJcbiAgYWRkUHJvcGVydHlDb2xvcihwcm9wZXJ0eU5hbWU6IHN0cmluZywgY29sb3I6IENvbG9yLCBkZWZDb2xvcjogQ29sb3IgPSBudWxsKSB7XHJcbiAgICBpZiAoY29sb3IgPT0gbnVsbCB8fCBjb2xvci5lcXVhbHMoZGVmQ29sb3IpKSByZXR1cm47XHJcblxyXG4gICAgbGV0IHZhbHVlID0gY29sb3IuaXNOYW1lZENvbG9yID8gY29sb3IubmFtZS5yZXBsYWNlKFwiIFwiLCBcIlwiKSA6IGAke2NvbG9yLmF9LCR7Y29sb3Iucn0sJHtjb2xvci5nfSwke2NvbG9yLmJ9YDtcclxuXHJcbiAgICB0aGlzLmFkZFByb3BlcnR5U3RyaW5nKHByb3BlcnR5TmFtZSwgdmFsdWUpO1xyXG4gIH1cclxuXHJcbiAgYWRkUHJvcGVydHlTdHJpbmcocHJvcGVydHlOYW1lOiBzdHJpbmcsIHZhbHVlOiBzdHJpbmcsIGRlZmF1bHRWYWx1ZT86IHN0cmluZykge1xyXG4gICAgaWYgKHRoaXMudmFsdWUgPT0gbnVsbCkgdGhpcy52YWx1ZSA9IG5ldyBTdGlKc29uKFwiXCIsIG5ldyBMaXN0KCkpO1xyXG4gICAgdGhpcy5yZW1vdmVQcm9wZXJ0eShwcm9wZXJ0eU5hbWUpO1xyXG5cclxuICAgIGlmICh2YWx1ZSA9PSBkZWZhdWx0VmFsdWUpIHJldHVybjtcclxuXHJcbiAgICB0aGlzLnZhbHVlLnZhbHVlLnB1c2gobmV3IFN0aUpzb24ocHJvcGVydHlOYW1lLCB2YWx1ZSwgdHJ1ZSkpO1xyXG4gIH1cclxuXHJcbiAgYWRkUHJvcGVydHlTdHJpbmdOdWxsT3JFbXB0eShwcm9wZXJ0eU5hbWU6IHN0cmluZywgdmFsdWU6IHN0cmluZykge1xyXG4gICAgaWYgKHRoaXMudmFsdWUgPT0gbnVsbCkgdGhpcy52YWx1ZSA9IG5ldyBTdGlKc29uKFwiXCIsIG5ldyBMaXN0KCkpO1xyXG4gICAgdGhpcy5yZW1vdmVQcm9wZXJ0eShwcm9wZXJ0eU5hbWUpO1xyXG5cclxuICAgIGlmIChTdGlTdHJpbmcuaXNOdWxsT3JFbXB0eSh2YWx1ZSkpIHJldHVybjtcclxuXHJcbiAgICB0aGlzLnZhbHVlLnZhbHVlLnB1c2gobmV3IFN0aUpzb24ocHJvcGVydHlOYW1lLCB2YWx1ZSwgdHJ1ZSkpO1xyXG4gIH1cclxuXHJcbiAgZ2V0IGNvdW50KCk6IG51bWJlciB7XHJcbiAgICBsZXQgY291bnQgPSAwO1xyXG4gICAgaWYgKCF0aGlzLmlzUHJvcGVydHkgJiYgdGhpcy52YWx1ZSAhPSBudWxsKSB7XHJcbiAgICAgIC8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBAdHlwZXNjcmlwdC1lc2xpbnQvbm8tdW51c2VkLXZhcnNcclxuICAgICAgZm9yIChsZXQgcHJvcGVydHlOYW1lIG9mIHRoaXMudmFsdWUudmFsdWUpIHtcclxuICAgICAgICBjb3VudCsrO1xyXG4gICAgICB9XHJcbiAgICB9XHJcblxyXG4gICAgcmV0dXJuIGNvdW50O1xyXG4gIH1cclxuXHJcbiAgZ2V0QXJyYXkoKTogYW55W10ge1xyXG4gICAgbGV0IHJlc3VsdCA9IFtdO1xyXG4gICAgdGhpcy52YWx1ZS5mb3JFYWNoKChlKSA9PiByZXN1bHQucHVzaChlLnZhbHVlKSk7XHJcbiAgICByZXR1cm4gcmVzdWx0O1xyXG4gIH1cclxuXHJcbiAgc2VyaWFsaXplKGluZGVudCA9IDApOiBzdHJpbmcge1xyXG4gICAgbGV0IGpzb25TdHJpbmcgPSBcIlwiO1xyXG4gICAgbGV0IHRhYnMgPSBTdGlKc29uLnByZXR0eVByaW50ID8gQXJyYXkoaW5kZW50ICsgMSkuam9pbihcIiBcIikgOiBcIlwiO1xyXG4gICAgZm9yIChsZXQgcHJvcGVydHkgb2YgdGhpcy52YWx1ZS5wcm9wZXJ0aWVzKCkpIHtcclxuICAgICAgbGV0IHZhbHVlU3RyaW5nID0gXCJcIjtcclxuICAgICAgaWYgKHByb3BlcnR5LmlzUHJvcGVydHkpIHtcclxuICAgICAgICBpZiAocHJvcGVydHkudmFsdWUgIT0gbnVsbCAmJiB0eXBlb2YgcHJvcGVydHkudmFsdWUgPT0gXCJzdHJpbmdcIilcclxuICAgICAgICAgIHZhbHVlU3RyaW5nID1cclxuICAgICAgICAgICAgJ1wiJyArXHJcbiAgICAgICAgICAgICg8c3RyaW5nPnByb3BlcnR5LnZhbHVlKVxyXG4gICAgICAgICAgICAgIC5yZXBsYWNlKC9bXFxcXF0vZywgXCJcXFxcXFxcXFwiKVxyXG4gICAgICAgICAgICAgIC5yZXBsYWNlKC9bXFxcIl0vZywgJ1xcXFxcIicpXHJcbiAgICAgICAgICAgICAgLy8gLnJlcGxhY2UoL1tcXC9dL2csICdcXFxcLycpXHJcbiAgICAgICAgICAgICAgLnJlcGxhY2UoL1tcXGJdL2csIFwiXFxcXGJcIilcclxuICAgICAgICAgICAgICAucmVwbGFjZSgvW1xcZl0vZywgXCJcXFxcZlwiKVxyXG4gICAgICAgICAgICAgIC5yZXBsYWNlKC9cXHJcXG4vZywgXCJcXG5cIilcclxuICAgICAgICAgICAgICAucmVwbGFjZSgvW1xccl0vZywgXCJcXG5cIilcclxuICAgICAgICAgICAgICAucmVwbGFjZSgvW1xcclxcbl0vZywgXCJcXFxcclxcXFxuXCIpXHJcbiAgICAgICAgICAgICAgLnJlcGxhY2UoL1tcXHRdL2csIFwiXFxcXHRcIilcclxuICAgICAgICAgICAgICAvLyBlc2xpbnQtZGlzYWJsZS1uZXh0LWxpbmUgbm8tY29udHJvbC1yZWdleFxyXG4gICAgICAgICAgICAgIC5yZXBsYWNlKC9bXFx1MDAwMS1cXHUwMDFGXS9nLCAoY2hyKSA9PiBcIlxcXFx1XCIgKyAoXCIwMDAwXCIgKyBjaHIuY2hhckNvZGVBdCgwKS50b1N0cmluZygxNikpLnN1YnN0cigtNCkpICtcclxuICAgICAgICAgICAgJ1wiJztcclxuICAgICAgICAvLyBcIlxcXCJcIiArIHByb3BlcnR5LnZhbHVlLnJlcGxhY2UoL1xcXCIvZ2ksIFwiXFxcXFxcXCJcIikucmVwbGFjZSgvXFxyXFxuL2dpLCBcIlxcblwiKS5yZXBsYWNlKC9cXHIvZ2ksIFwiXFxuXCIpLnJlcGxhY2UoL1xcbi9naSwgXCJcXFxcclxcXFxuXCIpLnJlcGxhY2UoL1xcXFwvZ2ksIFwiXFxcXFxcXFxcIikgKyBcIlxcXCJcIjtcclxuICAgICAgICBlbHNlIHZhbHVlU3RyaW5nID0gcHJvcGVydHkudmFsdWU7XHJcbiAgICAgIH0gZWxzZSBpZiAocHJvcGVydHkuaXNBcnJheSkge1xyXG4gICAgICAgIGlmIChwcm9wZXJ0eS52YWx1ZS5sZW5ndGggPT0gMCkge1xyXG4gICAgICAgICAgdmFsdWVTdHJpbmcgKz0gXCJbXCI7XHJcbiAgICAgICAgfVxyXG4gICAgICAgIGZvciAobGV0IGVsZW1lbnQgb2YgcHJvcGVydHkudmFsdWUpIHtcclxuICAgICAgICAgIGlmICh2YWx1ZVN0cmluZykgdmFsdWVTdHJpbmcgKz0gXCIsXCI7XHJcbiAgICAgICAgICBlbHNlIHZhbHVlU3RyaW5nICs9IFwiW1wiO1xyXG4gICAgICAgICAgdmFsdWVTdHJpbmcgKz0gZWxlbWVudD8uc2VyaWFsaXplICE9IG51bGwgPyBlbGVtZW50LnNlcmlhbGl6ZShpbmRlbnQgKyA0KSA6IGVsZW1lbnQ7XHJcbiAgICAgICAgfVxyXG4gICAgICAgIHZhbHVlU3RyaW5nICs9IFwiXVwiO1xyXG4gICAgICB9IGVsc2Uge1xyXG4gICAgICAgIHZhbHVlU3RyaW5nID0gcHJvcGVydHkudmFsdWUuc2VyaWFsaXplKGluZGVudCArIDIpO1xyXG4gICAgICB9XHJcblxyXG4gICAgICBpZiAoanNvblN0cmluZykganNvblN0cmluZyArPSBcIixcIjtcclxuICAgICAgaWYgKFN0aUpzb24ucHJldHR5UHJpbnQpIGpzb25TdHJpbmcgKz0gXCJcXG5cIiArIHRhYnMgKyBcIiAgXCI7XHJcbiAgICAgIGpzb25TdHJpbmcgKz0gJ1wiJyArIDxzdHJpbmc+cHJvcGVydHkubmFtZSArICdcIjonO1xyXG4gICAgICBpZiAoU3RpSnNvbi5wcmV0dHlQcmludCkganNvblN0cmluZyArPSBcIiBcIjtcclxuICAgICAganNvblN0cmluZyArPSB2YWx1ZVN0cmluZztcclxuICAgIH1cclxuXHJcbiAgICBpZiAoIWpzb25TdHJpbmcpIHJldHVybiBcIlwiO1xyXG4gICAgaWYgKFN0aUpzb24ucHJldHR5UHJpbnQpIHJldHVybiBcIntcIiArIGpzb25TdHJpbmcgKyBcIlxcblwiICsgdGFicyArIFwifVwiO1xyXG4gICAgcmV0dXJuIFwie1wiICsganNvblN0cmluZyArIFwifVwiO1xyXG4gIH1cclxuXHJcbiAgZGVzZXJpYWxpemUodGV4dDogYW55KSB7XHJcbiAgICBpZiAodHlwZW9mIHRleHQgPT0gXCJzdHJpbmdcIikge1xyXG4gICAgICBsZXQgcG9zID0gMDtcclxuICAgICAgbGV0IHN0YXJ0ID0gLTE7XHJcbiAgICAgIHdoaWxlIChwb3MgPCB0ZXh0Lmxlbmd0aCkge1xyXG4gICAgICAgIGlmIChcclxuICAgICAgICAgIHRleHRbcG9zXSA9PSAnXCInICYmXHJcbiAgICAgICAgICAocG9zID4gMCAmJiB0ZXh0W3BvcyAtIDFdICE9IFwiXFxcXFwiXHJcbiAgICAgICAgICAgID8gdHJ1ZVxyXG4gICAgICAgICAgICA6IHBvcyA+IDEgJiYgdGV4dFtwb3MgLSAyXSA9PSBcIlxcXFxcIiAmJiBwb3MgPiAyICYmIHRleHRbcG9zIC0gM10gIT0gXCJcXFxcXCIpXHJcbiAgICAgICAgKSB7XHJcbiAgICAgICAgICBpZiAoc3RhcnQgPiAtMSkge1xyXG4gICAgICAgICAgICBsZXQgc3ViVGV4dDogc3RyaW5nID0gdGV4dC5zdWJzdHJpbmcoc3RhcnQsIHBvcyk7XHJcbiAgICAgICAgICAgIGxldCBtb2RUZXh0OiBzdHJpbmcgPSBzdWJUZXh0XHJcbiAgICAgICAgICAgICAgLnJlcGxhY2UoL1xcXFxcXFxcL2dpLCBcIiMhISNcIilcclxuICAgICAgICAgICAgICAucmVwbGFjZSgvXFxcXHQvZ2ksIFwiIyF0I1wiKVxyXG4gICAgICAgICAgICAgIC5yZXBsYWNlKC9cXFxcXFxcIi9naSwgXCIjIXEjXCIpXHJcbiAgICAgICAgICAgICAgLnJlcGxhY2UoL1xcXFxyXFxcXG4vZ2ksIFwiIyFybiNcIilcclxuICAgICAgICAgICAgICAucmVwbGFjZSgvXFxcXHIvZ2ksIFwiIyFybiNcIilcclxuICAgICAgICAgICAgICAucmVwbGFjZSgvXFxcXG4vZ2ksIFwiIyFybiNcIik7XHJcbiAgICAgICAgICAgIGlmIChtb2RUZXh0Lmxlbmd0aCAhPSBzdWJUZXh0Lmxlbmd0aCkge1xyXG4gICAgICAgICAgICAgIC8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBAdHlwZXNjcmlwdC1lc2xpbnQvcmVzdHJpY3QtcGx1cy1vcGVyYW5kc1xyXG4gICAgICAgICAgICAgIHRleHQgPSB0ZXh0LnN1YnN0cmluZygwLCBzdGFydCkgKyBtb2RUZXh0ICsgdGV4dC5zdWJzdHJpbmcocG9zKTtcclxuICAgICAgICAgICAgICBwb3MgKz0gbW9kVGV4dC5sZW5ndGggLSBzdWJUZXh0Lmxlbmd0aDtcclxuICAgICAgICAgICAgfVxyXG4gICAgICAgICAgICBzdGFydCA9IC0xO1xyXG4gICAgICAgICAgfSBlbHNlIHN0YXJ0ID0gcG9zICsgMTtcclxuICAgICAgICB9XHJcbiAgICAgICAgcG9zKys7XHJcbiAgICAgIH1cclxuICAgIH1cclxuICAgIGxldCByZXBvcnRPYmplY3QgPSB0eXBlb2YgdGV4dCA9PSBcIm9iamVjdFwiID8gdGV4dCA6IEpTT04ucGFyc2UodGV4dCk7XHJcbiAgICB0aGlzLmRlc2VyaWFsaXplRnJvbU9iamVjdChyZXBvcnRPYmplY3QpO1xyXG4gICAgdGhpcy52YWx1ZSA9IHRoaXMudmFsdWUudmFsdWU7XHJcbiAgfVxyXG5cclxuICBwcml2YXRlIGRlc2VyaWFsaXplRnJvbU9iamVjdChjaGlsZDogYW55KSB7XHJcbiAgICBmb3IgKGxldCBwcm9wZXJ0eU5hbWUgaW4gY2hpbGQpIHtcclxuICAgICAgaWYgKHR5cGVvZiBjaGlsZFtwcm9wZXJ0eU5hbWVdID09IFwib2JqZWN0XCIpIHtcclxuICAgICAgICBsZXQgak9iamVjdDogU3RpSnNvbiA9IG5ldyBTdGlKc29uKHByb3BlcnR5TmFtZSk7XHJcbiAgICAgICAgaWYgKHRoaXMudmFsdWUgPT0gbnVsbCkgdGhpcy52YWx1ZSA9IG5ldyBTdGlKc29uKG51bGwsIG5ldyBMaXN0KCkpO1xyXG4gICAgICAgIHRoaXMudmFsdWUudmFsdWUucHVzaChqT2JqZWN0KTtcclxuICAgICAgICBqT2JqZWN0LmRlc2VyaWFsaXplRnJvbU9iamVjdChjaGlsZFtwcm9wZXJ0eU5hbWVdKTtcclxuICAgICAgfSBlbHNlIGlmICh0eXBlb2YgY2hpbGRbcHJvcGVydHlOYW1lXSAhPSBcImZ1bmN0aW9uXCIpIHtcclxuICAgICAgICBsZXQgak9iamVjdDogU3RpSnNvbiA9IG5ldyBTdGlKc29uKHByb3BlcnR5TmFtZSwgY2hpbGRbcHJvcGVydHlOYW1lXSwgdHJ1ZSk7XHJcbiAgICAgICAgaWYgKHRoaXMudmFsdWUgPT0gbnVsbCkgdGhpcy52YWx1ZSA9IG5ldyBTdGlKc29uKG51bGwsIG5ldyBMaXN0KCkpO1xyXG4gICAgICAgIHRoaXMudmFsdWUudmFsdWUucHVzaChqT2JqZWN0KTtcclxuICAgICAgfVxyXG4gICAgfVxyXG4gICAgaWYgKGNoaWxkICYmIEFycmF5LmlzQXJyYXkoY2hpbGQpICYmIGNoaWxkLmxlbmd0aCA9PSAwKSB7XHJcbiAgICAgIHRoaXMudmFsdWUgPSBuZXcgU3RpSnNvbihudWxsLCBuZXcgTGlzdCgpKTtcclxuICAgIH1cclxuICB9XHJcblxyXG4gIHRvU3RyaW5nKCk6IHN0cmluZyB7XHJcbiAgICByZXR1cm4gdGhpcy5uYW1lO1xyXG4gIH1cclxuXHJcbiAgY29uc3RydWN0b3IobmFtZTogc3RyaW5nID0gbnVsbCwgdmFsdWU6IGFueSA9IG51bGwsIGlzUHJvcGVydHkgPSBmYWxzZSwgb3B0aW9uczogYW55ID0ge30pIHtcclxuICAgIHRoaXMubmFtZSA9IG5hbWU7XHJcbiAgICBpZiAodmFsdWUgIT0gbnVsbCAmJiB0eXBlb2YgdmFsdWUgPT0gXCJzdHJpbmdcIilcclxuICAgICAgdmFsdWUgPSB2YWx1ZS5yZXBsYWNlKC8jIXJuIy9naSwgXCJcXHJcXG5cIikucmVwbGFjZSgvIyFxIy9naSwgJ1wiJykucmVwbGFjZSgvIyF0Iy9naSwgXCJcXHRcIikucmVwbGFjZSgvIyEhIy9naSwgXCJcXFxcXCIpO1xyXG4gICAgdGhpcy52YWx1ZSA9IHZhbHVlO1xyXG4gICAgdGhpcy5pc1Byb3BlcnR5ID0gaXNQcm9wZXJ0eTtcclxuICAgIHRoaXMub3B0aW9ucyA9IG9wdGlvbnM7XHJcbiAgfVxyXG59XHJcbiJdfQ==