import StiJson from "./StiJson";
export class Font {
    constructor(family, size, style = [], changesInterceptor) {
        this.changesInterceptor = changesInterceptor;
        this.fontSizeScale = 0.75;
        this.isChanged = true;
        this._family = family;
        this._size = size;
        this._style = style;
    }
    setProperty(propName, value) {
        switch (propName) {
            case "family":
                this.family = value;
                break;
            case "style":
                this.style = value;
                break;
            case "size":
                this.size = value;
                break;
        }
    }
    set style(value) {
        let this_ = this;
        let font = this.changesInterceptor ? this_.clone() : null;
        this._style = new Proxy(value, {
            deleteProperty: function (target, property) {
                let font = this.changesInterceptor ? this_.clone() : null;
                setTimeout(() => {
                    if (this.changesInterceptor) {
                        this.changesInterceptor.propertyChanged("font.style", this_.cloneStyle, font);
                    }
                    this_.isChanged = true;
                }, 0);
                return true;
            },
            set: function (target, property, value, receiver) {
                let font = this.changesInterceptor ? this_.clone() : null;
                target[property] = value;
                if (this.changesInterceptor) {
                    this.changesInterceptor.propertyChanged("font.style", this_.cloneStyle, font);
                }
                this_.isChanged = true;
                return true;
            },
        });
        if (this.changesInterceptor) {
            this.changesInterceptor.propertyChanged("font.style", this_.cloneStyle, font);
        }
        this_.isChanged = true;
    }
    get cloneStyle() {
        return this.style.map((s) => s);
    }
    get style() {
        return this._style;
    }
    resetChanged() {
        this.isChanged = false;
    }
    get family() {
        return this._family;
    }
    set family(value) {
        let font = this.changesInterceptor ? this.clone() : null;
        let oldValue = this._family;
        this._family = value;
        this.isChanged = true;
        if (this.changesInterceptor && oldValue != this._family) {
            this.changesInterceptor.propertyChanged("font.family", value, font);
        }
    }
    get size() {
        return this._size;
    }
    set size(value) {
        let font = this.changesInterceptor ? this.clone() : null;
        let oldValue = this._size;
        this._size = value;
        this.isChanged = true;
        if (this.changesInterceptor && oldValue != this._size) {
            this.changesInterceptor.propertyChanged("font.size", value, font);
        }
    }
    saveToJsonObject(options) {
        let json = new StiJson();
        json.addPropertyStringNullOrEmpty("Family", this.family);
        json.addPropertyNumber("Size", this.size);
        let style = this.style.filter((s) => s && s != "").join(",");
        json.addPropertyString("Style", style.length != 0 ? style : "Regular");
        return json;
    }
    loadFromJsonObject(json) {
        for (let property of json.properties()) {
            switch (property.name) {
                case "Family":
                    this._family = property.value;
                    break;
                case "Size":
                    this._size = property.value;
                    break;
                case "Style":
                    this._style = property.value.replace("Regular", "").split(",");
                    break;
            }
        }
    }
    isBold() {
        return this.style.indexOf("Bold") >= 0;
    }
    isItalic() {
        return this.style.indexOf("Italic") >= 0;
    }
    isUnderline() {
        return this.style.indexOf("Underline") >= 0;
    }
    htmlStyle() {
        return `font-weight:${this.isBold() ? "bold" : "normal"};font-style:${this.isItalic() ? "italic" : "normal"}; text-decoration:${this.isUnderline() ? "underline" : ""};font-family:"${this.family}";font-size:${this.size}pt`;
    }
    equals(font) {
        return this.htmlStyle() == font?.htmlStyle();
    }
    clone() {
        return new Font(this._family, this.size, this.cloneStyle);
    }
}
//# sourceMappingURL=data:application/json;base64,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