import { Injectable } from "@angular/core";
import { camelCase, indexOf } from "lodash";
import { Subject } from "rxjs";
import { StiForm } from "../elements/StiForm";
import { StiFormPageElement } from "../elements/StiFormPageElement";
import { StiPanelElement } from "../elements/StiPanelElement";
import { StiTableElement } from "../elements/StiTableElement";
import { StiNumberRangeFormat, StiThousandSeparator, StiUnitAlignment } from "../system/enums";
import { StiStaticHelperService } from "./sti-static-helper.service";
import * as i0 from "@angular/core";
export class StiModelService {
    static { this.PAGE_CLASS = "sti-page"; }
    static { this.PANEL_CLASS = "sti-panel"; }
    static { this.THEMES = ["Light", "Dark"]; }
    toggleTieredMenu(element) {
        let elementRect = this.getPositionRelativeToForm(element);
        let menuWidth = 104;
        let menuHeight = 165;
        this.tieredMenuY =
            window.innerHeight - elementRect.bottom >= menuHeight ? elementRect.bottom : elementRect.top - menuHeight;
        this.tieredMenuX =
            window.innerWidth - elementRect.left > menuWidth ? elementRect.left : elementRect.right - menuWidth;
        this.showTieredMenu = true;
    }
    get showTrialBar() {
        return this._showTrialBar;
    }
    set showTrialBar(value) {
        let lastState = this._showTrialBar;
        if (value != lastState) {
            this._showTrialBar = value;
        }
    }
    get showTrialModalWindow() {
        return this._showTrialModalWindow;
    }
    set showTrialModalWindow(value) {
        let lastState = this._showTrialModalWindow;
        if (value != lastState) {
            this._showTrialModalWindow = value;
        }
    }
    get showTrialTitle() {
        return this._showTrialTitle;
    }
    set showTrialTitle(value) {
        let lastState = this._showTrialTitle;
        if (value != lastState) {
            this._showTrialTitle = value;
        }
    }
    get canCloseTrialWindow() {
        return this._canCloseTrialWindow;
    }
    set canCloseTrialWindow(value) {
        let lastState = this._canCloseTrialWindow;
        if (value != lastState) {
            this._canCloseTrialWindow = value;
        }
    }
    set ribbon(value) {
        if ((value == "Insert" && !this.showInsertTab) || (this.viewerMode && value != "Preview"))
            return;
        if (value != this._ribbon) {
            let oldRibbon = this._ribbon;
            this._ribbon = value;
            this.interfaceEvent.next({ name: "setRibbon", data: { ribbon: value, oldRibbon: oldRibbon } });
        }
    }
    get ribbon() {
        return this._ribbon;
    }
    get previewMode() {
        return this._ribbon == "Preview";
    }
    get pdfPreviewMode() {
        return this._ribbon == "PDFPreview";
    }
    get fileMenuMode() {
        return this._ribbon == "File";
    }
    canDelete() {
        return (window.getSelection().toString().length == 0 &&
            document.activeElement.tagName.toLocaleLowerCase() == "body" &&
            !this.draggingElement &&
            !this.showSignatureEditor);
    }
    get showSaveToPdfButton() {
        return this.settings?.showSaveToPdfButton ?? true;
    }
    get showEditButton() {
        return this.settings?.showEditButton ?? false;
    }
    set embeddedMode(value) {
        this._embeddedMode = value;
    }
    get embeddedMode() {
        return this._embeddedMode || this.properties?.embeddedMode === true;
    }
    get cursor() {
        return this._cursor;
    }
    get zoom() {
        return this._zoom;
    }
    set requestUrl(value) {
        this._requestUrl = value;
        this.isWebView = value == "WebView";
    }
    get requestUrl() {
        return this._requestUrl;
    }
    set zoom(value) {
        if (value <= 0.1) {
            this._zoom = 0.1;
        }
        else if (value >= 4) {
            this._zoom = 4;
        }
        else {
            this._zoom = value;
        }
    }
    set cursor(value) {
        if (value == "default" && this.selectedComponent?.hover) {
            this._cursor = "move";
        }
        else {
            this._cursor = value;
        }
    }
    formattingDate(dateFormat) {
        let result = dateFormat
            .replace(/M/g, "k")
            .replace(/m/g, "r")
            .replace(/tt/g, "a")
            .replace(/k/g, "m")
            .replace(/r/g, "M");
        return result;
    }
    setZoomFromString(zoomString) {
        if (typeof zoomString != "string") {
            return;
        }
        if (zoomString.includes("%")) {
            let zoom = Number(zoomString.slice(0, zoomString.indexOf("%"))) / 100;
            if (zoom <= 0.1) {
                this.zoom = 0.1;
            }
            else if (zoom >= 4) {
                this.zoom = 4;
            }
            else {
                this.zoom = zoom;
            }
        }
        else {
            switch (zoomString.toLowerCase()) {
                case "pagewidth":
                    this.formEvents.next({ name: "SetZoomByPageWidth" });
                    break;
                case "onepage":
                    this.formEvents.next({ name: "SetZoomByOnePage" });
                    break;
                default:
                    this.formEvents.next({ name: "SetDefaultZoom" });
                    break;
            }
        }
    }
    getPositionRelativeToForm(element) {
        let elementRect = element?.getBoundingClientRect();
        let formRect = this.rootFormElement?.getBoundingClientRect();
        if (formRect) {
            return new DOMRect(elementRect?.x - formRect?.x, elementRect?.y - formRect?.y, elementRect?.width, elementRect?.height);
        }
        else {
            return elementRect;
        }
    }
    getBooleanSetting(settingName) {
        return this.settings && this.isNotEmpty(this.settings[settingName]) && this.isBoolean(this.settings[settingName])
            ? this.settings[settingName]
            : true;
    }
    isBoolean(value) {
        return typeof value === "boolean";
    }
    isNotEmpty(value) {
        return value?.toString()?.length > 0;
    }
    saveLastFormState() {
        this.form.reportVersion = this.buildVersion;
        this.form.reportChanged = StiStaticHelperService.dateToJsonDate(new Date());
        this.lastSavedFormState = this.form.saveToReportJsonObject().serialize();
    }
    toCamelCase(obj) {
        if (Array.isArray(obj)) {
            return obj.map((v) => this.toCamelCase(v));
        }
        else if (obj != null && obj.constructor === Object) {
            return Object.keys(obj).reduce((result, key) => ({
                ...result,
                [camelCase(key)]: this.toCamelCase(obj[key]),
            }), {});
        }
        return obj;
    }
    showColumnPreview(index, type) {
        if (!type &&
            this.selectedComponent?.element instanceof StiTableElement &&
            !this.selectedComponent?.element.showColumnsEditor) {
            this.selectedComponent.element.showColumnsEditor = true;
            this.needToRefresh.next(true);
        }
        else if (type == "totals" &&
            this.selectedComponent?.element instanceof StiTableElement &&
            !this.selectedComponent?.element.showTotalsEditor) {
            this.selectedComponent.element.showTotalsEditor = true;
            this.needToRefresh.next(true);
        }
        setTimeout(() => {
            if (this.currentContentEditableItemIndex != index || this.previousSelectedComponent != this.selectedComponent)
                this.currentContentEditableItem.next(index);
            this.currentContentEditableItemIndex = index;
        });
    }
    showError(message, sendedFromButton = false, buttonElement) {
        if (message instanceof ArrayBuffer) {
            var enc = new TextDecoder("utf-8");
            message = enc.decode(message);
        }
        else if (message instanceof Event) {
            message = message.type;
        }
        else if (message?.text) {
            message = message.text;
        }
        if (buttonElement) {
            this.currentUsedButtonElement = buttonElement;
        }
        this.sendedFromButton = sendedFromButton;
        this.errorWindowMessage = message;
        if (this.errorWindowMessage != "abort") {
            this.showErrorWindow = true;
        }
    }
    getImage(fileName) {
        return this.imagesArray && this.imagesArray[fileName] ? this.imagesArray[fileName] : "";
    }
    removeUnusedCells(element) {
        element.columns.forEach((column) => {
            if (column["cells"].length != element.rowLabels.length) {
                column["cells"].splice(element.rowLabels.length, column["cells"].length - 1);
            }
        });
    }
    startPdfExport(form) {
        if (form) {
            this.setFormFromString(JSON.stringify(form));
        }
        this.startPdfExportEvent.next(true);
    }
    setFormFromString(str, loadSavedSetings = false) {
        let form = new StiForm();
        form.loadFormJsonString(str);
        if (loadSavedSetings) {
            this.loadSavedSettings();
        }
        this.form = form;
    }
    get form() {
        return this._form;
    }
    set form(form) {
        this.updateForm(form);
        this.checkOnUniqueNames();
        form.formEvents.next({ name: "SetForm" });
        this.firstFormState = this.form.saveToReportJsonObject().serialize();
        if (this.viewerMode && this.isMobile()) {
            setTimeout(() => {
                this.isMobileViewer.next(true);
            });
        }
    }
    arrayOfStylesToString(array) {
        if (!array)
            return;
        let styles = [];
        if (array.length > 0) {
            array.forEach((item) => {
                if (item != "Underline") {
                    styles.push(item);
                }
            });
        }
        return styles.length > 0 ? styles.join(" ").toLocaleLowerCase() : "";
    }
    checkOnUniqueNames() {
        this.form.allElements.forEach((element) => {
            let currentName = element.name;
            let uniqueElements = this.form.allElements.filter((elem) => elem.name == currentName);
            if (uniqueElements.length > 1) {
                this.setUniqueName(element);
            }
        });
    }
    setUniqueName(element) {
        let regex = /\d+$/;
        let currentName = element.name ?? element.type.replace("_", "").replace("Element", "");
        let index = 0;
        let currentPosition = element.name?.match(regex) ? element.name?.match(regex)[0] : null;
        if (currentPosition) {
            currentName = currentName.slice(0, currentName.length - currentPosition.length);
        }
        while (this.form.allElements.find((element) => element.name == `${currentName + (index != 0 ? index + 1 : "")}`)) {
            index++;
        }
        element.name = currentName + (index != 0 ? index + 1 : "");
    }
    checkNumberRange(element) {
        if (element.rangeFormat == StiNumberRangeFormat.Digits) {
            let maximumValueString = "";
            for (let index = element.maximum; index > 0; index--) {
                maximumValueString += "9";
            }
            let maximumValue = Number(maximumValueString);
            let minimumValue = -Number(maximumValueString);
            if (element.value > maximumValue) {
                element.value = maximumValue;
            }
            else if (element.value < minimumValue) {
                element.value = minimumValue;
            }
        }
        else {
            if (element.value > element.maximum) {
                element.value = Number(element.maximum);
            }
            else if (element.value < element.minimum) {
                element.value = Number(element.minimum);
            }
        }
    }
    showTrialLog() {
        if (this.showTrialBar || this.showTrialTitle || this.showTrialModalWindow) {
            console.warn("You are using a trial version of the Stimulsoft PDF Forms!");
        }
    }
    loadSavedSettings() {
        if (localStorage.getItem("StiFormSettings") &&
            (this.isWebView || this.isCloudMode || this.properties?.cloudPlanIdent)) {
            this.form?.settings?.loadFormJsonString(localStorage.getItem("StiFormSettings"));
        }
    }
    saveSettings() {
        localStorage.removeItem("StiFormSettings");
        localStorage.setItem("StiFormSettings", this.form.settings.saveToJsonObject({}).serialize());
    }
    isMobile() {
        const toMatch = [/Android/i, /webOS/i, /iPhone/i, /iPad/i, /iPod/i, /BlackBerry/i, /Windows Phone/i];
        return toMatch.some((toMatchItem) => {
            return navigator.userAgent.match(toMatchItem);
        });
    }
    getBrowserType() {
        let userAgent = navigator.userAgent;
        let browserName;
        if (userAgent.match(/chrome|chromium|crios/i)) {
            browserName = "chrome";
        }
        else if (userAgent.match(/firefox|fxios/i)) {
            browserName = "firefox";
        }
        else if (userAgent.match(/safari/i)) {
            browserName = "safari";
        }
        else if (userAgent.match(/opr\//i)) {
            browserName = "opera";
        }
        else if (userAgent.match(/edg/i)) {
            browserName = "edge";
        }
        else {
            browserName = "No browser detection";
        }
        return browserName;
    }
    getScaleWithTranslate() {
        return `translate(${this.zoom <= 1 ? 0 : ((this.zoom - 1) / 2) * 100}%) scale(${this.zoom})`;
    }
    getScale() {
        return `scale(${this.zoom})`;
    }
    updateForm(form) {
        this._form = form;
        this.selectedComponent = null;
        this.lastUsedPage = null;
        form.formEvents.subscribe((e) => this.formEvents.next(e));
        if (form.pages?.length > 0) {
            this.previousPadding = form.pages[0].padding.clone();
            this.previousContentAreaWidth = form.pages[0].contentAreaWidth;
            setTimeout(() => {
                this.needToRefresh.next(true);
            }, 0);
        }
        form.formEvents.next({ name: "UpdateForm" });
    }
    set elementToUpdate(value) {
        let oldValue = this._elementToUpdate;
        this._elementToUpdate = value;
        if (oldValue != value) {
            this.formEvents.next({ name: "ElementToUpdateChanged", value: value });
        }
    }
    get elementToUpdate() {
        return this._elementToUpdate;
    }
    get selectedComponent() {
        return this._selectedComponent;
    }
    set selectedComponent(value) {
        let oldValue = this._selectedComponent;
        this._selectedPage = null;
        this._selectedComponent = value;
        this.needToRefresh.next(true);
        if (oldValue != value) {
            this.formEvents.next({ name: "SelectedComponentChanged", value: value });
        }
    }
    get selectedPage() {
        return this._selectedPage;
    }
    set selectedPage(value) {
        let oldValue = this._selectedPage;
        this._selectedComponent = null;
        this._selectedPage = value;
        this.needToRefresh.next(true);
        if (oldValue != value) {
            this.formEvents.next({ name: "SelectedPageChanged", value: value });
        }
    }
    get lastUsedPage() {
        return this._lastUsedPage ?? this.selectedPage ?? this.form?.pages[0];
    }
    set lastUsedPage(value) {
        this._lastUsedPage = value;
    }
    getSelectedComponentPage() {
        return this.getElementPage(this._selectedComponent?.element);
    }
    getElementPage(element) {
        return this.form?.pages.find((p) => p.elements.find((e) => e == element) != null);
    }
    getElementPageForScroll(element) {
        return this.form?.pages.find((p) => p.elements.find((e) => e == element) != null
            ? p.elements.find((e) => e == element)
            : this.getElementPanel(element));
    }
    overToolbar(x, y) {
        let elementsDiv = document.elementsFromPoint(x, y).find((elem) => elem?.classList.contains("elements"));
        return elementsDiv ? true : false;
    }
    getElementPanel(element) {
        let result;
        this.form.pages.forEach((page) => {
            page.allElements.forEach((panel) => {
                if (panel instanceof StiPanelElement && panel.elements.indexOf(element) >= 0) {
                    result = panel;
                }
            });
        });
        return result;
    }
    formatNumber(value, format, decimalDigits, unitAlignment, unit) {
        let formatedValue = this.formatNumberWithThousandSeparator(value, format, decimalDigits);
        switch (unitAlignment) {
            case StiUnitAlignment.PrefixInside:
                return `${unit}${formatedValue}`;
            case StiUnitAlignment.PrefixInsideSpace:
                return `${unit} ${formatedValue}`;
            case StiUnitAlignment.SuffixInside:
                return `${formatedValue}${unit}`;
            case StiUnitAlignment.SuffixInsideSpace:
                return `${formatedValue} ${unit}`;
        }
        return formatedValue;
    }
    getFormattedValue(element, inValue) {
        let value = inValue;
        try {
            if (typeof value === "number" && !isNaN(value)) {
                value = this.formatNumberWithThousandSeparator(parseFloat(value.toFixed(element.decimalDigits)), element.thousandSeparator, element.decimalDigits);
            }
        }
        catch {
            value = "";
        }
        return value;
    }
    formatNumberWithThousandSeparator(value, format, decimalDigits) {
        let isNegative = false;
        if (value == undefined) {
            return "";
        }
        let result;
        if (decimalDigits > 0) {
            result = value.toFixed(decimalDigits);
        }
        else {
            result = value.toString();
        }
        if (result.indexOf("-") != -1) {
            isNegative = true;
        }
        if (isNegative) {
            result = result.slice(1, result.length);
        }
        if (format == StiThousandSeparator.None) {
            return (isNegative ? "-" : "") + result;
        }
        else if (format == StiThousandSeparator.NoneComma) {
            return (isNegative ? "-" : "") + result.replace(".", ",");
        }
        else if (format == StiThousandSeparator.Comma) {
            var chars = (decimalDigits > 0 ? result.slice(0, indexOf(result, ".")) : result).split("").reverse();
            var withCommas = [];
            for (var i = 1; i <= chars.length; i++) {
                withCommas.push(chars[i - 1]);
                if (i % 3 == 0 && i != chars.length) {
                    withCommas.push(",");
                }
            }
            result =
                decimalDigits > 0
                    ? withCommas.reverse().join("") + result.slice(indexOf(result, "."), result.length)
                    : withCommas.reverse().join("");
            return (isNegative ? "-" : "") + result;
        }
        else if (format == StiThousandSeparator.Dot) {
            var chars = (decimalDigits > 0 ? result.slice(0, indexOf(result, ".")) : result).split("").reverse();
            var withDotts = [];
            for (var i = 1; i <= chars.length; i++) {
                withDotts.push(chars[i - 1]);
                if (i % 3 == 0 && i != chars.length) {
                    withDotts.push(".");
                }
            }
            result =
                decimalDigits > 0
                    ? withDotts.reverse().join("") + "," + result.slice(indexOf(result, ".") + 1, result.length)
                    : withDotts.reverse().join("");
            return (isNegative ? "-" : "") + result;
        }
        else if (format == StiThousandSeparator.Apostrophe) {
            var chars = (decimalDigits > 0 ? result.slice(0, indexOf(result, ".")) : result).split("").reverse();
            var withSpaces = [];
            for (var i = 1; i <= chars.length; i++) {
                withSpaces.push(chars[i - 1]);
                if (i % 3 == 0 && i != chars.length) {
                    withSpaces.push("`");
                }
            }
            result =
                decimalDigits > 0
                    ? withSpaces.reverse().join("") + "." + result.slice(indexOf(result, ".") + 1, result.length)
                    : withSpaces.reverse().join("");
            return (isNegative ? "-" : "") + result;
        }
    }
    getWindowScaling() {
        return window.devicePixelRatio * 100;
    }
    getScaling() {
        if (this.getWindowScaling() < 125) {
            return "";
        }
        else if (125 <= this.getWindowScaling() && this.getWindowScaling() < 150) {
            return "_x1_25";
        }
        else if (150 <= this.getWindowScaling() && this.getWindowScaling() < 175) {
            return "_x1_5";
        }
        else if (175 <= this.getWindowScaling() && this.getWindowScaling() < 200) {
            return "_x1_75";
        }
        else if (200 <= this.getWindowScaling() && this.getWindowScaling() < 300) {
            return "_x2";
        }
        else if (300 <= this.getWindowScaling() && this.getWindowScaling() < 400) {
            return "_x3";
        }
        else if (this.getWindowScaling() > 400) {
            return "_x4";
        }
    }
    getPageFromPoint(x, y) {
        let pageDiv;
        for (let elem of document.elementsFromPoint(x, y)) {
            if (elem?.classList.contains(StiModelService.PAGE_CLASS)) {
                pageDiv = elem;
                break;
            }
        }
        if (pageDiv) {
            for (let page of this.form.pages) {
                if (page.element?.nativeElement == pageDiv) {
                    pageDiv = page;
                    break;
                }
            }
        }
        // let pageDiv = document.elementsFromPoint(x, y).find((elem) => elem?.classList.contains(StiModelService.PAGE_CLASS));
        return pageDiv ? pageDiv : null;
    }
    getPanelFromPoint(x, y, excludeId) {
        let page = this.getPageFromPoint(x, y);
        let panelDivs = document
            .elementsFromPoint(x, y)
            .filter((elem) => elem?.classList.contains(StiModelService.PANEL_CLASS) && elem.id != excludeId);
        return panelDivs && page
            ? page.allElements.find((e) => panelDivs.find((p) => e.guid == p.id))
            : null;
    }
    getMouseOverPagePosition(event) {
        let page = this.getPageFromPoint(event.clientX, event.clientY);
        if (page) {
            let rect = this.getPositionRelativeToForm(page.element.nativeElement);
            return {
                x: event.clientX - (rect.left + page.padding.left * this.zoom),
                y: event.clientY - (rect.top + page.padding.top * this.zoom),
            };
        }
        return null;
    }
    getMouseOverPanelPosition(event, excludeId) {
        let panel = this.getPanelFromPoint(event.clientX, event.clientY, excludeId);
        if (panel) {
            let rect = this.getPositionRelativeToForm(panel.element.nativeElement);
            return { x: event.clientX - rect.left, y: event.clientY - rect.top };
        }
        return null;
    }
    loadAttachedFile() {
        let input = document.createElement("input");
        input.type = "file";
        let this_ = this;
        input.onchange = (_) => {
            let file = Array.from(input.files)[0];
            var reader = new FileReader();
            reader.readAsDataURL(file);
            reader.onload = function () {
                this_.form.attachedFiles.push(reader.result);
                this_.interfaceEvent.next({ name: "FileAttached", data: reader.result });
            };
        };
        input.click();
    }
    downloadFiles(data, file_name) {
        let file = new Blob([data]);
        let a = document.createElement("a"), url = URL.createObjectURL(file);
        a.href = url;
        a.download = file_name;
        document.body.appendChild(a);
        a.click();
        setTimeout(function () {
            document.body.removeChild(a);
            window.URL.revokeObjectURL(url);
        }, 0);
    }
    constructor() {
        this._showTrialBar = false;
        this._showTrialModalWindow = false;
        this._showTrialTitle = false;
        this._canCloseTrialWindow = true;
        this._zoom = 1;
        this._cursor = "default";
        this._ribbon = "Home";
        this.interfaceEvent = new Subject();
        this.isMobileViewer = new Subject();
        this.needToRefresh = new Subject();
        this.needToUpdateHelper = new Subject();
        this.startPdfExportEvent = new Subject();
        this.needToRefreshInnerProperties = new Subject();
        this.currentContentEditableItem = new Subject();
        this.retryButtonRequest = new Subject();
        this.fontFamilies = [];
        this.assemblyInfo = [];
        this.formTreeInnerItemsStates = [];
        this.userName = "";
        this.currentPropertyGridMode = "Properties";
        this.showFormElementsList = false;
        this.sortFormElementsList = false;
        this.showComponentsTab = true;
        this.showInsertTab = true;
        this.hideActiveDropdown = new Subject();
        this.showSignatureEditor = false;
        this.showDraggableModalWindow = false;
        this.showFormTreeWarningWindow = false;
        this.hideActiveAutocompleteList = new Subject();
        this.formTreeWarningMessage = "";
        this.needToSelectedComponent = new Subject();
        this.showButtonMessageWindow = false;
        this.showTieredMenu = false;
        this.showTrialTimerWindow = false;
        this.isInitialized = false;
        this.showContextMenu = false;
        this.showHelpWindow = false;
        this.showSaveDialog = false;
        this.showErrorWindow = false;
        this.basePlanIsTrial = false;
        this.viewerMode = false;
        this.isCloudMode = false;
        this.disableCloseButton = false;
        this.showCornerRadius = true;
        this.showAboutWindow = false;
        this.showGroupPreview = false;
        this.showToolboxSetup = false;
        this.showModalWindow = false;
        this.showProgressAfterUpdatePage = false;
        this.canDrag = true;
        this.tieredMenuX = 0;
        this.tieredMenuY = 0;
        this.currentExpressionInfo = 0; //0-none,1-label column,2-totals,3-label component.
        this.buildVersion = "";
        this.buildDate = "";
        this.errorWindowMessage = "";
        this.propertyGridWidth = 395;
        this.currentTablePropertyHeader = new Subject();
        this.currentTableHeader = "Main";
        this.formEvents = new Subject();
        this.saveDialogFromNew = false;
        this.saveDialogFromOpen = false;
        this.sendedFromButton = false;
        this.loadingFromCloud = false;
        this.showColorDialog = false;
        this.showMoreColorDialog = false;
        this.userIsAuthorized = false;
        this.userWasChecked = false;
        this.showSaveDialogFromCloud = false;
        this.hasEmptyReuiredFields = false;
        this.showExpressionEditor = false;
        this.showProgress = false;
        this.isWebView = false;
        this.currentSelectedFileOption = "Open";
        this._embeddedMode = false;
        this.embeddedEvent = function (data) { };
        this.fileName = "Form.mrt";
        this.enteredUrls = [];
        let savedData = {};
        this.interfaceEvent.subscribe(e => {
            switch (e.name) {
                case "setRibbon":
                    if (!this.viewerMode) {
                        if (e.data.ribbon == "Preview") {
                            savedData = { form: this.form, selectedGuid: this.selectedComponent?.element.guid ?? this.selectedPage?.guid };
                            let form = new StiForm();
                            form.loadFormJsonString(this.form.saveToJsonObject().serialize());
                            this._form = form;
                            this.formEvents.next({ name: "ValueChanged" });
                        }
                        else if (e.data.oldRibbon == "Preview") {
                            let form = new StiForm();
                            form.loadFormJsonString(savedData["form"].saveToJsonObject().serialize());
                            let selectedElement = form.getElementByGuid(savedData["selectedGuid"]);
                            if (selectedElement) {
                                if (!(selectedElement instanceof StiFormPageElement)) {
                                    selectedElement.needToSelectComponent = true;
                                }
                                else {
                                    this.selectedPage = selectedElement;
                                }
                            }
                            this._form = form;
                            this.formEvents.next({ name: "ValueChanged" });
                        }
                    }
                    break;
            }
        });
    }
    static { this.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "18.2.14", ngImport: i0, type: StiModelService, deps: [], target: i0.ɵɵFactoryTarget.Injectable }); }
    static { this.ɵprov = i0.ɵɵngDeclareInjectable({ minVersion: "12.0.0", version: "18.2.14", ngImport: i0, type: StiModelService }); }
}
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "18.2.14", ngImport: i0, type: StiModelService, decorators: [{
            type: Injectable
        }], ctorParameters: () => [] });
//# sourceMappingURL=data:application/json;base64,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