import { StiInputTextItem } from "../helper/StiInputTextItem";
import StiRectangleGeometry from "../properties/StiRectangleGeometry";
import { StiLocalizationService } from "../services/sti-localization.service";
import Color from "../system/Color";
import { Font } from "../system/Font";
import { StiFormElement } from "./base/StiFormElement";
import { StiLabelItem } from "./text/StiLabelItem";
import { StiOptionItem } from "./text/StiOptionItem";
import { StiTextItem } from "./text/StiTextItem";
export class StiSingleSelectionElement extends StiFormElement {
    constructor() {
        super();
        this.type = "Single_SelectionElement";
        this.tooltip = "";
        this.required = false;
        this.description = new StiTextItem();
        this.font = new Font("Segoe UI", 10);
        this._columns = 1;
        this.backgroundColor = Color.white();
        this.textColor = Color.black();
        this.checkColor = Color.black();
        this.borderColor = Color.lightGray();
        this.prefix = "";
        this.suffix = "";
        this.optionsContentEditable = [];
        this.useCustomOption = false;
        this.customOptionText = "";
        this.selectedOption = "";
        this.geometry = new StiRectangleGeometry(150, 80);
        this.label = new StiLabelItem(this, StiLocalizationService.get("Components", "StiSingleSelection"));
        this.options = [];
        this.options.push(StiLocalizationService.get("PropertyMain", "First"));
        this.options.push(StiLocalizationService.get("PropertyMain", "Second"));
        this.options.push(StiLocalizationService.get("PropertyMain", "Third"));
    }
    setDefaultValues() {
        this.selectedOption = "";
    }
    setTabIndex(value) {
        this.tabIndex = value;
        return value + (this.options?.length ?? 0);
    }
    get columns() {
        return this._columns;
    }
    set columns(value) {
        if (value <= 1) {
            this._columns = 1;
        }
        else {
            this._columns = value;
        }
    }
    set options(value) {
        this.nativeOptions = value;
        let this_ = this;
        this._options = new Proxy(value, {
            deleteProperty: function (target, property) {
                setTimeout(() => {
                    this_.updateOptionsContentEditable();
                }, 0);
                return true;
            },
            set: function (target, property, value, receiver) {
                target[property] = value;
                this_.updateOptionsContentEditable();
                return true;
            },
        });
        this.updateOptionsContentEditable();
    }
    getUsedFonts(usedFonts) {
        super.getUsedFonts(usedFonts);
        usedFonts.add(this.font.family, this.arrayOfStylesToString(this.font.style), this.font.size);
        usedFonts.add(this.description.font.family, this.arrayOfStylesToString(this.description.font.style), this.description.font.size);
    }
    updateOptionsContentEditable() {
        this.optionsContentEditable = [];
        this.options.forEach((l, index) => this.optionsContentEditable.push(new StiOptionItem(l, this, index)));
    }
    getDefaultEditableContent() {
        let textItem = new StiInputTextItem("", this.font, this.textColor);
        textItem.element = this;
        return textItem;
    }
    get value() {
        return this.selectedOption;
    }
    get options() {
        return this._options;
    }
    htmlStyle() {
        return `${this.font.htmlStyle()};color:${this.textColor.toHtmlColor()};`;
    }
    saveToJsonObject(options) {
        let jObject = super.saveToJsonObject(options);
        jObject.addPropertyJObject("Label", this.label.saveToJsonObject(options));
        jObject.addPropertyJObject("Description", this.description.saveToJsonObject(options));
        jObject.addPropertyString("Tooltip", this.tooltip);
        jObject.addPropertyBool("Required", this.required);
        jObject.addPropertyJObject("Font", this.font.saveToJsonObject(options));
        jObject.addPropertyNumber("Columns", this.columns);
        jObject.addPropertyColor("BackgroundColor", this.backgroundColor);
        jObject.addPropertyColor("TextColor", this.textColor);
        jObject.addPropertyColor("CheckColor", this.checkColor);
        jObject.addPropertyColor("BorderColor", this.borderColor);
        jObject.addPropertyArray("Options", this.options);
        jObject.addPropertyBool("UseCustomOption", this.useCustomOption);
        jObject.addPropertyString("CustomOptionText", this.customOptionText);
        jObject.addPropertyString("SelectedOption", this.selectedOption);
        jObject.addPropertyString("Prefix", this.prefix);
        jObject.addPropertyString("Suffix", this.suffix);
        return jObject;
    }
    loadFromJsonObject(json) {
        super.loadFromJsonObject(json);
        for (let property of json.properties()) {
            switch (property.name) {
                case "BackgroundColor":
                    this.backgroundColor = new Color();
                    this.backgroundColor.deserializeFromString(property.value);
                    break;
                case "TextColor":
                    this.textColor = new Color();
                    this.textColor.deserializeFromString(property.value);
                    break;
                case "Label":
                    this.label = new StiLabelItem(this);
                    this.label.loadFromJsonObject(property.value);
                    break;
                case "Font":
                    this.font = new Font();
                    this.font.loadFromJsonObject(property.value);
                    break;
                case "Description":
                    this.description = new StiTextItem();
                    this.description.loadFromJsonObject(property.value);
                    break;
                case "Tooltip":
                    this.tooltip = property.value;
                    break;
                case "Required":
                    this.required = property.value;
                    break;
                case "Columns":
                    this.columns = property.value;
                    break;
                case "BorderColor":
                    this.borderColor = new Color();
                    this.borderColor.deserializeFromString(property.value);
                    break;
                case "CheckColor":
                    this.checkColor = new Color();
                    this.checkColor.deserializeFromString(property.value);
                    break;
                case "CustomOptionText":
                    this.customOptionText = property.value;
                    break;
                case "UseCustomOption":
                    this.useCustomOption = property.value;
                    break;
                case "SelectedOption":
                    this.selectedOption = property.value;
                    break;
                case "Prefix":
                    this.prefix = property.value;
                    break;
                case "Suffix":
                    this.suffix = property.value;
                    break;
                case "Options":
                    this.options = [];
                    property.value.properties().forEach((p) => this.options.push(p.value));
                    break;
            }
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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