import { StiInputTextItem } from "../helper/StiInputTextItem";
import StiRectangleGeometry from "../properties/StiRectangleGeometry";
import { StiLocalizationService } from "../services/sti-localization.service";
import Color from "../system/Color";
import { Enum } from "../system/Enum";
import { StiLineWeight } from "../system/enums";
import { Font } from "../system/Font";
import { StiFormElement } from "./base/StiFormElement";
import { StiLabelItem } from "./text/StiLabelItem";
import { StiTextItem } from "./text/StiTextItem";
export class StiComboBoxElement extends StiFormElement {
    constructor() {
        super();
        this.type = "Combo_BoxElement";
        this.tooltip = "";
        this.required = false;
        this.description = new StiTextItem();
        this.font = new Font("Segoe UI", 10);
        this.backgroundColor = Color.white();
        this.textColor = Color.black();
        this.borderColor = Color.lightGray();
        this.borderWeight = StiLineWeight.Thin;
        this.editable = false;
        this.options = [];
        this.selectedOption = "";
        this.geometry = new StiRectangleGeometry(150, 40);
        this.label = new StiLabelItem(this, StiLocalizationService.get("Components", "StiComboBox"));
        this.options.push(StiLocalizationService.get("PropertyMain", "First"));
        this.options.push(StiLocalizationService.get("PropertyMain", "Second"));
        this.options.push(StiLocalizationService.get("PropertyMain", "Third"));
    }
    setDefaultValues() {
        this.selectedOption = "";
    }
    setTabIndex(value) {
        this.tabIndex = value++;
        return value;
    }
    getUsedFonts(usedFonts) {
        super.getUsedFonts(usedFonts);
        usedFonts.add(this.font.family, this.arrayOfStylesToString(this.font.style), this.font.size);
        usedFonts.add(this.description.font.family, this.arrayOfStylesToString(this.description.font.style), this.description.font.size);
    }
    getDefaultEditableContent() {
        let textItem = new StiInputTextItem(this.selectedOption, this.font, this.textColor, this.backgroundColor);
        textItem.element = this;
        return textItem;
    }
    get eFont() {
        return this.font;
    }
    set eFont(value) {
        this.font = value;
    }
    get eBackgroundColor() {
        return this.backgroundColor;
    }
    set eBackgroundColor(value) {
        this.backgroundColor = value;
    }
    get eColor() {
        return this.textColor;
    }
    set eColor(value) {
        this.textColor = value;
    }
    get value() {
        return this.selectedOption;
    }
    haveBackgoundColor() {
        return true;
    }
    haveContentAlignment() {
        return false;
    }
    haveVerticalAlignment() {
        return false;
    }
    htmlStyle() {
        return `${this.font.htmlStyle()};color:${this.eColor.toHtmlColor()}}`;
    }
    saveToJsonObject(options) {
        let jObject = super.saveToJsonObject(options);
        jObject.addPropertyJObject("Label", this.label.saveToJsonObject(options));
        jObject.addPropertyJObject("Description", this.description.saveToJsonObject(options));
        jObject.addPropertyJObject("Font", this.font.saveToJsonObject(options));
        jObject.addPropertyColor("BackgroundColor", this.backgroundColor);
        jObject.addPropertyColor("TextColor", this.textColor);
        jObject.addPropertyString("Tooltip", this.tooltip);
        jObject.addPropertyBool("Required", this.required);
        jObject.addPropertyColor("BorderColor", this.borderColor);
        jObject.addPropertyEnum("BorderWeight", StiLineWeight, this.borderWeight);
        jObject.addPropertyBool("Editable", this.editable);
        jObject.addPropertyString("SelectedOption", this.selectedOption);
        jObject.addPropertyArray("Options", this.options);
        return jObject;
    }
    loadFromJsonObject(json) {
        super.loadFromJsonObject(json);
        for (let property of json.properties()) {
            switch (property.name) {
                case "BackgroundColor":
                    this.backgroundColor = new Color();
                    this.backgroundColor.deserializeFromString(property.value);
                    break;
                case "TextColor":
                    this.textColor = new Color();
                    this.textColor.deserializeFromString(property.value);
                    break;
                case "Label":
                    this.label = new StiLabelItem(this);
                    this.label.loadFromJsonObject(property.value);
                    break;
                case "Font":
                    this.font = new Font();
                    this.font.loadFromJsonObject(property.value);
                    break;
                case "Description":
                    this.description = new StiTextItem();
                    this.description.loadFromJsonObject(property.value);
                    break;
                case "Tooltip":
                    this.tooltip = property.value;
                    break;
                case "Required":
                    this.required = property.value;
                    break;
                case "BorderColor":
                    this.borderColor = new Color();
                    this.borderColor.deserializeFromString(property.value);
                    break;
                case "BorderWeight":
                    this.borderWeight = Enum.parse(StiLineWeight, property.value);
                    break;
                case "Editable":
                    this.editable = property.value;
                    break;
                case "SelectedOption":
                    this.selectedOption = property.value;
                    break;
                case "Options":
                    this.options = [];
                    property.value.properties().forEach((p) => this.options.push(p.value));
                    break;
            }
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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