import { Component, ViewChild } from "@angular/core";
import Color from "../../system/Color";
import { StiTextItem } from "../../elements/text/StiTextItem";
import { StiHorizontalAlignment } from "../../system/enums";
import { Enum } from "../../system/Enum";
import { StiImageEditorResult } from "../../elements/signature/StiImageEditorResult";
import * as i0 from "@angular/core";
import * as i1 from "../../services/model.service";
import * as i2 from "../../services/sti-localization.service";
import * as i3 from "../../services/mouse.service";
import * as i4 from "../../services/sti-signature.service";
import * as i5 from "../../services/sti-draggable-modal-window.service";
import * as i6 from "../../services/sti-color-dialog.service";
import * as i7 from "@angular/common";
import * as i8 from "@angular/forms";
import * as i9 from "../sti-tooltip/sti-tooltip.directive";
export class StiElectronicSignatureEditorComponent {
    get brushColor() {
        return this.signatureService?.currentEditableSignatureElement?.penColor
            ? this.signatureService?.currentEditableSignatureElement?.penColor
            : Color.black();
    }
    constructor(model, loc, mouseService, signatureService, modalWindowService, colorService) {
        this.model = model;
        this.loc = loc;
        this.mouseService = mouseService;
        this.signatureService = signatureService;
        this.modalWindowService = modalWindowService;
        this.colorService = colorService;
        this.drawingMode = false;
        this.isDrawing = false;
        this.defaultPreviewText = "FullName FN";
    }
    get hasTemplates() {
        return this.signatureService.savedTemplates.length > 0;
    }
    get insertedText() {
        return this.signatureService.currentEditableText;
    }
    get insertedImage() {
        return this.signatureService.currentEditableImage;
    }
    get fullName() {
        return this.signatureService.currentEditableFullName;
    }
    set fullName(value) {
        this.signatureService.currentEditableFullName = value;
    }
    get initials() {
        return this.signatureService.currentEditableInitials;
    }
    set initials(value) {
        this.signatureService.currentEditableInitials = value;
    }
    isEmptyContent() {
        if (this.signatureService.selectedMode == "Draw") {
            return (!this.signatureService.currentEditableImage?.loadedImage &&
                !this.signatureService.currentEditableText?.value &&
                !this.signatureService.currentCanvasContent);
        }
        else {
            return !this.signatureService.currentEditableFullName && !this.signatureService.currentEditableInitials;
        }
    }
    saveTemplate() {
        if (this.signatureService.selectedMode == "Draw") {
            let imageObj = {
                type: "Image",
                image: this.signatureService.currentEditableImage,
                text: this.signatureService.currentEditableText,
                drawedImage: this.signatureService.currentCanvasContent,
            };
            if (!this.signatureService.savedTemplates.find((item) => item == imageObj)) {
                this.signatureService.savedTemplates.push(imageObj);
            }
        }
        else {
            let textObj = {
                type: "Text",
                fullName: this.signatureService.currentEditableFullName,
                initials: this.signatureService.currentEditableInitials,
                specialFont: this.signatureService.currentPreviewFont,
            };
            if (!this.signatureService.savedTemplates.find((item) => item == textObj)) {
                this.signatureService.savedTemplates.push(textObj);
            }
        }
        localStorage.removeItem("StiSignatureTemplates");
        localStorage.setItem("StiSignatureTemplates", JSON.stringify(this.signatureService.savedTemplates));
    }
    showSavedTemplatesPreview(button) {
        let buttonRect = this.model.getPositionRelativeToForm(button);
        this.signatureService.templatesPreviewX = buttonRect.right - 335;
        this.signatureService.templatesPreviewY = buttonRect.bottom + 1;
        this.signatureService.showSavedTemplates = !this.signatureService.showSavedTemplates;
    }
    ngOnDestroy() {
        this.signatureService.currentEditableText = new StiTextItem();
        this.signatureService.currentCanvasContent = null;
        this.signatureService.currentEditableImage = new StiImageEditorResult();
        this.signatureService.currentEditableSignatureElement = null;
    }
    isCanvasBlank(canvas) {
        const context = canvas.getContext("2d");
        const pixelBuffer = new Uint32Array(context.getImageData(0, 0, canvas.width, canvas.height).data.buffer);
        return !pixelBuffer.some((color) => color !== 0);
    }
    ngAfterViewInit() {
        const onResize = (event) => {
            this.signatureService.editorComponentX = (window.innerWidth - this.signatureService.editorComponentWidth) / 2;
            this.signatureService.editorComponentY = (window.innerHeight - this.signatureService.editorComponentHeight) / 2;
        };
        window.addEventListener("resize", onResize);
        this.fillDrawingCanvas();
    }
    get image() {
        return "url(" + this.insertedImage?.loadedImage + ")";
    }
    getHorizontalAlignment() {
        if (this.insertedImage?.horizontalAlignment == StiHorizontalAlignment.Center) {
            return "center";
        }
        else if (this.insertedImage?.horizontalAlignment == StiHorizontalAlignment.Left) {
            return "start";
        }
        else {
            return "end";
        }
    }
    onMouseDown(event, canvasElement) {
        if (this.drawingMode) {
            event.preventDefault();
            event.stopPropagation();
            this.isDrawing = true;
            this.context = canvasElement?.getContext("2d");
            this.context.strokeStyle = this.brushColor.toHtmlColor();
            const coords = this.relativeCoords(event, canvasElement);
            this.context.moveTo(coords.x, coords.y);
        }
    }
    relativeCoords(event, canvasElement) {
        const bounds = this.model.getPositionRelativeToForm(canvasElement);
        const x = ((event.clientX ? event.clientX : (event.touches && event.touches.length > 0 ? event.touches[0].clientX : 0)) - bounds.left) / this.model.zoom;
        const y = ((event.clientY ? event.clientY : (event.touches && event.touches.length > 0 ? event.touches[0].clientY : 0)) - bounds.top) / this.model.zoom;
        return { x: x, y: y };
    }
    onCanvasMove(event, canvasElement) {
        if (this.isDrawing) {
            const coords = this.relativeCoords(event, canvasElement);
            this.context.lineTo(coords.x, coords.y);
            this.context.stroke();
        }
    }
    onMouseUp() {
        if (this.isDrawing && !this.isCanvasBlank(this.canvas?.nativeElement)) {
            this.signatureService.currentCanvasContent = this.canvas?.nativeElement?.toDataURL("image/png");
        }
        this.isDragging = false;
        this.isDrawing = false;
    }
    getBackgroundSize(container) {
        if (!this.insertedImage?.loadedImage) {
            return;
        }
        if (this.insertedImage?.aspectRatio && this.insertedImage?.stretch) {
            return "contain";
        }
        if (this.insertedImage?.stretch) {
            return `${container.offsetWidth}px ${container.offsetHeight}px`;
        }
        else {
            return `${this.insertedImage?.naturalWidth}px ${this.insertedImage?.naturalHeight}px`;
        }
    }
    getBackgroundPosition() {
        if (this.insertedImage?.stretch && !this.insertedImage?.aspectRatio) {
            return `center`;
        }
        else {
            return `${this.signatureService.getVerticalPosition(this.insertedImage?.verticalAlignment)} ${this.signatureService.getHorizontalPosition(this.insertedImage?.horizontalAlignment)}`;
        }
    }
    get previewText() {
        return (this.fullName + " " + this.initials).length > 1
            ? this.fullName + " " + this.initials
            : this.defaultPreviewText;
    }
    get alignment() {
        if (this.insertedText.horizontalAlignment == StiHorizontalAlignment.Center) {
            return "center";
        }
        else if (this.insertedText.horizontalAlignment == StiHorizontalAlignment.Right) {
            return "end";
        }
        else {
            return "start";
        }
    }
    clear() {
        this.signatureService.currentEditableText.value = "";
        if (this.signatureService?.currentEditableImage?.loadedImage) {
            this.signatureService.currentEditableImage.loadedImage = "";
        }
        if (!this.context) {
            this.context = this.canvas?.nativeElement?.getContext("2d");
        }
        this.signatureService.currentCanvasContent = "";
        this.context?.clearRect(0, 0, this.canvas?.nativeElement?.width, this.canvas?.nativeElement?.height);
        this.context?.beginPath();
    }
    showEditor(editorType) {
        this.model.showDraggableModalWindow = true;
        if (editorType == "Text") {
            this.modalWindowService.isTextEditor = true;
            this.signatureService.currentEditableText = this.insertedText;
        }
        else {
            this.modalWindowService.isImageEditor = true;
            this.signatureService.currentEditableImage = this.insertedImage;
        }
    }
    showColorPicker(element, object, isRichText = false) {
        this.colorService.openDialog(element, object, isRichText);
    }
    fillDrawingCanvas() {
        let ctx = this.canvas?.nativeElement?.getContext("2d");
        if (ctx && this.signatureService.currentCanvasContent) {
            ctx.clearRect(0, 0, this.canvas?.nativeElement?.width, this.canvas?.nativeElement?.height);
            let image = new Image();
            image.onload = () => {
                ctx.drawImage(image, 0, 0, this.canvas?.nativeElement?.width, this.canvas?.nativeElement?.height);
            };
            image.src = this.signatureService.currentCanvasContent;
        }
    }
    showFontsPreview(element) {
        this.signatureService.showFontsPreview = !this.signatureService.showFontsPreview;
        let rect = this.model.getPositionRelativeToForm(element);
        this.signatureService.fontsPreviewX = rect.right - 200;
        this.signatureService.fontsPreviewY = rect.bottom + 2;
    }
    acceptChanges() {
        this.signatureService.currentActiveLoadingImages = 0;
        this.signatureService.completedLoadingImages = 0;
        this.signatureService.currentEditableSignatureElement.fullName = this.signatureService.currentEditableFullName;
        this.signatureService.currentEditableSignatureElement.initials = this.signatureService.currentEditableInitials;
        this.signatureService.currentEditableSignatureElement.specialFont = this.signatureService.currentPreviewFont;
        let canvasRect = this.model.getPositionRelativeToForm(this.canvas?.nativeElement);
        if (canvasRect && this.signatureService.selectedMode == "Draw") {
            let resultCanvas = document.createElement("canvas");
            this.canvasWidth = canvasRect.width;
            this.canvasHeight = canvasRect.height;
            resultCanvas.width = this.canvasWidth;
            resultCanvas.height = this.canvasHeight;
            let ctx = resultCanvas.getContext("2d");
            this.resultCanvasContext = ctx;
            this.insertedImageLayer = this.createLayer(resultCanvas);
            this.insertedTextLayer = this.createLayer(resultCanvas);
            this.drawedLayer = this.createLayer(resultCanvas);
            if (this.insertedImage?.loadedImage) {
                this.fillLayer(this.insertedImageLayer, this.insertedImage);
            }
            if (this.insertedText?.value) {
                this.fillLayer(this.insertedTextLayer, this.insertedText);
            }
            if (!this.isCanvasBlank(this.canvas?.nativeElement)) {
                this.fillLayer(this.drawedLayer, this.canvas?.nativeElement?.toDataURL("image/png"));
            }
            else {
                this.signatureService.currentCanvasContent = "";
            }
            this.signatureService.checkLoading();
        }
        else {
            this.updateResultCanvas("Text");
        }
        if (this.signatureService.selectedMode == "Draw") {
            this.signatureService.currentEditableSignatureElement.currentSignatureMode = "Draw";
        }
        else {
            this.signatureService.currentEditableSignatureElement.currentSignatureMode = "Type";
        }
    }
    createLayer(canvas) {
        let layer = document.createElement("canvas");
        layer.width = canvas.width;
        layer.height = canvas.height;
        return layer.getContext("2d");
    }
    drawLayer(context, layer) {
        if (layer) {
            context.drawImage(layer.canvas, 0, 0);
        }
    }
    fillLayer(layer, content) {
        if (content instanceof StiTextItem) {
            if (content.value) {
                layer.font = `${content.font.isItalic() ? "italic " : ""}${content.font.isBold() ? "bold " : ""}${content.font.size + "px "}${content.font.family}`;
                layer.textBaseline = "center";
                layer.textAlign = Enum.getName(StiHorizontalAlignment, content.horizontalAlignment).toLocaleLowerCase();
                layer.fillText(content.value, Enum.getName(StiHorizontalAlignment, content.horizontalAlignment).toLocaleLowerCase() == "left"
                    ? 0
                    : Enum.getName(StiHorizontalAlignment, content.horizontalAlignment).toLocaleLowerCase() == "center"
                        ? this.canvasWidth / 2
                        : this.canvasWidth - layer.measureText(content.value).width, this.canvasHeight / 2);
            }
        }
        else if (content?.loadedImage) {
            this.signatureService.currentActiveLoadingImages += 1;
            let image = new Image();
            image.onload = () => {
                if (content?.aspectRatio && content?.stretch) {
                    let hRatio = this.canvasWidth / image.naturalWidth;
                    let vRatio = this.canvasHeight / image.naturalHeight;
                    let ratio = Math.min(hRatio, vRatio);
                    let centerShiftX = (this.canvasWidth - image.naturalWidth * ratio) / 2;
                    let centerShiftY = (this.canvasHeight - image.naturalHeight * ratio) / 2;
                    layer.drawImage(image, 0, 0, image.naturalWidth, image.naturalHeight, centerShiftX, centerShiftY, image.naturalWidth * ratio, image.naturalHeight * ratio);
                    this.signatureService.completedLoadingImages += 1;
                }
                else {
                    layer.drawImage(image, 0, 0, content.stretch ? this.canvasWidth : image.naturalWidth, content.stretch ? this.canvasHeight : image.naturalHeight);
                    this.signatureService.completedLoadingImages += 1;
                }
            };
            image.src = content.loadedImage;
        }
        else {
            if (typeof content === "string") {
                this.signatureService.currentActiveLoadingImages += 1;
                let image = new Image();
                image.onload = () => {
                    layer.drawImage(image, 0, 0, this.canvasWidth, this.canvasHeight);
                    this.signatureService.completedLoadingImages += 1;
                };
                image.src = content;
            }
        }
    }
    changeMode(mode) {
        this.signatureService.selectedMode = mode;
        if (mode == "Draw") {
            setTimeout(() => {
                this.fillDrawingCanvas();
            });
        }
    }
    updateResultCanvas(type = "Draw") {
        if (type == "Draw") {
            this.drawLayer(this.resultCanvasContext, this.insertedImageLayer);
            this.drawLayer(this.resultCanvasContext, this.insertedTextLayer);
            this.drawLayer(this.resultCanvasContext, this.drawedLayer);
            // console.log(this.resultCanvasContext.canvas.toDataURL("image/png"));
            if (this.resultCanvasContext?.canvas?.toDataURL("image/png")) {
                this.signatureService.currentEditableSignatureElement.backgroundImage =
                    this.resultCanvasContext.canvas.toDataURL("image/png");
            }
            this.signatureService.currentEditableSignatureElement.insertedImage = this.signatureService.currentEditableImage;
            this.signatureService.currentEditableSignatureElement.insertedText = this.signatureService.currentEditableText;
            if (this.signatureService.currentCanvasContent) {
                this.signatureService.currentEditableSignatureElement.drawedImage = this.signatureService.currentCanvasContent;
            }
            else {
                this.signatureService.currentEditableSignatureElement.drawedImage = "";
            }
        }
        this.signatureService.currentEditableSignatureElement.isSigned = true;
        this.model.showSignatureEditor = false;
        // this.signatureService.currentSignatureElement = null;
    }
    onMouseMove(event) {
        if (this.isDragging) {
            let deltaX = event.clientX - this.startX;
            let deltaY = event.clientY - this.startY;
            if (this.startComponentX + deltaX < 0) {
                this.signatureService.editorComponentX = 0;
            }
            else if (this.startComponentX + deltaX > window.innerWidth - this.signatureService.editorComponentWidth) {
                this.signatureService.editorComponentX = window.innerWidth - this.signatureService.editorComponentWidth;
            }
            else {
                this.signatureService.editorComponentX = this.startComponentX + deltaX;
            }
            if (this.startComponentY + deltaY < 0) {
                this.signatureService.editorComponentY = 0;
            }
            else if (this.startComponentY + deltaY > window.innerHeight - this.signatureService.editorComponentHeight) {
                this.signatureService.editorComponentY = window.innerHeight - this.signatureService.editorComponentHeight;
            }
            else {
                this.signatureService.editorComponentY = this.startComponentY + deltaY;
            }
        }
    }
    startDrag(event) {
        this.startX = event.clientX;
        this.startY = event.clientY;
        this.startComponentX = this.signatureService.editorComponentX;
        this.startComponentY = this.signatureService.editorComponentY;
        this.isDragging = true;
    }
    ngOnInit() {
        this.drawingMode = true;
        if (localStorage.getItem("StiSignatureTemplates")) {
            this.signatureService.savedTemplates = JSON.parse(localStorage.getItem("StiSignatureTemplates"));
        }
        this.signatureService.editorComponentX = (window.innerWidth - this.signatureService.editorComponentWidth) / 2;
        this.signatureService.editorComponentY = (window.innerHeight - this.signatureService.editorComponentHeight) / 2;
        this.updateSignatureSubscription = this.signatureService.updateSignature.subscribe((event) => {
            if (event.type == "Text") {
                this.signatureService.currentEditableText = event.value;
            }
            else if (event.type == "Image") {
                this.signatureService.currentEditableImage = event.value;
            }
        });
        this.updateResultCanvasSubscription = this.signatureService.completedLoading.subscribe(() => {
            this.updateResultCanvas();
        });
        this.mouseUpSubscription = this.mouseService.getDocumentMouseUp().subscribe(() => {
            this.onMouseUp();
        });
        this.mouseService.getDocumentTouchEnd().subscribe(() => {
            this.onMouseUp();
        });
        this.mouseMoveSubscription = this.mouseService.getDocumentMouseMove().subscribe((event) => {
            this.onMouseMove(event);
        });
    }
    static { this.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "18.2.14", ngImport: i0, type: StiElectronicSignatureEditorComponent, deps: [{ token: i1.StiModelService }, { token: i2.StiLocalizationService }, { token: i3.StiMouseService }, { token: i4.SignatureService }, { token: i5.StiDraggableModalWindowService }, { token: i6.StiColorDialogService }], target: i0.ɵɵFactoryTarget.Component }); }
    static { this.ɵcmp = i0.ɵɵngDeclareComponent({ minVersion: "14.0.0", version: "18.2.14", type: StiElectronicSignatureEditorComponent, selector: "sti-electronic-signature-editor", viewQueries: [{ propertyName: "canvas", first: true, predicate: ["canvas"], descendants: true }], ngImport: i0, template: `<div
      class="editor-container"
      [style.left.px]="signatureService.editorComponentX"
      [style.top.px]="signatureService.editorComponentY"
    >
      <div class="editor-navbar" (mousedown)="startDrag($event)">
        <span [style.flex]="1">{{ loc.get2("Components", "StiSignature") }}</span>
        <div class="close-button" (click)="model.showSignatureEditor = false">
          <img [src]="model.getImage('images.Other.CloseForm.png')" />
        </div>
      </div>
      <div class="content-container">
        <div class="content-navigation">
          <div
            class="navigation-button {{ signatureService.selectedMode == 'Type' ? 'current' : '' }}"
            (click)="changeMode('Type')"
          >
            <div class="navigation-label">
              <span>{{ loc.get2("PropertyEnum", "StiSignatureTypeType") }}</span>
            </div>
            <div class="indicator {{ signatureService.selectedMode == 'Type' ? 'current-indicator' : '' }}"></div>
          </div>
          <div
            class="navigation-button {{ signatureService.selectedMode == 'Draw' ? 'current' : '' }}"
            (click)="changeMode('Draw')"
          >
            <div class="navigation-label">
              <span>{{ loc.get2("PropertyEnum", "StiSignatureTypeDraw") }}</span>
            </div>
            <div class="indicator {{ signatureService.selectedMode == 'Draw' ? 'current-indicator' : '' }}"></div>
          </div>
          <div class="templates-navigation-container">
            <div (click)="saveTemplate()" class="template-navigation-button {{ isEmptyContent() ? 'disabled' : '' }}">
              <img [src]="model.getImage('signature.StiSignatureSave.png')" />
            </div>
            <div
              #button
              (click)="showSavedTemplatesPreview(button)"
              class="template-navigation-button show-button"
              [style.width.px]="38"
            >
              <img [src]="model.getImage('signature.StiSignatureOpen.png')" /><i class="pi pi-chevron-down"></i>
            </div>
          </div>
        </div>
        <div class="bottom-divider"></div>
        <div class="type-container" *ngIf="signatureService.selectedMode == 'Type'">
          <div class="input-field">
            <span>{{ loc.getP("FullName") }}</span
            ><input type="text" [(ngModel)]="fullName" />
          </div>
          <div class="input-field">
            <span>{{ loc.getP("Initials") }}</span
            ><input type="text" [(ngModel)]="initials" />
          </div>
          <div class="button-panel">
            <div #selectButton class="select-button" (click)="showFontsPreview(selectButton)">
              <span class="button-label">{{ loc.get2("ChartRibbon", "Style") }}</span
              ><i class="pi pi-chevron-down"></i>
            </div>
          </div>
          <div class="preview-panel" [style.font-family]="signatureService.currentPreviewFont">
            <span>{{ previewText }}</span>
          </div>
        </div>
        <div class="draw-container" *ngIf="signatureService.selectedMode == 'Draw'">
          <div class="draw-buttons-panel">
            <div
              class="button-container {{ drawingMode ? 'active' : '' }}"
              [stiTooltip]="loc.getP('ForeColor')"
              tooltipPosition="bottom"
              showDelay="500"
              (mousedown)="$event.preventDefault()"
              #fontColorEditor
            >
              <div class="preview-container" (click)="drawingMode = !drawingMode">
                <div class="foreground" [style]="{ backgroundPosition: 'center', backgroundRepeat: 'no-repeat' }">
                  <img [src]="model.getImage('images.Brush.png')" alt="" />
                </div>
                <span
                  class="color-preview"
                  [style.background-color]="brushColor != null ? brushColor?.toHtmlColor() : 'rgb(182, 181, 180)'"
                ></span>
              </div>
              <div
                (click)="showColorPicker(fontColorEditor, brushColor)"
                [style]="{ position: 'relative', display: 'flex', alignItems: 'center' }"
              >
                <i class="pi pi-chevron-down"></i>
              </div>
            </div>
            <div class="draw-panel-button" (click)="showEditor('Text')">
              <img [src]="model.getImage('signature.Text.png')" />
            </div>
            <div class="draw-panel-button" (click)="showEditor('Image')">
              <img [src]="model.getImage('images.Toolbox.StiImageElement.png')" />
            </div>
            <div class="draw-panel-button" (click)="clear()">
              <img [src]="model.getImage('images.Page.ClearContent.png')" />
            </div>
          </div>
          <div class="draw-panel" #previewContainer>
            <div class="canvas-container">
              <canvas
                [width]="previewContainer.clientWidth"
                [height]="previewContainer.clientHeight"
                (mousedown)="onMouseDown($event, canvas)"
                (mousemove)="onCanvasMove($event, canvas)"
                (touchstart)="onMouseDown($event, canvas)"
                (touchmove)="onCanvasMove($event, canvas)"
                #canvas
              >
              </canvas>
            </div>
            <div class="preview-inserted-text" [style]="insertedText.htmlStyle()" [style.justify-content]="alignment">
              <span>{{ insertedText.value }}</span>
            </div>
            <div class="inserted-image-container" [style.justify-content]="getHorizontalAlignment()">
              <div
                class="preview-inserted-image"
                [style.width.px]="
                  insertedImage?.aspectRatio && insertedImage?.stretch
                    ? previewContainer.clientHeight
                    : previewContainer.clientWidth
                "
                [style.height.px]="previewContainer.clientHeight"
                [style.background-image]="image"
                [style.background-position]="getBackgroundPosition()"
                [style.background-size]="getBackgroundSize(previewContainer)"
                [style.background-repeat]="image ? 'no-repeat' : ''"
              ></div>
            </div>
            <div class="signature-description">
              <span>{{ "Signature" }}</span>
            </div>
          </div>
        </div>
      </div>
      <div class="setup-accept">
        <div class="accept button" (click)="acceptChanges()">
          <span [style.margin-left.px]="0">{{
            model.previewMode
              ? loc.get2("Buttons", "Sign")
              : loc.get2("Buttons", "Save").replace("&amp", "").replace(";", "")
          }}</span>
        </div>
        <div class="cancel button" (click)="model.showSignatureEditor = false">
          <span [style.margin-left.px]="0">{{ loc.getP("Cancel") }}</span>
        </div>
      </div>
    </div>
    <div
      class="fonts-preview-container"
      *ngIf="signatureService.showFontsPreview"
      [style.left.px]="signatureService.fontsPreviewX"
      [style.top.px]="signatureService.fontsPreviewY"
    >
      <div
        *ngFor="let font of signatureService.previewFonts"
        class="font-preview"
        [style.font-family]="font"
        (click)="signatureService.currentPreviewFont = font; signatureService.showFontsPreview = false"
      >
        <div class="vertical-line {{ signatureService.currentPreviewFont == font ? 'current-indicator' : '' }}"></div>
        <span> {{ previewText }}</span>
      </div>
    </div>
    <div
      *ngIf="signatureService.showSavedTemplates"
      class="saved-templates-container"
      [style.left.px]="signatureService.templatesPreviewX"
      [style.top.px]="signatureService.templatesPreviewY"
      [style.overflow-y]="hasTemplates ? 'scroll' : 'hidden'"
    >
      <div class="empty-container" *ngIf="!hasTemplates">
        <span>{{ "No saved items" }}</span>
      </div>
      <div
        class="template-item"
        *ngFor="let temp of signatureService.savedTemplates; let i = index"
        [style.margin-top.px]="i == 0 ? 6 : 0"
        (click)="signatureService.setTemplate(temp)"
      >
        <div class="delete-button" (click)="signatureService.deleteTemplate(i)">
          <img [src]="model.getImage('images.Other.CloseForm.png')" />
        </div>
        <div
          class="image-template"
          *ngIf="temp.type == 'Image'"
          #parentContainer
          [style.justify-content]="signatureService.getTemplateHorizontalAlignment(temp)"
        >
          <div
            class="loadedImage image-content"
            [style.background-image]="signatureService.getTemplateImage(temp)"
            [style.background-position]="signatureService.getTemplateBackgroundPosition(temp)"
            [style.background-size]="signatureService.getTemplateBackgroundSize(temp, parentContainer)"
            [style.background-repeat]="'no-repeat'"
          ></div>
          <div class="insertedText image-content">
            <span [style]="temp.text.htmlStyle()">{{ temp?.text?.value }}</span>
          </div>
          <div
            class="drawedImage image-content"
            [style.background-size]="'contain'"
            [style.background-image]="signatureService.getTemplateDrawedImage(temp)"
          ></div>
        </div>
        <div class="text-template" *ngIf="temp.type == 'Text'">
          <span [style.font-family]="temp.specialFont" [style.font-size.pt]="16">{{
            temp.fullName + " " + temp.initials
          }}</span>
        </div>
      </div>
    </div>`, isInline: true, styles: [".image-content{height:100%;width:100%;top:0;left:0;position:absolute}.drawedImage{z-index:3}.loadedImage{z-index:1}.insertedText{z-index:2;display:flex;align-items:center;justify-content:center}.text-template{display:flex;align-items:center;justify-content:center;height:50px}.image-template{position:relative;height:130px}.template-item{width:calc(100% - 12px);height:fit-content;padding:2px;margin:0 6px 6px;border:1px solid lightgray;border-radius:3px;position:relative}.delete-button{position:absolute;z-index:999;right:0;top:0;width:20px;height:20px;display:flex;align-items:center;justify-content:center;border-radius:3px}.delete-button img{height:16px;width:16px}.delete-button:hover img{filter:saturate(500%) contrast(100%) brightness(900%) invert(900%) sepia(50%)}.template-item:hover{background-color:#d7d7d7;cursor:pointer}.empty-container{height:100%;width:100%;background-color:#fafafa;display:flex;align-items:center;justify-content:center;color:#454545;font-size:13px;font-family:Arial}.saved-templates-container{position:absolute;overflow-x:hidden;overflow-y:scroll;width:335px;height:270px;border-radius:3px;-moz-box-shadow:0px 0px 7px rgba(0,0,0,.6);-webkit-box-shadow:0px 0px 7px rgb(0 0 0 / 60%);box-shadow:0 0 7px #0000004d;border:1px solid rgb(25,71,138);display:flex;flex-direction:column;background-color:#fff;z-index:99999999999999999}.active{background-color:#d3d3d3;outline:1px solid rgb(61,107,186)}.canvas-container{position:absolute;left:0;top:0;height:100%;width:100%;z-index:99999}.preview-inserted-image{position:absolute}.inserted-image-container{position:absolute;left:0;top:0;display:flex;height:100%;width:100%}.preview-inserted-text{position:absolute;height:100%;width:100%;display:flex;align-items:center;justify-content:center;z-index:999}.signature-content{z-index:9999;height:100%;width:100%}.signature-description{width:calc(100% - 10px);height:50px;margin:0 5px;padding:0 5px;border-top:1px solid lightgray;display:flex;align-items:center;overflow:hidden;z-index:999}.disabled{pointer-events:none;opacity:.5}.preview-container{display:flex;width:100%;position:relative;flex-direction:column}.preview-container img{height:16px;width:16px}.button-container{display:flex;height:26px;width:40px;position:relative;margin-right:5px;border-radius:3px}.button-container:hover{background-color:#d3d3d3}.button-container:active{background-color:var(--columnDeleteHover-color)}.color-preview{width:20px;height:5px;margin-left:4px}i{height:12px;width:12px;right:3px;top:50%;vertical-align:middle;color:var(--text-color)}.button-container:hover i{color:#fff}.foreground{margin-left:8px;margin-top:2px;width:16px;height:16px}.template-navigation-button{height:26px;width:26px;margin-right:2px;border-radius:3px;display:flex;align-items:center;justify-content:center}.template-navigation-button:hover{background-color:#d3d3d3}.template-navigation-button img{height:16px;width:16px}.templates-navigation-container{width:100%;display:flex;align-items:center;justify-content:flex-end}.vertical-line{width:3px;height:calc(100% - 14px);margin:7px 10px 7px 5px}.font-preview{font-size:16px;height:42px;display:flex;align-items:center;padding-left:5px;border-radius:3px;margin:1px}.font-preview:hover{background-color:#dadada}.fonts-preview-container{position:absolute;width:200px;height:134px;z-index:9999999999;padding:1px;border-radius:3px;border:1px solid #c6c6c6;background:#fff;color:#444;-moz-box-shadow:0px 0px 7px rgba(0,0,0,.6);-webkit-box-shadow:0px 0px 7px rgba(0,0,0,.6);box-shadow:0 0 7px #0000004d;cursor:default}.draw-buttons-panel{height:26px;margin-bottom:5px;margin-top:10px;display:flex;padding-left:24px}.draw-panel-button:hover{background-color:#d3d3d3}.draw-panel-button{height:26px;width:26px;margin-right:5px;border-radius:3px;display:flex;align-items:center;justify-content:center}.draw-panel-button img{height:16px;width:16px}.draw-panel{border:1px dashed #ababab;border-radius:3px;margin:0 24px;width:calc(100% - 48px);height:186px;display:flex;flex-direction:column-reverse;position:relative}i{margin-left:5px;font-size:7px;padding-top:4px}.select-button{border:1px solid #dddddd;border-radius:3px;min-width:30px;padding-left:10px;padding-right:6px;display:flex;color:#444;align-items:center;margin-right:24px;margin-top:24px;height:24px}.select-button:hover{background-color:#d3d3d3}.button-panel{width:100%;display:flex;justify-content:flex-end}.preview-panel{height:80px;width:calc(100% - 48px);border:1px solid #ababab;border-radius:3px;font-size:16pt;margin:10px 24px;padding-left:12px;display:flex;align-items:center;overflow:hidden}.input-field{width:calc(100% - 24px);display:flex;margin:12px 12px 6px;padding:0 12px;align-items:center}.input-field span{min-width:100px;font-size:12px}.input-field input{border-radius:3px;padding-left:5px;height:24px;outline:none;border:1px solid lightgray;width:100%}.input-field input:hover{border-color:#454545}.current-indicator{background-color:#19478a}.other-indicator{background-color:transparent}.current{background-color:#e5e5e5}.navigation-button:hover{background-color:#dadada}.navigation-button{border-radius:3px;height:26px;min-width:50px;display:flex;flex-direction:column;color:#444;font-size:12px;font-family:Arial;margin-right:2px}.navigation-label{height:27px;width:100%;height:18px;padding:0 10px;display:flex;align-items:center;justify-content:center;white-space:nowrap}.navigation-label span{margin-top:7px}.indicator{height:3px;margin:1px 9px}.bottom-divider{height:1px;width:calc(100% - 24px);background-color:#d3d3d3;margin:5px 12px}.content-navigation{display:flex;padding:12px 12px 0}.content-container{height:100%;width:100%}.editor-navbar{display:flex;height:30px;border-top-left-radius:3px;border-top-right-radius:3px;background-color:#515c6a;-webkit-user-select:none;color:#fff;align-items:center}.editor-navbar span{font-size:13px;margin-left:10px}.close-button{height:30px;width:30px;border-radius:3px;display:flex;align-items:center;justify-content:center}.close-button:hover{background-color:#dadada}.close-button img{height:16px;width:16px;filter:brightness(0) invert(1)}.editor-container{background-color:var(--back-color);position:absolute;left:calc(50% - 182px);top:calc(50% - 175px);z-index:999;width:365px;height:350px;overflow:hidden;display:flex;flex-direction:column;-moz-box-shadow:0px 0px 7px rgba(0,0,0,.6);-webkit-box-shadow:0px 0px 7px rgb(0 0 0 / 60%);box-shadow:0 0 7px #0000004d;border:1px solid rgb(25,71,138);color:var(--text-color);font-size:12px;border-radius:5px;font-family:Arial;-webkit-user-select:none}.button{height:24px;width:80px;border-radius:3px;margin:3px 5px;display:flex;align-items:center;justify-content:center;border:1px solid transparent}.accept:active{background-color:#355da0;border-color:#264c7d}.cancel:active{background-color:#e2e2e2;border-color:#838383}.accept{background-color:#19478a;color:#fff}.accept:hover{background-color:#3d6298}.cancel{background-color:#ddd;color:#444}.cancel:hover{background-color:#d5d5d5}.setup-accept{padding:8px;display:flex;align-items:flex-end;justify-content:flex-end}\n"], dependencies: [{ kind: "directive", type: i7.NgForOf, selector: "[ngFor][ngForOf]", inputs: ["ngForOf", "ngForTrackBy", "ngForTemplate"] }, { kind: "directive", type: i7.NgIf, selector: "[ngIf]", inputs: ["ngIf", "ngIfThen", "ngIfElse"] }, { kind: "directive", type: i8.DefaultValueAccessor, selector: "input:not([type=checkbox])[formControlName],textarea[formControlName],input:not([type=checkbox])[formControl],textarea[formControl],input:not([type=checkbox])[ngModel],textarea[ngModel],[ngDefaultControl]" }, { kind: "directive", type: i8.NgControlStatus, selector: "[formControlName],[ngModel],[formControl]" }, { kind: "directive", type: i8.NgModel, selector: "[ngModel]:not([formControlName]):not([formControl])", inputs: ["name", "disabled", "ngModel", "ngModelOptions"], outputs: ["ngModelChange"], exportAs: ["ngModel"] }, { kind: "directive", type: i9.StiTooltipDirective, selector: "[stiTooltip]", inputs: ["tooltipPosition", "stiTooltip", "showDelay"] }] }); }
}
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "18.2.14", ngImport: i0, type: StiElectronicSignatureEditorComponent, decorators: [{
            type: Component,
            args: [{ selector: "sti-electronic-signature-editor", template: `<div
      class="editor-container"
      [style.left.px]="signatureService.editorComponentX"
      [style.top.px]="signatureService.editorComponentY"
    >
      <div class="editor-navbar" (mousedown)="startDrag($event)">
        <span [style.flex]="1">{{ loc.get2("Components", "StiSignature") }}</span>
        <div class="close-button" (click)="model.showSignatureEditor = false">
          <img [src]="model.getImage('images.Other.CloseForm.png')" />
        </div>
      </div>
      <div class="content-container">
        <div class="content-navigation">
          <div
            class="navigation-button {{ signatureService.selectedMode == 'Type' ? 'current' : '' }}"
            (click)="changeMode('Type')"
          >
            <div class="navigation-label">
              <span>{{ loc.get2("PropertyEnum", "StiSignatureTypeType") }}</span>
            </div>
            <div class="indicator {{ signatureService.selectedMode == 'Type' ? 'current-indicator' : '' }}"></div>
          </div>
          <div
            class="navigation-button {{ signatureService.selectedMode == 'Draw' ? 'current' : '' }}"
            (click)="changeMode('Draw')"
          >
            <div class="navigation-label">
              <span>{{ loc.get2("PropertyEnum", "StiSignatureTypeDraw") }}</span>
            </div>
            <div class="indicator {{ signatureService.selectedMode == 'Draw' ? 'current-indicator' : '' }}"></div>
          </div>
          <div class="templates-navigation-container">
            <div (click)="saveTemplate()" class="template-navigation-button {{ isEmptyContent() ? 'disabled' : '' }}">
              <img [src]="model.getImage('signature.StiSignatureSave.png')" />
            </div>
            <div
              #button
              (click)="showSavedTemplatesPreview(button)"
              class="template-navigation-button show-button"
              [style.width.px]="38"
            >
              <img [src]="model.getImage('signature.StiSignatureOpen.png')" /><i class="pi pi-chevron-down"></i>
            </div>
          </div>
        </div>
        <div class="bottom-divider"></div>
        <div class="type-container" *ngIf="signatureService.selectedMode == 'Type'">
          <div class="input-field">
            <span>{{ loc.getP("FullName") }}</span
            ><input type="text" [(ngModel)]="fullName" />
          </div>
          <div class="input-field">
            <span>{{ loc.getP("Initials") }}</span
            ><input type="text" [(ngModel)]="initials" />
          </div>
          <div class="button-panel">
            <div #selectButton class="select-button" (click)="showFontsPreview(selectButton)">
              <span class="button-label">{{ loc.get2("ChartRibbon", "Style") }}</span
              ><i class="pi pi-chevron-down"></i>
            </div>
          </div>
          <div class="preview-panel" [style.font-family]="signatureService.currentPreviewFont">
            <span>{{ previewText }}</span>
          </div>
        </div>
        <div class="draw-container" *ngIf="signatureService.selectedMode == 'Draw'">
          <div class="draw-buttons-panel">
            <div
              class="button-container {{ drawingMode ? 'active' : '' }}"
              [stiTooltip]="loc.getP('ForeColor')"
              tooltipPosition="bottom"
              showDelay="500"
              (mousedown)="$event.preventDefault()"
              #fontColorEditor
            >
              <div class="preview-container" (click)="drawingMode = !drawingMode">
                <div class="foreground" [style]="{ backgroundPosition: 'center', backgroundRepeat: 'no-repeat' }">
                  <img [src]="model.getImage('images.Brush.png')" alt="" />
                </div>
                <span
                  class="color-preview"
                  [style.background-color]="brushColor != null ? brushColor?.toHtmlColor() : 'rgb(182, 181, 180)'"
                ></span>
              </div>
              <div
                (click)="showColorPicker(fontColorEditor, brushColor)"
                [style]="{ position: 'relative', display: 'flex', alignItems: 'center' }"
              >
                <i class="pi pi-chevron-down"></i>
              </div>
            </div>
            <div class="draw-panel-button" (click)="showEditor('Text')">
              <img [src]="model.getImage('signature.Text.png')" />
            </div>
            <div class="draw-panel-button" (click)="showEditor('Image')">
              <img [src]="model.getImage('images.Toolbox.StiImageElement.png')" />
            </div>
            <div class="draw-panel-button" (click)="clear()">
              <img [src]="model.getImage('images.Page.ClearContent.png')" />
            </div>
          </div>
          <div class="draw-panel" #previewContainer>
            <div class="canvas-container">
              <canvas
                [width]="previewContainer.clientWidth"
                [height]="previewContainer.clientHeight"
                (mousedown)="onMouseDown($event, canvas)"
                (mousemove)="onCanvasMove($event, canvas)"
                (touchstart)="onMouseDown($event, canvas)"
                (touchmove)="onCanvasMove($event, canvas)"
                #canvas
              >
              </canvas>
            </div>
            <div class="preview-inserted-text" [style]="insertedText.htmlStyle()" [style.justify-content]="alignment">
              <span>{{ insertedText.value }}</span>
            </div>
            <div class="inserted-image-container" [style.justify-content]="getHorizontalAlignment()">
              <div
                class="preview-inserted-image"
                [style.width.px]="
                  insertedImage?.aspectRatio && insertedImage?.stretch
                    ? previewContainer.clientHeight
                    : previewContainer.clientWidth
                "
                [style.height.px]="previewContainer.clientHeight"
                [style.background-image]="image"
                [style.background-position]="getBackgroundPosition()"
                [style.background-size]="getBackgroundSize(previewContainer)"
                [style.background-repeat]="image ? 'no-repeat' : ''"
              ></div>
            </div>
            <div class="signature-description">
              <span>{{ "Signature" }}</span>
            </div>
          </div>
        </div>
      </div>
      <div class="setup-accept">
        <div class="accept button" (click)="acceptChanges()">
          <span [style.margin-left.px]="0">{{
            model.previewMode
              ? loc.get2("Buttons", "Sign")
              : loc.get2("Buttons", "Save").replace("&amp", "").replace(";", "")
          }}</span>
        </div>
        <div class="cancel button" (click)="model.showSignatureEditor = false">
          <span [style.margin-left.px]="0">{{ loc.getP("Cancel") }}</span>
        </div>
      </div>
    </div>
    <div
      class="fonts-preview-container"
      *ngIf="signatureService.showFontsPreview"
      [style.left.px]="signatureService.fontsPreviewX"
      [style.top.px]="signatureService.fontsPreviewY"
    >
      <div
        *ngFor="let font of signatureService.previewFonts"
        class="font-preview"
        [style.font-family]="font"
        (click)="signatureService.currentPreviewFont = font; signatureService.showFontsPreview = false"
      >
        <div class="vertical-line {{ signatureService.currentPreviewFont == font ? 'current-indicator' : '' }}"></div>
        <span> {{ previewText }}</span>
      </div>
    </div>
    <div
      *ngIf="signatureService.showSavedTemplates"
      class="saved-templates-container"
      [style.left.px]="signatureService.templatesPreviewX"
      [style.top.px]="signatureService.templatesPreviewY"
      [style.overflow-y]="hasTemplates ? 'scroll' : 'hidden'"
    >
      <div class="empty-container" *ngIf="!hasTemplates">
        <span>{{ "No saved items" }}</span>
      </div>
      <div
        class="template-item"
        *ngFor="let temp of signatureService.savedTemplates; let i = index"
        [style.margin-top.px]="i == 0 ? 6 : 0"
        (click)="signatureService.setTemplate(temp)"
      >
        <div class="delete-button" (click)="signatureService.deleteTemplate(i)">
          <img [src]="model.getImage('images.Other.CloseForm.png')" />
        </div>
        <div
          class="image-template"
          *ngIf="temp.type == 'Image'"
          #parentContainer
          [style.justify-content]="signatureService.getTemplateHorizontalAlignment(temp)"
        >
          <div
            class="loadedImage image-content"
            [style.background-image]="signatureService.getTemplateImage(temp)"
            [style.background-position]="signatureService.getTemplateBackgroundPosition(temp)"
            [style.background-size]="signatureService.getTemplateBackgroundSize(temp, parentContainer)"
            [style.background-repeat]="'no-repeat'"
          ></div>
          <div class="insertedText image-content">
            <span [style]="temp.text.htmlStyle()">{{ temp?.text?.value }}</span>
          </div>
          <div
            class="drawedImage image-content"
            [style.background-size]="'contain'"
            [style.background-image]="signatureService.getTemplateDrawedImage(temp)"
          ></div>
        </div>
        <div class="text-template" *ngIf="temp.type == 'Text'">
          <span [style.font-family]="temp.specialFont" [style.font-size.pt]="16">{{
            temp.fullName + " " + temp.initials
          }}</span>
        </div>
      </div>
    </div>`, styles: [".image-content{height:100%;width:100%;top:0;left:0;position:absolute}.drawedImage{z-index:3}.loadedImage{z-index:1}.insertedText{z-index:2;display:flex;align-items:center;justify-content:center}.text-template{display:flex;align-items:center;justify-content:center;height:50px}.image-template{position:relative;height:130px}.template-item{width:calc(100% - 12px);height:fit-content;padding:2px;margin:0 6px 6px;border:1px solid lightgray;border-radius:3px;position:relative}.delete-button{position:absolute;z-index:999;right:0;top:0;width:20px;height:20px;display:flex;align-items:center;justify-content:center;border-radius:3px}.delete-button img{height:16px;width:16px}.delete-button:hover img{filter:saturate(500%) contrast(100%) brightness(900%) invert(900%) sepia(50%)}.template-item:hover{background-color:#d7d7d7;cursor:pointer}.empty-container{height:100%;width:100%;background-color:#fafafa;display:flex;align-items:center;justify-content:center;color:#454545;font-size:13px;font-family:Arial}.saved-templates-container{position:absolute;overflow-x:hidden;overflow-y:scroll;width:335px;height:270px;border-radius:3px;-moz-box-shadow:0px 0px 7px rgba(0,0,0,.6);-webkit-box-shadow:0px 0px 7px rgb(0 0 0 / 60%);box-shadow:0 0 7px #0000004d;border:1px solid rgb(25,71,138);display:flex;flex-direction:column;background-color:#fff;z-index:99999999999999999}.active{background-color:#d3d3d3;outline:1px solid rgb(61,107,186)}.canvas-container{position:absolute;left:0;top:0;height:100%;width:100%;z-index:99999}.preview-inserted-image{position:absolute}.inserted-image-container{position:absolute;left:0;top:0;display:flex;height:100%;width:100%}.preview-inserted-text{position:absolute;height:100%;width:100%;display:flex;align-items:center;justify-content:center;z-index:999}.signature-content{z-index:9999;height:100%;width:100%}.signature-description{width:calc(100% - 10px);height:50px;margin:0 5px;padding:0 5px;border-top:1px solid lightgray;display:flex;align-items:center;overflow:hidden;z-index:999}.disabled{pointer-events:none;opacity:.5}.preview-container{display:flex;width:100%;position:relative;flex-direction:column}.preview-container img{height:16px;width:16px}.button-container{display:flex;height:26px;width:40px;position:relative;margin-right:5px;border-radius:3px}.button-container:hover{background-color:#d3d3d3}.button-container:active{background-color:var(--columnDeleteHover-color)}.color-preview{width:20px;height:5px;margin-left:4px}i{height:12px;width:12px;right:3px;top:50%;vertical-align:middle;color:var(--text-color)}.button-container:hover i{color:#fff}.foreground{margin-left:8px;margin-top:2px;width:16px;height:16px}.template-navigation-button{height:26px;width:26px;margin-right:2px;border-radius:3px;display:flex;align-items:center;justify-content:center}.template-navigation-button:hover{background-color:#d3d3d3}.template-navigation-button img{height:16px;width:16px}.templates-navigation-container{width:100%;display:flex;align-items:center;justify-content:flex-end}.vertical-line{width:3px;height:calc(100% - 14px);margin:7px 10px 7px 5px}.font-preview{font-size:16px;height:42px;display:flex;align-items:center;padding-left:5px;border-radius:3px;margin:1px}.font-preview:hover{background-color:#dadada}.fonts-preview-container{position:absolute;width:200px;height:134px;z-index:9999999999;padding:1px;border-radius:3px;border:1px solid #c6c6c6;background:#fff;color:#444;-moz-box-shadow:0px 0px 7px rgba(0,0,0,.6);-webkit-box-shadow:0px 0px 7px rgba(0,0,0,.6);box-shadow:0 0 7px #0000004d;cursor:default}.draw-buttons-panel{height:26px;margin-bottom:5px;margin-top:10px;display:flex;padding-left:24px}.draw-panel-button:hover{background-color:#d3d3d3}.draw-panel-button{height:26px;width:26px;margin-right:5px;border-radius:3px;display:flex;align-items:center;justify-content:center}.draw-panel-button img{height:16px;width:16px}.draw-panel{border:1px dashed #ababab;border-radius:3px;margin:0 24px;width:calc(100% - 48px);height:186px;display:flex;flex-direction:column-reverse;position:relative}i{margin-left:5px;font-size:7px;padding-top:4px}.select-button{border:1px solid #dddddd;border-radius:3px;min-width:30px;padding-left:10px;padding-right:6px;display:flex;color:#444;align-items:center;margin-right:24px;margin-top:24px;height:24px}.select-button:hover{background-color:#d3d3d3}.button-panel{width:100%;display:flex;justify-content:flex-end}.preview-panel{height:80px;width:calc(100% - 48px);border:1px solid #ababab;border-radius:3px;font-size:16pt;margin:10px 24px;padding-left:12px;display:flex;align-items:center;overflow:hidden}.input-field{width:calc(100% - 24px);display:flex;margin:12px 12px 6px;padding:0 12px;align-items:center}.input-field span{min-width:100px;font-size:12px}.input-field input{border-radius:3px;padding-left:5px;height:24px;outline:none;border:1px solid lightgray;width:100%}.input-field input:hover{border-color:#454545}.current-indicator{background-color:#19478a}.other-indicator{background-color:transparent}.current{background-color:#e5e5e5}.navigation-button:hover{background-color:#dadada}.navigation-button{border-radius:3px;height:26px;min-width:50px;display:flex;flex-direction:column;color:#444;font-size:12px;font-family:Arial;margin-right:2px}.navigation-label{height:27px;width:100%;height:18px;padding:0 10px;display:flex;align-items:center;justify-content:center;white-space:nowrap}.navigation-label span{margin-top:7px}.indicator{height:3px;margin:1px 9px}.bottom-divider{height:1px;width:calc(100% - 24px);background-color:#d3d3d3;margin:5px 12px}.content-navigation{display:flex;padding:12px 12px 0}.content-container{height:100%;width:100%}.editor-navbar{display:flex;height:30px;border-top-left-radius:3px;border-top-right-radius:3px;background-color:#515c6a;-webkit-user-select:none;color:#fff;align-items:center}.editor-navbar span{font-size:13px;margin-left:10px}.close-button{height:30px;width:30px;border-radius:3px;display:flex;align-items:center;justify-content:center}.close-button:hover{background-color:#dadada}.close-button img{height:16px;width:16px;filter:brightness(0) invert(1)}.editor-container{background-color:var(--back-color);position:absolute;left:calc(50% - 182px);top:calc(50% - 175px);z-index:999;width:365px;height:350px;overflow:hidden;display:flex;flex-direction:column;-moz-box-shadow:0px 0px 7px rgba(0,0,0,.6);-webkit-box-shadow:0px 0px 7px rgb(0 0 0 / 60%);box-shadow:0 0 7px #0000004d;border:1px solid rgb(25,71,138);color:var(--text-color);font-size:12px;border-radius:5px;font-family:Arial;-webkit-user-select:none}.button{height:24px;width:80px;border-radius:3px;margin:3px 5px;display:flex;align-items:center;justify-content:center;border:1px solid transparent}.accept:active{background-color:#355da0;border-color:#264c7d}.cancel:active{background-color:#e2e2e2;border-color:#838383}.accept{background-color:#19478a;color:#fff}.accept:hover{background-color:#3d6298}.cancel{background-color:#ddd;color:#444}.cancel:hover{background-color:#d5d5d5}.setup-accept{padding:8px;display:flex;align-items:flex-end;justify-content:flex-end}\n"] }]
        }], ctorParameters: () => [{ type: i1.StiModelService }, { type: i2.StiLocalizationService }, { type: i3.StiMouseService }, { type: i4.SignatureService }, { type: i5.StiDraggableModalWindowService }, { type: i6.StiColorDialogService }], propDecorators: { canvas: [{
                type: ViewChild,
                args: ["canvas"]
            }] } });
//# sourceMappingURL=data:application/json;base64,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