import { StiInputTextItem } from "../helper/StiInputTextItem";
import StiRectangleGeometry from "../properties/StiRectangleGeometry";
import { StiLocalizationService } from "../services/sti-localization.service";
import Color from "../system/Color";
import { Enum } from "../system/Enum";
import { StiHorizontalAlignment, StiTextRangeFormat, StiLineWeight } from "../system/enums";
import { Font } from "../system/Font";
import { StiFormElement } from "./base/StiFormElement";
import { StiLabelItem } from "./text/StiLabelItem";
import { StiTextItem } from "./text/StiTextItem";
export class StiTextBoxElement extends StiFormElement {
    constructor() {
        super();
        this._linesCount = 1;
        this._maximum = 255;
        this.type = "Text_BoxElement";
        this.tooltip = "";
        this.required = false;
        this.description = new StiTextItem();
        this.font = new Font("Segoe UI", 10);
        this.text = "";
        this.backgroundColor = Color.white();
        this.textColor = Color.black();
        this.borderColor = Color.lightGray();
        this.contentAlignment = StiHorizontalAlignment.Left;
        this.isMultiline = false;
        this.password = "";
        this.isPassword = false;
        this.useRange = false;
        this.minimum = 0;
        this.rangeFormat = StiTextRangeFormat.Characters;
        this.borderWeight = StiLineWeight.Thin;
        this.mask = "";
        this.geometry = new StiRectangleGeometry(150, 40);
        this.label = new StiLabelItem(this, StiLocalizationService.get("Dialogs", "StiTextBoxControl"));
    }
    setDefaultValues() {
        this.value = "";
    }
    setTabIndex(value) {
        this.tabIndex = value++;
        return value;
    }
    getDefaultEditableContent() {
        let textItem = new StiInputTextItem(this.text, this.font, this.textColor, this.backgroundColor, this.contentAlignment);
        textItem.element = this;
        return textItem;
    }
    get maximum() {
        return this._maximum;
    }
    set maximum(value) {
        this._maximum = Math.max(1, value);
    }
    get linesCount() {
        return this._linesCount;
    }
    set linesCount(value) {
        if (value <= 1) {
            this._linesCount = 1;
        }
        else if (value >= 10) {
            this._linesCount = 10;
        }
        else {
            this._linesCount = value;
        }
    }
    get value() {
        return this.text;
    }
    set value(value) {
        this.text = value;
    }
    get eValue() {
        return this.value;
    }
    set eValue(value) {
        this.value = value;
    }
    get eFont() {
        return this.font;
    }
    set eFont(value) {
        this.font = value;
    }
    get eColor() {
        return this.textColor;
    }
    set eColor(value) {
        this.textColor = value;
    }
    get eBackgroundColor() {
        return this.backgroundColor;
    }
    set eBackgroundColor(value) {
        this.backgroundColor = value;
    }
    get eContentAlignment() {
        return this.contentAlignment;
    }
    set eContentAlignment(value) {
        this.contentAlignment = value;
    }
    haveContentAlignment() {
        return true;
    }
    haveVerticalAlignment() {
        return false;
    }
    getUsedFonts(usedFonts) {
        super.getUsedFonts(usedFonts);
        usedFonts.add(this.font.family, this.arrayOfStylesToString(this.font.style), this.font.size);
        usedFonts.add(this.description.font.family, this.arrayOfStylesToString(this.description.font.style), this.description.font.size);
    }
    haveBackgoundColor() {
        return true;
    }
    htmlStyle() {
        return `${this.font.htmlStyle()};color:${this.eColor.toHtmlColor()};text-align:${Enum.getName(StiHorizontalAlignment, this.contentAlignment)}`;
    }
    saveToJsonObject(options) {
        if (this.element?.nativeElement) {
            this.heightForExport = this.element.nativeElement.offsetHeight;
        }
        let jObject = super.saveToJsonObject(options);
        jObject.addPropertyJObject("Label", this.label.saveToJsonObject(options));
        jObject.addPropertyJObject("Description", this.description.saveToJsonObject(options));
        jObject.addPropertyJObject("Font", this.font.saveToJsonObject(options));
        jObject.addPropertyString("Text", this.text);
        jObject.addPropertyString("Mask", this.mask);
        jObject.addPropertyColor("BackgroundColor", this.backgroundColor);
        jObject.addPropertyColor("TextColor", this.textColor);
        jObject.addPropertyEnum("ContentAlignment", StiHorizontalAlignment, this.contentAlignment);
        jObject.addPropertyString("Tooltip", this.tooltip);
        jObject.addPropertyBool("Required", this.required);
        jObject.addPropertyBool("IsPassword", this.isPassword);
        jObject.addPropertyBool("IsMultiline", this.isMultiline);
        jObject.addPropertyNumber("LinesCount", this.linesCount);
        jObject.addPropertyBool("UseRange", this.useRange);
        jObject.addPropertyNumber("Minimum", this.minimum);
        jObject.addPropertyNumber("Maximum", this.maximum);
        jObject.addPropertyEnum("RangeFormat", StiTextRangeFormat, this.rangeFormat);
        jObject.addPropertyColor("BorderColor", this.borderColor);
        jObject.addPropertyEnum("BorderWeight", StiLineWeight, this.borderWeight);
        if (this.element?.nativeElement) {
            jObject.addPropertyNumber("HeightForExport", this.heightForExport);
        }
        return jObject;
    }
    loadFromJsonObject(json) {
        super.loadFromJsonObject(json);
        for (let property of json.properties()) {
            switch (property.name) {
                case "ContentAlignment":
                    this.contentAlignment = Enum.parse(StiHorizontalAlignment, property.value);
                    break;
                case "BackgroundColor":
                    this.backgroundColor = new Color();
                    this.backgroundColor.deserializeFromString(property.value);
                    break;
                case "TextColor":
                    this.textColor = new Color();
                    this.textColor.deserializeFromString(property.value);
                    break;
                case "Text":
                    this.text = property.value;
                    break;
                case "Mask":
                    this.mask = property.value;
                    break;
                case "Label":
                    this.label = new StiLabelItem(this);
                    this.label.loadFromJsonObject(property.value);
                    break;
                case "Font":
                    this.font = new Font();
                    this.font.loadFromJsonObject(property.value);
                    break;
                case "Description":
                    this.description = new StiTextItem();
                    this.description.loadFromJsonObject(property.value);
                    break;
                case "Tooltip":
                    this.tooltip = property.value;
                    break;
                case "Required":
                    this.required = property.value;
                    break;
                case "IsPassword":
                    this.isPassword = property.value;
                    break;
                case "IsMultiline":
                    this.isMultiline = property.value;
                    break;
                case "LinesCount":
                    this.linesCount = property.value;
                    break;
                case "UseRange":
                    this.useRange = property.value;
                    break;
                case "Minimum":
                    this.minimum = property.value;
                    break;
                case "Maximum":
                    this.maximum = property.value;
                    break;
                case "RangeFormat":
                    this.rangeFormat = Enum.parse(StiTextRangeFormat, property.value);
                    break;
                case "BorderColor":
                    this.borderColor = new Color();
                    this.borderColor.deserializeFromString(property.value);
                    break;
                case "BorderWeight":
                    this.borderWeight = Enum.parse(StiLineWeight, property.value);
                    break;
            }
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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