import StiRectangleGeometry from "../properties/StiRectangleGeometry";
import { StiLocalizationService } from "../services/sti-localization.service";
import Color from "../system/Color";
import { StiHorizontalAlignment } from "../system/enums";
import { StiFormElement } from "./base/StiFormElement";
import { StiImageEditorResult } from "./signature/StiImageEditorResult";
import { StiLabelItem } from "./text/StiLabelItem";
import { StiTextItem } from "./text/StiTextItem";
export class StiSignatureElement extends StiFormElement {
    get value() {
        return this.currentSignatureMode == "Draw" ? this.backgroundImage : this.getTypeImage();
    }
    getTypeImage() {
        if (this.element?.nativeElement) {
            let canvas = document.createElement('canvas');
            canvas.width = this.element?.nativeElement.offsetWidth;
            canvas.height = this.element?.nativeElement.offsetHeight;
            var ctx = canvas.getContext("2d");
            ctx.font = "16pt " + this.specialFont;
            ctx.fillText(`${this.fullName} ${this.initials}`, 0, this.element?.nativeElement.offsetHeight - (this.specialFont == "Teddy Bear" ? 20 : (this.specialFont == "MADE Likes Script" ? 11 : 4)));
            return canvas.toDataURL('image/png');
        }
        return "";
    }
    constructor() {
        super();
        this.type = "SignatureElement";
        this.required = false;
        this.description = new StiTextItem();
        this.signaturePanelWidth = 185;
        this.signaturePanelHeight = 100;
        this.insertedImage = new StiImageEditorResult();
        this.insertedText = new StiTextItem();
        this.penColor = Color.black();
        this.fullName = "";
        this.initials = "";
        this.specialFont = "Teddy Bear";
        this.currentSignatureMode = "Type";
        this.canvasHeight = 120;
        this.isSigned = false;
        this.geometry = new StiRectangleGeometry(180, 145);
        this.label = new StiLabelItem(this, StiLocalizationService.get("Components", "StiElectronicSignature"));
        this.description.value = "Signature";
        this.insertedText.font.size = 20;
        this.insertedText.horizontalAlignment = StiHorizontalAlignment.Center;
    }
    saveToJsonObject(options) {
        if (this.element) {
            if (this.element.nativeElement?.offsetWidth > 0 && this.element.nativeElement?.offsetHeight > 0) {
                this.signaturePanelWidth = this.element.nativeElement?.offsetWidth;
                this.signaturePanelHeight = this.element.nativeElement?.offsetHeight;
            }
        }
        let jObject = super.saveToJsonObject(options);
        jObject.addPropertyJObject("Label", this.label.saveToJsonObject(options));
        jObject.addPropertyJObject("Description", this.description.saveToJsonObject(options));
        jObject.addPropertyNumber("SignaturePanelWidth", this.signaturePanelWidth);
        jObject.addPropertyNumber("SignaturePanelHeight", this.signaturePanelHeight);
        jObject.addPropertyColor("PenColor", this.penColor);
        jObject.addPropertyJObject("InsertedImage", this.insertedImage?.saveToJsonObject(options));
        jObject.addPropertyJObject("InsertedText", this.insertedText?.saveToJsonObject(options));
        if (!(options && options["notSerializeImages"])) {
            jObject.addPropertyStringNullOrEmpty("DrawedImage", this.drawedImage?.substring(this.drawedImage.indexOf("base64,") + 7));
        }
        jObject.addPropertyStringNullOrEmpty("FullName", this.fullName);
        jObject.addPropertyStringNullOrEmpty("Initials", this.initials);
        jObject.addPropertyStringNullOrEmpty("SpecialFont", this.specialFont);
        jObject.addPropertyStringNullOrEmpty("CurrentSignatureMode", this.currentSignatureMode);
        if (!(options && options["notSerializeImages"])) {
            jObject.addPropertyStringNullOrEmpty("BackgroundImage", this.backgroundImage?.substring(this.backgroundImage.indexOf("base64,") + 7));
        }
        return jObject;
    }
    loadFromJsonObject(json) {
        super.loadFromJsonObject(json);
        for (let property of json.properties()) {
            switch (property.name) {
                case "Label":
                    this.label = new StiLabelItem(this);
                    this.label.loadFromJsonObject(property.value);
                    break;
                case "Description":
                    this.description = new StiTextItem();
                    this.description.loadFromJsonObject(property.value);
                    break;
                case "SignaturePanelWidth":
                    this.signaturePanelWidth = property.value;
                    break;
                case "PenColor":
                    this.penColor = new Color();
                    this.penColor.deserializeFromString(property.value);
                    break;
                case "SignaturePanelHeight":
                    this.signaturePanelHeight = property.value;
                    break;
                case "InsertedImage":
                    this.insertedImage = new StiImageEditorResult();
                    this.insertedImage.loadFromJsonObject(property.value);
                    break;
                case "InsertedText":
                    this.insertedText = new StiTextItem();
                    this.insertedText.loadFromJsonObject(property.value);
                    break;
                case "DrawedImage":
                    this.drawedImage = `data:image/png;base64,${property.value}`;
                    break;
                case "FullName":
                    this.fullName = property.value;
                    break;
                case "Initials":
                    this.initials = property.value;
                    break;
                case "SpecialFont":
                    this.specialFont = property.value;
                    break;
                case "CurrentSignatureMode":
                    this.currentSignatureMode = property.value;
                    break;
                case "BackgroundImage":
                    this.backgroundImage = `data:image/png;base64,${property.value}`;
                    break;
            }
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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