import { StiInputTextItem } from "../helper/StiInputTextItem";
import StiRectangleGeometry from "../properties/StiRectangleGeometry";
import { StiLocalizationService } from "../services/sti-localization.service";
import Color from "../system/Color";
import { Font } from "../system/Font";
import { StiFormElement } from "./base/StiFormElement";
import { StiLabelItem } from "./text/StiLabelItem";
import { StiTextItem } from "./text/StiTextItem";
export class StiCheckBoxElement extends StiFormElement {
    constructor() {
        super();
        this.type = "Check_BoxElement";
        this.tooltip = "";
        this.required = false;
        this.description = new StiTextItem();
        this.text = new StiTextItem(StiLocalizationService.get("Components", "StiCheckBox"), new Font("Segoe UI", 9));
        this.isChecked = false;
        this.backgroundColor = Color.white();
        this.checkColor = Color.black();
        this.borderColor = Color.lightGray();
        this.initialState = false;
        this.geometry = new StiRectangleGeometry(100, 40);
        this.label = new StiLabelItem(this, StiLocalizationService.get("Components", "StiCheckBox"));
    }
    setDefaultValues() {
        this.isChecked = false;
    }
    setTabIndex(value) {
        this.tabIndex = value++;
        return value;
    }
    getUsedFonts(usedFonts) {
        super.getUsedFonts(usedFonts);
        usedFonts.add(this.text.font.family, this.arrayOfStylesToString(this.text.font.style), this.text.font.size);
        usedFonts.add(this.description.font.family, this.arrayOfStylesToString(this.description.font.style), this.description.font.size);
    }
    get value() {
        return this.isChecked;
    }
    getDefaultEditableContent() {
        let textItem = new StiInputTextItem("", this.text.font, this.text.color);
        textItem.element = this;
        return textItem;
    }
    setProperty(propName, value, oldObject) {
        super.setProperty(propName, value, oldObject);
        if (propName == "innerText.backgroundColor" &&
            (oldObject === null || this.text.backgroundColor.equals(oldObject))) {
            this.text.backgroundColor = value.clone();
        }
        else if (propName == "innerText.borderColor" && (oldObject === null || this.borderColor.equals(oldObject))) {
            this.borderColor = value.clone();
        }
        else if (propName.startsWith("innerText.font") || propName == "innerText.color") {
            this.text.setProperty(propName.replace("innerText.", ""), value.clone ? value.clone() : value, oldObject);
        }
    }
    saveToJsonObject(options) {
        let jObject = super.saveToJsonObject(options);
        jObject.addPropertyJObject("Label", this.label.saveToJsonObject(options));
        jObject.addPropertyJObject("Description", this.description.saveToJsonObject(options));
        jObject.addPropertyString("Tooltip", this.tooltip);
        jObject.addPropertyBool("Required", this.required);
        jObject.addPropertyJObject("Text", this.text.saveToJsonObject(options));
        jObject.addPropertyBool("IsChecked", this.isChecked);
        jObject.addPropertyColor("BackgroundColor", this.backgroundColor);
        jObject.addPropertyColor("CheckColor", this.checkColor);
        jObject.addPropertyColor("BorderColor", this.borderColor);
        return jObject;
    }
    loadFromJsonObject(json) {
        super.loadFromJsonObject(json);
        for (let property of json.properties()) {
            switch (property.name) {
                case "BackgroundColor":
                    this.backgroundColor = new Color();
                    this.backgroundColor.deserializeFromString(property.value);
                    break;
                case "Label":
                    this.label = new StiLabelItem(this);
                    this.label.loadFromJsonObject(property.value);
                    break;
                case "Text":
                    this.text = new StiTextItem();
                    this.text.loadFromJsonObject(property.value);
                    break;
                case "Description":
                    this.description = new StiTextItem();
                    this.description.loadFromJsonObject(property.value);
                    break;
                case "Tooltip":
                    this.tooltip = property.value;
                    break;
                case "Required":
                    this.required = property.value;
                    break;
                case "IsChecked":
                    this.isChecked = property.value;
                    break;
                case "BorderColor":
                    this.borderColor = new Color();
                    this.borderColor.deserializeFromString(property.value);
                    break;
                case "CheckColor":
                    this.checkColor = new Color();
                    this.checkColor.deserializeFromString(property.value);
                    break;
            }
        }
    }
    htmlStyle() {
        return `${this.text.htmlStyle()};
    )}`;
    }
}
//# sourceMappingURL=data:application/json;base64,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