import { Injectable } from '@angular/core';
import { Subject } from 'rxjs';
import * as i0 from "@angular/core";
import * as i1 from "../forms/form.service";
import * as i2 from "./helper.service";
import * as i3 from "./model.service";
import * as i4 from "./collections.service";
import * as i5 from "./controller.service";
export class ExportService {
    constructor(formService, helper, model, collections, controller) {
        this.formService = formService;
        this.helper = helper;
        this.model = model;
        this.collections = collections;
        this.controller = controller;
        this.exportSettings = {};
        this.sendMail = false;
        this.subject = new Subject();
        controller.getActionMessage().subscribe((message) => {
            switch (message.action) {
                case 'Export':
                    this.export(message.data.format);
                    break;
                case 'SendEmail':
                    this.export(message.data.format, false, true);
                    break;
            }
        });
    }
    getMessage() {
        return this.subject.asObservable();
    }
    export(format, update = false, sendMail = false) {
        if (!this.helper.checkCloudAuthorization('export')) {
            return;
        }
        this.sendMail = sendMail;
        this.format = format;
        this.exportSettings = this.getDefaultExportSettings(this.format);
        if ((this.model.options.exports.showExportDialog && !sendMail) || (this.model.options.email.showExportDialog && sendMail)) {
            this.exportFormSettings = this.getExportSetings(update);
            if (!update) {
                this.formService.closeForm('exportForm');
                setTimeout(() => {
                    this.formService.showForm('exportForm');
                });
            }
        }
        else {
            this.postExport();
        }
    }
    postExport() {
        if (!this.sendMail) {
            this.controller.postExport(this.format, this.exportSettings);
        }
        else {
            this.subject.next('postMail');
        }
    }
    getExportSetings(update) {
        let result;
        this.getDefaultSettings(update);
        switch (this.format) {
            case 'Document':
                result = { components: this.getComponents(['SaveReportMdc', 'SaveReportMdz', 'SaveReportMdx', 'SaveReportPassword']) };
                break;
            case 'Pdf':
                result = {
                    components: this.getComponents(['PageRangeGroup', 'PageRangeAll', 'PageRangeCurrentPage', 'PageRangePages', 'PageRangePagesText', 'SettingsGroup', 'ImageResolution', 'ImageCompressionMethod',
                        'AllowEditable', 'ImageQuality', /*'StandardPdfFonts',*/ 'EmbeddedFonts', /*'UseUnicode', 'Compressed',*/ 'ExportRtfTextAsImage', 'PdfACompliance', 'UseDigitalSignature', 'DocumentSecurityButton', 'DigitalSignatureButton',
                        'OpenAfterExport', 'PasswordInputUser', 'PasswordInputOwner', 'PrintDocument', 'ModifyContents', 'CopyTextAndGraphics',
                        'AddOrModifyTextAnnotations', 'KeyLength', 'GetCertificateFromCryptoUI', 'SubjectNameString']),
                    openAfterExport: true
                };
                break;
            case 'Xps':
                result = {
                    components: this.getComponents(['PageRangeGroup', 'PageRangeAll', 'PageRangeCurrentPage', 'PageRangePages', 'PageRangePagesText', 'SettingsGroup', 'ImageResolution', 'ImageQuality', 'OpenAfterExport',
                        'ExportRtfTextAsImage']),
                    openAfterExport: true
                };
                break;
            case 'PowerPoint':
                result = { components: this.getComponents(['PageRangeGroup', 'PageRangeAll', 'PageRangeCurrentPage', 'PageRangePages', 'PageRangePagesText', 'SettingsGroup', 'ImageResolution', 'ImageQuality']) };
                break;
            case 'Html':
                result = {
                    components: this.getComponents(['HtmlType', 'Zoom', 'ImageFormatForHtml', 'ExportMode', 'UseEmbeddedImages', 'AddPageBreaks']),
                    openAfterExport: true
                };
                break;
            case 'Html5':
                result = {
                    components: this.getComponents(['PageRangeGroup', 'PageRangeAll', 'PageRangeCurrentPage', 'PageRangePages', 'PageRangePagesText', 'SettingsGroup', 'HtmlType', 'ImageFormatForHtml', 'ImageResolution',
                        'ImageQuality', 'ContinuousPages', 'OpenAfterExport']),
                    openAfterExport: true
                };
                break;
            case 'Mht':
                result = {
                    components: this.getComponents(['PageRangeGroup', 'PageRangeAll', 'PageRangeCurrentPage', 'PageRangePages', 'PageRangePagesText', 'SettingsGroup', 'HtmlType', 'Zoom', 'ImageFormatForHtml',
                        'ExportMode', 'AddPageBreaks'])
                };
                break;
            case 'Text':
                result = {
                    components: this.getComponents(['PageRangeGroup', 'PageRangeAll', 'PageRangeCurrentPage', 'PageRangePages', 'PageRangePagesText', 'SettingsGroup', 'KillSpaceLines',
                        'PutFeedPageCode', 'DrawBorder', 'CutLongLines', 'BorderType', 'ZoomX', 'ZoomY', 'EncodingTextOrCsvFile'])
                };
                break;
            case 'Rtf':
                result = {
                    components: this.getComponents(['PageRangeGroup', 'PageRangeAll', 'PageRangeCurrentPage', 'PageRangePages', 'PageRangePagesText', 'SettingsGroup', 'ImageResolution',
                        'ImageQuality', 'ExportModeRtf', 'UsePageHeadersAndFooters', 'RemoveEmptySpaceAtBottom'])
                };
                break;
            case 'Word':
                result = {
                    components: this.getComponents(['PageRangeGroup', 'PageRangeAll', 'PageRangeCurrentPage', 'PageRangePages', 'PageRangePagesText', 'SettingsGroup', 'ImageResolution',
                        'ImageQuality', 'UsePageHeadersAndFooters', 'RemoveEmptySpaceAtBottom'])
                };
                break;
            case 'Odt':
                result = {
                    components: this.getComponents(['PageRangeGroup', 'PageRangeAll', 'PageRangeCurrentPage', 'PageRangePages', 'PageRangePagesText', 'SettingsGroup', 'ImageResolution',
                        'ImageQuality', 'RemoveEmptySpaceAtBottom'])
                };
                break;
            case 'ExcelBiff':
            case 'Excel':
                result = {
                    components: this.getComponents(['PageRangeGroup', 'PageRangeAll', 'PageRangeCurrentPage', 'PageRangePages', 'PageRangePagesText', 'SettingsGroup', 'ExcelType', 'ImageResolution',
                        'ImageQuality', 'DataExportMode', 'ExportObjectFormatting', 'UseOnePageHeaderAndFooter', 'ExportEachPageToSheet', 'ExportPageBreaks'])
                };
                break;
            case 'ExcelXml':
                result = {
                    components: this.getComponents(['PageRangeGroup', 'PageRangeAll', 'PageRangeCurrentPage', 'PageRangePages', 'PageRangePagesText', 'SettingsGroup', 'ExcelType'])
                };
                break;
            case 'Ods':
                result = {
                    components: this.getComponents(['PageRangeGroup', 'PageRangeAll', 'PageRangeCurrentPage', 'PageRangePages', 'PageRangePagesText', 'SettingsGroup', 'ImageResolution',
                        'ImageQuality'])
                };
                break;
            case 'Csv':
                result = {
                    components: this.getComponents(['PageRangeGroup', 'PageRangeAll', 'PageRangeCurrentPage', 'PageRangePages', 'PageRangePagesText', 'SettingsGroup', 'DataType', 'EncodingTextOrCsvFile',
                        'Separator', 'DataExportMode', 'SkipColumnHeaders'])
                };
                break;
            case 'Dbf':
                result = {
                    components: this.getComponents(['PageRangeGroup', 'PageRangeAll', 'PageRangeCurrentPage', 'PageRangePages', 'PageRangePagesText', 'SettingsGroup', 'DataType', 'EncodingDbfFile'])
                };
                break;
            case 'Dif':
                result = {
                    components: this.getComponents(['PageRangeGroup', 'PageRangeAll', 'PageRangeCurrentPage', 'PageRangePages', 'PageRangePagesText', 'SettingsGroup', 'DataType', 'ExportDataOnly',
                        'UseDefaultSystemEncoding', 'EncodingDifFile'])
                };
                break;
            case 'Sylk':
                result = {
                    components: this.getComponents(['PageRangeGroup', 'PageRangeAll', 'PageRangeCurrentPage', 'PageRangePages', 'PageRangePagesText', 'SettingsGroup', 'DataType', 'ExportDataOnly',
                        'UseDefaultSystemEncoding', 'EncodingDifFile'])
                };
                break;
            case 'Json':
                result = {
                    components: this.getComponents(['PageRangeGroup', 'PageRangeAll', 'PageRangeCurrentPage', 'PageRangePages', 'PageRangePagesText', 'SettingsGroup', 'DataType', 'DataExportMode'])
                };
                break;
            case 'Xml':
                result = {
                    components: this.getComponents(['PageRangeGroup', 'PageRangeAll', 'PageRangeCurrentPage', 'PageRangePages', 'PageRangePagesText', 'SettingsGroup', 'DataType'])
                };
                break;
            case 'ImageBmp':
            case 'ImageGif':
            case 'ImageJpeg':
            case 'ImagePcx':
            case 'ImagePng':
            case 'ImageTiff':
            case 'ImageEmf':
            case 'ImageSvg':
            case 'ImageSvgz':
                result = {
                    components: this.getComponents(['PageRangeGroup', 'PageRangeAll', 'PageRangeCurrentPage', 'PageRangePages', 'PageRangePagesText', 'SettingsGroup', 'ImageType', 'ImageZoom', 'ImageResolution',
                        'ImageFormat', 'DitheringType', 'TiffCompressionScheme', 'CompressToArchive', 'CutEdges'])
                };
                break;
        }
        result.groups = this.getGroups(this.format);
        result.update = update;
        return result;
    }
    getComponents(names) {
        const mrgn = '8px';
        const components = [];
        names.forEach(name => {
            switch (name) {
                case 'ImageType':
                    components.push({
                        name, type: 'DropDownListForExportForm', label: this.model.loc('Type'), width: 160, tooltip: this.model.loc('TypeTooltip'),
                        margin: '2px ' + mrgn + ' 2px ' + mrgn, items: this.collections.getImageTypesItems()
                    });
                    break;
                case 'DataType':
                    components.push({
                        name, type: 'DropDownListForExportForm', label: this.model.loc('Type'), width: 160, tooltip: this.model.loc('TypeTooltip'),
                        margin: '2px ' + mrgn + ' 2px ' + mrgn, items: this.collections.getDataTypesItems()
                    });
                    break;
                case 'ExcelType':
                    components.push({
                        name, type: 'DropDownListForExportForm', label: this.model.loc('Type'), width: 160, tooltip: this.model.loc('TypeTooltip'),
                        margin: '2px ' + mrgn + ' 2px ' + mrgn, items: this.collections.getExcelTypesItems()
                    });
                    break;
                case 'HtmlType':
                    components.push({
                        name, type: 'DropDownListForExportForm', label: this.model.loc('Type'), width: 160, tooltip: this.model.loc('TypeTooltip'),
                        margin: '2px ' + mrgn + ' 2px ' + mrgn, items: this.collections.getHtmlTypesItems()
                    });
                    break;
                case 'Zoom':
                    components.push({
                        name, type: 'DropDownListForExportForm', label: this.model.loc('ZoomHtml'), width: 160, tooltip: this.model.loc('ZoomHtmlTooltip'),
                        margin: '2px ' + mrgn + ' 2px ' + mrgn, items: this.collections.getZoomItems()
                    });
                    break;
                case 'ImageZoom':
                    components.push({
                        name, type: 'DropDownListForExportForm', label: this.model.loc('ZoomHtml'), width: 160, tooltip: this.model.loc('ZoomHtmlTooltip'),
                        margin: '2px ' + mrgn + ' 2px ' + mrgn, items: this.collections.getZoomItems()
                    });
                    break;
                case 'ImageFormatForHtml':
                    components.push({
                        name, type: 'DropDownListForExportForm', label: this.model.loc('ImageFormatForHtml'), width: 160, tooltip: this.model.loc('ImageFormatForHtmlTooltip'),
                        margin: '2px ' + mrgn + ' 2px ' + mrgn, items: this.collections.getImageFormatForHtmlItems()
                    });
                    break;
                case 'ExportMode':
                    components.push({
                        name, type: 'DropDownListForExportForm', label: this.model.loc('ExportMode'), width: 160, tooltip: this.model.loc('ExportModeTooltip'),
                        margin: '2px ' + mrgn + ' 2px ' + mrgn, items: this.collections.getExportModeItems()
                    });
                    break;
                case 'CompressToArchive':
                    components.push({
                        name, type: 'CheckBox', caption: this.model.loc('CompressToArchive'), tooltip: this.model.loc('CompressToArchiveTooltip'),
                        margin: '4px ' + mrgn + ' 4px ' + mrgn
                    });
                    break;
                case 'UseEmbeddedImages':
                    components.push({
                        name, type: 'CheckBox', caption: this.model.loc('EmbeddedImageData'), tooltip: this.model.loc('EmbeddedImageDataTooltip'),
                        margin: '4px ' + mrgn + ' 4px ' + mrgn
                    });
                    break;
                case 'AddPageBreaks':
                    components.push({
                        name, type: 'CheckBox', caption: this.model.loc('AddPageBreaks'), tooltip: this.model.loc('AddPageBreaksTooltip'),
                        margin: '4px ' + mrgn + ' 4px ' + mrgn
                    });
                    break;
                case 'ImageResolution':
                    components.push({
                        name, type: 'DropDownListForExportForm', label: this.model.loc('ImageResolution'), width: 160, tooltip: this.model.loc('ImageResolutionTooltip'),
                        margin: '2px 4px 2px ' + mrgn, items: this.collections.getImageResolutionItems()
                    });
                    break;
                case 'ImageCompressionMethod':
                    components.push({
                        name, type: 'DropDownListForExportForm', label: this.model.loc('ImageCompressionMethod'), width: 160, tooltip: this.model.loc('ImageCompressionMethodTooltip'),
                        margin: '2px ' + mrgn + ' 2px ' + mrgn, items: this.collections.getImageCompressionMethodItems()
                    });
                    break;
                case 'AllowEditable':
                    components.push({
                        name, type: 'DropDownListForExportForm', label: this.model.loc('AllowEditable'), width: 160, tooltip: this.model.loc('AllowEditableTooltip'),
                        margin: '2px ' + mrgn + ' 2px ' + mrgn, items: this.collections.getAllowEditableItems()
                    });
                    break;
                case 'ImageQuality':
                    components.push({
                        name, type: 'DropDownListForExportForm', label: this.model.loc('ImageQuality'), width: 160, tooltip: this.model.loc('ImageQualityTooltip'),
                        margin: '2px 4px 2px ' + mrgn, items: this.collections.getImageQualityItems()
                    });
                    break;
                case 'ContinuousPages':
                    components.push({
                        name, type: 'CheckBox', caption: this.model.loc('ContinuousPages'), tooltip: this.model.loc('ContinuousPagesTooltip'),
                        margin: '4px ' + mrgn + ' 4px ' + mrgn
                    });
                    break;
                case 'StandardPdfFonts':
                    components.push({
                        name, type: 'CheckBox', caption: this.model.loc('StandardPDFFonts'), tooltip: this.model.loc('StandardPDFFontsTooltip'),
                        margin: '4px ' + mrgn + ' 4px ' + mrgn
                    });
                    break;
                case 'EmbeddedFonts':
                    components.push({
                        name, type: 'CheckBox', caption: this.model.loc('EmbeddedFonts'), tooltip: this.model.loc('EmbeddedFontsTooltip'),
                        margin: '4px ' + mrgn + ' 4px ' + mrgn
                    });
                    break;
                case 'UseUnicode':
                    components.push({
                        name, type: 'CheckBox', caption: this.model.loc('UseUnicode'), tooltip: this.model.loc('UseUnicodeTooltip'),
                        margin: '4px ' + mrgn + ' 4px ' + mrgn
                    });
                    break;
                case 'Compressed':
                    components.push({
                        name, type: 'CheckBox', caption: this.model.loc('Compressed'), tooltip: this.model.loc('CompressedTooltip'),
                        margin: '4px ' + mrgn + ' 4px ' + mrgn
                    });
                    break;
                case 'ExportRtfTextAsImage':
                    components.push({
                        name, type: 'CheckBox', caption: this.model.loc('ExportRtfTextAsImage'), tooltip: this.model.loc('ExportRtfTextAsImageTooltip'),
                        margin: '4px ' + mrgn + ' 4px ' + mrgn
                    });
                    break;
                case 'PdfACompliance':
                    components.push({
                        name, type: 'CheckBox', caption: this.model.loc('PdfACompliance'), tooltip: this.model.loc('PdfAComplianceTooltip'),
                        margin: '4px ' + mrgn + ' 4px ' + mrgn
                    });
                    break;
                case 'KillSpaceLines':
                    components.push({
                        name, type: 'CheckBox', caption: this.model.loc('KillSpaceLines'), tooltip: this.model.loc('KillSpaceLinesTooltip'),
                        margin: '4px ' + mrgn + ' 4px ' + mrgn
                    });
                    break;
                case 'PutFeedPageCode':
                    components.push({
                        name, type: 'CheckBox', caption: this.model.loc('PutFeedPageCode'), tooltip: this.model.loc('PutFeedPageCodeTooltip'),
                        margin: '4px ' + mrgn + ' 4px ' + mrgn
                    });
                    break;
                case 'DrawBorder':
                    components.push({
                        name, type: 'CheckBox', caption: this.model.loc('DrawBorder'), tooltip: this.model.loc('DrawBorderTooltip'),
                        margin: '4px ' + mrgn + ' 4px ' + mrgn
                    });
                    break;
                case 'CutLongLines':
                    components.push({
                        name, type: 'CheckBox', caption: this.model.loc('CutLongLines'), tooltip: this.model.loc('CutLongLinesTooltip'),
                        margin: '4px ' + mrgn + ' 4px ' + mrgn
                    });
                    break;
                case 'BorderType':
                    components.push({
                        name, type: 'DropDownListForExportForm', label: this.model.loc('BorderType'), width: 160, tooltip: this.model.loc('BorderTypeTooltip'),
                        margin: '2px ' + mrgn + ' 2px ' + mrgn, items: this.collections.getBorderTypeItems()
                    });
                    break;
                case 'ZoomX':
                    components.push({
                        name, type: 'DropDownListForExportForm', label: this.model.loc('ZoomXY').replace(':', '') + ' X ', width: 160, tooltip: this.model.loc('ZoomXYTooltip'),
                        margin: '2px ' + mrgn + ' 2px ' + mrgn, items: this.collections.getZoomItems()
                    });
                    break;
                case 'ZoomY':
                    components.push({
                        name, type: 'DropDownListForExportForm', label: this.model.loc('ZoomXY').replace(':', '') + ' Y ', width: 160, tooltip: this.model.loc('ZoomXYTooltip'),
                        margin: '2px ' + mrgn + ' 2px ' + mrgn, items: this.collections.getZoomItems()
                    });
                    break;
                case 'EncodingTextOrCsvFile':
                    components.push({
                        name, type: 'DropDownListForExportForm', label: this.model.loc('EncodingData'), width: 160, tooltip: this.model.loc('EncodingDataTooltip'),
                        margin: '2px ' + mrgn + ' 2px ' + mrgn, items: this.collections.getEncodingDataItems()
                    });
                    break;
                case 'ImageFormat':
                    components.push({
                        name, type: 'DropDownListForExportForm', label: this.model.loc('ImageFormat'), width: 160, tooltip: this.model.loc('ImageFormatTooltip'),
                        margin: '2px ' + mrgn + ' 2px ' + mrgn, items: this.collections.getImageFormatItems()
                    });
                    break;
                case 'DitheringType':
                    components.push({
                        name, type: 'DropDownListForExportForm', label: this.model.loc('MonochromeDitheringType'), width: 160, tooltip: this.model.loc('MonochromeDitheringTypeTooltip'),
                        margin: '2px ' + mrgn + ' 2px ' + mrgn, items: this.collections.getMonochromeDitheringTypeItems()
                    });
                    break;
                case 'TiffCompressionScheme':
                    components.push({
                        name, type: 'DropDownListForExportForm', label: this.model.loc('TiffCompressionScheme'), width: 160, tooltip: this.model.loc('TiffCompressionSchemeTooltip'),
                        margin: '2px ' + mrgn + ' 2px ' + mrgn, items: this.collections.getTiffCompressionSchemeItems()
                    });
                    break;
                case 'CutEdges':
                    components.push({
                        name, type: 'CheckBox', caption: this.model.loc('CutEdges'), tooltip: this.model.loc('CutEdgesTooltip'),
                        margin: '4px ' + mrgn + ' 4px ' + mrgn
                    });
                    break;
                case 'MultipleFiles':
                    components.push({
                        name, type: 'CheckBox', caption: this.model.loc('MultipleFiles'), tooltip: this.model.loc('MultipleFilesTooltip'),
                        margin: '4px ' + mrgn + ' 4px ' + mrgn
                    });
                    break;
                case 'ExportDataOnly':
                    components.push({
                        name, type: 'CheckBox', caption: this.model.loc('ExportDataOnly'), tooltip: this.model.loc('ExportDataOnlyTooltip'),
                        margin: '4px ' + mrgn + ' 4px ' + mrgn
                    });
                    break;
                case 'UseDefaultSystemEncoding':
                    components.push({
                        name, type: 'CheckBox', caption: this.model.loc('UseDefaultSystemEncoding'), tooltip: this.model.loc('UseDefaultSystemEncodingTooltip'),
                        margin: '4px ' + mrgn + ' 4px ' + mrgn
                    });
                    break;
                case 'EncodingDifFile':
                    components.push({
                        name, type: 'DropDownListForExportForm', label: this.model.loc('EncodingDifFile'), width: 160, tooltip: this.model.loc('EncodingDifFileTooltip'),
                        margin: '2px ' + mrgn + ' 2px ' + mrgn, items: this.collections.getEncodingDifFileItems()
                    });
                    break;
                case 'ExportModeRtf':
                    components.push({
                        name, type: 'DropDownListForExportForm', label: this.model.loc('ExportModeRtf'), width: 160, tooltip: this.model.loc('ExportModeRtfTooltip'),
                        margin: '2px ' + mrgn + ' 2px ' + mrgn, items: this.collections.getExportModeRtfItems()
                    });
                    break;
                case 'UsePageHeadersAndFooters':
                    components.push({
                        name, type: 'CheckBox', caption: this.model.loc('UsePageHeadersFooters'), tooltip: this.model.loc('UsePageHeadersFootersTooltip'),
                        margin: '4px ' + mrgn + ' 4px ' + mrgn
                    });
                    break;
                case 'RemoveEmptySpaceAtBottom':
                    components.push({
                        name, type: 'CheckBox', caption: this.model.loc('RemoveEmptySpace'), tooltip: this.model.loc('RemoveEmptySpaceTooltip'),
                        margin: '4px ' + mrgn + ' 4px ' + mrgn
                    });
                    break;
                case 'Separator':
                    components.push({
                        name, type: 'TextBox', label: this.model.loc('Separator'), tooltip: this.model.loc('SeparatorTooltip'), width: 160,
                        margin: '2px ' + mrgn + ' 2px ' + mrgn
                    });
                    break;
                case 'DataExportMode':
                    components.push({
                        name, type: 'DropDownListForExportForm', label: this.model.loc('BandsFilter'), width: 160, tooltip: this.model.loc('BandsFilterTooltip'),
                        margin: '2px ' + mrgn + ' 2px ' + mrgn, items: this.collections.getDataExportModeItems()
                    });
                    break;
                case 'SkipColumnHeaders':
                    components.push({
                        name, type: 'CheckBox', caption: this.model.loc('SkipColumnHeaders'), tooltip: this.model.loc('SkipColumnHeadersTooltip'),
                        margin: '4px ' + mrgn + ' 4px ' + mrgn
                    });
                    break;
                case 'ExportObjectFormatting':
                    components.push({
                        name, type: 'CheckBox', caption: this.model.loc('ExportObjectFormatting'), tooltip: this.model.loc('ExportObjectFormattingTooltip'),
                        margin: '4px ' + mrgn + ' 4px ' + mrgn
                    });
                    break;
                case 'UseOnePageHeaderAndFooter':
                    components.push({
                        name, type: 'CheckBox', caption: this.model.loc('UseOnePageHeaderFooter'), tooltip: this.model.loc('UseOnePageHeaderFooterTooltip'),
                        margin: '4px ' + mrgn + ' 4px ' + mrgn
                    });
                    break;
                case 'ExportEachPageToSheet':
                    components.push({
                        name, type: 'CheckBox', caption: this.model.loc('ExportEachPageToSheet'), tooltip: this.model.loc('ExportEachPageToSheetTooltip'),
                        margin: '4px ' + mrgn + ' 4px ' + mrgn
                    });
                    break;
                case 'ExportPageBreaks':
                    components.push({
                        name, type: 'CheckBox', caption: this.model.loc('ExportPageBreaks'), tooltip: this.model.loc('ExportPageBreaksTooltip'),
                        margin: '4px ' + mrgn + ' 4px ' + mrgn
                    });
                    break;
                case 'EncodingDbfFile':
                    components.push({
                        name, type: 'DropDownListForExportForm', label: this.model.loc('EncodingDbfFile'), width: 160, tooltip: this.model.loc('EncodingDbfFileTooltip'),
                        margin: '2px ' + mrgn + ' 2px ' + mrgn, items: this.collections.getEncodingDbfFileItems()
                    });
                    break;
                case 'DocumentSecurityButton':
                    components.push({
                        name, type: 'SmallButton', caption: this.model.loc('DocumentSecurityButton'), width: 160,
                        margin: '2px ' + mrgn + ' 2px ' + mrgn, items: this.collections.getEncodingDbfFileItems()
                    });
                    break;
                case 'UseDigitalSignature':
                    components.push({
                        name, caption: this.model.loc('DigitalSignatureButton'), type: 'DropDownListForExportForm', tooltip: this.model.loc('UseDigitalSignatureTooltip'), width: 160,
                        items: this.collections.getPdfSecurityCertificatesItems(), margin: '4px ' + mrgn + ' 4px ' + mrgn
                    });
                    break;
                default:
                    break;
            }
        });
        return components;
    }
    getGroups(format) {
        const openingGroupsStr = this.helper.getCookie('StimulsoftWebViewerExportSettingsOpeningGroups');
        const openingGroups = openingGroupsStr ? JSON.parse(openingGroupsStr) : null;
        const pageRangeAllIsDisabled = format.indexOf('Image') === 0 && format !== 'ImageTiff';
        return {
            savingReportGroup: { visible: format === 'Document', opened: openingGroups ? openingGroups.SavingReportGroup : true },
            pageRangeGroup: { visible: format !== 'Document', opened: openingGroups ? openingGroups.PageRangeGroup : true, pageRangeAllIsDisabled },
            settingsGroup: { visible: format !== 'Document', opened: openingGroups ? openingGroups.SettingsGroup : false }
        };
    }
    getStoredSettings(format) {
        let exportSettingsStr = this.model.options.exports.storeExportSettings ? this.helper.getCookie('StimulsoftWebViewerExportSettings' + this.getCommonExportFormat(format)) : null;
        return exportSettingsStr != null ? JSON.parse(exportSettingsStr) : null;
    }
    getDefaultSettings(update) {
        let settings;
        const exportSettings = this.getStoredSettings();
        if (exportSettings != null) {
            let exportFormat = exportSettings.ImageType || exportSettings.DataType || exportSettings.HtmlType;
            delete exportSettings.ExcelType;
            this.format = !update && exportFormat ? (exportSettings.ImageType ? 'Image' + exportFormat : exportFormat) : this.format;
            const defSettings = this.getDefaultExportSettings(this.format);
            const resultSettings = {};
            Object.keys(defSettings).forEach(key => resultSettings[key] = exportSettings[key] || defSettings[key]);
            Object.keys(exportSettings).forEach(key => resultSettings[key] = exportSettings[key] || defSettings[key]);
            if (this.model.options.exports.openAfterExport === true || this.model.options.exports.openAfterExport === false) {
                resultSettings.OpenAfterExport = this.model.options.exports.openAfterExport;
            }
            if (!(this.format === 'Pdf' || this.format === 'Html')) {
                resultSettings.OpenAfterExport = false;
            }
            settings = resultSettings;
        }
        else {
            settings = this.getDefaultExportSettings(this.format);
        }
        this.updateTypes(settings);
        this.exportSettings = this.helper.copyObject(settings);
    }
    updateTypes(defaultExportSettings) {
        const types = ['ImageType', 'DataType', 'ExcelType', 'HtmlType'];
        types.forEach((propertyName) => {
            if (defaultExportSettings[propertyName]) {
                switch (propertyName) {
                    case 'ImageType':
                        if (!this.model.options.exports.showExportToImageBmp && defaultExportSettings[propertyName] === 'Bmp') {
                            defaultExportSettings[propertyName] = 'Gif';
                        }
                        if (!this.model.options.exports.showExportToImageGif && defaultExportSettings[propertyName] === 'Gif') {
                            defaultExportSettings[propertyName] = 'Jpeg';
                        }
                        if (!this.model.options.exports.showExportToImageJpeg && defaultExportSettings[propertyName] === 'Jpeg') {
                            defaultExportSettings[propertyName] = 'Pcx';
                        }
                        if (!this.model.options.exports.showExportToImagePcx && defaultExportSettings[propertyName] === 'Pcx') {
                            defaultExportSettings[propertyName] = 'Png';
                        }
                        if (!this.model.options.exports.showExportToImagePng && defaultExportSettings[propertyName] === 'Png') {
                            defaultExportSettings[propertyName] = 'Tiff';
                        }
                        if (!this.model.options.exports.showExportToImageTiff && defaultExportSettings[propertyName] === 'Tiff') {
                            defaultExportSettings[propertyName] = 'Emf';
                        }
                        if (!this.model.options.exports.showExportToImageMetafile && defaultExportSettings[propertyName] === 'Emf') {
                            defaultExportSettings[propertyName] = 'Svg';
                        }
                        if (!this.model.options.exports.showExportToImageSvg && defaultExportSettings[propertyName] === 'Svg') {
                            defaultExportSettings[propertyName] = 'Svgz';
                        }
                        if (!this.model.options.exports.showExportToImageSvgz && defaultExportSettings[propertyName] === 'Svgz') {
                            defaultExportSettings[propertyName] = 'Bmp';
                        }
                        break;
                    case 'DataType':
                        if (!this.model.options.exports.showExportToCsv && defaultExportSettings[propertyName] === 'Csv') {
                            defaultExportSettings[propertyName] = 'Dbf';
                        }
                        if (!this.model.options.exports.showExportToDbf && defaultExportSettings[propertyName] === 'Dbf') {
                            defaultExportSettings[propertyName] = 'Xml';
                        }
                        if (!this.model.options.exports.showExportToXml && defaultExportSettings[propertyName] === 'Xml') {
                            defaultExportSettings[propertyName] = 'Dif';
                        }
                        if (!this.model.options.exports.showExportToDif && defaultExportSettings[propertyName] === 'Dif') {
                            defaultExportSettings[propertyName] = 'Sylk';
                        }
                        if (!this.model.options.exports.showExportToSylk && defaultExportSettings[propertyName] === 'Sylk') {
                            defaultExportSettings[propertyName] = 'Csv';
                        }
                        if (!this.model.options.exports.showExportToJson && defaultExportSettings[propertyName] === 'Json') {
                            defaultExportSettings[propertyName] = 'Json';
                        }
                        break;
                    case 'ExcelType':
                        if (!this.model.options.exports.showExportToExcel && defaultExportSettings[propertyName] === 'Excel') {
                            defaultExportSettings[propertyName] = 'ExcelBinary';
                        }
                        if (!this.model.options.exports.showExportToExcelBiff && defaultExportSettings[propertyName] === 'ExcelBinary') {
                            defaultExportSettings[propertyName] = 'ExcelXml';
                        }
                        if (!this.model.options.exports.showExportToExcelXml && defaultExportSettings[propertyName] === 'ExcelXml') {
                            defaultExportSettings[propertyName] = 'Excel';
                        }
                        break;
                    case 'HtmlType':
                        if (!this.model.options.exports.showExportToHtml && defaultExportSettings[propertyName] === 'Html') {
                            defaultExportSettings[propertyName] = 'Html5';
                        }
                        if (!this.model.options.exports.showExportToHtml5 && defaultExportSettings[propertyName] === 'Html5') {
                            defaultExportSettings[propertyName] = 'Mht';
                        }
                        if (!this.model.options.exports.showExportToMht && defaultExportSettings[propertyName] === 'Mht') {
                            defaultExportSettings[propertyName] = 'Html';
                        }
                        break;
                }
            }
        });
    }
    getCommonExportFormat(format) {
        format = format ?? this.format;
        if (format === 'Html' || format === 'Html5' || format === 'Mht') {
            return 'Html';
        }
        if (format === 'ExcelBiff' || format === 'Excel' || format === 'ExcelXml' || format === 'ExcelBinary') {
            return 'Excel';
        }
        if (format === 'Csv' || format === 'Dbf' || format === 'Xml' || format === 'Dif' || format === 'Sylk') {
            return 'Data';
        }
        if (format === 'ImageBmp' || format === 'ImageGif' || format === 'ImageJpeg' || format === 'ImagePcx' || format === 'ImagePng' ||
            format === 'ImageTiff' || format === 'ImageEmf' || format === 'ImageSvg' || format === 'ImageSvgz') {
            return 'Image';
        }
        return format;
    }
    getDefaultExportSettings(format, isDashboardExport = false) {
        let exportSettings = null;
        if (isDashboardExport) {
            return this.model.options.exports.defaultSettings['Dashboard' + format];
        }
        switch (format) {
            case 'Document':
                exportSettings = { Format: 'Mdc' };
                break;
            case 'Pdf':
                exportSettings = this.model.options.exports.defaultSettings['StiPdfExportSettings'];
                break;
            case 'Xps':
                exportSettings = this.model.options.exports.defaultSettings['StiXpsExportSettings'];
                break;
            case 'PowerPoint':
            case 'Ppt2007':
                exportSettings = this.model.options.exports.defaultSettings['StiPowerPointExportSettings'] || this.model.options.exports.defaultSettings['StiPpt2007ExportSettings'];
                break;
            case 'Html':
                exportSettings = this.model.options.exports.defaultSettings['StiHtmlExportSettings'];
                exportSettings.HtmlType = 'Html';
                break;
            case 'Html5':
                exportSettings = this.model.options.exports.defaultSettings['StiHtmlExportSettings'];
                exportSettings.HtmlType = 'Html5';
                break;
            case 'Mht':
                exportSettings = this.model.options.exports.defaultSettings['StiHtmlExportSettings'];
                exportSettings.HtmlType = 'Mht';
                break;
            case 'Text':
                exportSettings = this.model.options.exports.defaultSettings['StiTxtExportSettings'];
                break;
            case 'Rtf':
                exportSettings = this.model.options.exports.defaultSettings['StiRtfExportSettings'];
                break;
            case 'Word':
            case 'Word2007':
                exportSettings = this.model.options.exports.defaultSettings['StiWordExportSettings'] || this.model.options.exports.defaultSettings['StiWord2007ExportSettings'];
                break;
            case 'Odt':
                exportSettings = this.model.options.exports.defaultSettings['StiOdtExportSettings'];
                break;
            case 'ExcelBiff':
                exportSettings = this.model.options.exports.defaultSettings['StiExcelExportSettings'];
                exportSettings.ExcelType = 'ExcelBiff';
                break;
            case 'ExcelXml':
                exportSettings = this.model.options.exports.defaultSettings['StiExcelExportSettings'];
                exportSettings.ExcelType = 'ExcelXml';
                break;
            case 'Excel':
                exportSettings = this.model.options.exports.defaultSettings['StiExcelExportSettings'];
                exportSettings.ExcelType = 'Excel';
                break;
            case 'Ods':
                exportSettings = this.model.options.exports.defaultSettings['StiOdsExportSettings'];
                break;
            case 'ImageBmp':
                exportSettings = this.model.options.exports.defaultSettings['StiImageExportSettings'];
                exportSettings.ImageType = 'Bmp';
                break;
            case 'ImageGif':
                exportSettings = this.model.options.exports.defaultSettings['StiImageExportSettings'];
                exportSettings.ImageType = 'Gif';
                break;
            case 'ImageJpeg':
                exportSettings = this.model.options.exports.defaultSettings['StiImageExportSettings'];
                exportSettings.ImageType = 'Jpeg';
                break;
            case 'ImagePcx':
                exportSettings = this.model.options.exports.defaultSettings['StiImageExportSettings'];
                exportSettings.ImageType = 'Pcx';
                break;
            case 'ImagePng':
                exportSettings = this.model.options.exports.defaultSettings['StiImageExportSettings'];
                exportSettings.ImageType = 'Png';
                break;
            case 'ImageTiff':
                exportSettings = this.model.options.exports.defaultSettings['StiImageExportSettings'];
                exportSettings.ImageType = 'Tiff';
                break;
            case 'ImageSvg':
                exportSettings = this.model.options.exports.defaultSettings['StiImageExportSettings'];
                exportSettings.ImageType = 'Svg';
                break;
            case 'ImageSvgz':
                exportSettings = this.model.options.exports.defaultSettings['StiImageExportSettings'];
                exportSettings.ImageType = 'Svgz';
                break;
            case 'ImageEmf':
                exportSettings = this.model.options.exports.defaultSettings['StiImageExportSettings'];
                exportSettings.ImageType = 'Emf';
                break;
            case 'Xml':
                exportSettings = this.model.options.exports.defaultSettings['StiDataExportSettings'];
                exportSettings.DataType = 'Xml';
                break;
            case 'Csv':
                exportSettings = this.model.options.exports.defaultSettings['StiDataExportSettings'];
                exportSettings.DataType = 'Csv';
                break;
            case 'Dbf':
                exportSettings = this.model.options.exports.defaultSettings['StiDataExportSettings'];
                exportSettings.DataType = 'Dbf';
                break;
            case 'Dif':
                exportSettings = this.model.options.exports.defaultSettings['StiDataExportSettings'];
                exportSettings.DataType = 'Dif';
                break;
            case 'Sylk':
                exportSettings = this.model.options.exports.defaultSettings['StiDataExportSettings'];
                exportSettings.DataType = 'Sylk';
                break;
        }
        exportSettings.OpenAfterExport = !(this.model.options.exports.showOpenAfterExport === false);
        if (!(format === 'Pdf' || format === 'Html')) {
            exportSettings.OpenAfterExport = false;
        }
        return exportSettings;
    }
    static { this.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "18.2.14", ngImport: i0, type: ExportService, deps: [{ token: i1.FormService }, { token: i2.HelperService }, { token: i3.ModelService }, { token: i4.CollectionsService }, { token: i5.ControllerService }], target: i0.ɵɵFactoryTarget.Injectable }); }
    static { this.ɵprov = i0.ɵɵngDeclareInjectable({ minVersion: "12.0.0", version: "18.2.14", ngImport: i0, type: ExportService }); }
}
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "18.2.14", ngImport: i0, type: ExportService, decorators: [{
            type: Injectable
        }], ctorParameters: () => [{ type: i1.FormService }, { type: i2.HelperService }, { type: i3.ModelService }, { type: i4.CollectionsService }, { type: i5.ControllerService }] });
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZXhwb3J0LnNlcnZpY2UuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi8uLi8uLi9wcm9qZWN0cy9zdGltdWxzb2Z0LXZpZXdlci1hbmd1bGFyL3NyYy9saWIvc2VydmljZXMvZXhwb3J0LnNlcnZpY2UudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQUEsT0FBTyxFQUFFLFVBQVUsRUFBRSxNQUFNLGVBQWUsQ0FBQztBQU8zQyxPQUFPLEVBQUUsT0FBTyxFQUFjLE1BQU0sTUFBTSxDQUFDOzs7Ozs7O0FBRzNDLE1BQU0sT0FBTyxhQUFhO0lBU3hCLFlBQW1CLFdBQXdCLEVBQVMsTUFBcUIsRUFBUyxLQUFtQixFQUFTLFdBQStCLEVBQ3BJLFVBQTZCO1FBRG5CLGdCQUFXLEdBQVgsV0FBVyxDQUFhO1FBQVMsV0FBTSxHQUFOLE1BQU0sQ0FBZTtRQUFTLFVBQUssR0FBTCxLQUFLLENBQWM7UUFBUyxnQkFBVyxHQUFYLFdBQVcsQ0FBb0I7UUFDcEksZUFBVSxHQUFWLFVBQVUsQ0FBbUI7UUFQL0IsbUJBQWMsR0FBUSxFQUFFLENBQUM7UUFFekIsYUFBUSxHQUFHLEtBQUssQ0FBQztRQUVoQixZQUFPLEdBQUcsSUFBSSxPQUFPLEVBQU8sQ0FBQztRQUtuQyxVQUFVLENBQUMsZ0JBQWdCLEVBQUUsQ0FBQyxTQUFTLENBQUMsQ0FBQyxPQUFnQixFQUFFLEVBQUU7WUFDM0QsUUFBUSxPQUFPLENBQUMsTUFBTSxFQUFFLENBQUM7Z0JBQ3ZCLEtBQUssUUFBUTtvQkFDWCxJQUFJLENBQUMsTUFBTSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUM7b0JBQ2pDLE1BQU07Z0JBQ1IsS0FBSyxXQUFXO29CQUNkLElBQUksQ0FBQyxNQUFNLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxNQUFNLEVBQUUsS0FBSyxFQUFFLElBQUksQ0FBQyxDQUFDO29CQUM5QyxNQUFNO1lBQ1YsQ0FBQztRQUNILENBQUMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVNLFVBQVU7UUFDZixPQUFPLElBQUksQ0FBQyxPQUFPLENBQUMsWUFBWSxFQUFFLENBQUM7SUFDckMsQ0FBQztJQUVNLE1BQU0sQ0FBQyxNQUFjLEVBQUUsU0FBa0IsS0FBSyxFQUFFLFdBQW9CLEtBQUs7UUFDOUUsSUFBSSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsdUJBQXVCLENBQUMsUUFBUSxDQUFDLEVBQUUsQ0FBQztZQUNuRCxPQUFPO1FBQ1QsQ0FBQztRQUVELElBQUksQ0FBQyxRQUFRLEdBQUcsUUFBUSxDQUFDO1FBQ3pCLElBQUksQ0FBQyxNQUFNLEdBQUcsTUFBTSxDQUFDO1FBQ3JCLElBQUksQ0FBQyxjQUFjLEdBQUcsSUFBSSxDQUFDLHdCQUF3QixDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUVqRSxJQUFJLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLGdCQUFnQixJQUFJLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsZ0JBQWdCLElBQUksUUFBUSxDQUFDLEVBQUUsQ0FBQztZQUMxSCxJQUFJLENBQUMsa0JBQWtCLEdBQUcsSUFBSSxDQUFDLGdCQUFnQixDQUFDLE1BQU0sQ0FBQyxDQUFDO1lBQ3hELElBQUksQ0FBQyxNQUFNLEVBQUUsQ0FBQztnQkFDWixJQUFJLENBQUMsV0FBVyxDQUFDLFNBQVMsQ0FBQyxZQUFZLENBQUMsQ0FBQztnQkFDekMsVUFBVSxDQUFDLEdBQUcsRUFBRTtvQkFDZCxJQUFJLENBQUMsV0FBVyxDQUFDLFFBQVEsQ0FBQyxZQUFZLENBQUMsQ0FBQztnQkFDMUMsQ0FBQyxDQUFDLENBQUM7WUFDTCxDQUFDO1FBQ0gsQ0FBQzthQUFNLENBQUM7WUFDTixJQUFJLENBQUMsVUFBVSxFQUFFLENBQUM7UUFDcEIsQ0FBQztJQUNILENBQUM7SUFFTSxVQUFVO1FBQ2YsSUFBSSxDQUFDLElBQUksQ0FBQyxRQUFRLEVBQUUsQ0FBQztZQUNuQixJQUFJLENBQUMsVUFBVSxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsTUFBTSxFQUFFLElBQUksQ0FBQyxjQUFjLENBQUMsQ0FBQztRQUMvRCxDQUFDO2FBQU0sQ0FBQztZQUNOLElBQUksQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxDQUFDO1FBQ2hDLENBQUM7SUFDSCxDQUFDO0lBRUQsZ0JBQWdCLENBQUMsTUFBZTtRQUM5QixJQUFJLE1BQTBCLENBQUM7UUFFL0IsSUFBSSxDQUFDLGtCQUFrQixDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBRWhDLFFBQVEsSUFBSSxDQUFDLE1BQU0sRUFBRSxDQUFDO1lBQ3BCLEtBQUssVUFBVTtnQkFDYixNQUFNLEdBQUcsRUFBRSxVQUFVLEVBQUUsSUFBSSxDQUFDLGFBQWEsQ0FBQyxDQUFDLGVBQWUsRUFBRSxlQUFlLEVBQUUsZUFBZSxFQUFFLG9CQUFvQixDQUFDLENBQUMsRUFBRSxDQUFDO2dCQUN2SCxNQUFNO1lBRVIsS0FBSyxLQUFLO2dCQUNSLE1BQU0sR0FBRztvQkFDUCxVQUFVLEVBQUUsSUFBSSxDQUFDLGFBQWEsQ0FBQyxDQUFDLGdCQUFnQixFQUFFLGNBQWMsRUFBRSxzQkFBc0IsRUFBRSxnQkFBZ0IsRUFBRSxvQkFBb0IsRUFBRSxlQUFlLEVBQUUsaUJBQWlCLEVBQUUsd0JBQXdCO3dCQUM1TCxlQUFlLEVBQUUsY0FBYyxFQUFFLHVCQUF1QixDQUFDLGVBQWUsRUFBRSwrQkFBK0IsQ0FBQyxzQkFBc0IsRUFBRSxnQkFBZ0IsRUFBRSxxQkFBcUIsRUFBRSx3QkFBd0IsRUFBRSx3QkFBd0I7d0JBQzdOLGlCQUFpQixFQUFFLG1CQUFtQixFQUFFLG9CQUFvQixFQUFFLGVBQWUsRUFBRSxnQkFBZ0IsRUFBRSxxQkFBcUI7d0JBQ3RILDRCQUE0QixFQUFFLFdBQVcsRUFBRSw0QkFBNEIsRUFBRSxtQkFBbUIsQ0FBQyxDQUFDO29CQUNoRyxlQUFlLEVBQUUsSUFBSTtpQkFDdEIsQ0FBQztnQkFDRixNQUFNO1lBQ1IsS0FBSyxLQUFLO2dCQUNSLE1BQU0sR0FBRztvQkFDUCxVQUFVLEVBQUUsSUFBSSxDQUFDLGFBQWEsQ0FBQyxDQUFDLGdCQUFnQixFQUFFLGNBQWMsRUFBRSxzQkFBc0IsRUFBRSxnQkFBZ0IsRUFBRSxvQkFBb0IsRUFBRSxlQUFlLEVBQUUsaUJBQWlCLEVBQUUsY0FBYyxFQUFFLGlCQUFpQjt3QkFDck0sc0JBQXNCLENBQUMsQ0FBQztvQkFDMUIsZUFBZSxFQUFFLElBQUk7aUJBQ3RCLENBQUM7Z0JBQ0YsTUFBTTtZQUNSLEtBQUssWUFBWTtnQkFDZixNQUFNLEdBQUcsRUFBRSxVQUFVLEVBQUUsSUFBSSxDQUFDLGFBQWEsQ0FBQyxDQUFDLGdCQUFnQixFQUFFLGNBQWMsRUFBRSxzQkFBc0IsRUFBRSxnQkFBZ0IsRUFBRSxvQkFBb0IsRUFBRSxlQUFlLEVBQUUsaUJBQWlCLEVBQUUsY0FBYyxDQUFDLENBQUMsRUFBRSxDQUFDO2dCQUNwTSxNQUFNO1lBRVIsS0FBSyxNQUFNO2dCQUNULE1BQU0sR0FBRztvQkFDUCxVQUFVLEVBQUUsSUFBSSxDQUFDLGFBQWEsQ0FBQyxDQUFDLFVBQVUsRUFBRSxNQUFNLEVBQUUsb0JBQW9CLEVBQUUsWUFBWSxFQUFFLG1CQUFtQixFQUFFLGVBQWUsQ0FBQyxDQUFDO29CQUM5SCxlQUFlLEVBQUUsSUFBSTtpQkFDdEIsQ0FBQztnQkFDRixNQUFNO1lBRVIsS0FBSyxPQUFPO2dCQUNWLE1BQU0sR0FBRztvQkFDUCxVQUFVLEVBQUUsSUFBSSxDQUFDLGFBQWEsQ0FBQyxDQUFDLGdCQUFnQixFQUFFLGNBQWMsRUFBRSxzQkFBc0IsRUFBRSxnQkFBZ0IsRUFBRSxvQkFBb0IsRUFBRSxlQUFlLEVBQUUsVUFBVSxFQUFFLG9CQUFvQixFQUFFLGlCQUFpQjt3QkFDcE0sY0FBYyxFQUFFLGlCQUFpQixFQUFFLGlCQUFpQixDQUFDLENBQUM7b0JBQ3hELGVBQWUsRUFBRSxJQUFJO2lCQUN0QixDQUFDO2dCQUNGLE1BQU07WUFDUixLQUFLLEtBQUs7Z0JBQ1IsTUFBTSxHQUFHO29CQUNQLFVBQVUsRUFBRSxJQUFJLENBQUMsYUFBYSxDQUFDLENBQUMsZ0JBQWdCLEVBQUUsY0FBYyxFQUFFLHNCQUFzQixFQUFFLGdCQUFnQixFQUFFLG9CQUFvQixFQUFFLGVBQWUsRUFBRSxVQUFVLEVBQUUsTUFBTSxFQUFFLG9CQUFvQjt3QkFDekwsWUFBWSxFQUFFLGVBQWUsQ0FBQyxDQUFDO2lCQUNsQyxDQUFDO2dCQUNGLE1BQU07WUFDUixLQUFLLE1BQU07Z0JBQ1QsTUFBTSxHQUFHO29CQUNQLFVBQVUsRUFBRSxJQUFJLENBQUMsYUFBYSxDQUFDLENBQUMsZ0JBQWdCLEVBQUUsY0FBYyxFQUFFLHNCQUFzQixFQUFFLGdCQUFnQixFQUFFLG9CQUFvQixFQUFFLGVBQWUsRUFBRSxnQkFBZ0I7d0JBQ2pLLGlCQUFpQixFQUFFLFlBQVksRUFBRSxjQUFjLEVBQUUsWUFBWSxFQUFFLE9BQU8sRUFBRSxPQUFPLEVBQUUsdUJBQXVCLENBQUMsQ0FBQztpQkFDN0csQ0FBQztnQkFDRixNQUFNO1lBQ1IsS0FBSyxLQUFLO2dCQUNSLE1BQU0sR0FBRztvQkFDUCxVQUFVLEVBQUUsSUFBSSxDQUFDLGFBQWEsQ0FBQyxDQUFDLGdCQUFnQixFQUFFLGNBQWMsRUFBRSxzQkFBc0IsRUFBRSxnQkFBZ0IsRUFBRSxvQkFBb0IsRUFBRSxlQUFlLEVBQUUsaUJBQWlCO3dCQUNsSyxjQUFjLEVBQUUsZUFBZSxFQUFFLDBCQUEwQixFQUFFLDBCQUEwQixDQUFDLENBQUM7aUJBQzVGLENBQUM7Z0JBQ0YsTUFBTTtZQUNSLEtBQUssTUFBTTtnQkFDVCxNQUFNLEdBQUc7b0JBQ1AsVUFBVSxFQUFFLElBQUksQ0FBQyxhQUFhLENBQUMsQ0FBQyxnQkFBZ0IsRUFBRSxjQUFjLEVBQUUsc0JBQXNCLEVBQUUsZ0JBQWdCLEVBQUUsb0JBQW9CLEVBQUUsZUFBZSxFQUFFLGlCQUFpQjt3QkFDbEssY0FBYyxFQUFFLDBCQUEwQixFQUFFLDBCQUEwQixDQUFDLENBQUM7aUJBQzNFLENBQUM7Z0JBQ0YsTUFBTTtZQUNSLEtBQUssS0FBSztnQkFDUixNQUFNLEdBQUc7b0JBQ1AsVUFBVSxFQUFFLElBQUksQ0FBQyxhQUFhLENBQUMsQ0FBQyxnQkFBZ0IsRUFBRSxjQUFjLEVBQUUsc0JBQXNCLEVBQUUsZ0JBQWdCLEVBQUUsb0JBQW9CLEVBQUUsZUFBZSxFQUFFLGlCQUFpQjt3QkFDbEssY0FBYyxFQUFFLDBCQUEwQixDQUFDLENBQUM7aUJBQy9DLENBQUM7Z0JBQ0YsTUFBTTtZQUNSLEtBQUssV0FBVyxDQUFDO1lBQ2pCLEtBQUssT0FBTztnQkFDVixNQUFNLEdBQUc7b0JBQ1AsVUFBVSxFQUFFLElBQUksQ0FBQyxhQUFhLENBQUMsQ0FBQyxnQkFBZ0IsRUFBRSxjQUFjLEVBQUUsc0JBQXNCLEVBQUUsZ0JBQWdCLEVBQUUsb0JBQW9CLEVBQUUsZUFBZSxFQUFFLFdBQVcsRUFBRSxpQkFBaUI7d0JBQy9LLGNBQWMsRUFBRSxnQkFBZ0IsRUFBRSx3QkFBd0IsRUFBRSwyQkFBMkIsRUFBRSx1QkFBdUIsRUFBRSxrQkFBa0IsQ0FBQyxDQUFDO2lCQUN6SSxDQUFDO2dCQUNGLE1BQU07WUFDUixLQUFLLFVBQVU7Z0JBQ2IsTUFBTSxHQUFHO29CQUNQLFVBQVUsRUFBRSxJQUFJLENBQUMsYUFBYSxDQUFDLENBQUMsZ0JBQWdCLEVBQUUsY0FBYyxFQUFFLHNCQUFzQixFQUFFLGdCQUFnQixFQUFFLG9CQUFvQixFQUFFLGVBQWUsRUFBRSxXQUFXLENBQUMsQ0FBQztpQkFDakssQ0FBQztnQkFDRixNQUFNO1lBQ1IsS0FBSyxLQUFLO2dCQUNSLE1BQU0sR0FBRztvQkFDUCxVQUFVLEVBQUUsSUFBSSxDQUFDLGFBQWEsQ0FBQyxDQUFDLGdCQUFnQixFQUFFLGNBQWMsRUFBRSxzQkFBc0IsRUFBRSxnQkFBZ0IsRUFBRSxvQkFBb0IsRUFBRSxlQUFlLEVBQUUsaUJBQWlCO3dCQUNsSyxjQUFjLENBQUMsQ0FBQztpQkFDbkIsQ0FBQztnQkFDRixNQUFNO1lBQ1IsS0FBSyxLQUFLO2dCQUNSLE1BQU0sR0FBRztvQkFDUCxVQUFVLEVBQUUsSUFBSSxDQUFDLGFBQWEsQ0FBQyxDQUFDLGdCQUFnQixFQUFFLGNBQWMsRUFBRSxzQkFBc0IsRUFBRSxnQkFBZ0IsRUFBRSxvQkFBb0IsRUFBRSxlQUFlLEVBQUUsVUFBVSxFQUFFLHVCQUF1Qjt3QkFDcEwsV0FBVyxFQUFFLGdCQUFnQixFQUFFLG1CQUFtQixDQUFDLENBQUM7aUJBQ3ZELENBQUM7Z0JBQ0YsTUFBTTtZQUNSLEtBQUssS0FBSztnQkFDUixNQUFNLEdBQUc7b0JBQ1AsVUFBVSxFQUFFLElBQUksQ0FBQyxhQUFhLENBQUMsQ0FBQyxnQkFBZ0IsRUFBRSxjQUFjLEVBQUUsc0JBQXNCLEVBQUUsZ0JBQWdCLEVBQUUsb0JBQW9CLEVBQUUsZUFBZSxFQUFFLFVBQVUsRUFBRSxpQkFBaUIsQ0FBQyxDQUFDO2lCQUNuTCxDQUFDO2dCQUNGLE1BQU07WUFDUixLQUFLLEtBQUs7Z0JBQ1IsTUFBTSxHQUFHO29CQUNQLFVBQVUsRUFBRSxJQUFJLENBQUMsYUFBYSxDQUFDLENBQUMsZ0JBQWdCLEVBQUUsY0FBYyxFQUFFLHNCQUFzQixFQUFFLGdCQUFnQixFQUFFLG9CQUFvQixFQUFFLGVBQWUsRUFBRSxVQUFVLEVBQUUsZ0JBQWdCO3dCQUM3SywwQkFBMEIsRUFBRSxpQkFBaUIsQ0FBQyxDQUFDO2lCQUNsRCxDQUFDO2dCQUNGLE1BQU07WUFDUixLQUFLLE1BQU07Z0JBQ1QsTUFBTSxHQUFHO29CQUNQLFVBQVUsRUFBRSxJQUFJLENBQUMsYUFBYSxDQUFDLENBQUMsZ0JBQWdCLEVBQUUsY0FBYyxFQUFFLHNCQUFzQixFQUFFLGdCQUFnQixFQUFFLG9CQUFvQixFQUFFLGVBQWUsRUFBRSxVQUFVLEVBQUUsZ0JBQWdCO3dCQUM3SywwQkFBMEIsRUFBRSxpQkFBaUIsQ0FBQyxDQUFDO2lCQUNsRCxDQUFDO2dCQUNGLE1BQU07WUFDUixLQUFLLE1BQU07Z0JBQ1QsTUFBTSxHQUFHO29CQUNQLFVBQVUsRUFBRSxJQUFJLENBQUMsYUFBYSxDQUFDLENBQUMsZ0JBQWdCLEVBQUUsY0FBYyxFQUFFLHNCQUFzQixFQUFFLGdCQUFnQixFQUFFLG9CQUFvQixFQUFFLGVBQWUsRUFBRSxVQUFVLEVBQUUsZ0JBQWdCLENBQUMsQ0FBQztpQkFDbEwsQ0FBQztnQkFDRixNQUFNO1lBQ1IsS0FBSyxLQUFLO2dCQUNSLE1BQU0sR0FBRztvQkFDUCxVQUFVLEVBQUUsSUFBSSxDQUFDLGFBQWEsQ0FBQyxDQUFDLGdCQUFnQixFQUFFLGNBQWMsRUFBRSxzQkFBc0IsRUFBRSxnQkFBZ0IsRUFBRSxvQkFBb0IsRUFBRSxlQUFlLEVBQUUsVUFBVSxDQUFDLENBQUM7aUJBQ2hLLENBQUM7Z0JBQ0YsTUFBTTtZQUNSLEtBQUssVUFBVSxDQUFDO1lBQ2hCLEtBQUssVUFBVSxDQUFDO1lBQ2hCLEtBQUssV0FBVyxDQUFDO1lBQ2pCLEtBQUssVUFBVSxDQUFDO1lBQ2hCLEtBQUssVUFBVSxDQUFDO1lBQ2hCLEtBQUssV0FBVyxDQUFDO1lBQ2pCLEtBQUssVUFBVSxDQUFDO1lBQ2hCLEtBQUssVUFBVSxDQUFDO1lBQ2hCLEtBQUssV0FBVztnQkFDZCxNQUFNLEdBQUc7b0JBQ1AsVUFBVSxFQUFFLElBQUksQ0FBQyxhQUFhLENBQUMsQ0FBQyxnQkFBZ0IsRUFBRSxjQUFjLEVBQUUsc0JBQXNCLEVBQUUsZ0JBQWdCLEVBQUUsb0JBQW9CLEVBQUUsZUFBZSxFQUFFLFdBQVcsRUFBRSxXQUFXLEVBQUUsaUJBQWlCO3dCQUM1TCxhQUFhLEVBQUUsZUFBZSxFQUFFLHVCQUF1QixFQUFFLG1CQUFtQixFQUFFLFVBQVUsQ0FBQyxDQUFDO2lCQUM3RixDQUFDO2dCQUNGLE1BQU07UUFDVixDQUFDO1FBRUQsTUFBTSxDQUFDLE1BQU0sR0FBRyxJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUM1QyxNQUFNLENBQUMsTUFBTSxHQUFHLE1BQU0sQ0FBQztRQUN2QixPQUFPLE1BQU0sQ0FBQztJQUNoQixDQUFDO0lBRUQsYUFBYSxDQUFDLEtBQWU7UUFDM0IsTUFBTSxJQUFJLEdBQUcsS0FBSyxDQUFDO1FBQ25CLE1BQU0sVUFBVSxHQUFzQixFQUFFLENBQUM7UUFDekMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsRUFBRTtZQUNuQixRQUFRLElBQUksRUFBRSxDQUFDO2dCQUNiLEtBQUssV0FBVztvQkFDZCxVQUFVLENBQUMsSUFBSSxDQUFDO3dCQUNkLElBQUksRUFBRSxJQUFJLEVBQUUsMkJBQTJCLEVBQUUsS0FBSyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLE1BQU0sQ0FBQyxFQUFFLEtBQUssRUFBRSxHQUFHLEVBQUUsT0FBTyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLGFBQWEsQ0FBQzt3QkFDMUgsTUFBTSxFQUFFLE1BQU0sR0FBRyxJQUFJLEdBQUcsT0FBTyxHQUFHLElBQUksRUFBRSxLQUFLLEVBQUUsSUFBSSxDQUFDLFdBQVcsQ0FBQyxrQkFBa0IsRUFBRTtxQkFDckYsQ0FBQyxDQUFDO29CQUNILE1BQU07Z0JBRVIsS0FBSyxVQUFVO29CQUNiLFVBQVUsQ0FBQyxJQUFJLENBQUM7d0JBQ2QsSUFBSSxFQUFFLElBQUksRUFBRSwyQkFBMkIsRUFBRSxLQUFLLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsTUFBTSxDQUFDLEVBQUUsS0FBSyxFQUFFLEdBQUcsRUFBRSxPQUFPLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsYUFBYSxDQUFDO3dCQUMxSCxNQUFNLEVBQUUsTUFBTSxHQUFHLElBQUksR0FBRyxPQUFPLEdBQUcsSUFBSSxFQUFFLEtBQUssRUFBRSxJQUFJLENBQUMsV0FBVyxDQUFDLGlCQUFpQixFQUFFO3FCQUNwRixDQUFDLENBQUM7b0JBQ0gsTUFBTTtnQkFFUixLQUFLLFdBQVc7b0JBQ2QsVUFBVSxDQUFDLElBQUksQ0FBQzt3QkFDZCxJQUFJLEVBQUUsSUFBSSxFQUFFLDJCQUEyQixFQUFFLEtBQUssRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxNQUFNLENBQUMsRUFBRSxLQUFLLEVBQUUsR0FBRyxFQUFFLE9BQU8sRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxhQUFhLENBQUM7d0JBQzFILE1BQU0sRUFBRSxNQUFNLEdBQUcsSUFBSSxHQUFHLE9BQU8sR0FBRyxJQUFJLEVBQUUsS0FBSyxFQUFFLElBQUksQ0FBQyxXQUFXLENBQUMsa0JBQWtCLEVBQUU7cUJBQ3JGLENBQUMsQ0FBQztvQkFDSCxNQUFNO2dCQUVSLEtBQUssVUFBVTtvQkFDYixVQUFVLENBQUMsSUFBSSxDQUFDO3dCQUNkLElBQUksRUFBRSxJQUFJLEVBQUUsMkJBQTJCLEVBQUUsS0FBSyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLE1BQU0sQ0FBQyxFQUFFLEtBQUssRUFBRSxHQUFHLEVBQUUsT0FBTyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLGFBQWEsQ0FBQzt3QkFDMUgsTUFBTSxFQUFFLE1BQU0sR0FBRyxJQUFJLEdBQUcsT0FBTyxHQUFHLElBQUksRUFBRSxLQUFLLEVBQUUsSUFBSSxDQUFDLFdBQVcsQ0FBQyxpQkFBaUIsRUFBRTtxQkFDcEYsQ0FBQyxDQUFDO29CQUNILE1BQU07Z0JBRVIsS0FBSyxNQUFNO29CQUNULFVBQVUsQ0FBQyxJQUFJLENBQUM7d0JBQ2QsSUFBSSxFQUFFLElBQUksRUFBRSwyQkFBMkIsRUFBRSxLQUFLLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsVUFBVSxDQUFDLEVBQUUsS0FBSyxFQUFFLEdBQUcsRUFBRSxPQUFPLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsaUJBQWlCLENBQUM7d0JBQ2xJLE1BQU0sRUFBRSxNQUFNLEdBQUcsSUFBSSxHQUFHLE9BQU8sR0FBRyxJQUFJLEVBQUUsS0FBSyxFQUFFLElBQUksQ0FBQyxXQUFXLENBQUMsWUFBWSxFQUFFO3FCQUMvRSxDQUFDLENBQUM7b0JBQ0gsTUFBTTtnQkFFUixLQUFLLFdBQVc7b0JBQ2QsVUFBVSxDQUFDLElBQUksQ0FBQzt3QkFDZCxJQUFJLEVBQUUsSUFBSSxFQUFFLDJCQUEyQixFQUFFLEtBQUssRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxVQUFVLENBQUMsRUFBRSxLQUFLLEVBQUUsR0FBRyxFQUFFLE9BQU8sRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxpQkFBaUIsQ0FBQzt3QkFDbEksTUFBTSxFQUFFLE1BQU0sR0FBRyxJQUFJLEdBQUcsT0FBTyxHQUFHLElBQUksRUFBRSxLQUFLLEVBQUUsSUFBSSxDQUFDLFdBQVcsQ0FBQyxZQUFZLEVBQUU7cUJBQy9FLENBQUMsQ0FBQztvQkFDSCxNQUFNO2dCQUVSLEtBQUssb0JBQW9CO29CQUN2QixVQUFVLENBQUMsSUFBSSxDQUFDO3dCQUNkLElBQUksRUFBRSxJQUFJLEVBQUUsMkJBQTJCLEVBQUUsS0FBSyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLG9CQUFvQixDQUFDLEVBQUUsS0FBSyxFQUFFLEdBQUcsRUFBRSxPQUFPLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsMkJBQTJCLENBQUM7d0JBQ3RKLE1BQU0sRUFBRSxNQUFNLEdBQUcsSUFBSSxHQUFHLE9BQU8sR0FBRyxJQUFJLEVBQUUsS0FBSyxFQUFFLElBQUksQ0FBQyxXQUFXLENBQUMsMEJBQTBCLEVBQUU7cUJBQzdGLENBQUMsQ0FBQztvQkFDSCxNQUFNO2dCQUVSLEtBQUssWUFBWTtvQkFDZixVQUFVLENBQUMsSUFBSSxDQUFDO3dCQUNkLElBQUksRUFBRSxJQUFJLEVBQUUsMkJBQTJCLEVBQUUsS0FBSyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLFlBQVksQ0FBQyxFQUFFLEtBQUssRUFBRSxHQUFHLEVBQUUsT0FBTyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLG1CQUFtQixDQUFDO3dCQUN0SSxNQUFNLEVBQUUsTUFBTSxHQUFHLElBQUksR0FBRyxPQUFPLEdBQUcsSUFBSSxFQUFFLEtBQUssRUFBRSxJQUFJLENBQUMsV0FBVyxDQUFDLGtCQUFrQixFQUFFO3FCQUNyRixDQUFDLENBQUM7b0JBQ0gsTUFBTTtnQkFFUixLQUFLLG1CQUFtQjtvQkFDdEIsVUFBVSxDQUFDLElBQUksQ0FBQzt3QkFDZCxJQUFJLEVBQUUsSUFBSSxFQUFFLFVBQVUsRUFBRSxPQUFPLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsbUJBQW1CLENBQUMsRUFBRSxPQUFPLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsMEJBQTBCLENBQUM7d0JBQ3pILE1BQU0sRUFBRSxNQUFNLEdBQUcsSUFBSSxHQUFHLE9BQU8sR0FBRyxJQUFJO3FCQUN2QyxDQUFDLENBQUM7b0JBQ0gsTUFBTTtnQkFFUixLQUFLLG1CQUFtQjtvQkFDdEIsVUFBVSxDQUFDLElBQUksQ0FBQzt3QkFDZCxJQUFJLEVBQUUsSUFBSSxFQUFFLFVBQVUsRUFBRSxPQUFPLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsbUJBQW1CLENBQUMsRUFBRSxPQUFPLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsMEJBQTBCLENBQUM7d0JBQ3pILE1BQU0sRUFBRSxNQUFNLEdBQUcsSUFBSSxHQUFHLE9BQU8sR0FBRyxJQUFJO3FCQUN2QyxDQUFDLENBQUM7b0JBQ0gsTUFBTTtnQkFFUixLQUFLLGVBQWU7b0JBQ2xCLFVBQVUsQ0FBQyxJQUFJLENBQUM7d0JBQ2QsSUFBSSxFQUFFLElBQUksRUFBRSxVQUFVLEVBQUUsT0FBTyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLGVBQWUsQ0FBQyxFQUFFLE9BQU8sRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQzt3QkFDakgsTUFBTSxFQUFFLE1BQU0sR0FBRyxJQUFJLEdBQUcsT0FBTyxHQUFHLElBQUk7cUJBQ3ZDLENBQUMsQ0FBQztvQkFDSCxNQUFNO2dCQUVSLEtBQUssaUJBQWlCO29CQUNwQixVQUFVLENBQUMsSUFBSSxDQUFDO3dCQUNkLElBQUksRUFBRSxJQUFJLEVBQUUsMkJBQTJCLEVBQUUsS0FBSyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLGlCQUFpQixDQUFDLEVBQUUsS0FBSyxFQUFFLEdBQUcsRUFBRSxPQUFPLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsd0JBQXdCLENBQUM7d0JBQ2hKLE1BQU0sRUFBRSxjQUFjLEdBQUcsSUFBSSxFQUFFLEtBQUssRUFBRSxJQUFJLENBQUMsV0FBVyxDQUFDLHVCQUF1QixFQUFFO3FCQUNqRixDQUFDLENBQUM7b0JBQ0gsTUFBTTtnQkFFUixLQUFLLHdCQUF3QjtvQkFDM0IsVUFBVSxDQUFDLElBQUksQ0FBQzt3QkFDZCxJQUFJLEVBQUUsSUFBSSxFQUFFLDJCQUEyQixFQUFFLEtBQUssRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyx3QkFBd0IsQ0FBQyxFQUFFLEtBQUssRUFBRSxHQUFHLEVBQUUsT0FBTyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLCtCQUErQixDQUFDO3dCQUM5SixNQUFNLEVBQUUsTUFBTSxHQUFHLElBQUksR0FBRyxPQUFPLEdBQUcsSUFBSSxFQUFFLEtBQUssRUFBRSxJQUFJLENBQUMsV0FBVyxDQUFDLDhCQUE4QixFQUFFO3FCQUNqRyxDQUFDLENBQUM7b0JBQ0gsTUFBTTtnQkFFUixLQUFLLGVBQWU7b0JBQ2xCLFVBQVUsQ0FBQyxJQUFJLENBQUM7d0JBQ2QsSUFBSSxFQUFFLElBQUksRUFBRSwyQkFBMkIsRUFBRSxLQUFLLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsZUFBZSxDQUFDLEVBQUUsS0FBSyxFQUFFLEdBQUcsRUFBRSxPQUFPLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsc0JBQXNCLENBQUM7d0JBQzVJLE1BQU0sRUFBRSxNQUFNLEdBQUcsSUFBSSxHQUFHLE9BQU8sR0FBRyxJQUFJLEVBQUUsS0FBSyxFQUFFLElBQUksQ0FBQyxXQUFXLENBQUMscUJBQXFCLEVBQUU7cUJBQ3hGLENBQUMsQ0FBQztvQkFDSCxNQUFNO2dCQUVSLEtBQUssY0FBYztvQkFDakIsVUFBVSxDQUFDLElBQUksQ0FBQzt3QkFDZCxJQUFJLEVBQUUsSUFBSSxFQUFFLDJCQUEyQixFQUFFLEtBQUssRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxjQUFjLENBQUMsRUFBRSxLQUFLLEVBQUUsR0FBRyxFQUFFLE9BQU8sRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxxQkFBcUIsQ0FBQzt3QkFDMUksTUFBTSxFQUFFLGNBQWMsR0FBRyxJQUFJLEVBQUUsS0FBSyxFQUFFLElBQUksQ0FBQyxXQUFXLENBQUMsb0JBQW9CLEVBQUU7cUJBQzlFLENBQUMsQ0FBQztvQkFDSCxNQUFNO2dCQUVSLEtBQUssaUJBQWlCO29CQUNwQixVQUFVLENBQUMsSUFBSSxDQUFDO3dCQUNkLElBQUksRUFBRSxJQUFJLEVBQUUsVUFBVSxFQUFFLE9BQU8sRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxpQkFBaUIsQ0FBQyxFQUFFLE9BQU8sRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyx3QkFBd0IsQ0FBQzt3QkFDckgsTUFBTSxFQUFFLE1BQU0sR0FBRyxJQUFJLEdBQUcsT0FBTyxHQUFHLElBQUk7cUJBQ3ZDLENBQUMsQ0FBQztvQkFDSCxNQUFNO2dCQUVSLEtBQUssa0JBQWtCO29CQUNyQixVQUFVLENBQUMsSUFBSSxDQUFDO3dCQUNkLElBQUksRUFBRSxJQUFJLEVBQUUsVUFBVSxFQUFFLE9BQU8sRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxFQUFFLE9BQU8sRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyx5QkFBeUIsQ0FBQzt3QkFDdkgsTUFBTSxFQUFFLE1BQU0sR0FBRyxJQUFJLEdBQUcsT0FBTyxHQUFHLElBQUk7cUJBQ3ZDLENBQUMsQ0FBQztvQkFDSCxNQUFNO2dCQUVSLEtBQUssZUFBZTtvQkFDbEIsVUFBVSxDQUFDLElBQUksQ0FBQzt3QkFDZCxJQUFJLEVBQUUsSUFBSSxFQUFFLFVBQVUsRUFBRSxPQUFPLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsZUFBZSxDQUFDLEVBQUUsT0FBTyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLHNCQUFzQixDQUFDO3dCQUNqSCxNQUFNLEVBQUUsTUFBTSxHQUFHLElBQUksR0FBRyxPQUFPLEdBQUcsSUFBSTtxQkFDdkMsQ0FBQyxDQUFDO29CQUNILE1BQU07Z0JBRVIsS0FBSyxZQUFZO29CQUNmLFVBQVUsQ0FBQyxJQUFJLENBQUM7d0JBQ2QsSUFBSSxFQUFFLElBQUksRUFBRSxVQUFVLEVBQUUsT0FBTyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLFlBQVksQ0FBQyxFQUFFLE9BQU8sRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxtQkFBbUIsQ0FBQzt3QkFDM0csTUFBTSxFQUFFLE1BQU0sR0FBRyxJQUFJLEdBQUcsT0FBTyxHQUFHLElBQUk7cUJBQ3ZDLENBQUMsQ0FBQztvQkFDSCxNQUFNO2dCQUVSLEtBQUssWUFBWTtvQkFDZixVQUFVLENBQUMsSUFBSSxDQUFDO3dCQUNkLElBQUksRUFBRSxJQUFJLEVBQUUsVUFBVSxFQUFFLE9BQU8sRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxZQUFZLENBQUMsRUFBRSxPQUFPLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsbUJBQW1CLENBQUM7d0JBQzNHLE1BQU0sRUFBRSxNQUFNLEdBQUcsSUFBSSxHQUFHLE9BQU8sR0FBRyxJQUFJO3FCQUN2QyxDQUFDLENBQUM7b0JBQ0gsTUFBTTtnQkFFUixLQUFLLHNCQUFzQjtvQkFDekIsVUFBVSxDQUFDLElBQUksQ0FBQzt3QkFDZCxJQUFJLEVBQUUsSUFBSSxFQUFFLFVBQVUsRUFBRSxPQUFPLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsc0JBQXNCLENBQUMsRUFBRSxPQUFPLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsNkJBQTZCLENBQUM7d0JBQy9ILE1BQU0sRUFBRSxNQUFNLEdBQUcsSUFBSSxHQUFHLE9BQU8sR0FBRyxJQUFJO3FCQUN2QyxDQUFDLENBQUM7b0JBQ0gsTUFBTTtnQkFFUixLQUFLLGdCQUFnQjtvQkFDbkIsVUFBVSxDQUFDLElBQUksQ0FBQzt3QkFDZCxJQUFJLEVBQUUsSUFBSSxFQUFFLFVBQVUsRUFBRSxPQUFPLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsZ0JBQWdCLENBQUMsRUFBRSxPQUFPLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsdUJBQXVCLENBQUM7d0JBQ25ILE1BQU0sRUFBRSxNQUFNLEdBQUcsSUFBSSxHQUFHLE9BQU8sR0FBRyxJQUFJO3FCQUN2QyxDQUFDLENBQUM7b0JBQ0gsTUFBTTtnQkFFUixLQUFLLGdCQUFnQjtvQkFDbkIsVUFBVSxDQUFDLElBQUksQ0FBQzt3QkFDZCxJQUFJLEVBQUUsSUFBSSxFQUFFLFVBQVUsRUFBRSxPQUFPLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsZ0JBQWdCLENBQUMsRUFBRSxPQUFPLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsdUJBQXVCLENBQUM7d0JBQ25ILE1BQU0sRUFBRSxNQUFNLEdBQUcsSUFBSSxHQUFHLE9BQU8sR0FBRyxJQUFJO3FCQUN2QyxDQUFDLENBQUM7b0JBQ0gsTUFBTTtnQkFFUixLQUFLLGlCQUFpQjtvQkFDcEIsVUFBVSxDQUFDLElBQUksQ0FBQzt3QkFDZCxJQUFJLEVBQUUsSUFBSSxFQUFFLFVBQVUsRUFBRSxPQUFPLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsaUJBQWlCLENBQUMsRUFBRSxPQUFPLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsd0JBQXdCLENBQUM7d0JBQ3JILE1BQU0sRUFBRSxNQUFNLEdBQUcsSUFBSSxHQUFHLE9BQU8sR0FBRyxJQUFJO3FCQUN2QyxDQUFDLENBQUM7b0JBQ0gsTUFBTTtnQkFFUixLQUFLLFlBQVk7b0JBQ2YsVUFBVSxDQUFDLElBQUksQ0FBQzt3QkFDZCxJQUFJLEVBQUUsSUFBSSxFQUFFLFVBQVUsRUFBRSxPQUFPLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsWUFBWSxDQUFDLEVBQUUsT0FBTyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLG1CQUFtQixDQUFDO3dCQUMzRyxNQUFNLEVBQUUsTUFBTSxHQUFHLElBQUksR0FBRyxPQUFPLEdBQUcsSUFBSTtxQkFDdkMsQ0FBQyxDQUFDO29CQUNILE1BQU07Z0JBRVIsS0FBSyxjQUFjO29CQUNqQixVQUFVLENBQUMsSUFBSSxDQUFDO3dCQUNkLElBQUksRUFBRSxJQUFJLEVBQUUsVUFBVSxFQUFFLE9BQU8sRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxjQUFjLENBQUMsRUFBRSxPQUFPLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMscUJBQXFCLENBQUM7d0JBQy9HLE1BQU0sRUFBRSxNQUFNLEdBQUcsSUFBSSxHQUFHLE9BQU8sR0FBRyxJQUFJO3FCQUN2QyxDQUFDLENBQUM7b0JBQ0gsTUFBTTtnQkFFUixLQUFLLFlBQVk7b0JBQ2YsVUFBVSxDQUFDLElBQUksQ0FBQzt3QkFDZCxJQUFJLEVBQUUsSUFBSSxFQUFFLDJCQUEyQixFQUFFLEtBQUssRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxZQUFZLENBQUMsRUFBRSxLQUFLLEVBQUUsR0FBRyxFQUFFLE9BQU8sRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxtQkFBbUIsQ0FBQzt3QkFDdEksTUFBTSxFQUFFLE1BQU0sR0FBRyxJQUFJLEdBQUcsT0FBTyxHQUFHLElBQUksRUFBRSxLQUFLLEVBQUUsSUFBSSxDQUFDLFdBQVcsQ0FBQyxrQkFBa0IsRUFBRTtxQkFDckYsQ0FBQyxDQUFDO29CQUNILE1BQU07Z0JBRVIsS0FBSyxPQUFPO29CQUNWLFVBQVUsQ0FBQyxJQUFJLENBQUM7d0JBQ2QsSUFBSSxFQUFFLElBQUksRUFBRSwyQkFBMkIsRUFBRSxLQUFLLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsUUFBUSxDQUFDLENBQUMsT0FBTyxDQUFDLEdBQUcsRUFBRSxFQUFFLENBQUMsR0FBRyxLQUFLLEVBQUUsS0FBSyxFQUFFLEdBQUcsRUFBRSxPQUFPLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsZUFBZSxDQUFDO3dCQUN2SixNQUFNLEVBQUUsTUFBTSxHQUFHLElBQUksR0FBRyxPQUFPLEdBQUcsSUFBSSxFQUFFLEtBQUssRUFBRSxJQUFJLENBQUMsV0FBVyxDQUFDLFlBQVksRUFBRTtxQkFDL0UsQ0FBQyxDQUFDO29CQUNILE1BQU07Z0JBRVIsS0FBSyxPQUFPO29CQUNWLFVBQVUsQ0FBQyxJQUFJLENBQUM7d0JBQ2QsSUFBSSxFQUFFLElBQUksRUFBRSwyQkFBMkIsRUFBRSxLQUFLLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsUUFBUSxDQUFDLENBQUMsT0FBTyxDQUFDLEdBQUcsRUFBRSxFQUFFLENBQUMsR0FBRyxLQUFLLEVBQUUsS0FBSyxFQUFFLEdBQUcsRUFBRSxPQUFPLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsZUFBZSxDQUFDO3dCQUN2SixNQUFNLEVBQUUsTUFBTSxHQUFHLElBQUksR0FBRyxPQUFPLEdBQUcsSUFBSSxFQUFFLEtBQUssRUFBRSxJQUFJLENBQUMsV0FBVyxDQUFDLFlBQVksRUFBRTtxQkFDL0UsQ0FBQyxDQUFDO29CQUNILE1BQU07Z0JBRVIsS0FBSyx1QkFBdUI7b0JBQzFCLFVBQVUsQ0FBQyxJQUFJLENBQUM7d0JBQ2QsSUFBSSxFQUFFLElBQUksRUFBRSwyQkFBMkIsRUFBRSxLQUFLLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsY0FBYyxDQUFDLEVBQUUsS0FBSyxFQUFFLEdBQUcsRUFBRSxPQUFPLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMscUJBQXFCLENBQUM7d0JBQzFJLE1BQU0sRUFBRSxNQUFNLEdBQUcsSUFBSSxHQUFHLE9BQU8sR0FBRyxJQUFJLEVBQUUsS0FBSyxFQUFFLElBQUksQ0FBQyxXQUFXLENBQUMsb0JBQW9CLEVBQUU7cUJBQ3ZGLENBQUMsQ0FBQztvQkFDSCxNQUFNO2dCQUVSLEtBQUssYUFBYTtvQkFDaEIsVUFBVSxDQUFDLElBQUksQ0FBQzt3QkFDZCxJQUFJLEVBQUUsSUFBSSxFQUFFLDJCQUEyQixFQUFFLEtBQUssRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxhQUFhLENBQUMsRUFBRSxLQUFLLEVBQUUsR0FBRyxFQUFFLE9BQU8sRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxvQkFBb0IsQ0FBQzt3QkFDeEksTUFBTSxFQUFFLE1BQU0sR0FBRyxJQUFJLEdBQUcsT0FBTyxHQUFHLElBQUksRUFBRSxLQUFLLEVBQUUsSUFBSSxDQUFDLFdBQVcsQ0FBQyxtQkFBbUIsRUFBRTtxQkFDdEYsQ0FBQyxDQUFDO29CQUNILE1BQU07Z0JBRVIsS0FBSyxlQUFlO29CQUNsQixVQUFVLENBQUMsSUFBSSxDQUFDO3dCQUNkLElBQUksRUFBRSxJQUFJLEVBQUUsMkJBQTJCLEVBQUUsS0FBSyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLHlCQUF5QixDQUFDLEVBQUUsS0FBSyxFQUFFLEdBQUcsRUFBRSxPQUFPLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsZ0NBQWdDLENBQUM7d0JBQ2hLLE1BQU0sRUFBRSxNQUFNLEdBQUcsSUFBSSxHQUFHLE9BQU8sR0FBRyxJQUFJLEVBQUUsS0FBSyxFQUFFLElBQUksQ0FBQyxXQUFXLENBQUMsK0JBQStCLEVBQUU7cUJBQ2xHLENBQUMsQ0FBQztvQkFDSCxNQUFNO2dCQUVSLEtBQUssdUJBQXVCO29CQUMxQixVQUFVLENBQUMsSUFBSSxDQUFDO3dCQUNkLElBQUksRUFBRSxJQUFJLEVBQUUsMkJBQTJCLEVBQUUsS0FBSyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLHVCQUF1QixDQUFDLEVBQUUsS0FBSyxFQUFFLEdBQUcsRUFBRSxPQUFPLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsOEJBQThCLENBQUM7d0JBQzVKLE1BQU0sRUFBRSxNQUFNLEdBQUcsSUFBSSxHQUFHLE9BQU8sR0FBRyxJQUFJLEVBQUUsS0FBSyxFQUFFLElBQUksQ0FBQyxXQUFXLENBQUMsNkJBQTZCLEVBQUU7cUJBQ2hHLENBQUMsQ0FBQztvQkFDSCxNQUFNO2dCQUVSLEtBQUssVUFBVTtvQkFDYixVQUFVLENBQUMsSUFBSSxDQUFDO3dCQUNkLElBQUksRUFBRSxJQUFJLEVBQUUsVUFBVSxFQUFFLE9BQU8sRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxVQUFVLENBQUMsRUFBRSxPQUFPLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsaUJBQWlCLENBQUM7d0JBQ3ZHLE1BQU0sRUFBRSxNQUFNLEdBQUcsSUFBSSxHQUFHLE9BQU8sR0FBRyxJQUFJO3FCQUN2QyxDQUFDLENBQUM7b0JBQ0gsTUFBTTtnQkFFUixLQUFLLGVBQWU7b0JBQ2xCLFVBQVUsQ0FBQyxJQUFJLENBQUM7d0JBQ2QsSUFBSSxFQUFFLElBQUksRUFBRSxVQUFVLEVBQUUsT0FBTyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLGVBQWUsQ0FBQyxFQUFFLE9BQU8sRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQzt3QkFDakgsTUFBTSxFQUFFLE1BQU0sR0FBRyxJQUFJLEdBQUcsT0FBTyxHQUFHLElBQUk7cUJBQ3ZDLENBQUMsQ0FBQztvQkFDSCxNQUFNO2dCQUVSLEtBQUssZ0JBQWdCO29CQUNuQixVQUFVLENBQUMsSUFBSSxDQUFDO3dCQUNkLElBQUksRUFBRSxJQUFJLEVBQUUsVUFBVSxFQUFFLE9BQU8sRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxFQUFFLE9BQU8sRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyx1QkFBdUIsQ0FBQzt3QkFDbkgsTUFBTSxFQUFFLE1BQU0sR0FBRyxJQUFJLEdBQUcsT0FBTyxHQUFHLElBQUk7cUJBQ3ZDLENBQUMsQ0FBQztvQkFDSCxNQUFNO2dCQUVSLEtBQUssMEJBQTBCO29CQUM3QixVQUFVLENBQUMsSUFBSSxDQUFDO3dCQUNkLElBQUksRUFBRSxJQUFJLEVBQUUsVUFBVSxFQUFFLE9BQU8sRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQywwQkFBMEIsQ0FBQyxFQUFFLE9BQU8sRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxpQ0FBaUMsQ0FBQzt3QkFDdkksTUFBTSxFQUFFLE1BQU0sR0FBRyxJQUFJLEdBQUcsT0FBTyxHQUFHLElBQUk7cUJBQ3ZDLENBQUMsQ0FBQztvQkFDSCxNQUFNO2dCQUVSLEtBQUssaUJBQWlCO29CQUNwQixVQUFVLENBQUMsSUFBSSxDQUFDO3dCQUNkLElBQUksRUFBRSxJQUFJLEVBQUUsMkJBQTJCLEVBQUUsS0FBSyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLGlCQUFpQixDQUFDLEVBQUUsS0FBSyxFQUFFLEdBQUcsRUFBRSxPQUFPLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsd0JBQXdCLENBQUM7d0JBQ2hKLE1BQU0sRUFBRSxNQUFNLEdBQUcsSUFBSSxHQUFHLE9BQU8sR0FBRyxJQUFJLEVBQUUsS0FBSyxFQUFFLElBQUksQ0FBQyxXQUFXLENBQUMsdUJBQXVCLEVBQUU7cUJBQzFGLENBQUMsQ0FBQztvQkFDSCxNQUFNO2dCQUVSLEtBQUssZUFBZTtvQkFDbEIsVUFBVSxDQUFDLElBQUksQ0FBQzt3QkFDZCxJQUFJLEVBQUUsSUFBSSxFQUFFLDJCQUEyQixFQUFFLEtBQUssRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxlQUFlLENBQUMsRUFBRSxLQUFLLEVBQUUsR0FBRyxFQUFFLE9BQU8sRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQzt3QkFDNUksTUFBTSxFQUFFLE1BQU0sR0FBRyxJQUFJLEdBQUcsT0FBTyxHQUFHLElBQUksRUFBRSxLQUFLLEVBQUUsSUFBSSxDQUFDLFdBQVcsQ0FBQyxxQkFBcUIsRUFBRTtxQkFDeEYsQ0FBQyxDQUFDO29CQUNILE1BQU07Z0JBRVIsS0FBSywwQkFBMEI7b0JBQzdCLFVBQVUsQ0FBQyxJQUFJLENBQUM7d0JBQ2QsSUFBSSxFQUFFLElBQUksRUFBRSxVQUFVLEVBQUUsT0FBTyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLHVCQUF1QixDQUFDLEVBQUUsT0FBTyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLDhCQUE4QixDQUFDO3dCQUNqSSxNQUFNLEVBQUUsTUFBTSxHQUFHLElBQUksR0FBRyxPQUFPLEdBQUcsSUFBSTtxQkFDdkMsQ0FBQyxDQUFDO29CQUNILE1BQU07Z0JBRVIsS0FBSywwQkFBMEI7b0JBQzdCLFVBQVUsQ0FBQyxJQUFJLENBQUM7d0JBQ2QsSUFBSSxFQUFFLElBQUksRUFBRSxVQUFVLEVBQUUsT0FBTyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLGtCQUFrQixDQUFDLEVBQUUsT0FBTyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLHlCQUF5QixDQUFDO3dCQUN2SCxNQUFNLEVBQUUsTUFBTSxHQUFHLElBQUksR0FBRyxPQUFPLEdBQUcsSUFBSTtxQkFDdkMsQ0FBQyxDQUFDO29CQUNILE1BQU07Z0JBRVIsS0FBSyxXQUFXO29CQUNkLFVBQVUsQ0FBQyxJQUFJLENBQUM7d0JBQ2QsSUFBSSxFQUFFLElBQUksRUFBRSxTQUFTLEVBQUUsS0FBSyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLFdBQVcsQ0FBQyxFQUFFLE9BQU8sRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxFQUFFLEtBQUssRUFBRSxHQUFHO3dCQUNsSCxNQUFNLEVBQUUsTUFBTSxHQUFHLElBQUksR0FBRyxPQUFPLEdBQUcsSUFBSTtxQkFDdkMsQ0FBQyxDQUFDO29CQUNILE1BQU07Z0JBRVIsS0FBSyxnQkFBZ0I7b0JBQ25CLFVBQVUsQ0FBQyxJQUFJLENBQUM7d0JBQ2QsSUFBSSxFQUFFLElBQUksRUFBRSwyQkFBMkIsRUFBRSxLQUFLLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsYUFBYSxDQUFDLEVBQUUsS0FBSyxFQUFFLEdBQUcsRUFBRSxPQUFPLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsb0JBQW9CLENBQUM7d0JBQ3hJLE1BQU0sRUFBRSxNQUFNLEdBQUcsSUFBSSxHQUFHLE9BQU8sR0FBRyxJQUFJLEVBQUUsS0FBSyxFQUFFLElBQUksQ0FBQyxXQUFXLENBQUMsc0JBQXNCLEVBQUU7cUJBQ3pGLENBQUMsQ0FBQztvQkFDSCxNQUFNO2dCQUVSLEtBQUssbUJBQW1CO29CQUN0QixVQUFVLENBQUMsSUFBSSxDQUFDO3dCQUNkLElBQUksRUFBRSxJQUFJLEVBQUUsVUFBVSxFQUFFLE9BQU8sRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxtQkFBbUIsQ0FBQyxFQUFFLE9BQU8sRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQywwQkFBMEIsQ0FBQzt3QkFDekgsTUFBTSxFQUFFLE1BQU0sR0FBRyxJQUFJLEdBQUcsT0FBTyxHQUFHLElBQUk7cUJBQ3ZDLENBQUMsQ0FBQztvQkFDSCxNQUFNO2dCQUVSLEtBQUssd0JBQXdCO29CQUMzQixVQUFVLENBQUMsSUFBSSxDQUFDO3dCQUNkLElBQUksRUFBRSxJQUFJLEVBQUUsVUFBVSxFQUFFLE9BQU8sRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyx3QkFBd0IsQ0FBQyxFQUFFLE9BQU8sRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQywrQkFBK0IsQ0FBQzt3QkFDbkksTUFBTSxFQUFFLE1BQU0sR0FBRyxJQUFJLEdBQUcsT0FBTyxHQUFHLElBQUk7cUJBQ3ZDLENBQUMsQ0FBQztvQkFDSCxNQUFNO2dCQUVSLEtBQUssMkJBQTJCO29CQUM5QixVQUFVLENBQUMsSUFBSSxDQUFDO3dCQUNkLElBQUksRUFBRSxJQUFJLEVBQUUsVUFBVSxFQUFFLE9BQU8sRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyx3QkFBd0IsQ0FBQyxFQUFFLE9BQU8sRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQywrQkFBK0IsQ0FBQzt3QkFDbkksTUFBTSxFQUFFLE1BQU0sR0FBRyxJQUFJLEdBQUcsT0FBTyxHQUFHLElBQUk7cUJBQ3ZDLENBQUMsQ0FBQztvQkFDSCxNQUFNO2dCQUVSLEtBQUssdUJBQXVCO29CQUMxQixVQUFVLENBQUMsSUFBSSxDQUFDO3dCQUNkLElBQUksRUFBRSxJQUFJLEVBQUUsVUFBVSxFQUFFLE9BQU8sRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyx1QkFBdUIsQ0FBQyxFQUFFLE9BQU8sRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyw4QkFBOEIsQ0FBQzt3QkFDakksTUFBTSxFQUFFLE1BQU0sR0FBRyxJQUFJLEdBQUcsT0FBTyxHQUFHLElBQUk7cUJBQ3ZDLENBQUMsQ0FBQztvQkFDSCxNQUFNO2dCQUVSLEtBQUssa0JBQWtCO29CQUNyQixVQUFVLENBQUMsSUFBSSxDQUFDO3dCQUNkLElBQUksRUFBRSxJQUFJLEVBQUUsVUFBVSxFQUFFLE9BQU8sRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxFQUFFLE9BQU8sRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyx5QkFBeUIsQ0FBQzt3QkFDdkgsTUFBTSxFQUFFLE1BQU0sR0FBRyxJQUFJLEdBQUcsT0FBTyxHQUFHLElBQUk7cUJBQ3ZDLENBQUMsQ0FBQztvQkFDSCxNQUFNO2dCQUVSLEtBQUssaUJBQWlCO29CQUNwQixVQUFVLENBQUMsSUFBSSxDQUFDO3dCQUNkLElBQUksRUFBRSxJQUFJLEVBQUUsMkJBQTJCLEVBQUUsS0FBSyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLGlCQUFpQixDQUFDLEVBQUUsS0FBSyxFQUFFLEdBQUcsRUFBRSxPQUFPLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsd0JBQXdCLENBQUM7d0JBQ2hKLE1BQU0sRUFBRSxNQUFNLEdBQUcsSUFBSSxHQUFHLE9BQU8sR0FBRyxJQUFJLEVBQUUsS0FBSyxFQUFFLElBQUksQ0FBQyxXQUFXLENBQUMsdUJBQXVCLEVBQUU7cUJBQzFGLENBQUMsQ0FBQztvQkFDSCxNQUFNO2dCQUVSLEtBQUssd0JBQXdCO29CQUMzQixVQUFVLENBQUMsSUFBSSxDQUFDO3dCQUNkLElBQUksRUFBRSxJQUFJLEVBQUUsYUFBYSxFQUFFLE9BQU8sRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyx3QkFBd0IsQ0FBQyxFQUFFLEtBQUssRUFBRSxHQUFHO3dCQUN4RixNQUFNLEVBQUUsTUFBTSxHQUFHLElBQUksR0FBRyxPQUFPLEdBQUcsSUFBSSxFQUFFLEtBQUssRUFBRSxJQUFJLENBQUMsV0FBVyxDQUFDLHVCQUF1QixFQUFFO3FCQUMxRixDQUFDLENBQUM7b0JBQ0gsTUFBTTtnQkFFUixLQUFLLHFCQUFxQjtvQkFDeEIsVUFBVSxDQUFDLElBQUksQ0FBQzt3QkFDZCxJQUFJLEVBQUUsT0FBTyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLHdCQUF3QixDQUFDLEVBQUUsSUFBSSxFQUFFLDJCQUEyQixFQUFFLE9BQU8sRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyw0QkFBNEIsQ0FBQyxFQUFFLEtBQUssRUFBRSxHQUFHO3dCQUM3SixLQUFLLEVBQUUsSUFBSSxDQUFDLFdBQVcsQ0FBQywrQkFBK0IsRUFBRSxFQUFFLE1BQU0sRUFBRSxNQUFNLEdBQUcsSUFBSSxHQUFHLE9BQU8sR0FBRyxJQUFJO3FCQUNsRyxDQUFDLENBQUM7b0JBQ0gsTUFBTTtnQkFFUjtvQkFDRSxNQUFNO1lBQ1YsQ0FBQztRQUNILENBQUMsQ0FBQyxDQUFDO1FBQ0gsT0FBTyxVQUFVLENBQUM7SUFDcEIsQ0FBQztJQUVELFNBQVMsQ0FBQyxNQUFjO1FBQ3RCLE1BQU0sZ0JBQWdCLEdBQUcsSUFBSSxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUMsZ0RBQWdELENBQUMsQ0FBQztRQUNqRyxNQUFNLGFBQWEsR0FBRyxnQkFBZ0IsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUM7UUFDN0UsTUFBTSxzQkFBc0IsR0FBRyxNQUFNLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsSUFBSSxNQUFNLEtBQUssV0FBVyxDQUFDO1FBRXZGLE9BQU87WUFDTCxpQkFBaUIsRUFBRSxFQUFFLE9BQU8sRUFBRSxNQUFNLEtBQUssVUFBVSxFQUFFLE1BQU0sRUFBRSxhQUFhLENBQUMsQ0FBQyxDQUFDLGFBQWEsQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDLENBQUMsSUFBSSxFQUFFO1lBQ3JILGNBQWMsRUFBRSxFQUFFLE9BQU8sRUFBRSxNQUFNLEtBQUssVUFBVSxFQUFFLE1BQU0sRUFBRSxhQUFhLENBQUMsQ0FBQyxDQUFDLGFBQWEsQ0FBQyxjQUFjLENBQUMsQ0FBQyxDQUFDLElBQUksRUFBRSxzQkFBc0IsRUFBRTtZQUN2SSxhQUFhLEVBQUUsRUFBRSxPQUFPLEVBQUUsTUFBTSxLQUFLLFVBQVUsRUFBRSxNQUFNLEVBQUUsYUFBYSxDQUFDLENBQUMsQ0FBQyxhQUFhLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQyxLQUFLLEVBQUU7U0FDL0csQ0FBQztJQUNKLENBQUM7SUFFRCxpQkFBaUIsQ0FBQyxNQUFlO1FBQy9CLElBQUksaUJBQWlCLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLG1CQUFtQixDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxtQ0FBbUMsR0FBRyxJQUFJLENBQUMscUJBQXFCLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDO1FBQ2hMLE9BQU8saUJBQWlCLElBQUksSUFBSSxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLGlCQUFpQixDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQztJQUMxRSxDQUFDO0lBRUQsa0JBQWtCLENBQUMsTUFBZTtRQUNoQyxJQUFJLFFBQWEsQ0FBQztRQUNsQixNQUFNLGNBQWMsR0FBRyxJQUFJLENBQUMsaUJBQWlCLEVBQUUsQ0FBQztRQUNoRCxJQUFJLGNBQWMsSUFBSSxJQUFJLEVBQUUsQ0FBQztZQUMzQixJQUFJLFlBQVksR0FBRyxjQUFjLENBQUMsU0FBUyxJQUFJLGNBQWMsQ0FBQyxRQUFRLElBQUksY0FBYyxDQUFDLFFBQVEsQ0FBQztZQUNsRyxPQUFPLGNBQWMsQ0FBQyxTQUFTLENBQUM7WUFDaEMsSUFBSSxDQUFDLE1BQU0sR0FBRyxDQUFDLE1BQU0sSUFBSSxZQUFZLENBQUMsQ0FBQyxDQUFDLENBQUMsY0FBYyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsT0FBTyxHQUFHLFlBQVksQ0FBQyxDQUFDLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUM7WUFFekgsTUFBTSxXQUFXLEdBQUcsSUFBSSxDQUFDLHdCQUF3QixDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQztZQUMvRCxNQUFNLGNBQWMsR0FBUSxFQUFFLENBQUM7WUFDL0IsTUFBTSxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxjQUFjLENBQUMsR0FBRyxDQUFDLEdBQUcsY0FBYyxDQUFDLEdBQUcsQ0FBQyxJQUFJLFdBQVcsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDO1lBQ3ZHLE1BQU0sQ0FBQyxJQUFJLENBQUMsY0FBYyxDQUFDLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsY0FBYyxDQUFDLEdBQUcsQ0FBQyxHQUFHLGNBQWMsQ0FBQyxHQUFHLENBQUMsSUFBSSxXQUFXLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQztZQUUxRyxJQUFJLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxlQUFlLEtBQUssSUFBSSxJQUFJLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxlQUFlLEtBQUssS0FBSyxFQUFFLENBQUM7Z0JBQ2hILGNBQWMsQ0FBQyxlQUFlLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLGVBQWUsQ0FBQztZQUM5RSxDQUFDO1lBRUQsSUFBSSxDQUFDLENBQUMsSUFBSSxDQUFDLE1BQU0sS0FBSyxLQUFLLElBQUksSUFBSSxDQUFDLE1BQU0sS0FBSyxNQUFNLENBQUMsRUFBRSxDQUFDO2dCQUN2RCxjQUFjLENBQUMsZUFBZSxHQUFHLEtBQUssQ0FBQztZQUN6QyxDQUFDO1lBRUQsUUFBUSxHQUFHLGNBQWMsQ0FBQztRQUM1QixDQUFDO2FBQU0sQ0FBQztZQUNOLFFBQVEsR0FBRyxJQUFJLENBQUMsd0JBQXdCLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBQ3hELENBQUM7UUFFRCxJQUFJLENBQUMsV0FBVyxDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBRTNCLElBQUksQ0FBQyxjQUFjLEdBQUcsSUFBSSxDQUFDLE1BQU0sQ0FBQyxVQUFVLENBQUMsUUFBUSxDQUFDLENBQUM7SUFDekQsQ0FBQztJQUVELFdBQVcsQ0FBQyxxQkFBMEI7UUFDcEMsTUFBTSxLQUFLLEdBQUcsQ0FBQyxXQUFXLEVBQUUsVUFBVSxFQUFFLFdBQVcsRUFBRSxVQUFVLENBQUMsQ0FBQztRQUNqRSxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsWUFBWSxFQUFFLEVBQUU7WUFDN0IsSUFBSSxxQkFBcUIsQ0FBQyxZQUFZLENBQUMsRUFBRSxDQUFDO2dCQUN4QyxRQUFRLFlBQVksRUFBRSxDQUFDO29CQUNyQixLQUFLLFdBQVc7d0JBQ2QsSUFBSSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxvQkFBb0IsSUFBSSxxQkFBcUIsQ0FBQyxZQUFZLENBQUMsS0FBSyxLQUFLLEVBQUUsQ0FBQzs0QkFBQyxxQkFBcUIsQ0FBQyxZQUFZLENBQUMsR0FBRyxLQUFLLENBQUM7d0JBQUMsQ0FBQzt3QkFDdkosSUFBSSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxvQkFBb0IsSUFBSSxxQkFBcUIsQ0FBQyxZQUFZLENBQUMsS0FBSyxLQUFLLEVBQUUsQ0FBQzs0QkFBQyxxQkFBcUIsQ0FBQyxZQUFZLENBQUMsR0FBRyxNQUFNLENBQUM7d0JBQUMsQ0FBQzt3QkFDeEosSUFBSSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxxQkFBcUIsSUFBSSxxQkFBcUIsQ0FBQyxZQUFZLENBQUMsS0FBSyxNQUFNLEVBQUUsQ0FBQzs0QkFBQyxxQkFBcUIsQ0FBQyxZQUFZLENBQUMsR0FBRyxLQUFLLENBQUM7d0JBQUMsQ0FBQzt3QkFDekosSUFBSSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxvQkFBb0IsSUFBSSxxQkFBcUIsQ0FBQyxZQUFZLENBQUMsS0FBSyxLQUFLLEVBQUUsQ0FBQzs0QkFBQyxxQkFBcUIsQ0FBQyxZQUFZLENBQUMsR0FBRyxLQUFLLENBQUM7d0JBQUMsQ0FBQzt3QkFDdkosSUFBSSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxvQkFBb0IsSUFBSSxxQkFBcUIsQ0FBQyxZQUFZLENBQUMsS0FBSyxLQUFLLEVBQUUsQ0FBQzs0QkFBQyxxQkFBcUIsQ0FBQyxZQUFZLENBQUMsR0FBRyxNQUFNLENBQUM7d0JBQUMsQ0FBQzt3QkFDeEosSUFBSSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxxQkFBcUIsSUFBSSxxQkFBcUIsQ0FBQyxZQUFZLENBQUMsS0FBSyxNQUFNLEVBQUUsQ0FBQzs0QkFBQyxxQkFBcUIsQ0FBQyxZQUFZLENBQUMsR0FBRyxLQUFLLENBQUM7d0JBQUMsQ0FBQzt3QkFDekosSUFBSSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyx5QkFBeUIsSUFBSSxxQkFBcUIsQ0FBQyxZQUFZLENBQUMsS0FBSyxLQUFLLEVBQUUsQ0FBQzs0QkFBQyxxQkFBcUIsQ0FBQyxZQUFZLENBQUMsR0FBRyxLQUFLLENBQUM7d0JBQUMsQ0FBQzt3QkFDNUosSUFBSSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxvQkFBb0IsSUFBSSxxQkFBcUIsQ0FBQyxZQUFZLENBQUMsS0FBSyxLQUFLLEVBQUUsQ0FBQzs0QkFBQyxxQkFBcUIsQ0FBQyxZQUFZLENBQUMsR0FBRyxNQUFNLENBQUM7d0JBQUMsQ0FBQzt3QkFDeEosSUFBSSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxxQkFBcUIsSUFBSSxxQkFBcUIsQ0FBQyxZQUFZLENBQUMsS0FBSyxNQUFNLEVBQUUsQ0FBQzs0QkFBQyxxQkFBcUIsQ0FBQyxZQUFZLENBQUMsR0FBRyxLQUFLLENBQUM7d0JBQUMsQ0FBQzt3QkFDekosTUFBTTtvQkFFUixLQUFLLFVBQVU7d0JBQ2IsSUFBSSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxlQUFlLElBQUkscUJBQXFCLENBQUMsWUFBWSxDQUFDLEtBQUssS0FBSyxFQUFFLENBQUM7NEJBQUMscUJBQXFCLENBQUMsWUFBWSxDQUFDLEdBQUcsS0FBSyxDQUFDO3dCQUFDLENBQUM7d0JBQ2xKLElBQUksQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsZUFBZSxJQUFJLHFCQUFxQixDQUFDLFlBQVksQ0FBQyxLQUFLLEtBQUssRUFBRSxDQUFDOzRCQUFDLHFCQUFxQixDQUFDLFlBQVksQ0FBQyxHQUFHLEtBQUssQ0FBQzt3QkFBQyxDQUFDO3dCQUNsSixJQUFJLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLGVBQWUsSUFBSSxxQkFBcUIsQ0FBQyxZQUFZLENBQUMsS0FBSyxLQUFLLEVBQUUsQ0FBQzs0QkFBQyxxQkFBcUIsQ0FBQyxZQUFZLENBQUMsR0FBRyxLQUFLLENBQUM7d0JBQUMsQ0FBQzt3QkFDbEosSUFBSSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxlQUFlLElBQUkscUJBQXFCLENBQUMsWUFBWSxDQUFDLEtBQUssS0FBSyxFQUFFLENBQUM7NEJBQUMscUJBQXFCLENBQUMsWUFBWSxDQUFDLEdBQUcsTUFBTSxDQUFDO3dCQUFDLENBQUM7d0JBQ25KLElBQUksQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsZ0JBQWdCLElBQUkscUJBQXFCLENBQUMsWUFBWSxDQUFDLEtBQUssTUFBTSxFQUFFLENBQUM7NEJBQUMscUJBQXFCLENBQUMsWUFBWSxDQUFDLEdBQUcsS0FBSyxDQUFDO3dCQUFDLENBQUM7d0JBQ3BKLElBQUksQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsZ0JBQWdCLElBQUkscUJBQXFCLENBQUMsWUFBWSxDQUFDLEtBQUssTUFBTSxFQUFFLENBQUM7NEJBQUMscUJBQXFCLENBQUMsWUFBWSxDQUFDLEdBQUcsTUFBTSxDQUFDO3dCQUFDLENBQUM7d0JBQ3JKLE1BQU07b0JBRVIsS0FBSyxXQUFXO3dCQUNkLElBQUksQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsaUJBQWlCLElBQUkscUJBQXFCLENBQUMsWUFBWSxDQUFDLEtBQUssT0FBTyxFQUFFLENBQUM7NEJBQUMscUJBQXFCLENBQUMsWUFBWSxDQUFDLEdBQUcsYUFBYSxDQUFDO3dCQUFDLENBQUM7d0JBQzlKLElBQUksQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMscUJBQXFCLElBQUkscUJBQXFCLENBQUMsWUFBWSxDQUFDLEtBQUssYUFBYSxFQUFFLENBQUM7NEJBQUMscUJBQXFCLENBQUMsWUFBWSxDQUFDLEdBQUcsVUFBVSxDQUFDO3dCQUFDLENBQUM7d0JBQ3JLLElBQUksQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsb0JBQW9CLElBQUkscUJBQXFCLENBQUMsWUFBWSxDQUFDLEtBQUssVUFBVSxFQUFFLENBQUM7NEJBQUMscUJBQXFCLENBQUMsWUFBWSxDQUFDLEdBQUcsT0FBTyxDQUFDO3dCQUFDLENBQUM7d0JBQzlKLE1BQU07b0JBRVIsS0FBSyxVQUFVO3dCQUNiLElBQUksQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsZ0JBQWdCLElBQUkscUJBQXFCLENBQUMsWUFBWSxDQUFDLEtBQUssTUFBTSxFQUFFLENBQUM7NEJBQUMscUJBQXFCLENBQUMsWUFBWSxDQUFDLEdBQUcsT0FBTyxDQUFDO3dCQUFDLENBQUM7d0JBQ3RKLElBQUksQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsaUJBQWlCLElBQUkscUJBQXFCLENBQUMsWUFBWSxDQUFDLEtBQUssT0FBTyxFQUFFLENBQUM7NEJBQUMscUJBQXFCLENBQUMsWUFBWSxDQUFDLEdBQUcsS0FBSyxDQUFDO3dCQUFDLENBQUM7d0JBQ3RKLElBQUksQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsZUFBZSxJQUFJLHFCQUFxQixDQUFDLFlBQVksQ0FBQyxLQUFLLEtBQUssRUFBRSxDQUFDOzRCQUFDLHFCQUFxQixDQUFDLFlBQVksQ0FBQyxHQUFHLE1BQU0sQ0FBQzt3QkFBQyxDQUFDO3dCQUNuSixNQUFNO2dCQUNWLENBQUM7WUFDSCxDQUFDO1FBQ0gsQ0FBQyxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQscUJBQXFCLENBQUMsTUFBZTtRQUNuQyxNQUFNLEdBQUcsTUFBTSxJQUFJLElBQUksQ0FBQyxNQUFNLENBQUM7UUFDL0IsSUFBSSxNQUFNLEtBQUssTUFBTSxJQUFJLE1BQU0sS0FBSyxPQUFPLElBQUksTUFBTSxLQUFLLEtBQUssRUFBRSxDQUFDO1lBQUMsT0FBTyxNQUFNLENBQUM7UUFBQyxDQUFDO1FBQ25GLElBQUksTUFBTSxLQUFLLFdBQVcsSUFBSSxNQUFNLEtBQUssT0FBTyxJQUFJLE1BQU0sS0FBSyxVQUFVLElBQUksTUFBTSxLQUFLLGFBQWEsRUFBRSxDQUFDO1lBQUMsT0FBTyxPQUFPLENBQUM7UUFBQyxDQUFDO1FBQzFILElBQUksTUFBTSxLQUFLLEtBQUssSUFBSSxNQUFNLEtBQUssS0FBSyxJQUFJLE1BQU0sS0FBSyxLQUFLLElBQUksTUFBTSxLQUFLLEtBQUssSUFBSSxNQUFNLEtBQUssTUFBTSxFQUFFLENBQUM7WUFBQyxPQUFPLE1BQU0sQ0FBQztRQUFDLENBQUM7UUFDekgsSUFBSSxNQUFNLEtBQUssVUFBVSxJQUFJLE1BQU0sS0FBSyxVQUFVLElBQUksTUFBTSxLQUFLLFdBQVcsSUFBSSxNQUFNLEtBQUssVUFBVSxJQUFJLE1BQU0sS0FBSyxVQUFVO1lBQzVILE1BQU0sS0FBSyxXQUFXLElBQUksTUFBTSxLQUFLLFVBQVUsSUFBSSxNQUFNLEtBQUssVUFBVSxJQUFJLE1BQU0sS0FBSyxXQUFXLEVBQUUsQ0FBQztZQUFDLE9BQU8sT0FBTyxDQUFDO1FBQUMsQ0FBQztRQUV6SCxPQUFPLE1BQU0sQ0FBQztJQUNoQixDQUFDO0lBRU0sd0JBQXdCLENBQUMsTUFBYyxFQUFFLG9CQUE2QixLQUFLO1FBQ2hGLElBQUksY0FBYyxHQUFHLElBQUksQ0FBQztRQUUxQixJQUFJLGlCQUFpQixFQUFFLENBQUM7WUFDdEIsT0FBTyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsZUFBZSxDQUFDLFdBQVcsR0FBRyxNQUFNLENBQUMsQ0FBQztRQUMxRSxDQUFDO1FBRUQsUUFBUSxNQUFNLEVBQUUsQ0FBQztZQUNmLEtBQUssVUFBVTtnQkFDYixjQUFjLEdBQUcsRUFBRSxNQUFNLEVBQUUsS0FBSyxFQUFFLENBQUM7Z0JBQ25DLE1BQU07WUFFUixLQUFLLEtBQUs7Z0JBQ1IsY0FBYyxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxlQUFlLENBQUMsc0JBQXNCLENBQUMsQ0FBQztnQkFDcEYsTUFBTTtZQUVSLEtBQUssS0FBSztnQkFDUixjQUFjLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLGVBQWUsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO2dCQUNwRixNQUFNO1lBRVIsS0FBSyxZQUFZLENBQUM7WUFDbEIsS0FBSyxTQUFTO2dCQUNaLGNBQWMsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsZUFBZSxDQUFDLDZCQUE2QixDQUFDLElBQUksSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLGVBQWUsQ0FBQywwQkFBMEIsQ0FBQyxDQUFDO2dCQUNySyxNQUFNO1lBRVIsS0FBSyxNQUFNO2dCQUNULGNBQWMsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsZUFBZSxDQUFDLHVCQUF1QixDQUFDLENBQUM7Z0JBQ3JGLGNBQWMsQ0FBQyxRQUFRLEdBQUcsTUFBTSxDQUFDO2dCQUNqQyxNQUFNO1lBRVIsS0FBSyxPQUFPO2dCQUNWLGNBQWMsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsZUFBZSxDQUFDLHVCQUF1QixDQUFDLENBQUM7Z0JBQ3JGLGNBQWMsQ0FBQyxRQUFRLEdBQUcsT0FBTyxDQUFDO2dCQUNsQyxNQUFNO1lBRVIsS0FBSyxLQUFLO2dCQUNSLGNBQWMsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsZUFBZSxDQUFDLHVCQUF1QixDQUFDLENBQUM7Z0JBQ3JGLGNBQWMsQ0FBQyxRQUFRLEdBQUcsS0FBSyxDQUFDO2dCQUNoQyxNQUFNO1lBRVIsS0FBSyxNQUFNO2dCQUNULGNBQWMsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsZUFBZSxDQUFDLHNCQUFzQixDQUFDLENBQUM7Z0JBQ3BGLE1BQU07WUFFUixLQUFLLEtBQUs7Z0JBQ1IsY0FBYyxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxlQUFlLENBQUMsc0JBQXNCLENBQUMsQ0FBQztnQkFDcEYsTUFBTTtZQUVSLEtBQUssTUFBTSxDQUFDO1lBQ1osS0FBSyxVQUFVO2dCQUNiLGNBQWMsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsZUFBZSxDQUFDLHVCQUF1QixDQUFDLElBQUksSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLGVBQWUsQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO2dCQUNoSyxNQUFNO1lBRVIsS0FBSyxLQUFLO2dCQUNSLGNBQWMsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsZUFBZSxDQUFDLHNCQUFzQixDQUFDLENBQUM7Z0JBQ3BGLE1BQU07WUFFUixLQUFLLFdBQVc7Z0JBQ2QsY0FBYyxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxlQUFlLENBQUMsd0JBQXdCLENBQUMsQ0FBQztnQkFDdEYsY0FBYyxDQUFDLFNBQVMsR0FBRyxXQUFXLENBQUM7Z0JBQ3ZDLE1BQU07WUFFUixLQUFLLFVBQVU7Z0JBQ2IsY0FBYyxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxlQUFlLENBQUMsd0JBQXdCLENBQUMsQ0FBQztnQkFDdEYsY0FBYyxDQUFDLFNBQVMsR0FBRyxVQUFVLENBQUM7Z0JBQ3RDLE1BQU07WUFFUixLQUFLLE9BQU87Z0JBQ1YsY0FBYyxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxlQUFlLENBQUMsd0JBQXdCLENBQUMsQ0FBQztnQkFDdEYsY0FBYyxDQUFDLFNBQVMsR0FBRyxPQUFPLENBQUM7Z0JBQ25DLE1BQU07WUFFUixLQUFLLEtBQUs7Z0JBQ1IsY0FBYyxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxlQUFlLENBQUMsc0JBQXNCLENBQUMsQ0FBQztnQkFDcEYsTUFBTTtZQUVSLEtBQUssVUFBVTtnQkFDYixjQUFjLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLGVBQWUsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO2dCQUN0RixjQUFjLENBQUMsU0FBUyxHQUFHLEtBQUssQ0FBQztnQkFDakMsTUFBTTtZQUVSLEtBQUssVUFBVTtnQkFDYixjQUFjLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLGVBQWUsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO2dCQUN0RixjQUFjLENBQUMsU0FBUyxHQUFHLEtBQUssQ0FBQztnQkFDakMsTUFBTTtZQUVSLEtBQUssV0FBVztnQkFDZCxjQUFjLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLGVBQWUsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO2dCQUN0RixjQUFjLENBQUMsU0FBUyxHQUFHLE1BQU0sQ0FBQztnQkFDbEMsTUFBTTtZQUVSLEtBQUssVUFBVTtnQkFDYixjQUFjLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLGVBQWUsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO2dCQUN0RixjQUFjLENBQUMsU0FBUyxHQUFHLEtBQUssQ0FBQztnQkFDakMsTUFBTTtZQUVSLEtBQUssVUFBVTtnQkFDYixjQUFjLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLGVBQWUsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO2dCQUN0RixjQUFjLENBQUMsU0FBUyxHQUFHLEtBQUssQ0FBQztnQkFDakMsTUFBTTtZQUVSLEtBQUssV0FBVztnQkFDZCxjQUFjLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLGVBQWUsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO2dCQUN0RixjQUFjLENBQUMsU0FBUyxHQUFHLE1BQU0sQ0FBQztnQkFDbEMsTUFBTTtZQUVSLEtBQUssVUFBVTtnQkFDYixjQUFjLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLGVBQWUsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO2dCQUN0RixjQUFjLENBQUMsU0FBUyxHQUFHLEtBQUssQ0FBQztnQkFDakMsTUFBTTtZQUVSLEtBQUssV0FBVztnQkFDZCxjQUFjLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLGVBQWUsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO2dCQUN0RixjQUFjLENBQUMsU0FBUyxHQUFHLE1BQU0sQ0FBQztnQkFDbEMsTUFBTTtZQUVSLEtBQUssVUFBVTtnQkFDYixjQUFjLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLGVBQWUsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO2dCQUN0RixjQUFjLENBQUMsU0FBUyxHQUFHLEtBQUssQ0FBQztnQkFDakMsTUFBTTtZQUVSLEtBQUssS0FBSztnQkFDUixjQUFjLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLGVBQWUsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO2dCQUNyRixjQUFjLENBQUMsUUFBUSxHQUFHLEtBQUssQ0FBQztnQkFDaEMsTUFBTTtZQUVSLEtBQUssS0FBSztnQkFDUixjQUFjLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLGVBQWUsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO2dCQUNyRixjQUFjLENBQUMsUUFBUSxHQUFHLEtBQUssQ0FBQztnQkFDaEMsTUFBTTtZQUVSLEtBQUssS0FBSztnQkFDUixjQUFjLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLGVBQWUsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO2dCQUNyRixjQUFjLENBQUMsUUFBUSxHQUFHLEtBQUssQ0FBQztnQkFDaEMsTUFBTTtZQUVSLEtBQUssS0FBSztnQkFDUixjQUFjLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLGVBQWUsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO2dCQUNyRixjQUFjLENBQUMsUUFBUSxHQUFHLEtBQUssQ0FBQztnQkFDaEMsTUFBTTtZQUVSLEtBQUssTUFBTTtnQkFDVCxjQUFjLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLGVBQWUsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO2dCQUNyRixjQUFjLENBQUMsUUFBUSxHQUFHLE1BQU0sQ0FBQztnQkFDakMsTUFBTTtRQUNWLENBQUM7UUFFRCxjQUFjLENBQUMsZUFBZSxHQUFHLENBQUMsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsbUJBQW1CLEtBQUssS0FBSyxDQUFDLENBQUM7UUFDN0YsSUFBSSxDQUFDLENBQUMsTUFBTSxLQUFLLEtBQUssSUFBSSxNQUFNLEtBQUssTUFBTSxDQUFDLEVBQUUsQ0FBQztZQUM3QyxjQUFjLENBQUMsZUFBZSxHQUFHLEtBQUssQ0FBQztRQUN6QyxDQUFDO1FBRUQsT0FBTyxjQUFjLENBQUM7SUFDeEIsQ0FBQzsrR0EzekJVLGFBQWE7bUhBQWIsYUFBYTs7NEZBQWIsYUFBYTtrQkFEekIsVUFBVSIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IEluamVjdGFibGUgfSBmcm9tICdAYW5ndWxhci9jb3JlJztcclxuaW1wb3J0IHsgRXhwb3J0Rm9ybVNldHRpbmdzLCBFeHBvcnRDb21wb25lbnQsIE1lc3NhZ2UgfSBmcm9tICcuL29iamVjdHMnO1xyXG5pbXBvcnQgeyBGb3JtU2VydmljZSB9IGZyb20gJy4uL2Zvcm1zL2Zvcm0uc2VydmljZSc7XHJcbmltcG9ydCB7IEhlbHBlclNlcnZpY2UgfSBmcm9tICcuL2hlbHBlci5zZXJ2aWNlJztcclxuaW1wb3J0IHsgTW9kZWxTZXJ2aWNlIH0gZnJvbSAnLi9tb2RlbC5zZXJ2aWNlJztcclxuaW1wb3J0IHsgQ29sbGVjdGlvbnNTZXJ2aWNlIH0gZnJvbSAnLi9jb2xsZWN0aW9ucy5zZXJ2aWNlJztcclxuaW1wb3J0IHsgQ29udHJvbGxlclNlcnZpY2UgfSBmcm9tICcuL2NvbnRyb2xsZXIuc2VydmljZSc7XHJcbmltcG9ydCB7IFN1YmplY3QsIE9ic2VydmFibGUgfSBmcm9tICdyeGpzJztcclxuXHJcbkBJbmplY3RhYmxlKClcclxuZXhwb3J0IGNsYXNzIEV4cG9ydFNlcnZpY2Uge1xyXG5cclxuICBwdWJsaWMgZXhwb3J0Rm9ybVNldHRpbmdzOiBFeHBvcnRGb3JtU2V0dGluZ3M7XHJcbiAgcHVibGljIGV4cG9ydFNldHRpbmdzOiBhbnkgPSB7fTtcclxuICBwdWJsaWMgZm9ybWF0OiBzdHJpbmc7XHJcbiAgcHVibGljIHNlbmRNYWlsID0gZmFsc2U7XHJcblxyXG4gIHByaXZhdGUgc3ViamVjdCA9IG5ldyBTdWJqZWN0PGFueT4oKTtcclxuXHJcbiAgY29uc3RydWN0b3IocHVibGljIGZvcm1TZXJ2aWNlOiBGb3JtU2VydmljZSwgcHVibGljIGhlbHBlcjogSGVscGVyU2VydmljZSwgcHVibGljIG1vZGVsOiBNb2RlbFNlcnZpY2UsIHB1YmxpYyBjb2xsZWN0aW9uczogQ29sbGVjdGlvbnNTZXJ2aWNlLFxyXG4gICAgcHVibGljIGNvbnRyb2xsZXI6IENvbnRyb2xsZXJTZXJ2aWNlKSB7XHJcblxyXG4gICAgY29udHJvbGxlci5nZXRBY3Rpb25NZXNzYWdlKCkuc3Vic2NyaWJlKChtZXNzYWdlOiBNZXNzYWdlKSA9PiB7XHJcbiAgICAgIHN3aXRjaCAobWVzc2FnZS5hY3Rpb24pIHtcclxuICAgICAgICBjYXNlICdFeHBvcnQnOlxyXG4gICAgICAgICAgdGhpcy5leHBvcnQobWVzc2FnZS5kYXRhLmZvcm1hdCk7XHJcbiAgICAgICAgICBicmVhaztcclxuICAgICAgICBjYXNlICdTZW5kRW1haWwnOlxyXG4gICAgICAgICAgdGhpcy5leHBvcnQobWVzc2FnZS5kYXRhLmZvcm1hdCwgZmFsc2UsIHRydWUpO1xyXG4gICAgICAgICAgYnJlYWs7XHJcbiAgICAgIH1cclxuICAgIH0pO1xyXG4gIH1cclxuXHJcbiAgcHVibGljIGdldE1lc3NhZ2UoKTogT2JzZXJ2YWJsZTxhbnk+IHtcclxuICAgIHJldHVybiB0aGlzLnN1YmplY3QuYXNPYnNlcnZhYmxlKCk7XHJcbiAgfVxyXG5cclxuICBwdWJsaWMgZXhwb3J0KGZvcm1hdDogc3RyaW5nLCB1cGRhdGU6IGJvb2xlYW4gPSBmYWxzZSwgc2VuZE1haWw6IGJvb2xlYW4gPSBmYWxzZSkge1xyXG4gICAgaWYgKCF0aGlzLmhlbHBlci5jaGVja0Nsb3VkQXV0aG9yaXphdGlvbignZXhwb3J0JykpIHtcclxuICAgICAgcmV0dXJuO1xyXG4gICAgfVxyXG5cclxuICAgIHRoaXMuc2VuZE1haWwgPSBzZW5kTWFpbDtcclxuICAgIHRoaXMuZm9ybWF0ID0gZm9ybWF0O1xyXG4gICAgdGhpcy5leHBvcnRTZXR0aW5ncyA9IHRoaXMuZ2V0RGVmYXVsdEV4cG9ydFNldHRpbmdzKHRoaXMuZm9ybWF0KTtcclxuXHJcbiAgICBpZiAoKHRoaXMubW9kZWwub3B0aW9ucy5leHBvcnRzLnNob3dFeHBvcnREaWFsb2cgJiYgIXNlbmRNYWlsKSB8fCAodGhpcy5tb2RlbC5vcHRpb25zLmVtYWlsLnNob3dFeHBvcnREaWFsb2cgJiYgc2VuZE1haWwpKSB7XHJcbiAgICAgIHRoaXMuZXhwb3J0Rm9ybVNldHRpbmdzID0gdGhpcy5nZXRFeHBvcnRTZXRpbmdzKHVwZGF0ZSk7XHJcbiAgICAgIGlmICghdXBkYXRlKSB7XHJcbiAgICAgICAgdGhpcy5mb3JtU2VydmljZS5jbG9zZUZvcm0oJ2V4cG9ydEZvcm0nKTtcclxuICAgICAgICBzZXRUaW1lb3V0KCgpID0+IHtcclxuICAgICAgICAgIHRoaXMuZm9ybVNlcnZpY2Uuc2hvd0Zvcm0oJ2V4cG9ydEZvcm0nKTtcclxuICAgICAgICB9KTtcclxuICAgICAgfVxyXG4gICAgfSBlbHNlIHtcclxuICAgICAgdGhpcy5wb3N0RXhwb3J0KCk7XHJcbiAgICB9XHJcbiAgfVxyXG5cclxuICBwdWJsaWMgcG9zdEV4cG9ydCgpIHtcclxuICAgIGlmICghdGhpcy5zZW5kTWFpbCkge1xyXG4gICAgICB0aGlzLmNvbnRyb2xsZXIucG9zdEV4cG9ydCh0aGlzLmZvcm1hdCwgdGhpcy5leHBvcnRTZXR0aW5ncyk7XHJcbiAgICB9IGVsc2Uge1xyXG4gICAgICB0aGlzLnN1YmplY3QubmV4dCgncG9zdE1haWwnKTtcclxuICAgIH1cclxuICB9XHJcblxyXG4gIGdldEV4cG9ydFNldGluZ3ModXBkYXRlOiBib29sZWFuKTogRXhwb3J0Rm9ybVNldHRpbmdzIHtcclxuICAgIGxldCByZXN1bHQ6IEV4cG9ydEZvcm1TZXR0aW5ncztcclxuXHJcbiAgICB0aGlzLmdldERlZmF1bHRTZXR0aW5ncyh1cGRhdGUpO1xyXG5cclxuICAgIHN3aXRjaCAodGhpcy5mb3JtYXQpIHtcclxuICAgICAgY2FzZSAnRG9jdW1lbnQnOlxyXG4gICAgICAgIHJlc3VsdCA9IHsgY29tcG9uZW50czogdGhpcy5nZXRDb21wb25lbnRzKFsnU2F2ZVJlcG9ydE1kYycsICdTYXZlUmVwb3J0TWR6JywgJ1NhdmVSZXBvcnRNZHgnLCAnU2F2ZVJlcG9ydFBhc3N3b3JkJ10pIH07XHJcbiAgICAgICAgYnJlYWs7XHJcblxyXG4gICAgICBjYXNlICdQZGYnOlxyXG4gICAgICAgIHJlc3VsdCA9IHtcclxuICAgICAgICAgIGNvbXBvbmVudHM6IHRoaXMuZ2V0Q29tcG9uZW50cyhbJ1BhZ2VSYW5nZUdyb3VwJywgJ1BhZ2VSYW5nZUFsbCcsICdQYWdlUmFuZ2VDdXJyZW50UGFnZScsICdQYWdlUmFuZ2VQYWdlcycsICdQYWdlUmFuZ2VQYWdlc1RleHQnLCAnU2V0dGluZ3NHcm91cCcsICdJbWFnZVJlc29sdXRpb24nLCAnSW1hZ2VDb21wcmVzc2lvbk1ldGhvZCcsXHJcbiAgICAgICAgICAgICdBbGxvd0VkaXRhYmxlJywgJ0ltYWdlUXVhbGl0eScsIC8qJ1N0YW5kYXJkUGRmRm9udHMnLCovICdFbWJlZGRlZEZvbnRzJywgLyonVXNlVW5pY29kZScsICdDb21wcmVzc2VkJywqLyAnRXhwb3J0UnRmVGV4dEFzSW1hZ2UnLCAnUGRmQUNvbXBsaWFuY2UnLCAnVXNlRGlnaXRhbFNpZ25hdHVyZScsICdEb2N1bWVudFNlY3VyaXR5QnV0dG9uJywgJ0RpZ2l0YWxTaWduYXR1cmVCdXR0b24nLFxyXG4gICAgICAgICAgICAnT3BlbkFmdGVyRXhwb3J0JywgJ1Bhc3N3b3JkSW5wdXRVc2VyJywgJ1Bhc3N3b3JkSW5wdXRPd25lcicsICdQcmludERvY3VtZW50JywgJ01vZGlmeUNvbnRlbnRzJywgJ0NvcHlUZXh0QW5kR3JhcGhpY3MnLFxyXG4gICAgICAgICAgICAnQWRkT3JNb2RpZnlUZXh0QW5ub3RhdGlvbnMnLCAnS2V5TGVuZ3RoJywgJ0dldENlcnRpZmljYXRlRnJvbUNyeXB0b1VJJywgJ1N1YmplY3ROYW1lU3RyaW5nJ10pLFxyXG4gICAgICAgICAgb3BlbkFmdGVyRXhwb3J0OiB0cnVlXHJcbiAgICAgICAgfTtcclxuICAgICAgICBicmVhaztcclxuICAgICAgY2FzZSAnWHBzJzpcclxuICAgICAgICByZXN1bHQgPSB7XHJcbiAgICAgICAgICBjb21wb25lbnRzOiB0aGlzLmdldENvbXBvbmVudHMoWydQYWdlUmFuZ2VHcm91cCcsICdQYWdlUmFuZ2VBbGwnLCAnUGFnZVJhbmdlQ3VycmVudFBhZ2UnLCAnUGFnZVJhbmdlUGFnZXMnLCAnUGFnZVJhbmdlUGFnZXNUZXh0JywgJ1NldHRpbmdzR3JvdXAnLCAnSW1hZ2VSZXNvbHV0aW9uJywgJ0ltYWdlUXVhbGl0eScsICdPcGVuQWZ0ZXJFeHBvcnQnLFxyXG4gICAgICAgICAgICAnRXhwb3J0UnRmVGV4dEFzSW1hZ2UnXSksXHJcbiAgICAgICAgICBvcGVuQWZ0ZXJFeHBvcnQ6IHRydWVcclxuICAgICAgICB9O1xyXG4gICAgICAgIGJyZWFrO1xyXG4gICAgICBjYXNlICdQb3dlclBvaW50JzpcclxuICAgICAgICByZXN1bHQgPSB7IGNvbXBvbmVudHM6IHRoaXMuZ2V0Q29tcG9uZW50cyhbJ1BhZ2VSYW5nZUdyb3VwJywgJ1BhZ2VSYW5nZUFsbCcsICdQYWdlUmFuZ2VDdXJyZW50UGFnZScsICdQYWdlUmFuZ2VQYWdlcycsICdQYWdlUmFuZ2VQYWdlc1RleHQnLCAnU2V0dGluZ3NHcm91cCcsICdJbWFnZVJlc29sdXRpb24nLCAnSW1hZ2VRdWFsaXR5J10pIH07XHJcbiAgICAgICAgYnJlYWs7XHJcblxyXG4gICAgICBjYXNlICdIdG1sJzpcclxuICAgICAgICByZXN1bHQgPSB7XHJcbiAgICAgICAgICBjb21wb25lbnRzOiB0aGlzLmdldENvbXBvbmVudHMoWydIdG1sVHlwZScsICdab29tJywgJ0ltYWdlRm9ybWF0Rm9ySHRtbCcsICdFeHBvcnRNb2RlJywgJ1VzZUVtYmVkZGVkSW1hZ2VzJywgJ0FkZFBhZ2VCcmVha3MnXSksXHJcbiAgICAgICAgICBvcGVuQWZ0ZXJFeHBvcnQ6IHRydWVcclxuICAgICAgICB9O1xyXG4gICAgICAgIGJyZWFrO1xyXG5cclxuICAgICAgY2FzZSAnSHRtbDUnOlxyXG4gICAgICAgIHJlc3VsdCA9IHtcclxuICAgICAgICAgIGNvbXBvbmVudHM6IHRoaXMuZ2V0Q29tcG9uZW50cyhbJ1BhZ2VSYW5nZUdyb3VwJywgJ1BhZ2VSYW5nZUFsbCcsICdQYWdlUmFuZ2VDdXJyZW50UGFnZScsICdQYWdlUmFuZ2VQYWdlcycsICdQYWdlUmFuZ2VQYWdlc1RleHQnLCAnU2V0dGluZ3NHcm91cCcsICdIdG1sVHlwZScsICdJbWFnZUZvcm1hdEZvckh0bWwnLCAnSW1hZ2VSZXNvbHV0aW9uJyxcclxuICAgICAgICAgICAgJ0ltYWdlUXVhbGl0eScsICdDb250aW51b3VzUGFnZXMnLCAnT3BlbkFmdGVyRXhwb3J0J10pLFxyXG4gICAgICAgICAgb3BlbkFmdGVyRXhwb3J0OiB0cnVlXHJcbiAgICAgICAgfTtcclxuICAgICAgICBicmVhaztcclxuICAgICAgY2FzZSAnTWh0JzpcclxuICAgICAgICByZXN1bHQgPSB7XHJcbiAgICAgICAgICBjb21wb25lbnRzOiB0aGlzLmdldENvbXBvbmVudHMoWydQYWdlUmFuZ2VHcm91cCcsICdQYWdlUmFuZ2VBbGwnLCAnUGFnZVJhbmdlQ3VycmVudFBhZ2UnLCAnUGFnZVJhbmdlUGFnZXMnLCAnUGFnZVJhbmdlUGFnZXNUZXh0JywgJ1NldHRpbmdzR3JvdXAnLCAnSHRtbFR5cGUnLCAnWm9vbScsICdJbWFnZUZvcm1hdEZvckh0bWwnLFxyXG4gICAgICAgICAgICAnRXhwb3J0TW9kZScsICdBZGRQYWdlQnJlYWtzJ10pXHJcbiAgICAgICAgfTtcclxuICAgICAgICBicmVhaztcclxuICAgICAgY2FzZSAnVGV4dCc6XHJcbiAgICAgICAgcmVzdWx0ID0ge1xyXG4gICAgICAgICAgY29tcG9uZW50czogdGhpcy5nZXRDb21wb25lbnRzKFsnUGFnZVJhbmdlR3JvdXAnLCAnUGFnZVJhbmdlQWxsJywgJ1BhZ2VSYW5nZUN1cnJlbnRQYWdlJywgJ1BhZ2VSYW5nZVBhZ2VzJywgJ1BhZ2VSYW5nZVBhZ2VzVGV4dCcsICdTZXR0aW5nc0dyb3VwJywgJ0tpbGxTcGFjZUxpbmVzJyxcclxuICAgICAgICAgICAgJ1B1dEZlZWRQYWdlQ29kZScsICdEcmF3Qm9yZGVyJywgJ0N1dExvbmdMaW5lcycsICdCb3JkZXJUeXBlJywgJ1pvb21YJywgJ1pvb21ZJywgJ0VuY29kaW5nVGV4dE9yQ3N2RmlsZSddKVxyXG4gICAgICAgIH07XHJcbiAgICAgICAgYnJlYWs7XHJcbiAgICAgIGNhc2UgJ1J0Zic6XHJcbiAgICAgICAgcmVzdWx0ID0ge1xyXG4gICAgICAgICAgY29tcG9uZW50czogdGhpcy5nZXRDb21wb25lbnRzKFsnUGFnZVJhbmdlR3JvdXAnLCAnUGFnZVJhbmdlQWxsJywgJ1BhZ2VSYW5nZUN1cnJlbnRQYWdlJywgJ1BhZ2VSYW5nZVBhZ2VzJywgJ1BhZ2VSYW5nZVBhZ2VzVGV4dCcsICdTZXR0aW5nc0dyb3VwJywgJ0ltYWdlUmVzb2x1dGlvbicsXHJcbiAgICAgICAgICAgICdJbWFnZVF1YWxpdHknLCAnRXhwb3J0TW9kZVJ0ZicsICdVc2VQYWdlSGVhZGVyc0FuZEZvb3RlcnMnLCAnUmVtb3ZlRW1wdHlTcGFjZUF0Qm90dG9tJ10pXHJcbiAgICAgICAgfTtcclxuICAgICAgICBicmVhaztcclxuICAgICAgY2FzZSAnV29yZCc6XHJcbiAgICAgICAgcmVzdWx0ID0ge1xyXG4gICAgICAgICAgY29tcG9uZW50czogdGhpcy5nZXRDb21wb25lbnRzKFsnUGFnZVJhbmdlR3JvdXAnLCAnUGFnZVJhbmdlQWxsJywgJ1BhZ2VSYW5nZUN1cnJlbnRQYWdlJywgJ1BhZ2VSYW5nZVBhZ2VzJywgJ1BhZ2VSYW5nZVBhZ2VzVGV4dCcsICdTZXR0aW5nc0dyb3VwJywgJ0ltYWdlUmVzb2x1dGlvbicsXHJcbiAgICAgICAgICAgICdJbWFnZVF1YWxpdHknLCAnVXNlUGFnZUhlYWRlcnNBbmRGb290ZXJzJywgJ1JlbW92ZUVtcHR5U3BhY2VBdEJvdHRvbSddKVxyXG4gICAgICAgIH07XHJcbiAgICAgICAgYnJlYWs7XHJcbiAgICAgIGNhc2UgJ09kdCc6XHJcbiAgICAgICAgcmVzdWx0ID0ge1xyXG4gICAgICAgICAgY29tcG9uZW50czogdGhpcy5nZXRDb21wb25lbnRzKFsnUGFnZVJhbmdlR3JvdXAnLCAnUGFnZVJhbmdlQWxsJywgJ1BhZ2VSYW5nZUN1cnJlbnRQYWdlJywgJ1BhZ2VSYW5nZVBhZ2VzJywgJ1BhZ2VSYW5nZVBhZ2VzVGV4dCcsICdTZXR0aW5nc0dyb3VwJywgJ0ltYWdlUmVzb2x1dGlvbicsXHJcbiAgICAgICAgICAgICdJbWFnZVF1YWxpdHknLCAnUmVtb3ZlRW1wdHlTcGFjZUF0Qm90dG9tJ10pXHJcbiAgICAgICAgfTtcclxuICAgICAgICBicmVhaztcclxuICAgICAgY2FzZSAnRXhjZWxCaWZmJzpcclxuICAgICAgY2FzZSAnRXhjZWwnOlxyXG4gICAgICAgIHJlc3VsdCA9IHtcclxuICAgICAgICAgIGNvbXBvbmVudHM6IHRoaXMuZ2V0Q29tcG9uZW50cyhbJ1BhZ2VSYW5nZUdyb3VwJywgJ1BhZ2VSYW5nZUFsbCcsICdQYWdlUmFuZ2VDdXJyZW50UGFnZScsICdQYWdlUmFuZ2VQYWdlcycsICdQYWdlUmFuZ2VQYWdlc1RleHQnLCAnU2V0dGluZ3NHcm91cCcsICdFeGNlbFR5cGUnLCAnSW1hZ2VSZXNvbHV0aW9uJyxcclxuICAgICAgICAgICAgJ0ltYWdlUXVhbGl0eScsICdEYXRhRXhwb3J0TW9kZScsICdFeHBvcnRPYmplY3RGb3JtYXR0aW5nJywgJ1VzZU9uZVBhZ2VIZWFkZXJBbmRGb290ZXInLCAnRXhwb3J0RWFjaFBhZ2VUb1NoZWV0JywgJ0V4cG9ydFBhZ2VCcmVha3MnXSlcclxuICAgICAgICB9O1xyXG4gICAgICAgIGJyZWFrO1xyXG4gICAgICBjYXNlICdFeGNlbFhtbCc6XHJcbiAgICAgICAgcmVzdWx0ID0ge1xyXG4gICAgICAgICAgY29tcG9uZW50czogdGhpcy5nZXRDb21wb25lbnRzKFsnUGFnZVJhbmdlR3JvdXAnLCAnUGFnZVJhbmdlQWxsJywgJ1BhZ2VSYW5nZUN1cnJlbnRQYWdlJywgJ1BhZ2VSYW5nZVBhZ2VzJywgJ1BhZ2VSYW5nZVBhZ2VzVGV4dCcsICdTZXR0aW5nc0dyb3VwJywgJ0V4Y2VsVHlwZSddKVxyXG4gICAgICAgIH07XHJcbiAgICAgICAgYnJlYWs7XHJcbiAgICAgIGNhc2UgJ09kcyc6XHJcbiAgICAgICAgcmVzdWx0ID0ge1xyXG4gICAgICAgICAgY29tcG9uZW50czogdGhpcy5nZXRDb21wb25lbnRzKFsnUGFnZVJhbmdlR3JvdXAnLCAnUGFnZVJhbmdlQWxsJywgJ1BhZ2VSYW5nZUN1cnJlbnRQYWdlJywgJ1BhZ2VSYW5nZVBhZ2VzJywgJ1BhZ2VSYW5nZVBhZ2VzVGV4dCcsICdTZXR0aW5nc0dyb3VwJywgJ0ltYWdlUmVzb2x1dGlvbicsXHJcbiAgICAgICAgICAgICdJbWFnZVF1YWxpdHknXSlcclxuICAgICAgICB9O1xyXG4gICAgICAgIGJyZWFrO1xyXG4gICAgICBjYXNlICdDc3YnOlxyXG4gICAgICAgIHJlc3VsdCA9IHtcclxuICAgICAgICAgIGNvbXBvbmVudHM6IHRoaXMuZ2V0Q29tcG9uZW50cyhbJ1BhZ2VSYW5nZUdyb3VwJywgJ1BhZ2VSYW5nZUFsbCcsICdQYWdlUmFuZ2VDdXJyZW50UGFnZScsICdQYWdlUmFuZ2VQYWdlcycsICdQYWdlUmFuZ2VQYWdlc1RleHQnLCAnU2V0dGluZ3NHcm91cCcsICdEYXRhVHlwZScsICdFbmNvZGluZ1RleHRPckNzdkZpbGUnLFxyXG4gICAgICAgICAgICAnU2VwYXJhdG9yJywgJ0RhdGFFeHBvcnRNb2RlJywgJ1NraXBDb2x1bW5IZWFkZXJzJ10pXHJcbiAgICAgICAgfTtcclxuICAgICAgICBicmVhaztcclxuICAgICAgY2FzZSAnRGJmJzpcclxuICAgICAgICByZXN1bHQgPSB7XHJcbiAgICAgICAgICBjb21wb25lbnRzOiB0aGlzLmdldENvbXBvbmVudHMoWydQYWdlUmFuZ2VHcm91cCcsICdQYWdlUmFuZ2VBbGwnLCAnUGFnZVJhbmdlQ3VycmVudFBhZ2UnLCAnUGFnZVJhbmdlUGFnZXMnLCAnUGFnZVJhbmdlUGFnZXNUZXh0JywgJ1NldHRpbmdzR3JvdXAnLCAnRGF0YVR5cGUnLCAnRW5jb2RpbmdEYmZGaWxlJ10pXHJcbiAgICAgICAgfTtcclxuICAgICAgICBicmVhaztcclxuICAgICAgY2FzZSAnRGlmJzpcclxuICAgICAgICByZXN1bHQgPSB7XHJcbiAgICAgICAgICBjb21wb25lbnRzOiB0aGlzLmdldENvbXBvbmVudHMoWydQYWdlUmFuZ2VHcm91cCcsICdQYWdlUmFuZ2VBbGwnLCAnUGFnZVJhbmdlQ3VycmVudFBhZ2UnLCAnUGFnZVJhbmdlUGFnZXMnLCAnUGFnZVJhbmdlUGFnZXNUZXh0JywgJ1NldHRpbmdzR3JvdXAnLCAnRGF0YVR5cGUnLCAnRXhwb3J0RGF0YU9ubHknLFxyXG4gICAgICAgICAgICAnVXNlRGVmYXVsdFN5c3RlbUVuY29kaW5nJywgJ0VuY29kaW5nRGlmRmlsZSddKVxyXG4gICAgICAgIH07XHJcbiAgICAgICAgYnJlYWs7XHJcbiAgICAgIGNhc2UgJ1N5bGsnOlxyXG4gICAgICAgIHJlc3VsdCA9IHtcclxuICAgICAgICAgIGNvbXBvbmVudHM6IHRoaXMuZ2V0Q29tcG9uZW50cyhbJ1BhZ2VSYW5nZUdyb3VwJywgJ1BhZ2VSYW5nZUFsbCcsICdQYWdlUmFuZ2VDdXJyZW50UGFnZScsICdQYWdlUmFuZ2VQYWdlcycsICdQYWdlUmFuZ2VQYWdlc1RleHQnLCAnU2V0dGluZ3NHcm91cCcsICdEYXRhVHlwZScsICdFeHBvcnREYXRhT25seScsXHJcbiAgICAgICAgICAgICdVc2VEZWZhdWx0U3lzdGVtRW5jb2RpbmcnLCAnRW5jb2RpbmdEaWZGaWxlJ10pXHJcbiAgICAgICAgfTtcclxuICAgICAgICBicmVhaztcclxuICAgICAgY2FzZSAnSnNvbic6XHJcbiAgICAgICAgcmVzdWx0ID0ge1xyXG4gICAgICAgICAgY29tcG9uZW50czogdGhpcy5nZXRDb21wb25lbnRzKFsnUGFnZVJhbmdlR3JvdXAnLCAnUGFnZVJhbmdlQWxsJywgJ1BhZ2VSYW5nZUN1cnJlbnRQYWdlJywgJ1BhZ2VSYW5nZVBhZ2VzJywgJ1BhZ2VSYW5nZVBhZ2VzVGV4dCcsICdTZXR0aW5nc0dyb3VwJywgJ0RhdGFUeXBlJywgJ0RhdGFFeHBvcnRNb2RlJ10pXHJcbiAgICAgICAgfTtcclxuICAgICAgICBicmVhaztcclxuICAgICAgY2FzZSAnWG1sJzpcclxuICAgICAgICByZXN1bHQgPSB7XHJcbiAgICAgICAgICBjb21wb25lbnRzOiB0aGlzLmdldENvbXBvbmVudHMoWydQYWdlUmFuZ2VHcm91cCcsICdQYWdlUmFuZ2VBbGwnLCAnUGFnZVJhbmdlQ3VycmVudFBhZ2UnLCAnUGFnZVJhbmdlUGFnZXMnLCAnUGFnZVJhbmdlUGFnZXNUZXh0JywgJ1NldHRpbmdzR3JvdXAnLCAnRGF0YVR5cGUnXSlcclxuICAgICAgICB9O1xyXG4gICAgICAgIGJyZWFrO1xyXG4gICAgICBjYXNlICdJbWFnZUJtcCc6XHJcbiAgICAgIGNhc2UgJ0ltYWdlR2lmJzpcclxuICAgICAgY2FzZSAnSW1hZ2VKcGVnJzpcclxuICAgICAgY2FzZSAnSW1hZ2VQY3gnOlxyXG4gICAgICBjYXNlICdJbWFnZVBuZyc6XHJcbiAgICAgIGNhc2UgJ0ltYWdlVGlmZic6XHJcbiAgICAgIGNhc2UgJ0ltYWdlRW1mJzpcclxuICAgICAgY2FzZSAnSW1hZ2VTdmcnOlxyXG4gICAgICBjYXNlICdJbWFnZVN2Z3onOlxyXG4gICAgICAgIHJlc3VsdCA9IHtcclxuICAgICAgICAgIGNvbXBvbmVudHM6IHRoaXMuZ2V0Q29tcG9uZW50cyhbJ1BhZ2VSYW5nZUdyb3VwJywgJ1BhZ2VSYW5nZUFsbCcsICdQYWdlUmFuZ2VDdXJyZW50UGFnZScsICdQYWdlUmFuZ2VQYWdlcycsICdQYWdlUmFuZ2VQYWdlc1RleHQnLCAnU2V0dGluZ3NHcm91cCcsICdJbWFnZVR5cGUnLCAnSW1hZ2Vab29tJywgJ0ltYWdlUmVzb2x1dGlvbicsXHJcbiAgICAgICAgICAgICdJbWFnZUZvcm1hdCcsICdEaXRoZXJpbmdUeXBlJywgJ1RpZmZDb21wcmVzc2lvblNjaGVtZScsICdDb21wcmVzc1RvQXJjaGl2ZScsICdDdXRFZGdlcyddKVxyXG4gICAgICAgIH07XHJcbiAgICAgICAgYnJlYWs7XHJcbiAgICB9XHJcblxyXG4gICAgcmVzdWx0Lmdyb3VwcyA9IHRoaXMuZ2V0R3JvdXBzKHRoaXMuZm9ybWF0KTtcclxuICAgIHJlc3VsdC51cGRhdGUgPSB1cGRhdGU7XHJcbiAgICByZXR1cm4gcmVzdWx0O1xyXG4gIH1cclxuXHJcbiAgZ2V0Q29tcG9uZW50cyhuYW1lczogc3RyaW5nW10pOiBFeHBvcnRDb21wb25lbnRbXSB7XHJcbiAgICBjb25zdCBtcmduID0gJzhweCc7XHJcbiAgICBjb25zdCBjb21wb25lbnRzOiBFeHBvcnRDb21wb25lbnRbXSA9IFtdO1xyXG4gICAgbmFtZXMuZm9yRWFjaChuYW1lID0+IHtcclxuICAgICAgc3dpdGNoIChuYW1lKSB7XHJcbiAgICAgICAgY2FzZSAnSW1hZ2VUeXBlJzpcclxuICAgICAgICAgIGNvbXBvbmVudHMucHVzaCh7XHJcbiAgICAgICAgICAgIG5hbWUsIHR5cGU6ICdEcm9wRG93bkxpc3RGb3JFeHBvcnRGb3JtJywgbGFiZWw6IHRoaXMubW9kZWwubG9jKCdUeXBlJyksIHdpZHRoOiAxNjAsIHRvb2x0aXA6IHRoaXMubW9kZWwubG9jKCdUeXBlVG9vbHRpcCcpLFxyXG4gICAgICAgICAgICBtYXJnaW46ICcycHggJyArIG1yZ24gKyAnIDJweCAnICsgbXJnbiwgaXRlbXM6IHRoaXMuY29sbGVjdGlvbnMuZ2V0SW1hZ2VUeXBlc0l0ZW1zKClcclxuICAgICAgICAgIH0pO1xyXG4gICAgICAgICAgYnJlYWs7XHJcblxyXG4gICAgICAgIGNhc2UgJ0RhdGFUeXBlJzpcclxuICAgICAgICAgIGNvbXBvbmVudHMucHVzaCh7XHJcbiAgICAgICAgICAgIG5hbWUsIHR5cGU6ICdEcm9wRG93bkxpc3RGb3JFeHBvcnRGb3JtJywgbGFiZWw6IHRoaXMubW9kZWwubG9jKCdUeXBlJyksIHdpZHRoOiAxNjAsIHRvb2x0aXA6IHRoaXMubW9kZWwubG9jKCdUeXBlVG9vbHRpcCcpLFxyXG4gICAgICAgICAgICBtYXJnaW46ICcycHggJyArIG1yZ24gKyAnIDJweCAnICsgbXJnbiwgaXRlbXM6IHRoaXMuY29sbGVjdGlvbnMuZ2V0RGF0YVR5cGVzSXRlbXMoKVxyXG4gICAgICAgICAgfSk7XHJcbiAgICAgICAgICBicmVhaztcclxuXHJcbiAgICAgICAgY2FzZSAnRXhjZWxUeXBlJzpcclxuICAgICAgICAgIGNvbXBvbmVudHMucHVzaCh7XHJcbiAgICAgICAgICAgIG5hbWUsIHR5cGU6ICdEcm9wRG93bkxpc3RGb3JFeHBvcnRGb3JtJywgbGFiZWw6IHRoaXMubW9kZWwubG9jKCdUeXBlJyksIHdpZHRoOiAxNjAsIHRvb2x0aXA6IHRoaXMubW9kZWwubG9jKCdUeXBlVG9vbHRpcCcpLFxyXG4gICAgICAgICAgICBtYXJnaW46ICcycHggJyArIG1yZ24gKyAnIDJweCAnICsgbXJnbiwgaXRlbXM6IHRoaXMuY29sbGVjdGlvbnMuZ2V0RXhjZWxUeXBlc0l0ZW1zKClcclxuICAgICAgICAgIH0pO1xyXG4gICAgICAgICAgYnJlYWs7XHJcblxyXG4gICAgICAgIGNhc2UgJ0h0bWxUeXBlJzpcclxuICAgICAgICAgIGNvbXBvbmVudHMucHVzaCh7XHJcbiAgICAgICAgICAgIG5hbWUsIHR5cGU6ICdEcm9wRG93bkxpc3RGb3JFeHBvcnRGb3JtJywgbGFiZWw6IHRoaXMubW9kZWwubG9jKCdUeXBlJyksIHdpZHRoOiAxNjAsIHRvb2x0aXA6IHRoaXMubW9kZWwubG9jKCdUeXBlVG9vbHRpcCcpLFxyXG4gICAgICAgICAgICBtYXJnaW46ICcycHggJyArIG1yZ24gKyAnIDJweCAnICsgbXJnbiwgaXRlbXM6IHRoaXMuY29sbGVjdGlvbnMuZ2V0SHRtbFR5cGVzSXRlbXMoKVxyXG4gICAgICAgICAgfSk7XHJcbiAgICAgICAgICBicmVhaztcclxuXHJcbiAgICAgICAgY2FzZSAnWm9vbSc6XHJcbiAgICAgICAgICBjb21wb25lbnRzLnB1c2goe1xyXG4gICAgICAgICAgICBuYW1lLCB0eXBlOiAnRHJvcERvd25MaXN0Rm9yRXhwb3J0Rm9ybScsIGxhYmVsOiB0aGlzLm1vZGVsLmxvYygnWm9vbUh0bWwnKSwgd2lkdGg6IDE2MCwgdG9vbHRpcDogdGhpcy5tb2RlbC5sb2MoJ1pvb21IdG1sVG9vbHRpcCcpLFxyXG4gICAgICAgICAgICBtYXJnaW46ICcycHggJyArIG1yZ24gKyAnIDJweCAnICsgbXJnbiwgaXRlbXM6IHRoaXMuY29sbGVjdGlvbnMuZ2V0Wm9vbUl0ZW1zKClcclxuICAgICAgICAgIH0pO1xyXG4gICAgICAgICAgYnJlYWs7XHJcblxyXG4gICAgICAgIGNhc2UgJ0ltYWdlWm9vbSc6XHJcbiAgICAgICAgICBjb21wb25lbnRzLnB1c2goe1xyXG4gICAgICAgICAgICBuYW1lLCB0eXBlOiAnRHJvcERvd25MaXN0Rm9yRXhwb3J0Rm9ybScsIGxhYmVsOiB0aGlzLm1vZGVsLmxvYygnWm9vbUh0bWwnKSwgd2lkdGg6IDE2MCwgdG9vbHRpcDogdGhpcy5tb2RlbC5sb2MoJ1pvb21IdG1sVG9vbHRpcCcpLFxyXG4gICAgICAgICAgICBtYXJnaW46ICcycHggJyArIG1yZ24gKyAnIDJweCAnICsgbXJnbiwgaXRlbXM6IHRoaXMuY29sbGVjdGlvbnMuZ2V0Wm9vbUl0ZW1zKClcclxuICAgICAgICAgIH0pO1xyXG4gICAgICAgICAgYnJlYWs7XHJcblxyXG4gICAgICAgIGNhc2UgJ0ltYWdlRm9ybWF0Rm9ySHRtbCc6XHJcbiAgICAgICAgICBjb21wb25lbnRzLnB1c2goe1xyXG4gICAgICAgICAgICBuYW1lLCB0eXBlOiAnRHJvcERvd25MaXN0Rm9yRXhwb3J0Rm9ybScsIGxhYmVsOiB0aGlzLm1vZGVsLmxvYygnSW1hZ2VGb3JtYXRGb3JIdG1sJyksIHdpZHRoOiAxNjAsIHRvb2x0aXA6IHRoaXMubW9kZWwubG9jKCdJbWFnZUZvcm1hdEZvckh0bWxUb29sdGlwJyksXHJcbiAgICAgICAgICAgIG1hcmdpbjogJzJweCAnICsgbXJnbiArICcgMnB4ICcgKyBtcmduLCBpdGVtczogdGhpcy5jb2xsZWN0aW9ucy5nZXRJbWFnZUZvcm1hdEZvckh0bWxJdGVtcygpXHJcbiAgICAgICAgICB9KTtcclxuICAgICAgICAgIGJyZWFrO1xyXG5cclxuICAgICAgICBjYXNlICdFeHBvcnRNb2RlJzpcclxuICAgICAgICAgIGNvbXBvbmVudHMucHVzaCh7XHJcbiAgICAgICAgICAgIG5hbWUsIHR5cGU6ICdEcm9wRG93bkxpc3RGb3JFeHBvcnRGb3JtJywgbGFiZWw6IHRoaXMubW9kZWwubG9jKCdFeHBvcnRNb2RlJyksIHdpZHRoOiAxNjAsIHRvb2x0aXA6IHRoaXMubW9kZWwubG9jKCdFeHBvcnRNb2RlVG9vbHRpcCcpLFxyXG4gICAgICAgICAgICBtYXJnaW46ICcycHggJyArIG1yZ24gKyAnIDJweCAnICsgbXJnbiwgaXRlbXM6IHRoaXMuY29sbGVjdGlvbnMuZ2V0RXhwb3J0TW9kZUl0ZW1zKClcclxuICAgICAgICAgIH0pO1xyXG4gICAgICAgICAgYnJlYWs7XHJcblxyXG4gICAgICAgIGNhc2UgJ0NvbXByZXNzVG9BcmNoaXZlJzpcclxuICAgICAgICAgIGNvbXBvbmVudHMucHVzaCh7XHJcbiAgICAgICAgICAgIG5hbWUsIHR5cGU6ICdDaGVja0JveCcsIGNhcHRpb246IHRoaXMubW9kZWwubG9jKCdDb21wcmVzc1RvQXJjaGl2ZScpLCB0b29sdGlwOiB0aGlzLm1vZGVsLmxvYygnQ29tcHJlc3NUb0FyY2hpdmVUb29sdGlwJyksXHJcbiAgICAgICAgICAgIG1hcmdpbjogJzRweCAnICsgbXJnbiArICcgNHB4ICcgKyBtcmduXHJcbiAgICAgICAgICB9KTtcclxuICAgICAgICAgIGJyZWFrO1xyXG5cclxuICAgICAgICBjYXNlICdVc2VFbWJlZGRlZEltYWdlcyc6XHJcbiAgICAgICAgICBjb21wb25lbnRzLnB1c2goe1xyXG4gICAgICAgICAgICBuYW1lLCB0eXBlOiAnQ2hlY2tCb3gnLCBjYXB0aW9uOiB0aGlzLm1vZGVsLmxvYygnRW1iZWRkZWRJbWFnZURhdGEnKSwgdG9vbHRpcDogdGhpcy5tb2RlbC5sb2MoJ0VtYmVkZGVkSW1hZ2VEYXRhVG9vbHRpcCcpLFxyXG4gICAgICAgICAgICBtYXJnaW46ICc0cHggJyArIG1yZ24gKyAnIDRweCAnICsgbXJnblxyXG4gICAgICAgICAgfSk7XHJcbiAgICAgICAgICBicmVhaztcclxuXHJcbiAgICAgICAgY2FzZSAnQWRkUGFnZUJyZWFrcyc6XHJcbiAgICAgICAgICBjb21wb25lbnRzLnB1c2goe1xyXG4gICAgICAgICAgICBuYW1lLCB0eXBlOiAnQ2hlY2tCb3gnLCBjYXB0aW9uOiB0aGlzLm1vZGVsLmxvYygnQWRkUGFnZUJyZWFrcycpLCB0b29sdGlwOiB0aGlzLm1vZGVsLmxvYygnQWRkUGFnZUJyZWFrc1Rvb2x0aXAnKSxcclxuICAgICAgICAgICAgbWFyZ2luOiAnNHB4ICcgKyBtcmduICsgJyA0cHggJyArIG1yZ25cclxuICAgICAgICAgIH0pO1xyXG4gICAgICAgICAgYnJlYWs7XHJcblxyXG4gICAgICAgIGNhc2UgJ0ltYWdlUmVzb2x1dGlvbic6XHJcbiAgICAgICAgICBjb21wb25lbnRzLnB1c2goe1xyXG4gICAgICAgICAgICBuYW1lLCB0eXBlOiAnRHJvcERvd25MaXN0Rm9yRXhwb3J0Rm9ybScsIGxhYmVsOiB0aGlzLm1vZGVsLmxvYygnSW1hZ2VSZXNvbHV0aW9uJyksIHdpZHRoOiAxNjAsIHRvb2x0aXA6IHRoaXMubW9kZWwubG9jKCdJbWFnZVJlc29sdXRpb25Ub29sdGlwJyksXHJcbiAgICAgICAgICAgIG1hcmdpbjogJzJweCA0cHggMnB4ICcgKyBtcmduLCBpdGVtczogdGhpcy5jb2xsZWN0aW9ucy5nZXRJbWFnZVJlc29sdXRpb25JdGVtcygpXHJcbiAgICAgICAgICB9KTtcclxuICAgICAgICAgIGJyZWFrO1xyXG5cclxuICAgICAgICBjYXNlICdJbWFnZUNvbXByZXNzaW9uTWV0aG9kJzpcclxuICAgICAgICAgIGNvbXBvbmVudHMucHVzaCh7XHJcbiAgICAgICAgICAgIG5hbWUsIHR5cGU6ICdEcm9wRG93bkxpc3RGb3JFeHBvcnRGb3JtJywgbGFiZWw6IHRoaXMubW9kZWwubG9jKCdJbWFnZUNvbXByZXNzaW9uTWV0aG9kJyksIHdpZHRoOiAxNjAsIHRvb2x0aXA6IHRoaXMubW9kZWwubG9jKCdJbWFnZUNvbXByZXNzaW9uTWV0aG9kVG9vbHRpcCcpLFxyXG4gICAgICAgICAgICBtYXJnaW46ICcycHggJyArIG1yZ24gKyAnIDJweCAnICsgbXJnbiwgaXRlbXM6IHRoaXMuY29sbGVjdGlvbnMuZ2V0SW1hZ2VDb21wcmVzc2lvbk1ldGhvZEl0ZW1zKClcclxuICAgICAgICAgIH0pO1xyXG4gICAgICAgICAgYnJlYWs7XHJcblxyXG4gICAgICAgIGNhc2UgJ0FsbG93RWRpdGFibGUnOlxyXG4gICAgICAgICAgY29tcG9uZW50cy5wdXNoKHtcclxuICAgICAgICAgICAgbmFtZSwgdHlwZTogJ0Ryb3BEb3duTGlzdEZvckV4cG9ydEZvcm0nLCBsYWJlbDogdGhpcy5tb2RlbC5sb2MoJ0FsbG93RWRpdGFibGUnKSwgd2lkdGg6IDE2MCwgdG9vbHRpcDogdGhpcy5tb2RlbC5sb2MoJ0FsbG93RWRpdGFibGVUb29sdGlwJyksXHJcbiAgICAgICAgICAgIG1hcmdpbjogJzJweCAnICsgbXJnbiArICcgMnB4ICcgKyBtcmduLCBpdGVtczogdGhpcy5jb2xsZWN0aW9ucy5nZXRBbGxvd0VkaXRhYmxlSXRlbXMoKVxyXG4gICAgICAgICAgfSk7XHJcbiAgICAgICAgICBicmVhaztcclxuXHJcbiAgICAgICAgY2FzZSAnSW1hZ2VRdWFsaXR5JzpcclxuICAgICAgICAgIGNvbXBvbmVudHMucHVzaCh7XHJcbiAgICAgICAgICAgIG5hbWUsIHR5cGU6ICdEcm9wRG93bkxpc3RGb3JFeHBvcnRGb3JtJywgbGFiZWw6IHRoaXMubW9kZWwubG9jKCdJbWFnZVF1YWxpdHknKSwgd2lkdGg6IDE2MCwgdG9vbHRpcDogdGhpcy5tb2RlbC5sb2MoJ0ltYWdlUXVhbGl0eVRvb2x0aXAnKSxcclxuICAgICAgICAgICAgbWFyZ2luOiAnMnB4IDRweCAycHggJyArIG1yZ24sIGl0ZW1zOiB0aGlzLmNvbGxlY3Rpb25zLmdldEltYWdlUXVhbGl0eUl0ZW1zKClcclxuICAgICAgICAgIH0pO1xyXG4gICAgICAgICAgYnJlYWs7XHJcblxyXG4gICAgICAgIGNhc2UgJ0NvbnRpbnVvdXNQYWdlcyc6XHJcbiAgICAgICAgICBjb21wb25lbnRzLnB1c2goe1xyXG4gICAgICAgICAgICBuYW1lLCB0eXBlOiAnQ2hlY2tCb3gnLCBjYXB0aW9uOiB0aGlzLm1vZGVsLmxvYygnQ29udGludW91c1BhZ2VzJyksIHRvb2x0aXA6IHRoaXMubW9kZWwubG9jKCdDb250aW51b3VzUGFnZXNUb29sdGlwJyksXHJcbiAgICAgICAgICAgIG1hcmdpbjogJzRweCAnICsgbXJnbiArICcgNHB4ICcgKyBtcmduXHJcbiAgICAgICAgICB9KTtcclxuICAgICAgICAgIGJyZWFrO1xyXG5cclxuICAgICAgICBjYXNlICdTdGFuZGFyZFBkZkZvbnRzJzpcclxuICAgICAgICAgIGNvbXBvbmVudHMucHVzaCh7XHJcbiAgICAgICAgICAgIG5hbWUsIHR5cGU6ICdDaGVja0JveCcsIGNhcHRpb246IHRoaXMubW9kZWwubG9jKCdTdGFuZGFyZFBERkZvbnRzJyksIHRvb2x0aXA6IHRoaXMubW9kZWwubG9jKCdTdGFuZGFyZFBERkZvbnRzVG9vbHRpcCcpLFxyXG4gICAgICAgICAgICBtYXJnaW46ICc0cHggJyArIG1yZ24gKyAnIDRweCAnICsgbXJnblxyXG4gICAgICAgICAgfSk7XHJcbiAgICAgICAgICBicmVhaztcclxuXHJcbiAgICAgICAgY2FzZSAnRW1iZWRkZWRGb250cyc6XHJcbiAgICAgICAgICBjb21wb25lbnRzLnB1c2goe1xyXG4gICAgICAgICAgICBuYW1lLCB0eXBlOiAnQ2hlY2tCb3gnLCBjYXB0aW9uOiB0aGlzLm1vZGVsLmxvYygnRW1iZWRkZWRGb250cycpLCB0b29sdGlwOiB0aGlzLm1vZGVsLmxvYygnRW1iZWRkZWRGb250c1Rvb2x0aXAnKSxcclxuICAgICAgICAgICAgbWFyZ2luOiAnNHB4ICcgKyBtcmduICsgJyA0cHggJyArIG1yZ25cclxuICAgICAgICAgIH0pO1xyXG4gICAgICAgICAgYnJlYWs7XHJcblxyXG4gICAgICAgIGNhc2UgJ1VzZVVuaWNvZGUnOlxyXG4gICAgICAgICAgY29tcG9uZW50cy5wdXNoKHtcclxuICAgICAgICAgICAgbmFtZSwgdHlwZTogJ0NoZWNrQm94JywgY2FwdGlvbjogdGhpcy5tb2RlbC5sb2MoJ1VzZVVuaWNvZGUnKSwgdG9vbHRpcDogdGhpcy5tb2RlbC5sb2MoJ1VzZVVuaWNvZGVUb29sdGlwJyksXHJcbiAgICAgICAgICAgIG1hcmdpbjogJzRweCAnICsgbXJnbiArICcgNHB4ICcgKyBtcmduXHJcbiAgICAgICAgICB9KTtcclxuICAgICAgICAgIGJyZWFrO1xyXG5cclxuICAgICAgICBjYXNlICdDb21wcmVzc2VkJzpcclxuICAgICAgICAgIGNvbXBvbmVudHMucHVzaCh7XHJcbiAgICAgICAgICAgIG5hbWUsIHR5cGU6ICdDaGVja0JveCcsIGNhcHRpb246IHRoaXMubW9kZWwubG9jKCdDb21wcmVzc2VkJyksIHRvb2x0aXA6IHRoaXMubW9kZWwubG9jKCdDb21wcmVzc2VkVG9vbHRpcCcpLFxyXG4gICAgICAgICAgICBtYXJnaW46ICc0cHggJyArIG1yZ24gKyAnIDRweCAnICsgbXJnblxyXG4gICAgICAgICAgfSk7XHJcbiAgICAgICAgICBicmVhaztcclxuXHJcbiAgICAgICAgY2FzZSAnRXhwb3J0UnRmVGV4dEFzSW1hZ2UnOlxyXG4gICAgICAgICAgY29tcG9uZW50cy5wdXNoKHtcclxuICAgICAgICAgICAgbmFtZSwgdHlwZTogJ0NoZWNrQm94JywgY2FwdGlvbjogdGhpcy5tb2RlbC5sb2MoJ0V4cG9ydFJ0ZlRleHRBc0ltYWdlJyksIHRvb2x0aXA6IHRoaXMubW9kZWwubG9jKCdFeHBvcnRSdGZUZXh0QXNJbWFnZVRvb2x0aXAnKSxcclxuICAgICAgICAgICAgbWFyZ2luOiAnNHB4ICcgKyBtcmduICsgJyA0cHggJyArIG1yZ25cclxuICAgICAgICAgIH0pO1xyXG4gICAgICAgICAgYnJlYWs7XHJcblxyXG4gICAgICAgIGNhc2UgJ1BkZkFDb21wbGlhbmNlJzpcclxuICAgICAgICAgIGNvbXBvbmVudHMucHVzaCh7XHJcbiAgICAgICAgICAgIG5hbWUsIHR5cGU6ICdDaGVja0JveCcsIGNhcHRpb246IHRoaXMubW9kZWwubG9jKCdQZGZBQ29tcGxpYW5jZScpLCB0b29sdGlwOiB0aGlzLm1vZGVsLmxvYygnUGRmQUNvbXBsaWFuY2VUb29sdGlwJyksXHJcbiAgICAgICAgICAgIG1hcmdpbjogJzRweCAnICsgbXJnbiArICcgNHB4ICcgKyBtcmduXHJcbiAgICAgICAgICB9KTtcclxuICAgICAgICAgIGJyZWFrO1xyXG5cclxuICAgICAgICBjYXNlICdLaWxsU3BhY2VMaW5lcyc6XHJcbiAgICAgICAgICBjb21wb25lbnRzLnB1c2goe1xyXG4gICAgICAgICAgICBuYW1lLCB0eXBlOiAnQ2hlY2tCb3gnLCBjYXB0aW9uOiB0aGlzLm1vZGVsLmxvYygnS2lsbFNwYWNlTGluZXMnKSwgdG9vbHRpcDogdGhpcy5tb2RlbC5sb2MoJ0tpbGxTcGFjZUxpbmVzVG9vbHRpcCcpLFxyXG4gICAgICAgICAgICBtYXJnaW46ICc0cHggJyArIG1yZ24gKyAnIDRweCAnICsgbXJnblxyXG4gICAgICAgICAgfSk7XHJcbiAgICAgICAgICBicmVhaztcclxuXHJcbiAgICAgICAgY2FzZSAnUHV0RmVlZFBhZ2VDb2RlJzpcclxuICAgICAgICAgIGNvbXBvbmVudHMucHVzaCh7XHJcbiAgICAgICAgICAgIG5hbWUsIHR5cGU6ICdDaGVja0JveCcsIGNhcHRpb246IHRoaXMubW9kZWwubG9jKCdQdXRGZWVkUGFnZUNvZGUnKSwgdG9vbHRpcDogdGhpcy5tb2RlbC5sb2MoJ1B1dEZlZWRQYWdlQ29kZVRvb2x0aXAnKSxcclxuICAgICAgICAgICAgbWFyZ2luOiAnNHB4ICcgKyBtcmduICsgJyA0cHggJyArIG1yZ25cclxuICAgICAgICAgIH0pO1xyXG4gICAgICAgICAgYnJlYWs7XHJcblxyXG4gICAgICAgIGNhc2UgJ0RyYXdCb3JkZXInOlxyXG4gICAgICAgICAgY29tcG9uZW50cy5wdXNoKHtcclxuICAgICAgICAgICAgbmFtZSwgdHlwZTogJ0NoZWNrQm94JywgY2FwdGlvbjogdGhpcy5tb2RlbC5sb2MoJ0RyYXdCb3JkZXInKSwgdG9vbHRpcDogdGhpcy5tb2RlbC5sb2MoJ0RyYXdCb3JkZXJUb29sdGlwJyksXHJcbiAgICAgICAgICAgIG1hcmdpbjogJzRweCAnICsgbXJnbiArICcgNHB4ICcgKyBtcmduXHJcbiAgICAgICAgICB9KTtcclxuICAgICAgICAgIGJyZWFrO1xyXG5cclxuICAgICAgICBjYXNlICdDdXRMb25nTGluZXMnOlxyXG4gICAgICAgICAgY29tcG9uZW50cy5wdXNoKHtcclxuICAgICAgICAgICAgbmFtZSwgdHlwZTogJ0NoZWNrQm94JywgY2FwdGlvbjogdGhpcy5tb2RlbC5sb2MoJ0N1dExvbmdMaW5lcycpLCB0b29sdGlwOiB0aGlzLm1vZGVsLmxvYygnQ3V0TG9uZ0xpbmVzVG9vbHRpcCcpLFxyXG4gICAgICAgICAgICBtYXJnaW46ICc0cHggJyArIG1yZ24gKyAnIDRweCAnICsgbXJnblxyXG4gICAgICAgICAgfSk7XHJcbiAgICAgICAgICBicmVhaztcclxuXHJcbiAgICAgICAgY2FzZSAnQm9yZGVyVHlwZSc6XHJcbiAgICAgICAgICBjb21wb25lbnRzLnB1c2goe1xyXG4gICAgICAgICAgICBuYW1lLCB0eXBlOiAnRHJvcERvd25MaXN0Rm9yRXhwb3J0Rm9ybScsIGxhYmVsOiB0aGlzLm1vZGVsLmxvYygnQm9yZGVyVHlwZScpLCB3aWR0aDogMTYwLCB0b29sdGlwOiB0aGlzLm1vZGVsLmxvYygnQm9yZGVyVHlwZVRvb2x0aXAnKSxcclxuICAgICAgICAgICAgbWFyZ2luOiAnMnB4ICcgKyBtcmduICsgJyAycHggJyArIG1yZ24sIGl0ZW1zOiB0aGlzLmNvbGxlY3Rpb25zLmdldEJvcmRlclR5cGVJdGVtcygpXHJcbiAgICAgICAgICB9KTtcclxuICAgICAgICAgIGJyZWFrO1xyXG5cclxuICAgICAgICBjYXNlICdab29tWCc6XHJcbiAgICAgICAgICBjb21wb25lbnRzLnB1c2goe1xyXG4gICAgICAgICAgICBuYW1lLCB0eXBlOiAnRHJvcERvd25MaXN0Rm9yRXhwb3J0Rm9ybScsIGxhYmVsOiB0aGlzLm1vZGVsLmxvYygnWm9vbVhZJykucmVwbGFjZSgnOicsICcnKSArICcgWCAnLCB3aWR0aDogMTYwLCB0b29sdGlwOiB0aGlzLm1vZGVsLmxvYygnWm9vbVhZVG9vbHRpcCcpLFxyXG4gICAgICAgICAgICBtYXJnaW46ICcycHggJyArIG1yZ24gKyAnIDJweCAnICsgbXJnbiwgaXRlbXM6IHRoaXMuY29sbGVjdGlvbnMuZ2V0Wm9vbUl0ZW1zKClcclxuICAgICAgICAgIH0pO1xyXG4gICAgICAgICAgYnJlYWs7XHJcblxyXG4gICAgICAgIGNhc2UgJ1pvb21ZJzpcclxuICAgICAgICAgIGNvbXBvbmVudHMucHVzaCh7XHJcbiAgICAgICAgICAgIG5hbWUsIHR5cGU6ICdEcm9wRG93bkxpc3RGb3JFeHBvcnRGb3JtJywgbGFiZWw6IHRoaXMubW9kZWwubG9jKCdab29tWFknKS5yZXBsYWNlKCc6JywgJycpICsgJyBZICcsIHdpZHRoOiAxNjAsIHRvb2x0aXA6IHRoaXMubW9kZWwubG9jKCdab29tWFlUb29sdGlwJyksXHJcbiAgICAgICAgICAgIG1hcmdpbjogJzJweCAnICsgbXJnbiArICcgMnB4ICcgKyBtcmduLCBpdGVtczogdGhpcy5jb2xsZWN0aW9ucy5nZXRab29tSXRlbXMoKVxyXG4gICAgICAgICAgfSk7XHJcbiAgICAgICAgICBicmVhaztcclxuXHJcbiAgICAgICAgY2FzZSAnRW5jb2RpbmdUZXh0T3JDc3ZGaWxlJzpcclxuICAgICAgICAgIGNvbXBvbmVudHMucHVzaCh7XHJcbiAgICAgICAgICAgIG5hbWUsIHR5cGU6ICdEcm9wRG93bkxpc3RGb3JFeHBvcnRGb3JtJywgbGFiZWw6IHRoaXMubW9kZWwubG9jKCdFbmNvZGluZ0RhdGEnKSwgd2lkdGg6IDE2MCwgdG9vbHRpcDogdGhpcy5tb2RlbC5sb2MoJ0VuY29kaW5nRGF0YVRvb2x0aXAnKSxcclxuICAgICAgICAgICAgbWFyZ2luOiAnMnB4ICcgKyBtcmduICsgJyAycHggJyArIG1yZ24sIGl0ZW1zOiB0aGlzLmNvbGxlY3Rpb25zLmdldEVuY29kaW5nRGF0YUl0ZW1zKClcclxuICAgICAgICAgIH0pO1xyXG4gICAgICAgICAgYnJlYWs7XHJcblxyXG4gICAgICAgIGNhc2UgJ0ltYWdlRm9ybWF0JzpcclxuICAgICAgICAgIGNvbXBvbmVudHMucHVzaCh7XHJcbiAgICAgICAgICAgIG5hbWUsIHR5cGU6ICdEcm9wRG93bkxpc3RGb3JFeHBvcnRGb3JtJywgbGFiZWw6IHRoaXMubW9kZWwubG9jKCdJbWFnZUZvcm1hdCcpLCB3aWR0aDogMTYwLCB0b29sdGlwOiB0aGlzLm1vZGVsLmxvYygnSW1hZ2VGb3JtYXRUb29sdGlwJyksXHJcbiAgICAgICAgICAgIG1hcmdpbjogJzJweCAnICsgbXJnbiArICcgMnB4ICcgKyBtcmduLCBpdGVtczogdGhpcy5jb2xsZWN0aW9ucy5nZXRJbWFnZUZvcm1hdEl0ZW1zKClcclxuICAgICAgICAgIH0pO1xyXG4gICAgICAgICAgYnJlYWs7XHJcblxyXG4gICAgICAgIGNhc2UgJ0RpdGhlcmluZ1R5cGUnOlxyXG4gICAgICAgICAgY29tcG9uZW50cy5wdXNoKHtcclxuICAgICAgICAgICAgbmFtZSwgdHlwZTogJ0Ryb3BEb3duTGlzdEZvckV4cG9ydEZvcm0nLCBsYWJlbDogdGhpcy5tb2RlbC5sb2MoJ01vbm9jaHJvbWVEaXRoZXJpbmdUeXBlJyksIHdpZHRoOiAxNjAsIHRvb2x0aXA6IHRoaXMubW9kZWwubG9jKCdNb25vY2hyb21lRGl0aGVyaW5nVHlwZVRvb2x0aXAnKSxcclxuICAgICAgICAgICAgbWFyZ2luOiAnMnB4ICcgKyBtcmduICsgJyAycHggJyArIG1yZ24sIGl0ZW1zOiB0aGlzLmNvbGxlY3Rpb25zLmdldE1vbm9jaHJvbWVEaXRoZXJpbmdUeXBlSXRlbXMoKVxyXG4gICAgICAgICAgfSk7XHJcbiAgICAgICAgICBicmVhaztcclxuXHJcbiAgICAgICAgY2FzZSAnVGlmZkNvbXByZXNzaW9uU2NoZW1lJzpcclxuICAgICAgICAgIGNvbXBvbmVudHMucHVzaCh7XHJcbiAgICAgICAgICAgIG5hbWUsIHR5cGU6ICdEcm9wRG93bkxpc3RGb3JFeHBvcnRGb3JtJywgbGFiZWw6IHRoaXMubW9kZWwubG9jKCdUaWZmQ29tcHJlc3Npb25TY2hlbWUnKSwgd2lkdGg6IDE2MCwgdG9vbHRpcDogdGhpcy5tb2RlbC5sb2MoJ1RpZmZDb21wcmVzc2lvblNjaGVtZVRvb2x0aXAnKSxcclxuICAgICAgICAgICAgbWFyZ2luOiAnMnB4ICcgKyBtcmduICsgJyAycHggJyArIG1yZ24sIGl0ZW1zOiB0aGlzLmNvbGxlY3Rpb25zLmdldFRpZmZDb21wcmVzc2lvblNjaGVtZUl0ZW1zKClcclxuICAgICAgICAgIH0pO1xyXG4gICAgICAgICAgYnJlYWs7XHJcblxyXG4gICAgICAgIGNhc2UgJ0N1dEVkZ2VzJzpcclxuICAgICAgICAgIGNvbXBvbmVudHMucHVzaCh7XHJcbiAgICAgICAgICAgIG5hbWUsIHR5cGU6ICdDaGVja0JveCcsIGNhcHRpb246IHRoaXMubW9kZWwubG9jKCdDdXRFZGdlcycpLCB0b29sdGlwOiB0aGlzLm1vZGVsLmxvYygnQ3V0RWRnZXNUb29sdGlwJyksXHJcbiAgICAgICAgICAgIG1hcmdpbjogJzRweCAnICsgbXJnbiArICcgNHB4ICcgKyBtcmduXHJcbiAgICAgICAgICB9KTtcclxuICAgICAgICAgIGJyZWFrO1xyXG5cclxuICAgICAgICBjYXNlICdNdWx0aXBsZUZpbGVzJzpcclxuICAgICAgICAgIGNvbXBvbmVudHMucHVzaCh7XHJcbiAgICAgICAgICAgIG5hbWUsIHR5cGU6ICdDaGVja0JveCcsIGNhcHRpb246IHRoaXMubW9kZWwubG9jKCdNdWx0aXBsZUZpbGVzJyksIHRvb2x0aXA6IHRoaXMubW9kZWwubG9jKCdNdWx0aXBsZUZpbGVzVG9vbHRpcCcpLFxyXG4gICAgICAgICAgICBtYXJnaW46ICc0cHggJyArIG1yZ24gKyAnIDRweCAnICsgbXJnblxyXG4gICAgICAgICAgfSk7XHJcbiAgICAgICAgICBicmVhaztcclxuXHJcbiAgICAgICAgY2FzZSAnRXhwb3J0RGF0YU9ubHknOlxyXG4gICAgICAgICAgY29tcG9uZW50cy5wdXNoKHtcclxuICAgICAgICAgICAgbmFtZSwgdHlwZTogJ0NoZWNrQm94JywgY2FwdGlvbjogdGhpcy5tb2RlbC5sb2MoJ0V4cG9ydERhdGFPbmx5JyksIHRvb2x0aXA6IHRoaXMubW9kZWwubG9jKCdFeHBvcnREYXRhT25seVRvb2x0aXAnKSxcclxuICAgICAgICAgICAgbWFyZ2luOiAnNHB4ICcgKyBtcmduICsgJyA0cHggJyArIG1yZ25cclxuICAgICAgICAgIH0pO1xyXG4gICAgICAgICAgYnJlYWs7XHJcblxyXG4gICAgICAgIGNhc2UgJ1VzZURlZmF1bHRTeXN0ZW1FbmNvZGluZyc6XHJcbiAgICAgICAgICBjb21wb25lbnRzLnB1c2goe1xyXG4gICAgICAgICAgICBuYW1lLCB0eXBlOiAnQ2hlY2tCb3gnLCBjYXB0aW9uOiB0aGlzLm1vZGVsLmxvYygnVXNlRGVmYXVsdFN5c3RlbUVuY29kaW5nJyksIHRvb2x0aXA6IHRoaXMubW9kZWwubG9jKCdVc2VEZWZhdWx0U3lzdGVtRW5jb2RpbmdUb29sdGlwJyksXHJcbiAgICAgICAgICAgIG1hcmdpbjogJzRweCAnICsgbXJnbiArICcgNHB4ICcgKyBtcmduXHJcbiAgICAgICAgICB9KTtcclxuICAgICAgICAgIGJyZWFrO1xyXG5cclxuICAgICAgICBjYXNlICdFbmNvZGluZ0RpZkZpbGUnOlxyXG4gICAgICAgICAgY29tcG9uZW50cy5wdXNoKHtcclxuICAgICAgICAgICAgbmFtZSwgdHlwZTogJ0Ryb3BEb3duTGlzdEZvckV4cG9ydEZvcm0nLCBsYWJlbDogdGhpcy5tb2RlbC5sb2MoJ0VuY29kaW5nRGlmRmlsZScpLCB3aWR0aDogMTYwLCB0b29sdGlwOiB0aGlzLm1vZGVsLmxvYygnRW5jb2RpbmdEaWZGaWxlVG9vbHRpcCcpLFxyXG4gICAgICAgICAgICBtYXJnaW46ICcycHggJyArIG1yZ24gKyAnIDJweCAnICsgbXJnbiwgaXRlbXM6IHRoaXMuY29sbGVjdGlvbnMuZ2V0RW5jb2RpbmdEaWZGaWxlSXRlbXMoKVxyXG4gICAgICAgICAgfSk7XHJcbiAgICAgICAgICBicmVhaztcclxuXHJcbiAgICAgICAgY2FzZSAnRXhwb3J0TW9kZVJ0Zic6XHJcbiAgICAgICAgICBjb21wb25lbnRzLnB1c2goe1xyXG4gICAgICAgICAgICBuYW1lLCB0eXBlOiAnRHJvcERvd25MaXN0Rm9yRXhwb3J0Rm9ybScsIGxhYmVsOiB0aGlzLm1vZGVsLmxvYygnRXhwb3J0TW9kZVJ0ZicpLCB3aWR0aDogMTYwLCB0b29sdGlwOiB0aGlzLm1vZGVsLmxvYygnRXhwb3J0TW9kZVJ0ZlRvb2x0aXAnKSxcclxuICAgICAgICAgICAgbWFyZ2luOiAnMnB4ICcgKyBtcmduICsgJyAycHggJyArIG1yZ24sIGl0ZW1zOiB0aGlzLmNvbGxlY3Rpb25zLmdldEV4cG9ydE1vZGVSdGZJdGVtcygpXHJcbiAgICAgICAgICB9KTtcclxuICAgICAgICAgIGJyZWFrO1xyXG5cclxuICAgICAgICBjYXNlICdVc2VQYWdlSGVhZGVyc0FuZEZvb3RlcnMnOlxyXG4gICAgICAgICAgY29tcG9uZW50cy5wdXNoKHtcclxuICAgICAgICAgICAgbmFtZSwgdHlwZTogJ0NoZWNrQm94JywgY2FwdGlvbjogdGhpcy5tb2RlbC5sb2MoJ1VzZVBhZ2VIZWFkZXJzRm9vdGVycycpLCB0b29sdGlwOiB0aGlzLm1vZGVsLmxvYygnVXNlUGFnZUhlYWRlcnNGb290ZXJzVG9vbHRpcCcpLFxyXG4gICAgICAgICAgICBtYXJnaW46ICc0cHggJyArIG1yZ24gKyAnIDRweCAnICsgbXJnblxyXG4gICAgICAgICAgfSk7XHJcbiAgICAgICAgICBicmVhaztcclxuXHJcbiAgICAgICAgY2FzZSAnUmVtb3ZlRW1wdHlTcGFjZUF0Qm90dG9tJzpcclxuICAgICAgICAgIGNvbXBvbmVudHMucHVzaCh7XHJcbiAgICAgICAgICAgIG5hbWUsIHR5cGU6ICdDaGVja0JveCcsIGNhcHRpb246IHRoaXMubW9kZWwubG9jKCdSZW1vdmVFbXB0eVNwYWNlJyksIHRvb2x0aXA6IHRoaXMubW9kZWwubG9jKCdSZW1vdmVFbXB0eVNwYWNlVG9vbHRpcCcpLFxyXG4gICAgICAgICAgICBtYXJnaW46ICc0cHggJyArIG1yZ24gKyAnIDRweCAnICsgbXJnblxyXG4gICAgICAgICAgfSk7XHJcbiAgICAgICAgICBicmVhaztcclxuXHJcbiAgICAgICAgY2FzZSAnU2VwYXJhdG9yJzpcclxuICAgICAgICAgIGNvbXBvbmVudHMucHVzaCh7XHJcbiAgICAgICAgICAgIG5hbWUsIHR5cGU6ICdUZXh0Qm94JywgbGFiZWw6IHRoaXMubW9kZWwubG9jKCdTZXBhcmF0b3InKSwgdG9vbHRpcDogdGhpcy5tb2RlbC5sb2MoJ1NlcGFyYXRvclRvb2x0aXAnKSwgd2lkdGg6IDE2MCxcclxuICAgICAgICAgICAgbWFyZ2luOiAnMnB4ICcgKyBtcmduICsgJyAycHggJyArIG1yZ25cclxuICAgICAgICAgIH0pO1xyXG4gICAgICAgICAgYnJlYWs7XHJcblxyXG4gICAgICAgIGNhc2UgJ0RhdGFFeHBvcnRNb2RlJzpcclxuICAgICAgICAgIGNvbXBvbmVudHMucHVzaCh7XHJcbiAgICAgICAgICAgIG5hbWUsIHR5cGU6ICdEcm9wRG93bkxpc3RGb3JFeHBvcnRGb3JtJywgbGFiZWw6IHRoaXMubW9kZWwubG9jKCdCYW5kc0ZpbHRlcicpLCB3aWR0aDogMTYwLCB0b29sdGlwOiB0aGlzLm1vZGVsLmxvYygnQmFuZHNGaWx0ZXJUb29sdGlwJyksXHJcbiAgICAgICAgICAgIG1hcmdpbjogJzJweCAnICsgbXJnbiArICcgMnB4ICcgKyBtcmduLCBpdGVtczogdGhpcy5jb2xsZWN0aW9ucy5nZXREYXRhRXhwb3J0TW9kZUl0ZW1zKClcclxuICAgICAgICAgIH0pO1xyXG4gICAgICAgICAgYnJlYWs7XHJcblxyXG4gICAgICAgIGNhc2UgJ1NraXBDb2x1bW5IZWFkZXJzJzpcclxuICAgICAgICAgIGNvbXBvbmVudHMucHVzaCh7XHJcbiAgICAgICAgICAgIG5hbWUsIHR5cGU6ICdDaGVja0JveCcsIGNhcHRpb246IHRoaXMubW9kZWwubG9jKCdTa2lwQ29sdW1uSGVhZGVycycpLCB0b29sdGlwOiB0aGlzLm1vZGVsLmxvYygnU2tpcENvbHVtbkhlYWRlcnNUb29sdGlwJyksXHJcbiAgICAgICAgICAgIG1hcmdpbjogJzRweCAnICsgbXJnbiArICcgNHB4ICcgKyBtcmduXHJcbiAgICAgICAgICB9KTtcclxuICAgICAgICAgIGJyZWFrO1xyXG5cclxuICAgICAgICBjYXNlICdFeHBvcnRPYmplY3RGb3JtYXR0aW5nJzpcclxuICAgICAgICAgIGNvbXBvbmVudHMucHVzaCh7XHJcbiAgICAgICAgICAgIG5hbWUsIHR5cGU6ICdDaGVja0JveCcsIGNhcHRpb246IHRoaXMubW9kZWwubG9jKCdFeHBvcnRPYmplY3RGb3JtYXR0aW5nJyksIHRvb2x0aXA6IHRoaXMubW9kZWwubG9jKCdFeHBvcnRPYmplY3RGb3JtYXR0aW5nVG9vbHRpcCcpLFxyXG4gICAgICAgICAgICBtYXJnaW46ICc0cHggJyArIG1yZ24gKyAnIDRweCAnICsgbXJnblxyXG4gICAgICAgICAgfSk7XHJcbiAgICAgICAgICBicmVhaztcclxuXHJcbiAgICAgICAgY2FzZSAnVXNlT25lUGFnZUhlYWRlckFuZEZvb3Rlcic6XHJcbiAgICAgICAgICBjb21wb25lbnRzLnB1c2goe1xyXG4gICAgICAgICAgICBuYW1lLCB0eXBlOiAnQ2hlY2tCb3gnLCBjYXB0aW9uOiB0aGlzLm1vZGVsLmxvYygnVXNlT25lUGFnZUhlYWRlckZvb3RlcicpLCB0b29sdGlwOiB0aGlzLm1vZGVsLmxvYygnVXNlT25lUGFnZUhlYWRlckZvb3RlclRvb2x0aXAnKSxcclxuICAgICAgICAgICAgbWFyZ2luOiAnNHB4ICcgKyBtcmduICsgJyA0cHggJyArIG1yZ25cclxuICAgICAgICAgIH0pO1xyXG4gICAgICAgICAgYnJlYWs7XHJcblxyXG4gICAgICAgIGNhc2UgJ0V4cG9ydEVhY2hQYWdlVG9TaGVldCc6XHJcbiAgICAgICAgICBjb21wb25lbnRzLnB1c2goe1xyXG4gICAgICAgICAgICBuYW1lLCB0eXBlOiAnQ2hlY2tCb3gnLCBjYXB0aW9uOiB0aGlzLm1vZGVsLmxvYygnRXhwb3J0RWFjaFBhZ2VUb1NoZWV0JyksIHRvb2x0aXA6IHRoaXMubW9kZWwubG9jKCdFeHBvcnRFYWNoUGFnZVRvU2hlZXRUb29sdGlwJyksXHJcbiAgICAgICAgICAgIG1hcmdpbjogJzRweCAnICsgbXJnbiArICcgNHB4ICcgKyBtcmduXHJcbiAgICAgICAgICB9KTtcclxuICAgICAgICAgIGJyZWFrO1xyXG5cclxuICAgICAgICBjYXNlICdFeHBvcnRQYWdlQnJlYWtzJzpcclxuICAgICAgICAgIGNvbXBvbmVudHMucHVzaCh7XHJcbiAgICAgICAgICAgIG5hbWUsIHR5cGU6ICdDaGVja0JveCcsIGNhcHRpb246IHRoaXMubW9kZWwubG9jKCdFeHBvcnRQYWdlQnJlYWtzJyksIHRvb2x0aXA6IHRoaXMubW9kZWwubG9jKCdFeHBvcnRQYWdlQnJlYWtzVG9vbHRpcCcpLFxyXG4gICAgICAgICAgICBtYXJnaW46ICc0cHggJyArIG1yZ24gKyAnIDRweCAnICsgbXJnblxyXG4gICAgICAgICAgfSk7XHJcbiAgICAgICAgICBicmVhaztcclxuXHJcbiAgICAgICAgY2FzZSAnRW5jb2RpbmdEYmZGaWxlJzpcclxuICAgICAgICAgIGNvbXBvbmVudHMucHVzaCh7XHJcbiAgICAgICAgICAgIG5hbWUsIHR5cGU6ICdEcm9wRG93bkxpc3RGb3JFeHBvcnRGb3JtJywgbGFiZWw6IHRoaXMubW9kZWwubG9jKCdFbmNvZGluZ0RiZkZpbGUnKSwgd2lkdGg6IDE2MCwgdG9vbHRpcDogdGhpcy5tb2RlbC5sb2MoJ0VuY29kaW5nRGJmRmlsZVRvb2x0aXAnKSxcclxuICAgICAgICAgICAgbWFyZ2luOiAnMnB4ICcgKyBtcmduICsgJyAycHggJyArIG1yZ24sIGl0ZW1zOiB0aGlzLmNvbGxlY3Rpb25zLmdldEVuY29kaW5nRGJmRmlsZUl0ZW1zKClcclxuICAgICAgICAgIH0pO1xyXG4gICAgICAgICAgYnJlYWs7XHJcblxyXG4gICAgICAgIGNhc2UgJ0RvY3VtZW50U2VjdXJpdHlCdXR0b24nOlxyXG4gICAgICAgICAgY29tcG9uZW50cy5wdXNoKHtcclxuICAgICAgICAgICAgbmFtZSwgdHlwZTogJ1NtYWxsQnV0dG9uJywgY2FwdGlvbjogdGhpcy5tb2RlbC5sb2MoJ0RvY3VtZW50U2VjdXJpdHlCdXR0b24nKSwgd2lkdGg6IDE2MCxcclxuICAgICAgICAgICAgbWFyZ2luOiAnMnB4ICcgKyBtcmduICsgJyAycHggJyArIG1yZ24sIGl0ZW1zOiB0aGlzLmNvbGxlY3Rpb25zLmdldEVuY29kaW5nRGJmRmlsZUl0ZW1zKClcclxuICAgICAgICAgIH0pO1xyXG4gICAgICAgICAgYnJlYWs7XHJcblxyXG4gICAgICAgIGNhc2UgJ1VzZURpZ2l0YWxTaWduYXR1cmUnOlxyXG4gICAgICAgICAgY29tcG9uZW50cy5wdXNoKHtcclxuICAgICAgICAgICAgbmFtZSwgY2FwdGlvbjogdGhpcy5tb2RlbC5sb2MoJ0RpZ2l0YWxTaWduYXR1cmVCdXR0b24nKSwgdHlwZTogJ0Ryb3BEb3duTGlzdEZvckV4cG9ydEZvcm0nLCB0b29sdGlwOiB0aGlzLm1vZGVsLmxvYygnVXNlRGlnaXRhbFNpZ25hdHVyZVRvb2x0aXAnKSwgd2lkdGg6IDE2MCxcclxuICAgICAgICAgICAgaXRlbXM6IHRoaXMuY29sbGVjdGlvbnMuZ2V0UGRmU2VjdXJpdHlDZXJ0aWZpY2F0ZXNJdGVtcygpLCBtYXJnaW46ICc0cHggJyArIG1yZ24gKyAnIDRweCAnICsgbXJnblxyXG4gICAgICAgICAgfSk7XHJcbiAgICAgICAgICBicmVhaztcclxuXHJcbiAgICAgICAgZGVmYXVsdDpcclxuICAgICAgICAgIGJyZWFrO1xyXG4gICAgICB9XHJcbiAgICB9KTtcclxuICAgIHJldHVybiBjb21wb25lbnRzO1xyXG4gIH1cclxuXHJcbiAgZ2V0R3JvdXBzKGZvcm1hdDogc3RyaW5nKTogYW55IHtcclxuICAgIGNvbnN0IG9wZW5pbmdHcm91cHNTdHIgPSB0aGlzLmhlbHBlci5nZXRDb29raWUoJ1N0aW11bHNvZnRXZWJWaWV3ZXJFeHBvcnRTZXR0aW5nc09wZW5pbmdHcm91cHMnKTtcclxuICAgIGNvbnN0IG9wZW5pbmdHcm91cHMgPSBvcGVuaW5nR3JvdXBzU3RyID8gSlNPTi5wYXJzZShvcGVuaW5nR3JvdXBzU3RyKSA6IG51bGw7XHJcbiAgICBjb25zdCBwYWdlUmFuZ2VBbGxJc0Rpc2FibGVkID0gZm9ybWF0LmluZGV4T2YoJ0ltYWdlJykgPT09IDAgJiYgZm9ybWF0ICE9PSAnSW1hZ2VUaWZmJztcclxuXHJcbiAgICByZXR1cm4ge1xyXG4gICAgICBzYXZpbmdSZXBvcnRHcm91cDogeyB2aXNpYmxlOiBmb3JtYXQgPT09ICdEb2N1bWVudCcsIG9wZW5lZDogb3BlbmluZ0dyb3VwcyA/IG9wZW5pbmdHcm91cHMuU2F2aW5nUmVwb3J0R3JvdXAgOiB0cnVlIH0sXHJcbiAgICAgIHBhZ2VSYW5nZUdyb3VwOiB7IHZpc2libGU6IGZvcm1hdCAhPT0gJ0RvY3VtZW50Jywgb3BlbmVkOiBvcGVuaW5nR3JvdXBzID8gb3BlbmluZ0dyb3Vwcy5QYWdlUmFuZ2VHcm91cCA6IHRydWUsIHBhZ2VSYW5nZUFsbElzRGlzYWJsZWQgfSxcclxuICAgICAgc2V0dGluZ3NHcm91cDogeyB2aXNpYmxlOiBmb3JtYXQgIT09ICdEb2N1bWVudCcsIG9wZW5lZDogb3BlbmluZ0dyb3VwcyA/IG9wZW5pbmdHcm91cHMuU2V0dGluZ3NHcm91cCA6IGZhbHNlIH1cclxuICAgIH07XHJcbiAgfVxyXG5cclxuICBnZXRTdG9yZWRTZXR0aW5ncyhmb3JtYXQ/OiBzdHJpbmcpIHtcclxuICAgIGxldCBleHBvcnRTZXR0aW5nc1N0ciA9IHRoaXMubW9kZWwub3B0aW9ucy5leHBvcnRzLnN0b3JlRXhwb3J0U2V0dGluZ3MgPyB0aGlzLmhlbHBlci5nZXRDb29raWUoJ1N0aW11bHNvZnRXZWJWaWV3ZXJFeHBvcnRTZXR0aW5ncycgKyB0aGlzLmdldENvbW1vbkV4cG9ydEZvcm1hdChmb3JtYXQpKSA6IG51bGw7XHJcbiAgICByZXR1cm4gZXhwb3J0U2V0dGluZ3NTdHIgIT0gbnVsbCA/IEpTT04ucGFyc2UoZXhwb3J0U2V0dGluZ3NTdHIpIDogbnVsbDtcclxuICB9XHJcblxyXG4gIGdldERlZmF1bHRTZXR0aW5ncyh1cGRhdGU6IGJvb2xlYW4pOiBhbnkge1xyXG4gICAgbGV0IHNldHRpbmdzOiBhbnk7XHJcbiAgICBjb25zdCBleHBvcnRTZXR0aW5ncyA9IHRoaXMuZ2V0U3RvcmVkU2V0dGluZ3MoKTtcclxuICAgIGlmIChleHBvcnRTZXR0aW5ncyAhPSBudWxsKSB7XHJcbiAgICAgIGxldCBleHBvcnRGb3JtYXQgPSBleHBvcnRTZXR0aW5ncy5JbWFnZVR5cGUgfHwgZXhwb3J0U2V0dGluZ3MuRGF0YVR5cGUgfHwgZXhwb3J0U2V0dGluZ3MuSHRtbFR5cGU7XHJcbiAgICAgIGRlbGV0ZSBleHBvcnRTZXR0aW5ncy5FeGNlbFR5cGU7XHJcbiAgICAgIHRoaXMuZm9ybWF0ID0gIXVwZGF0ZSAmJiBleHBvcnRGb3JtYXQgPyAoZXhwb3J0U2V0dGluZ3MuSW1hZ2VUeXBlID8gJ0ltYWdlJyArIGV4cG9ydEZvcm1hdCA6IGV4cG9ydEZvcm1hdCkgOiB0aGlzLmZvcm1hdDtcclxuXHJcbiAgICAgIGNvbnN0IGRlZlNldHRpbmdzID0gdGhpcy5nZXREZWZhdWx0RXhwb3J0U2V0dGluZ3ModGhpcy5mb3JtYXQpO1xyXG4gICAgICBjb25zdCByZXN1bHRTZXR0aW5nczogYW55ID0ge307XHJcbiAgICAgIE9iamVjdC5rZXlzKGRlZlNldHRpbmdzKS5mb3JFYWNoKGtleSA9PiByZXN1bHRTZXR0aW5nc1trZXldID0gZXhwb3J0U2V0dGluZ3Nba2V5XSB8fCBkZWZTZXR0aW5nc1trZXldKTtcclxuICAgICAgT2JqZWN0LmtleXMoZXhwb3J0U2V0dGluZ3MpLmZvckVhY2goa2V5ID0+IHJlc3VsdFNldHRpbmdzW2tleV0gPSBleHBvcnRTZXR0aW5nc1trZXldIHx8IGRlZlNldHRpbmdzW2tleV0pO1xyXG5cclxuICAgICAgaWYgKHRoaXMubW9kZWwub3B0aW9ucy5leHBvcnRzLm9wZW5BZnRlckV4cG9ydCA9PT0gdHJ1ZSB8fCB0aGlzLm1vZGVsLm9wdGlvbnMuZXhwb3J0cy5vcGVuQWZ0ZXJFeHBvcnQgPT09IGZhbHNlKSB7XHJcbiAgICAgICAgcmVzdWx0U2V0dGluZ3MuT3BlbkFmdGVyRXhwb3J0ID0gdGhpcy5tb2RlbC5vcHRpb25zLmV4cG9ydHMub3BlbkFmdGVyRXhwb3J0O1xyXG4gICAgICB9XHJcblxyXG4gICAgICBpZiAoISh0aGlzLmZvcm1hdCA9PT0gJ1BkZicgfHwgdGhpcy5mb3JtYXQgPT09ICdIdG1sJykpIHtcclxuICAgICAgICByZXN1bHRTZXR0aW5ncy5PcGVuQWZ0ZXJFeHBvcnQgPSBmYWxzZTtcclxuICAgICAgfVxyXG5cclxuICAgICAgc2V0dGluZ3MgPSByZXN1bHRTZXR0aW5ncztcclxuICAgIH0gZWxzZSB7XHJcbiAgICAgIHNldHRpbmdzID0gdGhpcy5nZXREZWZhdWx0RXhwb3J0U2V0dGluZ3ModGhpcy5mb3JtYXQpO1xyXG4gICAgfVxyXG5cclxuICAgIHRoaXMudXBkYXRlVHlwZXMoc2V0dGluZ3MpO1xyXG5cclxuICAgIHRoaXMuZXhwb3J0U2V0dGluZ3MgPSB0aGlzLmhlbHBlci5jb3B5T2JqZWN0KHNldHRpbmdzKTtcclxuICB9XHJcblxyXG4gIHVwZGF0ZVR5cGVzKGRlZmF1bHRFeHBvcnRTZXR0aW5nczogYW55KSB7XHJcbiAgICBjb25zdCB0eXBlcyA9IFsnSW1hZ2VUeXBlJywgJ0RhdGFUeXBlJywgJ0V4Y2VsVHlwZScsICdIdG1sVHlwZSddO1xyXG4gICAgdHlwZXMuZm9yRWFjaCgocHJvcGVydHlOYW1lKSA9PiB7XHJcbiAgICAgIGlmIChkZWZhdWx0RXhwb3J0U2V0dGluZ3NbcHJvcGVydHlOYW1lXSkge1xyXG4gICAgICAgIHN3aXRjaCAocHJvcGVydHlOYW1lKSB7XHJcbiAgICAgICAgICBjYXNlICdJbWFnZVR5cGUnOlxyXG4gICAgICAgICAgICBpZiAoIXRoaXMubW9kZWwub3B0aW9ucy5leHBvcnRzLnNob3dFeHBvcnRUb0ltYWdlQm1wICYmIGRlZmF1bHRFeHBvcnRTZXR0aW5nc1twcm9wZXJ0eU5hbWVdID09PSAnQm1wJykgeyBkZWZhdWx0RXhwb3J0U2V0dGluZ3NbcHJvcGVydHlOYW1lXSA9ICdHaWYnOyB9XHJcbiAgICAgICAgICAgIGlmICghdGhpcy5tb2RlbC5vcHRpb25zLmV4cG9ydHMuc2hvd0V4cG9ydFRvSW1hZ2VHaWYgJiYgZGVmYXVsdEV4cG9ydFNldHRpbmdzW3Byb3BlcnR5TmFtZV0gPT09ICdHaWYnKSB7IGRlZmF1bHRFeHBvcnRTZXR0aW5nc1twcm9wZXJ0eU5hbWVdID0gJ0pwZWcnOyB9XHJcbiAgICAgICAgICAgIGlmICghdGhpcy5tb2RlbC5vcHRpb25zLmV4cG9ydHMuc2hvd0V4cG9ydFRvSW1hZ2VKcGVnICYmIGRlZmF1bHRFeHBvcnRTZXR0aW5nc1twcm9wZXJ0eU5hbWVdID09PSAnSnBlZycpIHsgZGVmYXVsdEV4cG9ydFNldHRpbmdzW3Byb3BlcnR5TmFtZV0gPSAnUGN4JzsgfVxyXG4gICAgICAgICAgICBpZiAoIXRoaXMubW9kZWwub3B0aW9ucy5leHBvcnRzLnNob3dFeHBvcnRUb0ltYWdlUGN4ICYmIGRlZmF1bHRFeHBvcnRTZXR0aW5nc1twcm9wZXJ0eU5hbWVdID09PSAnUGN4JykgeyBkZWZhdWx0RXhwb3J0U2V0dGluZ3NbcHJvcGVydHlOYW1lXSA9ICdQbmcnOyB9XHJcbiAgICAgICAgICAgIGlmICghdGhpcy5tb2RlbC5vcHRpb25zLmV4cG9ydHMuc2hvd0V4cG9ydFRvSW1hZ2VQbmcgJiYgZGVmYXVsdEV4cG9ydFNldHRpbmdzW3Byb3BlcnR5TmFtZV0gPT09ICdQbmcnKSB7IGRlZmF1bHRFeHBvcnRTZXR0aW5nc1twcm9wZXJ0eU5hbWVdID0gJ1RpZmYnOyB9XHJcbiAgICAgICAgICAgIGlmICghdGhpcy5tb2RlbC5vcHRpb25zLmV4cG9ydHMuc2hvd0V4cG9ydFRvSW1hZ2VUaWZmICYmIGRlZmF1bHRFeHBvcnRTZXR0aW5nc1twcm9wZXJ0eU5hbWVdID09PSAnVGlmZicpIHsgZGVmYXVsdEV4cG9ydFNldHRpbmdzW3Byb3BlcnR5TmFtZV0gPSAnRW1mJzsgfVxyXG4gICAgICAgICAgICBpZiAoIXRoaXMubW9kZWwub3B0aW9ucy5leHBvcnRzLnNob3dFeHBvcnRUb0ltYWdlTWV0YWZpbGUgJiYgZGVmYXVsdEV4cG9ydFNldHRpbmdzW3Byb3BlcnR5TmFtZV0gPT09ICdFbWYnKSB7IGRlZmF1bHRFeHBvcnRTZXR0aW5nc1twcm9wZXJ0eU5hbWVdID0gJ1N2Zyc7IH1cclxuICAgICAgICAgICAgaWYgKCF0aGlzLm1vZGVsLm9wdGlvbnMuZXhwb3J0cy5zaG93RXhwb3J0VG9JbWFnZVN2ZyAmJiBkZWZhdWx0RXhwb3J0U2V0dGluZ3NbcHJvcGVydHlOYW1lXSA9PT0gJ1N2ZycpIHsgZGVmYXVsdEV4cG9ydFNldHRpbmdzW3Byb3BlcnR5TmFtZV0gPSAnU3Zneic7IH1cclxuICAgICAgICAgICAgaWYgKCF0aGlzLm1vZGVsLm9wdGlvbnMuZXhwb3J0cy5zaG93RXhwb3J0VG9JbWFnZVN2Z3ogJiYgZGVmYXVsdEV4cG9ydFNldHRpbmdzW3Byb3BlcnR5TmFtZV0gPT09ICdTdmd6JykgeyBkZWZhdWx0RXhwb3J0U2V0dGluZ3NbcHJvcGVydHlOYW1lXSA9ICdCbXAnOyB9XHJcbiAgICAgICAgICAgIGJyZWFrO1xyXG5cclxuICAgICAgICAgIGNhc2UgJ0RhdGFUeXBlJzpcclxuICAgICAgICAgICAgaWYgKCF0aGlzLm1vZGVsLm9wdGlvbnMuZXhwb3J0cy5zaG93RXhwb3J0VG9Dc3YgJiYgZGVmYXVsdEV4cG9ydFNldHRpbmdzW3Byb3BlcnR5TmFtZV0gPT09ICdDc3YnKSB7IGRlZmF1bHRFeHBvcnRTZXR0aW5nc1twcm9wZXJ0eU5hbWVdID0gJ0RiZic7IH1cclxuICAgICAgICAgICAgaWYgKCF0aGlzLm1vZGVsLm9wdGlvbnMuZXhwb3J0cy5zaG93RXhwb3J0VG9EYmYgJiYgZGVmYXVsdEV4cG9ydFNldHRpbmdzW3Byb3BlcnR5TmFtZV0gPT09ICdEYmYnKSB7IGRlZmF1bHRFeHBvcnRTZXR0aW5nc1twcm9wZXJ0eU5hbWVdID0gJ1htbCc7IH1cclxuICAgICAgICAgICAgaWYgKCF0aGlzLm1vZGVsLm9wdGlvbnMuZXhwb3J0cy5zaG93RXhwb3J0VG9YbWwgJiYgZGVmYXVsdEV4cG9ydFNldHRpbmdzW3Byb3BlcnR5TmFtZV0gPT09ICdYbWwnKSB7IGRlZmF1bHRFeHBvcnRTZXR0aW5nc1twcm9wZXJ0eU5hbWVdID0gJ0RpZic7IH1cclxuICAgICAgICAgICAgaWYgKCF0aGlzLm1vZGVsLm9wdGlvbnMuZXhwb3J0cy5zaG93RXhwb3J0VG9EaWYgJiYgZGVmYXVsdEV4cG9ydFNldHRpbmdzW3Byb3BlcnR5TmFtZV0gPT09ICdEaWYnKSB7IGRlZmF1bHRFeHBvcnRTZXR0aW5nc1twcm9wZXJ0eU5hbWVdID0gJ1N5bGsnOyB9XHJcbiAgICAgICAgICAgIGlmICghdGhpcy5tb2RlbC5vcHRpb25zLmV4cG9ydHMuc2hvd0V4cG9ydFRvU3lsayAmJiBkZWZhdWx0RXhwb3J0U2V0dGluZ3NbcHJvcGVydHlOYW1lXSA9PT0gJ1N5bGsnKSB7IGRlZmF1bHRFeHBvcnRTZXR0aW5nc1twcm9wZXJ0eU5hbWVdID0gJ0Nzdic7IH1cclxuICAgICAgICAgICAgaWYgKCF0aGlzLm1vZGVsLm9wdGlvbnMuZXhwb3J0cy5zaG93RXhwb3J0VG9Kc29uICYmIGRlZmF1bHRFeHBvcnRTZXR0aW5nc1twcm9wZXJ0eU5hbWVdID09PSAnSnNvbicpIHsgZGVmYXVsdEV4cG9ydFNldHRpbmdzW3Byb3BlcnR5TmFtZV0gPSAnSnNvbic7IH1cclxuICAgICAgICAgICAgYnJlYWs7XHJcblxyXG4gICAgICAgICAgY2FzZSAnRXhjZWxUeXBlJzpcclxuICAgICAgICAgICAgaWYgKCF0aGlzLm1vZGVsLm9wdGlvbnMuZXhwb3J0cy5zaG93RXhwb3J0VG9FeGNlbCAmJiBkZWZhdWx0RXhwb3J0U2V0dGluZ3NbcHJvcGVydHlOYW1lXSA9PT0gJ0V4Y2VsJykgeyBkZWZhdWx0RXhwb3J0U2V0dGluZ3NbcHJvcGVydHlOYW1lXSA9ICdFeGNlbEJpbmFyeSc7IH1cclxuICAgICAgICAgICAgaWYgKCF0aGlzLm1vZGVsLm9wdGlvbnMuZXhwb3J0cy5zaG93RXhwb3J0VG9FeGNlbEJpZmYgJiYgZGVmYXVsdEV4cG9ydFNldHRpbmdzW3Byb3BlcnR5TmFtZV0gPT09ICdFeGNlbEJpbmFyeScpIHsgZGVmYXVsdEV4cG9ydFNldHRpbmdzW3Byb3BlcnR5TmFtZV0gPSAnRXhjZWxYbWwnOyB9XHJcbiAgICAgICAgICAgIGlmICghdGhpcy5tb2RlbC5vcHRpb25zLmV4cG9ydHMuc2hvd0V4cG9ydFRvRXhjZWxYbWwgJiYgZGVmYXVsdEV4cG9ydFNldHRpbmdzW3Byb3BlcnR5TmFtZV0gPT09ICdFeGNlbFhtbCcpIHsgZGVmYXVsdEV4cG9ydFNldHRpbmdzW3Byb3BlcnR5TmFtZV0gPSAnRXhjZWwnOyB9XHJcbiAgICAgICAgICAgIGJyZWFrO1xyXG5cclxuICAgICAgICAgIGNhc2UgJ0h0bWxUeXBlJzpcclxuICAgICAgICAgICAgaWYgKCF0aGlzLm1vZGVsLm9wdGlvbnMuZXhwb3J0cy5zaG93RXhwb3J0VG9IdG1sICYmIGRlZmF1bHRFeHBvcnRTZXR0aW5nc1twcm9wZXJ0eU5hbWVdID09PSAnSHRtbCcpIHsgZGVmYXVsdEV4cG9ydFNldHRpbmdzW3Byb3BlcnR5TmFtZV0gPSAnSHRtbDUnOyB9XHJcbiAgICAgICAgICAgIGlmICghdGhpcy5tb2RlbC5vcHRpb25zLmV4cG9ydHMuc2hvd0V4cG9ydFRvSHRtbDUgJiYgZGVmYXVsdEV4cG9ydFNldHRpbmdzW3Byb3BlcnR5TmFtZV0gPT09ICdIdG1sNScpIHsgZGVmYXVsdEV4cG9ydFNldHRpbmdzW3Byb3BlcnR5TmFtZV0gPSAnTWh0JzsgfVxyXG4gICAgICAgICAgICBpZiAoIXRoaXMubW9kZWwub3B0aW9ucy5leHBvcnRzLnNob3dFeHBvcnRUb01odCAmJiBkZWZhdWx0RXhwb3J0U2V0dGluZ3NbcHJvcGVydHlOYW1lXSA9PT0gJ01odCcpIHsgZGVmYXVsdEV4cG9ydFNldHRpbmdzW3Byb3BlcnR5TmFtZV0gPSAnSHRtbCc7IH1cclxuICAgICAgICAgICAgYnJlYWs7XHJcbiAgICAgICAgfVxyXG4gICAgICB9XHJcbiAgICB9KTtcclxuICB9XHJcblxyXG4gIGdldENvbW1vbkV4cG9ydEZvcm1hdChmb3JtYXQ/OiBzdHJpbmcpOiBzdHJpbmcge1xyXG4gICAgZm9ybWF0ID0gZm9ybWF0ID8/IHRoaXMuZm9ybWF0O1xyXG4gICAgaWYgKGZvcm1hdCA9PT0gJ0h0bWwnIHx8IGZvcm1hdCA9PT0gJ0h0bWw1JyB8fCBmb3JtYXQgPT09ICdNaHQnKSB7IHJldHVybiAnSHRtbCc7IH1cclxuICAgIGlmIChmb3JtYXQgPT09ICdFeGNlbEJpZmYnIHx8IGZvcm1hdCA9PT0gJ0V4Y2VsJyB8fCBmb3JtYXQgPT09ICdFeGNlbFhtbCcgfHwgZm9ybWF0ID09PSAnRXhjZWxCaW5hcnknKSB7IHJldHVybiAnRXhjZWwnOyB9XHJcbiAgICBpZiAoZm9ybWF0ID09PSAnQ3N2JyB8fCBmb3JtYXQgPT09ICdEYmYnIHx8IGZvcm1hdCA9PT0gJ1htbCcgfHwgZm9ybWF0ID09PSAnRGlmJyB8fCBmb3JtYXQgPT09ICdTeWxrJykgeyByZXR1cm4gJ0RhdGEnOyB9XHJcbiAgICBpZiAoZm9ybWF0ID09PSAnSW1hZ2VCbXAnIHx8IGZvcm1hdCA9PT0gJ0ltYWdlR2lmJyB8fCBmb3JtYXQgPT09ICdJbWFnZUpwZWcnIHx8IGZvcm1hdCA9PT0gJ0ltYWdlUGN4JyB8fCBmb3JtYXQgPT09ICdJbWFnZVBuZycgfHxcclxuICAgICAgZm9ybWF0ID09PSAnSW1hZ2VUaWZmJyB8fCBmb3JtYXQgPT09ICdJbWFnZUVtZicgfHwgZm9ybWF0ID09PSAnSW1hZ2VTdmcnIHx8IGZvcm1hdCA9PT0gJ0ltYWdlU3ZneicpIHsgcmV0dXJuICdJbWFnZSc7IH1cclxuXHJcbiAgICByZXR1cm4gZm9ybWF0O1xyXG4gIH1cclxuXHJcbiAgcHVibGljIGdldERlZmF1bHRFeHBvcnRTZXR0aW5ncyhmb3JtYXQ6IHN0cmluZywgaXNEYXNoYm9hcmRFeHBvcnQ6IGJvb2xlYW4gPSBmYWxzZSkge1xyXG4gICAgbGV0IGV4cG9ydFNldHRpbmdzID0gbnVsbDtcclxuXHJcbiAgICBpZiAoaXNEYXNoYm9hcmRFeHBvcnQpIHtcclxuICAgICAgcmV0dXJuIHRoaXMubW9kZWwub3B0aW9ucy5leHBvcnRzLmRlZmF1bHRTZXR0aW5nc1snRGFzaGJvYXJkJyArIGZvcm1hdF07XHJcbiAgICB9XHJcblxyXG4gICAgc3dpdGNoIChmb3JtYXQpIHtcclxuICAgICAgY2FzZSAnRG9jdW1lbnQnOlxyXG4gICAgICAgIGV4cG9ydFNldHRpbmdzID0geyBGb3JtYXQ6ICdNZGMnIH07XHJcbiAgICAgICAgYnJlYWs7XHJcblxyXG4gICAgICBjYXNlICdQZGYnOlxyXG4gICAgICAgIGV4cG9ydFNldHRpbmdzID0gdGhpcy5tb2RlbC5vcHRpb25zLmV4cG9ydHMuZGVmYXVsdFNldHRpbmdzWydTdGlQZGZFeHBvcnRTZXR0aW5ncyddO1xyXG4gICAgICAgIGJyZWFrO1xyXG5cclxuICAgICAgY2FzZSAnWHBzJzpcclxuICAgICAgICBleHBvcnRTZXR0aW5ncyA9IHRoaXMubW9kZWwub3B0aW9ucy5leHBvcnRzLmRlZmF1bHRTZXR0aW5nc1snU3RpWHBzRXhwb3J0U2V0dGluZ3MnXTtcclxuICAgICAgICBicmVhaztcclxuXHJcbiAgICAgIGNhc2UgJ1Bvd2VyUG9pbnQnOlxyXG4gICAgICBjYXNlICdQcHQyMDA3JzpcclxuICAgICAgICBleHBvcnRTZXR0aW5ncyA9IHRoaXMubW9kZWwub3B0aW9ucy5leHBvcnRzLmRlZmF1bHRTZXR0aW5nc1snU3RpUG93ZXJQb2ludEV4cG9ydFNldHRpbmdzJ10gfHwgdGhpcy5tb2RlbC5vcHRpb25zLmV4cG9ydHMuZGVmYXVsdFNldHRpbmdzWydTdGlQcHQyMDA3RXhwb3J0U2V0dGluZ3MnXTtcclxuICAgICAgICBicmVhaztcclxuXHJcbiAgICAgIGNhc2UgJ0h0bWwnOlxyXG4gICAgICAgIGV4cG9ydFNldHRpbmdzID0gdGhpcy5tb2RlbC5vcHRpb25zLmV4cG9ydHMuZGVmYXVsdFNldHRpbmdzWydTdGlIdG1sRXhwb3J0U2V0dGluZ3MnXTtcclxuICAgICAgICBleHBvcnRTZXR0aW5ncy5IdG1sVHlwZSA9ICdIdG1sJztcclxuICAgICAgICBicmVhaztcclxuXHJcbiAgICAgIGNhc2UgJ0h0bWw1JzpcclxuICAgICAgICBleHBvcnRTZXR0aW5ncyA9IHRoaXMubW9kZWwub3B0aW9ucy5leHBvcnRzLmRlZmF1bHRTZXR0aW5nc1snU3RpSHRtbEV4cG9ydFNldHRpbmdzJ107XHJcbiAgICAgICAgZXhwb3J0U2V0dGluZ3MuSHRtbFR5cGUgPSAnSHRtbDUnO1xyXG4gICAgICAgIGJyZWFrO1xyXG5cclxuICAgICAgY2FzZSAnTWh0JzpcclxuICAgICAgICBleHBvcnRTZXR0aW5ncyA9IHRoaXMubW9kZWwub3B0aW9ucy5leHBvcnRzLmRlZmF1bHRTZXR0aW5nc1snU3RpSHRtbEV4cG9ydFNldHRpbmdzJ107XHJcbiAgICAgICAgZXhwb3J0U2V0dGluZ3MuSHRtbFR5cGUgPSAnTWh0JztcclxuICAgICAgICBicmVhaztcclxuXHJcbiAgICAgIGNhc2UgJ1RleHQnOlxyXG4gICAgICAgIGV4cG9ydFNldHRpbmdzID0gdGhpcy5tb2RlbC5vcHRpb25zLmV4cG9ydHMuZGVmYXVsdFNldHRpbmdzWydTdGlUeHRFeHBvcnRTZXR0aW5ncyddO1xyXG4gICAgICAgIGJyZWFrO1xyXG5cclxuICAgICAgY2FzZSAnUnRmJzpcclxuICAgICAgICBleHBvcnRTZXR0aW5ncyA9IHRoaXMubW9kZWwub3B0aW9ucy5leHBvcnRzLmRlZmF1bHRTZXR0aW5nc1snU3RpUnRmRXhwb3J0U2V0dGluZ3MnXTtcclxuICAgICAgICBicmVhaztcclxuXHJcbiAgICAgIGNhc2UgJ1dvcmQnOlxyXG4gICAgICBjYXNlICdXb3JkMjAwNyc6XHJcbiAgICAgICAgZXhwb3J0U2V0dGluZ3MgPSB0aGlzLm1vZGVsLm9wdGlvbnMuZXhwb3J0cy5kZWZhdWx0U2V0dGluZ3NbJ1N0aVdvcmRFeHBvcnRTZXR0aW5ncyddIHx8IHRoaXMubW9kZWwub3B0aW9ucy5leHBvcnRzLmRlZmF1bHRTZXR0aW5nc1snU3RpV29yZDIwMDdFeHBvcnRTZXR0aW5ncyddO1xyXG4gICAgICAgIGJyZWFrO1xyXG5cclxuICAgICAgY2FzZSAnT2R0JzpcclxuICAgICAgICBleHBvcnRTZXR0aW5ncyA9IHRoaXMubW9kZWwub3B0aW9ucy5leHBvcnRzLmRlZmF1bHRTZXR0aW5nc1snU3RpT2R0RXhwb3J0U2V0dGluZ3MnXTtcclxuICAgICAgICBicmVhaztcclxuXHJcbiAgICAgIGNhc2UgJ0V4Y2VsQmlmZic6XHJcbiAgICAgICAgZXhwb3J0U2V0dGluZ3MgPSB0aGlzLm1vZGVsLm9wdGlvbnMuZXhwb3J0cy5kZWZhdWx0U2V0dGluZ3NbJ1N0aUV4Y2VsRXhwb3J0U2V0dGluZ3MnXTtcclxuICAgICAgICBleHBvcnRTZXR0aW5ncy5FeGNlbFR5cGUgPSAnRXhjZWxCaWZmJztcclxuICAgICAgICBicmVhaztcclxuXHJcbiAgICAgIGNhc2UgJ0V4Y2VsWG1sJzpcclxuICAgICAgICBleHBvcnRTZXR0aW5ncyA9IHRoaXMubW9kZWwub3B0aW9ucy5leHBvcnRzLmRlZmF1bHRTZXR0aW5nc1snU3RpRXhjZWxFeHBvcnRTZXR0aW5ncyddO1xyXG4gICAgICAgIGV4cG9ydFNldHRpbmdzLkV4Y2VsVHlwZSA9ICdFeGNlbFhtbCc7XHJcbiAgICAgICAgYnJlYWs7XHJcblxyXG4gICAgICBjYXNlICdFeGNlbCc6XHJcbiAgICAgICAgZXhwb3J0U2V0dGluZ3MgPSB0aGlzLm1vZGVsLm9wdGlvbnMuZXhwb3J0cy5kZWZhdWx0U2V0dGluZ3NbJ1N0aUV4Y2VsRXhwb3J0U2V0dGluZ3MnXTtcclxuICAgICAgICBleHBvcnRTZXR0aW5ncy5FeGNlbFR5cGUgPSAnRXhjZWwnO1xyXG4gICAgICAgIGJyZWFrO1xyXG5cclxuICAgICAgY2FzZSAnT2RzJzpcclxuICAgICAgICBleHBvcnRTZXR0aW5ncyA9IHRoaXMubW9kZWwub3B0aW9ucy5leHBvcnRzLmRlZmF1bHRTZXR0aW5nc1snU3RpT2RzRXhwb3J0U2V0dGluZ3MnXTtcclxuICAgICAgICBicmVhaztcclxuXHJcbiAgICAgIGNhc2UgJ0ltYWdlQm1wJzpcclxuICAgICAgICBleHBvcnRTZXR0aW5ncyA9IHRoaXMubW9kZWwub3B0aW9ucy5leHBvcnRzLmRlZmF1bHRTZXR0aW5nc1snU3RpSW1hZ2VFeHBvcnRTZXR0aW5ncyddO1xyXG4gICAgICAgIGV4cG9ydFNldHRpbmdzLkltYWdlVHlwZSA9ICdCbXAnO1xyXG4gICAgICAgIGJyZWFrO1xyXG5cclxuICAgICAgY2FzZSAnSW1hZ2VHaWYnOlxyXG4gICAgICAgIGV4cG9ydFNldHRpbmdzID0gdGhpcy5tb2RlbC5vcHRpb25zLmV4cG9ydHMuZGVmYXVsdFNldHRpbmdzWydTdGlJbWFnZUV4cG9ydFNldHRpbmdzJ107XHJcbiAgICAgICAgZXhwb3J0U2V0dGluZ3MuSW1hZ2VUeXBlID0gJ0dpZic7XHJcbiAgICAgICAgYnJlYWs7XHJcblxyXG4gICAgICBjYXNlICdJbWFnZUpwZWcnOlxyXG4gICAgICAgIGV4cG9ydFNldHRpbmdzID0gdGhpcy5tb2RlbC5vcHRpb25zLmV4cG9ydHMuZGVmYXVsdFNldHRpbmdzWydTdGlJbWFnZUV4cG9ydFNldHRpbmdzJ107XHJcbiAgICAgICAgZXhwb3J0U2V0dGluZ3MuSW1hZ2VUeXBlID0gJ0pwZWcnO1xyXG4gICAgICAgIGJyZWFrO1xyXG5cclxuICAgICAgY2FzZSAnSW1hZ2VQY3gnOlxyXG4gICAgICAgIGV4cG9ydFNldHRpbmdzID0gdGhpcy5tb2RlbC5vcHRpb25zLmV4cG9ydHMuZGVmYXVsdFNldHRpbmdzWydTdGlJbWFnZUV4cG9ydFNldHRpbmdzJ107XHJcbiAgICAgICAgZXhwb3J0U2V0dGluZ3MuSW1hZ2VUeXBlID0gJ1BjeCc7XHJcbiAgICAgICAgYnJlYWs7XHJcblxyXG4gICAgICBjYXNlICdJbWFnZVBuZyc6XHJcbiAgICAgICAgZXhwb3J0U2V0dGluZ3MgPSB0aGlzLm1vZGVsLm9wdGlvbnMuZXhwb3J0cy5kZWZhdWx0U2V0dGluZ3NbJ1N0aUltYWdlRXhwb3J0U2V0dGluZ3MnXTtcclxuICAgICAgICBleHBvcnRTZXR0aW5ncy5JbWFnZVR5cGUgPSAnUG5nJztcclxuICAgICAgICBicmVhaztcclxuXHJcbiAgICAgIGNhc2UgJ0ltYWdlVGlmZic6XHJcbiAgICAgICAgZXhwb3J0U2V0dGluZ3MgPSB0aGlzLm1vZGVsLm9wdGlvbnMuZXhwb3J0cy5kZWZhdWx0U2V0dGluZ3NbJ1N0aUltYWdlRXhwb3J0U2V0dGluZ3MnXTtcclxuICAgICAgICBleHBvcnRTZXR0aW5ncy5JbWFnZVR5cGUgPSAnVGlmZic7XHJcbiAgICAgICAgYnJlYWs7XHJcblxyXG4gICAgICBjYXNlICdJbWFnZVN2Zyc6XHJcbiAgICAgICAgZXhwb3J0U2V0dGluZ3MgPSB0aGlzLm1vZGVsLm9wdGlvbnMuZXhwb3J0cy5kZWZhdWx0U2V0dGluZ3NbJ1N0aUltYWdlRXhwb3J0U2V0dGluZ3MnXTtcclxuICAgICAgICBleHBvcnRTZXR0aW5ncy5JbWFnZVR5cGUgPSAnU3ZnJztcclxuICAgICAgICBicmVhaztcclxuXHJcbiAgICAgIGNhc2UgJ0ltYWdlU3Zneic6XHJcbiAgICAgICAgZXhwb3J0U2V0dGluZ3MgPSB0aGlzLm1vZGVsLm9wdGlvbnMuZXhwb3J0cy5kZWZhdWx0U2V0dGluZ3NbJ1N0aUltYWdlRXhwb3J0U2V0dGluZ3MnXTtcclxuICAgICAgICBleHBvcnRTZXR0aW5ncy5JbWFnZVR5cGUgPSAnU3Zneic7XHJcbiAgICAgICAgYnJlYWs7XHJcblxyXG4gICAgICBjYXNlICdJbWFnZUVtZic6XHJcbiAgICAgICAgZXhwb3J0U2V0dGluZ3MgPSB0aGlzLm1vZGVsLm9wdGlvbnMuZXhwb3J0cy5kZWZhdWx0U2V0dGluZ3NbJ1N0aUltYWdlRXhwb3J0U2V0dGluZ3MnXTtcclxuICAgICAgICBleHBvcnRTZXR0aW5ncy5JbWFnZVR5cGUgPSAnRW1mJztcclxuICAgICAgICBicmVhaztcclxuXHJcbiAgICAgIGNhc2UgJ1htbCc6XHJcbiAgICAgICAgZXhwb3J0U2V0dGluZ3MgPSB0aGlzLm1vZGVsLm9wdGlvbnMuZXhwb3J0cy5kZWZhdWx0U2V0dGluZ3NbJ1N0aURhdGFFeHBvcnRTZXR0aW5ncyddO1xyXG4gICAgICAgIGV4cG9ydFNldHRpbmdzLkRhdGFUeXBlID0gJ1htbCc7XHJcbiAgICAgICAgYnJlYWs7XHJcblxyXG4gICAgICBjYXNlICdDc3YnOlxyXG4gICAgICAgIGV4cG9ydFNldHRpbmdzID0gdGhpcy5tb2RlbC5vcHRpb25zLmV4cG9ydHMuZGVmYXVsdFNldHRpbmdzWydTdGlEYXRhRXhwb3J0U2V0dGluZ3MnXTtcclxuICAgICAgICBleHBvcnRTZXR0aW5ncy5EYXRhVHlwZSA9ICdDc3YnO1xyXG4gICAgICAgIGJyZWFrO1xyXG5cclxuICAgICAgY2FzZSAnRGJmJzpcclxuICAgICAgICBleHBvcnRTZXR0aW5ncyA9IHRoaXMubW9kZWwub3B0aW9ucy5leHBvcnRzLmRlZmF1bHRTZXR0aW5nc1snU3RpRGF0YUV4cG9ydFNldHRpbmdzJ107XHJcbiAgICAgICAgZXhwb3J0U2V0dGluZ3MuRGF0YVR5cGUgPSAnRGJmJztcclxuICAgICAgICBicmVhaztcclxuXHJcbiAgICAgIGNhc2UgJ0RpZic6XHJcbiAgICAgICAgZXhwb3J0U2V0dGluZ3MgPSB0aGlzLm1vZGVsLm9wdGlvbnMuZXhwb3J0cy5kZWZhdWx0U2V0dGluZ3NbJ1N0aURhdGFFeHBvcnRTZXR0aW5ncyddO1xyXG4gICAgICAgIGV4cG9ydFNldHRpbmdzLkRhdGFUeXBlID0gJ0RpZic7XHJcbiAgICAgICAgYnJlYWs7XHJcblxyXG4gICAgICBjYXNlICdTeWxrJzpcclxuICAgICAgICBleHBvcnRTZXR0aW5ncyA9IHRoaXMubW9kZWwub3B0aW9ucy5leHBvcnRzLmRlZmF1bHRTZXR0aW5nc1snU3RpRGF0YUV4cG9ydFNldHRpbmdzJ107XHJcbiAgICAgICAgZXhwb3J0U2V0dGluZ3MuRGF0YVR5cGUgPSAnU3lsayc7XHJcbiAgICAgICAgYnJlYWs7XHJcbiAgICB9XHJcblxyXG4gICAgZXhwb3J0U2V0dGluZ3MuT3BlbkFmdGVyRXhwb3J0ID0gISh0aGlzLm1vZGVsLm9wdGlvbnMuZXhwb3J0cy5zaG93T3BlbkFmdGVyRXhwb3J0ID09PSBmYWxzZSk7XHJcbiAgICBpZiAoIShmb3JtYXQgPT09ICdQZGYnIHx8IGZvcm1hdCA9PT0gJ0h0bWwnKSkge1xyXG4gICAgICBleHBvcnRTZXR0aW5ncy5PcGVuQWZ0ZXJFeHBvcnQgPSBmYWxzZTtcclxuICAgIH1cclxuXHJcbiAgICByZXR1cm4gZXhwb3J0U2V0dGluZ3M7XHJcbiAgfVxyXG5cclxufVxyXG4iXX0=