import { Injectable } from '@angular/core';
import * as i0 from "@angular/core";
import * as i1 from "./model.service";
import * as i2 from "./controller.service";
import * as i3 from "./export.service";
import * as i4 from "./helper.service";
import * as i5 from "./mail.service";
import * as i6 from "./find.service";
import * as i7 from "./dashboard-proxy.service";
export class ApiDashboardService {
    constructor(model, controller, exportService, helper, mailService, findService, proxy) {
        this.model = model;
        this.controller = controller;
        this.exportService = exportService;
        this.helper = helper;
        this.mailService = mailService;
        this.findService = findService;
        this.proxy = proxy;
        this.EXPORT_FORMATS = ['Document', 'Pdf', 'Xps', 'PowerPoint', 'Html', 'Html5', 'Mht', 'Text', 'Rtf', 'Word', 'Odt', 'ExcelBiff',
            'Excel', 'ExcelXml', 'Excel', 'Ods', 'Csv', 'Dbf', 'Dif', 'Sylk', 'Json', 'Xml', 'ImageBmp',
            'ImageGif', 'ImageJpeg', 'ImagePcx', 'ImagePng', 'ImageTiff', 'ImageEmf', 'ImageSvg', 'ImageSvgz'];
    }
    /**
     * The current page number
     */
    get currentPage() {
        return this.proxy.jsObject?.reportParams?.pageNumber;
    }
    set currentPage(value) {
        let jsObject = this.proxy.jsObject;
        let pageNumber = Math.max(Math.max(value, 0), jsObject.reportParams.pagesCount - 1);
        jsObject.controls.toolbar.controls["PageControl"].textBox.value = pageNumber.toString();
        jsObject.postAction("GoToPage");
    }
    /**
     * The total pages count
     */
    get pagesCount() {
        return this.proxy.jsObject.reportParams.pagesCount;
    }
    /**
     * The view mode, can be 'SinglePage', 'Continuous' & 'MultiplePages'
     */
    get viewMode() {
        return this.proxy.jsObject?.reportParams?.viewMode;
    }
    set viewMode(value) {
        if (value === 'SinglePage' || value === 'Continuous' || value === 'MultiplePages') {
            this.proxy.jsObject.postAction('ViewMode' + value);
        }
    }
    /**
     * The page zoom in percent
     * From 1 to 200
     */
    get zoom() {
        return this.proxy.jsObject?.reportParams?.zoom;
    }
    set zoom(value) {
        if (value >= 1 && value <= 1000) {
            let zoom = 200;
            if (value < 37)
                zoom = 25;
            else if (value < 62)
                zoom = 50;
            else if (value < 87)
                zoom = 75;
            else if (value < 125)
                zoom = 100;
            else if (value < 175)
                zoom = 150;
            this.proxy.jsObject.postAction('Zoom' + zoom);
        }
    }
    /**
     * Zoom page in height
     */
    zoomPageHeight() {
        this.proxy.jsObject.postAction('ZoomOnePage');
    }
    /**
     * Zoom page in width
     */
    zoomPageWidth() {
        this.proxy.jsObject.postAction('ZoomPageWidth');
    }
    /**
     * Print to PDF
     */
    printPdf() {
        this.proxy.jsObject.postPrint('PrintPdf');
    }
    /**
     * Print with preview
     */
    printWithPreview() {
        this.proxy.jsObject.postPrint('PrintWithPreview');
    }
    /**
     * Print without preview
     */
    printWithoutPreview() {
        this.proxy.jsObject.postPrint('PrintWithoutPreview');
    }
    get reportType() {
        return this.proxy.jsObject?.reportParams?.type ?? "Report";
    }
    /**
     * Show export form
     *
     * @param format The format to export, can be 'Document', 'Pdf', 'Xps', 'PowerPoint', 'Html', 'Html5', 'Mht', 'Text', 'Rtf', 'Word', 'Odt', 'ExcelBiff',
     * 'Excel', 'ExcelXml', 'Excel', 'Ods', 'Csv', 'Dbf', 'Dif', 'Sylk', 'Json', 'Xml', 'ImageBmp',
     * 'ImageGif', 'ImageJpeg', 'ImagePcx', 'ImagePng', 'ImageTiff', 'ImageEmf', 'ImageSvg', 'ImageSvgz'
     */
    showExportForm(format) {
        if (this.EXPORT_FORMATS.some(f => f === format)) {
            this.proxy.jsObject.controls.forms.exportForm.show(format, this.proxy.jsObject.options.actions.exportReport);
        }
    }
    /**
     * Show export form & email
     *
     * @param format The format to export, can be 'Document', 'Pdf', 'Xps', 'PowerPoint', 'Html', 'Html5', 'Mht', 'Text', 'Rtf', 'Word', 'Odt', 'ExcelBiff',
     * 'Excel', 'ExcelXml', 'Excel', 'Ods', 'Csv', 'Dbf', 'Dif', 'Sylk', 'Json', 'Xml', 'ImageBmp',
     * 'ImageGif', 'ImageJpeg', 'ImagePcx', 'ImagePng', 'ImageTiff', 'ImageEmf', 'ImageSvg', 'ImageSvgz'
     */
    showExportEmailForm(format) {
        if (this.EXPORT_FORMATS.some(f => f === format)) {
            this.proxy.jsObject.controls.forms.exportForm.show(format, this.proxy.jsObject.options.actions.emailReport);
        }
    }
    /**
     * Export report to selected format
     * Use default settings if not specified
     *
     * @param format The format to export, can be 'Document', 'Pdf', 'Xps', 'PowerPoint', 'Html', 'Html5', 'Mht', 'Text', 'Rtf', 'Word', 'Odt', 'ExcelBiff',
     * 'Excel', 'ExcelXml', 'Excel', 'Ods', 'Csv', 'Dbf', 'Dif', 'Sylk', 'Json', 'Xml', 'ImageBmp',
     * 'ImageGif', 'ImageJpeg', 'ImagePcx', 'ImagePng', 'ImageTiff', 'ImageEmf', 'ImageSvg', 'ImageSvgz'
     *
     * @param settings The export settings
     */
    export(format, settings) {
        if (this.reportType == "Report") {
            if (this.EXPORT_FORMATS.some(f => f === format)) {
                const exportSettings = this.getExportSettings(format, settings);
                this.proxy.jsObject.postExport(format, exportSettings);
            }
        }
        else {
            this.dashboardExport(format, settings);
        }
    }
    /**
     * Export dashboard to selected format
     * Use default settings if not specified
     *
     * @param format The format to export, can be 'Pdf', 'Html', 'Excel', 'Image'
     *
     * @param settings The export settings
     */
    dashboardExport(format, settings) {
        if (this.EXPORT_FORMATS.some(f => f === format)) {
            settings = this.getDefaultDashboardSettings(format, settings);
            this.proxy.jsObject.postExport(format, settings, null, true);
        }
    }
    /**
     * Export report to seleted format & email
     * Use default settings if not specified
     * Use default email settings if not specified
     *
     * @param format The format to export, can be 'Document', 'Pdf', 'Xps', 'PowerPoint', 'Html', 'Html5', 'Mht', 'Text', 'Rtf', 'Word', 'Odt', 'ExcelBiff',
     * 'Excel', 'ExcelXml', 'Excel', 'Ods', 'Csv', 'Dbf', 'Dif', 'Sylk', 'Json', 'Xml', 'ImageBmp',
     * 'ImageGif', 'ImageJpeg', 'ImagePcx', 'ImagePng', 'ImageTiff', 'ImageEmf', 'ImageSvg', 'ImageSvgz'
     *
     * @param settings The export settings
     *
     * @param email The email
     *
     * @param message The email message
     *
     * @param subject The email subject
     */
    exportEmail(format, settings, email, subject, message) {
        if (this.EXPORT_FORMATS.some(f => f === format)) {
            this.exportService.exportSettings = this.getExportSettings(format, settings);
            this.exportService.format = format;
            this.mailService.fillDefaults();
            if (email) {
                this.exportService.exportSettings.Email = email;
            }
            if (message) {
                this.exportService.exportSettings.Message = message;
            }
            if (subject) {
                this.exportService.exportSettings.Subject = subject;
            }
            this.mailService.sendMail();
        }
    }
    /**
     * Search text in report
     *
     * @param text text to search
     *
     * @param matchCase
     *
     * @param matchWholeWord
     */
    search(text, matchCase, matchWholeWord) {
        this.proxy.jsObject.showFindLabels(text, matchCase, matchWholeWord);
    }
    /**
     * Select next or previous search
     *
     * @param direction 'Next' or 'Previous'
     *
     */
    selectSearchLabel(direction) {
        this.proxy.jsObject.selectFindLabel(direction);
    }
    /**
     * Hide search
     */
    hideSearchLabels() {
        this.proxy.jsObject.hideFindLabels();
    }
    /**
     * Refresh dashboard
     */
    dashboardRefresh() {
        this.proxy.jsObject.postAction("Refresh");
    }
    dashboardResetAllFilters() {
        this.proxy.jsObject.postInteraction({ action: "DashboardResetAllFilters" });
    }
    getExportSettings(format, settings) {
        this.exportService.sendMail = false;
        this.exportService.format = format;
        const exportSettings = this.helper.copyObject(this.exportService.getDefaultExportSettings(format));
        if (settings) {
            Object.keys(settings).forEach(key => exportSettings[key] = settings[key]);
        }
        return exportSettings;
    }
    getDefaultDashboardSettings(format, settings) {
        let exportSettings = {};
        switch (format) {
            case "Pdf":
                exportSettings = {
                    ImageQuality: 200,
                    OpenAfterExport: true,
                    Orientation: "Landscape",
                    PaperSize: "A4"
                };
                break;
            case "Excel":
                exportSettings = {
                    ImageQuality: 200
                };
                break;
            case "Html":
                exportSettings = {
                    EnableAnimation: true,
                    ImageQuality: 200,
                    OpenAfterExport: true,
                    Scale: 100
                };
                break;
            case "Image":
                exportSettings = {
                    ImageType: "Png",
                    OpenAfterExport: true,
                    Scale: 100
                };
                break;
        }
        if (settings) {
            Object.keys(settings).forEach(key => exportSettings[key] = settings[key]);
        }
        return exportSettings;
    }
    static { this.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "18.2.14", ngImport: i0, type: ApiDashboardService, deps: [{ token: i1.ModelService }, { token: i2.ControllerService }, { token: i3.ExportService }, { token: i4.HelperService }, { token: i5.MailService }, { token: i6.FindService }, { token: i7.DashboardProxyService }], target: i0.ɵɵFactoryTarget.Injectable }); }
    static { this.ɵprov = i0.ɵɵngDeclareInjectable({ minVersion: "12.0.0", version: "18.2.14", ngImport: i0, type: ApiDashboardService }); }
}
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "18.2.14", ngImport: i0, type: ApiDashboardService, decorators: [{
            type: Injectable
        }], ctorParameters: () => [{ type: i1.ModelService }, { type: i2.ControllerService }, { type: i3.ExportService }, { type: i4.HelperService }, { type: i5.MailService }, { type: i6.FindService }, { type: i7.DashboardProxyService }] });
//# sourceMappingURL=data:application/json;base64,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