import { Injectable } from "@angular/core";
import { camelCase, indexOf } from "lodash";
import { Subject } from "rxjs";
import { StiForm } from "../elements/StiForm";
import { StiFormPageElement } from "../elements/StiFormPageElement";
import { StiPanelElement } from "../elements/StiPanelElement";
import { StiTableElement } from "../elements/StiTableElement";
import { StiNumberRangeFormat, StiThousandSeparator, StiUnitAlignment } from "../system/enums";
import { StiStaticHelperService } from "./sti-static-helper.service";
import * as i0 from "@angular/core";
export class StiModelService {
    static { this.PAGE_CLASS = "sti-page"; }
    static { this.PANEL_CLASS = "sti-panel"; }
    static { this.THEMES = ["Light", "Dark"]; }
    toggleTieredMenu(element) {
        let elementRect = this.getPositionRelativeToForm(element);
        let menuWidth = 104;
        let menuHeight = 165;
        this.tieredMenuY =
            window.innerHeight - elementRect.bottom >= menuHeight ? elementRect.bottom : elementRect.top - menuHeight;
        this.tieredMenuX =
            window.innerWidth - elementRect.left > menuWidth ? elementRect.left : elementRect.right - menuWidth;
        this.showTieredMenu = true;
    }
    get showTrialBar() {
        return this._showTrialBar;
    }
    set showTrialBar(value) {
        let lastState = this._showTrialBar;
        if (value != lastState) {
            this._showTrialBar = value;
        }
    }
    get showTrialModalWindow() {
        return this._showTrialModalWindow;
    }
    set showTrialModalWindow(value) {
        let lastState = this._showTrialModalWindow;
        if (value != lastState) {
            this._showTrialModalWindow = value;
        }
    }
    get showTrialTitle() {
        return this._showTrialTitle;
    }
    set showTrialTitle(value) {
        let lastState = this._showTrialTitle;
        if (value != lastState) {
            this._showTrialTitle = value;
        }
    }
    get canCloseTrialWindow() {
        return this._canCloseTrialWindow;
    }
    set canCloseTrialWindow(value) {
        let lastState = this._canCloseTrialWindow;
        if (value != lastState) {
            this._canCloseTrialWindow = value;
        }
    }
    set ribbon(value) {
        if ((value == "Insert" && !this.showInsertTab) || (this.viewerMode && value != "Preview"))
            return;
        if (value != this._ribbon) {
            let oldRibbon = this._ribbon;
            this._ribbon = value;
            this.interfaceEvent.next({ name: "setRibbon", data: { ribbon: value, oldRibbon: oldRibbon } });
        }
    }
    get ribbon() {
        return this._ribbon;
    }
    get previewMode() {
        return this._ribbon == "Preview";
    }
    get pdfPreviewMode() {
        return this._ribbon == "PDFPreview";
    }
    get fileMenuMode() {
        return this._ribbon == "File";
    }
    canDelete() {
        return (window.getSelection().toString().length == 0 &&
            document.activeElement.tagName.toLocaleLowerCase() == "body" &&
            !this.draggingElement &&
            !this.showSignatureEditor);
    }
    get showSaveToPdfButton() {
        return this.settings?.showSaveToPdfButton ?? true;
    }
    get showEditButton() {
        return this.settings?.showEditButton ?? false;
    }
    set embeddedMode(value) {
        this._embeddedMode = value;
    }
    get embeddedMode() {
        return this._embeddedMode || this.properties?.embeddedMode === true;
    }
    get cursor() {
        return this._cursor;
    }
    get zoom() {
        return this._zoom;
    }
    set requestUrl(value) {
        this._requestUrl = value;
        this.isWebView = value == "WebView";
    }
    get requestUrl() {
        return this._requestUrl;
    }
    set zoom(value) {
        if (value <= 0.1) {
            this._zoom = 0.1;
        }
        else if (value >= 4) {
            this._zoom = 4;
        }
        else {
            this._zoom = value;
        }
    }
    set cursor(value) {
        if (value == "default" && this.selectedComponent?.hover) {
            this._cursor = "move";
        }
        else {
            this._cursor = value;
        }
    }
    formattingDate(dateFormat) {
        let result = dateFormat
            .replace(/M/g, "k")
            .replace(/m/g, "r")
            .replace(/tt/g, "a")
            .replace(/k/g, "m")
            .replace(/r/g, "M");
        return result;
    }
    setZoomFromString(zoomString) {
        if (typeof zoomString != "string") {
            return;
        }
        if (zoomString.includes("%")) {
            let zoom = Number(zoomString.slice(0, zoomString.indexOf("%"))) / 100;
            if (zoom <= 0.1) {
                this.zoom = 0.1;
            }
            else if (zoom >= 4) {
                this.zoom = 4;
            }
            else {
                this.zoom = zoom;
            }
        }
        else {
            switch (zoomString.toLowerCase()) {
                case "pagewidth":
                    this.formEvents.next({ name: "SetZoomByPageWidth" });
                    break;
                case "onepage":
                    this.formEvents.next({ name: "SetZoomByOnePage" });
                    break;
                default:
                    this.formEvents.next({ name: "SetDefaultZoom" });
                    break;
            }
        }
    }
    getPositionRelativeToForm(element) {
        let elementRect = element?.getBoundingClientRect();
        let formRect = this.rootFormElement?.getBoundingClientRect();
        if (formRect) {
            return new DOMRect(elementRect?.x - formRect?.x, elementRect?.y - formRect?.y, elementRect?.width, elementRect?.height);
        }
        else {
            return elementRect;
        }
    }
    getBooleanSetting(settingName) {
        return this.settings && this.isNotEmpty(this.settings[settingName]) && this.isBoolean(this.settings[settingName])
            ? this.settings[settingName]
            : true;
    }
    isBoolean(value) {
        return typeof value === "boolean";
    }
    isNotEmpty(value) {
        return value?.toString()?.length > 0;
    }
    saveLastFormState() {
        this.form.reportVersion = this.buildVersion;
        this.form.reportChanged = StiStaticHelperService.dateToJsonDate(new Date());
        this.lastSavedFormState = this.form.saveToReportJsonObject().serialize();
    }
    toCamelCase(obj) {
        if (Array.isArray(obj)) {
            return obj.map((v) => this.toCamelCase(v));
        }
        else if (obj != null && obj.constructor === Object) {
            return Object.keys(obj).reduce((result, key) => ({
                ...result,
                [camelCase(key)]: this.toCamelCase(obj[key]),
            }), {});
        }
        return obj;
    }
    showColumnPreview(index, type) {
        if (!type &&
            this.selectedComponent?.element instanceof StiTableElement &&
            !this.selectedComponent?.element.showColumnsEditor) {
            this.selectedComponent.element.showColumnsEditor = true;
            this.needToRefresh.next(true);
        }
        else if (type == "totals" &&
            this.selectedComponent?.element instanceof StiTableElement &&
            !this.selectedComponent?.element.showTotalsEditor) {
            this.selectedComponent.element.showTotalsEditor = true;
            this.needToRefresh.next(true);
        }
        setTimeout(() => {
            if (this.currentContentEditableItemIndex != index || this.previousSelectedComponent != this.selectedComponent)
                this.currentContentEditableItem.next(index);
            this.currentContentEditableItemIndex = index;
        });
    }
    showError(message, sendedFromButton = false, buttonElement) {
        if (message instanceof ArrayBuffer) {
            var enc = new TextDecoder("utf-8");
            message = enc.decode(message);
        }
        else if (message instanceof Event) {
            message = message.type;
        }
        else if (message?.text) {
            message = message.text;
        }
        if (buttonElement) {
            this.currentUsedButtonElement = buttonElement;
        }
        this.sendedFromButton = sendedFromButton;
        this.errorWindowMessage = message;
        if (this.errorWindowMessage != "abort") {
            this.showErrorWindow = true;
        }
    }
    getImage(fileName) {
        return this.imagesArray && this.imagesArray[fileName] ? this.imagesArray[fileName] : "";
    }
    removeUnusedCells(element) {
        element.columns.forEach((column) => {
            if (column["cells"].length != element.rowLabels.length) {
                column["cells"].splice(element.rowLabels.length, column["cells"].length - 1);
            }
        });
    }
    startPdfExport(form) {
        if (form) {
            this.setFormFromString(JSON.stringify(form));
        }
        this.startPdfExportEvent.next(true);
    }
    setFormFromString(str, loadSavedSetings = false) {
        let form = new StiForm();
        form.loadFormJsonString(str);
        if (loadSavedSetings) {
            this.loadSavedSettings();
        }
        this.form = form;
    }
    get form() {
        return this._form;
    }
    set form(form) {
        this.updateForm(form);
        this.checkOnUniqueNames();
        form.formEvents.next({ name: "SetForm" });
        this.firstFormState = this.form.saveToReportJsonObject().serialize();
        if (this.viewerMode && this.isMobile()) {
            setTimeout(() => {
                this.isMobileViewer.next(true);
            });
        }
    }
    arrayOfStylesToString(array) {
        if (!array)
            return;
        let styles = [];
        if (array.length > 0) {
            array.forEach((item) => {
                if (item != "Underline") {
                    styles.push(item);
                }
            });
        }
        return styles.length > 0 ? styles.join(" ").toLocaleLowerCase() : "";
    }
    checkOnUniqueNames() {
        this.form.allElements.forEach((element) => {
            let currentName = element.name;
            let uniqueElements = this.form.allElements.filter((elem) => elem.name == currentName);
            if (uniqueElements.length > 1) {
                this.setUniqueName(element);
            }
        });
    }
    setUniqueName(element) {
        let regex = /\d+$/;
        let currentName = element.name ?? element.type.replace("_", "").replace("Element", "");
        let index = 0;
        let currentPosition = element.name?.match(regex) ? element.name?.match(regex)[0] : null;
        if (currentPosition) {
            currentName = currentName.slice(0, currentName.length - currentPosition.length);
        }
        while (this.form.allElements.find((element) => element.name == `${currentName + (index != 0 ? index + 1 : "")}`)) {
            index++;
        }
        element.name = currentName + (index != 0 ? index + 1 : "");
    }
    checkNumberRange(element) {
        if (element.rangeFormat == StiNumberRangeFormat.Digits) {
            let maximumValueString = "";
            for (let index = element.maximum; index > 0; index--) {
                maximumValueString += "9";
            }
            let maximumValue = Number(maximumValueString);
            let minimumValue = -Number(maximumValueString);
            if (element.value > maximumValue) {
                element.value = maximumValue;
            }
            else if (element.value < minimumValue) {
                element.value = minimumValue;
            }
        }
        else {
            if (element.value > element.maximum) {
                element.value = Number(element.maximum);
            }
            else if (element.value < element.minimum) {
                element.value = Number(element.minimum);
            }
        }
    }
    showTrialLog() {
        if (this.showTrialBar || this.showTrialTitle || this.showTrialModalWindow) {
            console.warn("You are using a trial version of the Stimulsoft PDF Forms!");
        }
    }
    loadSavedSettings() {
        if (localStorage.getItem("StiFormSettings") &&
            (this.isWebView || this.isCloudMode || this.properties?.cloudPlanIdent)) {
            this.form?.settings?.loadFormJsonString(localStorage.getItem("StiFormSettings"));
        }
    }
    saveSettings() {
        localStorage.removeItem("StiFormSettings");
        localStorage.setItem("StiFormSettings", this.form.settings.saveToJsonObject({}).serialize());
    }
    isMobile() {
        const toMatch = [/Android/i, /webOS/i, /iPhone/i, /iPad/i, /iPod/i, /BlackBerry/i, /Windows Phone/i];
        return toMatch.some((toMatchItem) => {
            return navigator.userAgent.match(toMatchItem);
        });
    }
    getBrowserType() {
        let userAgent = navigator.userAgent;
        let browserName;
        if (userAgent.match(/chrome|chromium|crios/i)) {
            browserName = "chrome";
        }
        else if (userAgent.match(/firefox|fxios/i)) {
            browserName = "firefox";
        }
        else if (userAgent.match(/safari/i)) {
            browserName = "safari";
        }
        else if (userAgent.match(/opr\//i)) {
            browserName = "opera";
        }
        else if (userAgent.match(/edg/i)) {
            browserName = "edge";
        }
        else {
            browserName = "No browser detection";
        }
        return browserName;
    }
    getScaleWithTranslate() {
        return `translate(${this.zoom <= 1 ? 0 : ((this.zoom - 1) / 2) * 100}%) scale(${this.zoom})`;
    }
    getScale() {
        return `scale(${this.zoom})`;
    }
    updateForm(form) {
        this._form = form;
        this.selectedComponent = null;
        this.lastUsedPage = null;
        form.formEvents.subscribe((e) => this.formEvents.next(e));
        if (form.pages?.length > 0) {
            this.previousPadding = form.pages[0].padding.clone();
            this.previousContentAreaWidth = form.pages[0].contentAreaWidth;
            setTimeout(() => {
                this.needToRefresh.next(true);
            }, 0);
        }
        form.formEvents.next({ name: "UpdateForm" });
    }
    set elementToUpdate(value) {
        let oldValue = this._elementToUpdate;
        this._elementToUpdate = value;
        if (oldValue != value) {
            this.formEvents.next({ name: "ElementToUpdateChanged", value: value });
        }
    }
    get elementToUpdate() {
        return this._elementToUpdate;
    }
    get selectedComponent() {
        return this._selectedComponent;
    }
    set selectedComponent(value) {
        let oldValue = this._selectedComponent;
        this._selectedPage = null;
        this._selectedComponent = value;
        this.needToRefresh.next(true);
        if (oldValue != value) {
            this.formEvents.next({ name: "SelectedComponentChanged", value: value });
        }
    }
    get selectedPage() {
        return this._selectedPage;
    }
    set selectedPage(value) {
        let oldValue = this._selectedPage;
        this._selectedComponent = null;
        this._selectedPage = value;
        this.needToRefresh.next(true);
        if (oldValue != value) {
            this.formEvents.next({ name: "SelectedPageChanged", value: value });
        }
    }
    get lastUsedPage() {
        return this._lastUsedPage ?? this.selectedPage ?? this.form?.pages[0];
    }
    set lastUsedPage(value) {
        this._lastUsedPage = value;
    }
    getSelectedComponentPage() {
        return this.getElementPage(this._selectedComponent?.element);
    }
    getElementPage(element) {
        return this.form?.pages.find((p) => p.elements.find((e) => e == element) != null);
    }
    getElementPageForScroll(element) {
        return this.form?.pages.find((p) => p.elements.find((e) => e == element) != null
            ? p.elements.find((e) => e == element)
            : this.getElementPanel(element));
    }
    overToolbar(x, y) {
        let elementsDiv = document.elementsFromPoint(x, y).find((elem) => elem?.classList.contains("elements"));
        return elementsDiv ? true : false;
    }
    getElementPanel(element) {
        let result;
        this.form.pages.forEach((page) => {
            page.allElements.forEach((panel) => {
                if (panel instanceof StiPanelElement && panel.elements.indexOf(element) >= 0) {
                    result = panel;
                }
            });
        });
        return result;
    }
    formatNumber(value, format, decimalDigits, unitAlignment, unit) {
        let formatedValue = this.formatNumberWithThousandSeparator(value, format, decimalDigits);
        switch (unitAlignment) {
            case StiUnitAlignment.PrefixInside:
                return `${unit}${formatedValue}`;
            case StiUnitAlignment.PrefixInsideSpace:
                return `${unit} ${formatedValue}`;
            case StiUnitAlignment.SuffixInside:
                return `${formatedValue}${unit}`;
            case StiUnitAlignment.SuffixInsideSpace:
                return `${formatedValue} ${unit}`;
        }
        return formatedValue;
    }
    getFormattedValue(element, inValue) {
        let value = inValue;
        try {
            if (typeof value === "number" && !isNaN(value)) {
                value = this.formatNumberWithThousandSeparator(parseFloat(value.toFixed(element.decimalDigits)), element.thousandSeparator, element.decimalDigits);
            }
        }
        catch {
            value = "";
        }
        return value;
    }
    formatNumberWithThousandSeparator(value, format, decimalDigits) {
        let isNegative = false;
        if (value == undefined) {
            return "";
        }
        let result;
        if (decimalDigits > 0) {
            result = value.toFixed(decimalDigits);
        }
        else {
            result = value.toString();
        }
        if (result.indexOf("-") != -1) {
            isNegative = true;
        }
        if (isNegative) {
            result = result.slice(1, result.length);
        }
        if (format == StiThousandSeparator.None) {
            return (isNegative ? "-" : "") + result;
        }
        else if (format == StiThousandSeparator.NoneComma) {
            return (isNegative ? "-" : "") + result.replace(".", ",");
        }
        else if (format == StiThousandSeparator.Comma) {
            var chars = (decimalDigits > 0 ? result.slice(0, indexOf(result, ".")) : result).split("").reverse();
            var withCommas = [];
            for (var i = 1; i <= chars.length; i++) {
                withCommas.push(chars[i - 1]);
                if (i % 3 == 0 && i != chars.length) {
                    withCommas.push(",");
                }
            }
            result =
                decimalDigits > 0
                    ? withCommas.reverse().join("") + result.slice(indexOf(result, "."), result.length)
                    : withCommas.reverse().join("");
            return (isNegative ? "-" : "") + result;
        }
        else if (format == StiThousandSeparator.Dot) {
            var chars = (decimalDigits > 0 ? result.slice(0, indexOf(result, ".")) : result).split("").reverse();
            var withDotts = [];
            for (var i = 1; i <= chars.length; i++) {
                withDotts.push(chars[i - 1]);
                if (i % 3 == 0 && i != chars.length) {
                    withDotts.push(".");
                }
            }
            result =
                decimalDigits > 0
                    ? withDotts.reverse().join("") + "," + result.slice(indexOf(result, ".") + 1, result.length)
                    : withDotts.reverse().join("");
            return (isNegative ? "-" : "") + result;
        }
        else if (format == StiThousandSeparator.Apostrophe) {
            var chars = (decimalDigits > 0 ? result.slice(0, indexOf(result, ".")) : result).split("").reverse();
            var withSpaces = [];
            for (var i = 1; i <= chars.length; i++) {
                withSpaces.push(chars[i - 1]);
                if (i % 3 == 0 && i != chars.length) {
                    withSpaces.push("`");
                }
            }
            result =
                decimalDigits > 0
                    ? withSpaces.reverse().join("") + "." + result.slice(indexOf(result, ".") + 1, result.length)
                    : withSpaces.reverse().join("");
            return (isNegative ? "-" : "") + result;
        }
    }
    getWindowScaling() {
        return window.devicePixelRatio * 100;
    }
    getScaling() {
        if (this.getWindowScaling() < 125) {
            return "";
        }
        else if (125 <= this.getWindowScaling() && this.getWindowScaling() < 150) {
            return "_x1_25";
        }
        else if (150 <= this.getWindowScaling() && this.getWindowScaling() < 175) {
            return "_x1_5";
        }
        else if (175 <= this.getWindowScaling() && this.getWindowScaling() < 200) {
            return "_x1_75";
        }
        else if (200 <= this.getWindowScaling() && this.getWindowScaling() < 300) {
            return "_x2";
        }
        else if (300 <= this.getWindowScaling() && this.getWindowScaling() < 400) {
            return "_x3";
        }
        else if (this.getWindowScaling() > 400) {
            return "_x4";
        }
    }
    getPageFromPoint(x, y) {
        let pageDiv;
        for (let elem of document.elementsFromPoint(x, y)) {
            if (elem?.classList.contains(StiModelService.PAGE_CLASS)) {
                pageDiv = elem;
                break;
            }
        }
        if (pageDiv) {
            for (let page of this.form.pages) {
                if (page.element?.nativeElement == pageDiv) {
                    pageDiv = page;
                    break;
                }
            }
        }
        // let pageDiv = document.elementsFromPoint(x, y).find((elem) => elem?.classList.contains(StiModelService.PAGE_CLASS));
        return pageDiv ? pageDiv : null;
    }
    getPanelFromPoint(x, y, excludeId) {
        let page = this.getPageFromPoint(x, y);
        let panelDivs = document
            .elementsFromPoint(x, y)
            .filter((elem) => elem?.classList.contains(StiModelService.PANEL_CLASS) && elem.id != excludeId);
        return panelDivs && page
            ? page.allElements.find((e) => panelDivs.find((p) => e.guid == p.id))
            : null;
    }
    getMouseOverPagePosition(event) {
        let page = this.getPageFromPoint(event.clientX, event.clientY);
        if (page) {
            let rect = this.getPositionRelativeToForm(page.element.nativeElement);
            return {
                x: event.clientX - (rect.left + page.padding.left * this.zoom),
                y: event.clientY - (rect.top + page.padding.top * this.zoom),
            };
        }
        return null;
    }
    getMouseOverPanelPosition(event, excludeId) {
        let panel = this.getPanelFromPoint(event.clientX, event.clientY, excludeId);
        if (panel) {
            let rect = this.getPositionRelativeToForm(panel.element.nativeElement);
            return { x: event.clientX - rect.left, y: event.clientY - rect.top };
        }
        return null;
    }
    loadAttachedFile() {
        let input = document.createElement("input");
        input.type = "file";
        let this_ = this;
        input.onchange = (_) => {
            let file = Array.from(input.files)[0];
            var reader = new FileReader();
            reader.readAsDataURL(file);
            reader.onload = function () {
                this_.form.attachedFiles.push(reader.result);
                this_.interfaceEvent.next({ name: "FileAttached", data: reader.result });
            };
        };
        input.click();
    }
    downloadFiles(data, file_name) {
        let file = new Blob([data]);
        let a = document.createElement("a"), url = URL.createObjectURL(file);
        a.href = url;
        a.download = file_name;
        document.body.appendChild(a);
        a.click();
        setTimeout(function () {
            document.body.removeChild(a);
            window.URL.revokeObjectURL(url);
        }, 0);
    }
    constructor() {
        this._showTrialBar = false;
        this._showTrialModalWindow = false;
        this._showTrialTitle = false;
        this._canCloseTrialWindow = true;
        this._zoom = 1;
        this._cursor = "default";
        this._ribbon = "Home";
        this.interfaceEvent = new Subject();
        this.isMobileViewer = new Subject();
        this.needToRefresh = new Subject();
        this.needToUpdateHelper = new Subject();
        this.startPdfExportEvent = new Subject();
        this.needToRefreshInnerProperties = new Subject();
        this.currentContentEditableItem = new Subject();
        this.retryButtonRequest = new Subject();
        this.fontFamilies = [];
        this.assemblyInfo = [];
        this.formTreeInnerItemsStates = [];
        this.userName = "";
        this.currentPropertyGridMode = "Properties";
        this.showFormElementsList = false;
        this.sortFormElementsList = false;
        this.showComponentsTab = true;
        this.showInsertTab = true;
        this.hideActiveDropdown = new Subject();
        this.showSignatureEditor = false;
        this.showDraggableModalWindow = false;
        this.showFormTreeWarningWindow = false;
        this.hideActiveAutocompleteList = new Subject();
        this.formTreeWarningMessage = "";
        this.needToSelectedComponent = new Subject();
        this.showButtonMessageWindow = false;
        this.showTieredMenu = false;
        this.showTrialTimerWindow = false;
        this.isInitialized = false;
        this.showContextMenu = false;
        this.showHelpWindow = false;
        this.showSaveDialog = false;
        this.showErrorWindow = false;
        this.basePlanIsTrial = false;
        this.viewerMode = false;
        this.isCloudMode = false;
        this.disableCloseButton = false;
        this.showCornerRadius = true;
        this.showAboutWindow = false;
        this.showGroupPreview = false;
        this.showToolboxSetup = false;
        this.showModalWindow = false;
        this.showProgressAfterUpdatePage = false;
        this.canDrag = true;
        this.tieredMenuX = 0;
        this.tieredMenuY = 0;
        this.currentExpressionInfo = 0; //0-none,1-label column,2-totals,3-label component.
        this.buildVersion = "";
        this.buildDate = "";
        this.errorWindowMessage = "";
        this.propertyGridWidth = 395;
        this.currentTablePropertyHeader = new Subject();
        this.currentTableHeader = "Main";
        this.formEvents = new Subject();
        this.saveDialogFromNew = false;
        this.saveDialogFromOpen = false;
        this.sendedFromButton = false;
        this.loadingFromCloud = false;
        this.showColorDialog = false;
        this.showMoreColorDialog = false;
        this.userIsAuthorized = false;
        this.userWasChecked = false;
        this.showSaveDialogFromCloud = false;
        this.hasEmptyReuiredFields = false;
        this.showExpressionEditor = false;
        this.showProgress = false;
        this.isWebView = false;
        this.currentSelectedFileOption = "Open";
        this._embeddedMode = false;
        this.embeddedEvent = function (data) { };
        this.fileName = "Form.mrt";
        this.enteredUrls = [];
        let savedData = {};
        this.interfaceEvent.subscribe(e => {
            switch (e.name) {
                case "setRibbon":
                    if (!this.viewerMode) {
                        if (e.data.ribbon == "Preview") {
                            savedData = { form: this.form, selectedGuid: this.selectedComponent?.element.guid ?? this.selectedPage?.guid };
                            let form = new StiForm();
                            form.loadFormJsonString(this.form.saveToJsonObject().serialize());
                            this._form = form;
                            this.formEvents.next({ name: "ValueChanged" });
                        }
                        else if (e.data.oldRibbon == "Preview") {
                            let form = new StiForm();
                            form.loadFormJsonString(savedData["form"].saveToJsonObject().serialize());
                            let selectedElement = form.getElementByGuid(savedData["selectedGuid"]);
                            if (selectedElement) {
                                if (!(selectedElement instanceof StiFormPageElement)) {
                                    selectedElement.needToSelectComponent = true;
                                }
                                else {
                                    this.selectedPage = selectedElement;
                                }
                            }
                            this._form = form;
                            this.formEvents.next({ name: "ValueChanged" });
                        }
                    }
                    break;
            }
        });
    }
    static { this.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "18.2.14", ngImport: i0, type: StiModelService, deps: [], target: i0.ɵɵFactoryTarget.Injectable }); }
    static { this.ɵprov = i0.ɵɵngDeclareInjectable({ minVersion: "12.0.0", version: "18.2.14", ngImport: i0, type: StiModelService }); }
}
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "18.2.14", ngImport: i0, type: StiModelService, decorators: [{
            type: Injectable
        }], ctorParameters: () => [] });
//# sourceMappingURL=data:application/json;base64,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