import StiPadding from "../../properties/StiPadding";
import StiRectangleGeometry from "../../properties/StiRectangleGeometry";
import StiFormObject from "./StiFormObject";
import { StiLabelItem } from "../text/StiLabelItem";
import { StiLabelAlignment, StiVerticalAlignment } from "../../system/enums";
export class StiFormElement extends StiFormObject {
    constructor() {
        super(...arguments);
        this._visible = true;
        this._geometry = new StiRectangleGeometry(0, 0, this);
        this._padding = new StiPadding(8, 8, 8, 8, this);
        this._label = new StiLabelItem(this, "");
        this._readOnly = false;
        this._isChanged = true;
        this.render = true;
        this.position = "absolute";
        // public label: StiLabelItem = new StiLabelItem(this, "");
        this.hasGeometry = false;
        this.isAdded = false;
    }
    get maxWidth() {
        if (this.label.alignment == StiLabelAlignment.Top || this.label.eValue.length == 0) {
            return this.geometry.width - this.padding.left - this.padding.right;
        }
        else {
            return this.geometry.width - this.label.width - this.padding.left - this.padding.right;
        }
    }
    setDefaultValues() { }
    isChanged() {
        return this._isChanged || this._geometry.isChanged() || this._padding.isChanged();
    }
    getDefaultEditableContent() {
        return null;
    }
    canUpdateProperty(propName) {
        return true;
    }
    isLabelTopAlignment() {
        return this.label.alignment == StiLabelAlignment.Top ? true : false;
    }
    getVerticalAlignment() {
        if (this.label.verticalAlignment == StiVerticalAlignment.Top) {
            return "start";
        }
        else if (this.label.verticalAlignment == StiVerticalAlignment.Center) {
            return "center";
        }
        else {
            return "flex-end";
        }
    }
    resetChanged() {
        this._isChanged = false;
        this._geometry.resetChanged();
        this._padding.resetChanged();
    }
    get visible() {
        return this._visible;
    }
    set visible(value) {
        this._visible = value;
        this._isChanged = true;
    }
    get label() {
        return this._label;
    }
    set label(value) {
        let oldValue = this._label;
        if (oldValue != value) {
            this._label = value;
            this._isChanged = true;
        }
    }
    get geometry() {
        return this._geometry;
    }
    set geometry(value) {
        if (!this.geometry || !value || !this.geometry.equals(value)) {
            this._isChanged = true;
        }
        else {
            value.resetChanged();
        }
        this._geometry = value;
        this._geometry.changesInterceptor = this;
    }
    get padding() {
        return this._padding;
    }
    set padding(value) {
        let oldValue = this._padding;
        if (oldValue != value) {
            this._padding = value;
            this._padding.changesInterceptor = this;
            this._isChanged = true;
            this.propertyChanged("padding");
        }
    }
    get name() {
        return this._name;
    }
    set name(value) {
        this._name = value;
        this._isChanged = true;
    }
    get readOnly() {
        return this._readOnly;
    }
    set readOnly(value) {
        this._readOnly = value;
        this._isChanged = true;
    }
    get computedHeight() {
        return Number(this.height) + Number(this.padding.top) + Number(this.padding.bottom);
    }
    get computedWidth() {
        return Number(this.width) - Number(this.padding.left) - Number(this.padding.right);
    }
    get height() {
        return this.geometry.height;
    }
    set height(value) {
        this.geometry.height = value;
    }
    get width() {
        return this.geometry.width;
    }
    set width(value) {
        this.geometry.width = value;
    }
    get x() {
        return this.geometry.x;
    }
    set x(value) {
        this.geometry.x = value;
    }
    get y() {
        return this.geometry.y;
    }
    set y(value) {
        this.geometry.y = value;
    }
    hasText() {
        return false;
    }
    propertyChanged(propName) { }
    showLabelPlaceholder(isSelected) {
        return this.label.alignment == StiLabelAlignment.Top || this.label.width < 15 || !isSelected ? true : false;
    }
    getContentAlign() {
        return this.label.alignment == StiLabelAlignment.Right ? "end" : "start";
    }
    getLabelWidth() {
        return this.label.alignment == StiLabelAlignment.Top ? "100%" : this.label.width + "px";
    }
    getTopMargin() {
        return this.label.alignment == StiLabelAlignment.Top && this.label.value ? "8" : "0";
    }
    setProperty(propName, value, oldObject) {
        if (!this.canUpdateProperty(propName)) {
            return;
        }
        if (propName.startsWith("label") && "label" in this) {
            this.label.setProperty(propName.replace("label.", ""), value.clone ? value.clone() : value, oldObject);
        }
        else if (propName.startsWith("description") && "description" in this) {
            this.description.setProperty(propName.replace("description.", ""), value.clone ? value.clone() : value, oldObject);
        }
        else if (this[propName] &&
            (oldObject === null ||
                (propName == "padding" && this[propName].equals(oldObject)) ||
                this[propName] == oldObject)) {
            this[propName] = value.clone ? value.clone() : value;
        }
        else if (propName == "innerText.color" &&
            this["textColor"] &&
            (oldObject === null || oldObject.equals(this["textColor"]))) {
            this["textColor"] = value.clone();
        }
        else if (propName == "innerText.backgroundColor" &&
            this["backgroundColor"] &&
            (oldObject === null || oldObject.equals(this["backgroundColor"]))) {
            this["backgroundColor"] = value.clone();
        }
        else if (propName == "innerText.borderColor" &&
            this["borderColor"] &&
            (oldObject === null || oldObject.equals(this["borderColor"]))) {
            this["borderColor"] = value.clone();
        }
        else if (propName == "innerText.checkColor" &&
            this["checkColor"] &&
            (oldObject === null || oldObject.equals(this["checkColor"]))) {
            this["checkColor"] = value.clone();
        }
        else if (propName == "innerText.borderWeight" &&
            this["borderWeight"] &&
            (oldObject === null || this["borderWeight"] == oldObject)) {
            this["borderWeight"] = value;
        }
        else if (propName == "innerText.contentAlignment" &&
            this["contentAlignment"] &&
            (oldObject === null || this["contentAlignment"] == oldObject)) {
            this["contentAlignment"] = value;
        }
        else if (propName.startsWith("innerText.font") &&
            this["font"] &&
            (oldObject === null || oldObject.equals(this["font"]))) {
            this["font"].setProperty(propName.replace("innerText.font.", ""), value);
        }
        else if (propName.startsWith("padding.") &&
            this["padding"] &&
            (oldObject === null || oldObject.equals(this["padding"]))) {
            let opt = propName.replace("padding.", "");
            this["padding"][opt] = value;
        }
    }
    arrayOfStylesToString(array) {
        let styles = [];
        if (array.length > 0) {
            array.forEach((item) => {
                if (item != "Underline") {
                    styles.push(item);
                }
            });
        }
        return styles.length > 0 ? styles.join(" ").toLocaleLowerCase() : "";
    }
    getUsedFonts(usedFonts) {
        usedFonts.add(this._label.font.family, this.arrayOfStylesToString(this._label.font.style), this._label.font.size);
    }
    getParent(form) {
        let container;
        form.pages.forEach((page) => {
            if (page.elements.indexOf(this) >= 0) {
                container = page;
            }
            else {
                page.allElements.forEach((e) => {
                    let elements = e.elements;
                    if (elements && elements.indexOf(this) >= 0) {
                        container = e;
                    }
                });
            }
        });
        return container;
    }
    getOverPageGeometry(form) {
        let geometry = this.geometry.clone();
        form.pages.forEach((page) => {
            let allElements = page.allElements;
            let child = this;
            let find = true;
            while (find) {
                find = false;
                allElements.forEach((e) => {
                    let elements = e.elements;
                    if (elements && elements.indexOf(child) >= 0) {
                        geometry.x += e.geometry.x + e.padding.left;
                        geometry.y += e.geometry.y + e.padding.top;
                        child = e;
                        find = true;
                    }
                });
            }
        });
        return geometry;
    }
    saveToJsonObject(options) {
        let jObject = super.saveToJsonObject(options);
        jObject.addPropertyBool("Visible", this.visible);
        jObject.addPropertyBool("ReadOnly", this.readOnly);
        jObject.addPropertyJObject("Padding", this.padding.saveToJsonObject(options));
        jObject.addPropertyJObject("Geometry", this.geometry.saveToJsonObject(options));
        jObject.addPropertyStringNullOrEmpty("Name", this.name);
        return jObject;
    }
    loadFromJsonObject(json) {
        this.isAdded = true;
        super.loadFromJsonObject(json);
        for (let property of json.properties()) {
            switch (property.name) {
                case "Visible":
                    this.visible = property.value;
                    break;
                case "Enabled":
                case "ReadOnly":
                    if (property.name == "Enabled") {
                        this.readOnly = !property.value;
                    }
                    else {
                        this.readOnly = property.value;
                    }
                    break;
                case "Padding":
                    this.padding = new StiPadding();
                    this.padding.loadFromJsonObject(property.value);
                    break;
                case "Geometry":
                    this.geometry = new StiRectangleGeometry();
                    this.geometry.loadFromJsonObject(property.value);
                    break;
                case "Name":
                    this.name = property.value;
                    break;
            }
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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