import { StiInputTextItem } from "../helper/StiInputTextItem";
import StiRectangleGeometry from "../properties/StiRectangleGeometry";
import { StiLocalizationService } from "../services/sti-localization.service";
import { StiStaticHelperService } from "../services/sti-static-helper.service";
import Color from "../system/Color";
import { Enum } from "../system/Enum";
import { StiHorizontalAlignment, StiLineWeight, StiDateTimeBoxType, StiDateTimeBoxValueChoise } from "../system/enums";
import { Font } from "../system/Font";
import { StiFormElement } from "./base/StiFormElement";
import { StiLabelItem } from "./text/StiLabelItem";
import { StiTextItem } from "./text/StiTextItem";
export class StiDateTimeBoxElement extends StiFormElement {
    constructor(form) {
        super();
        this.type = "DateTime_BoxElement";
        this.tooltip = "";
        this.required = false;
        this.description = new StiTextItem();
        this.font = new Font("Segoe UI", 10);
        this.contentAlignment = StiHorizontalAlignment.Left;
        this.backgroundColor = Color.white();
        this.textColor = Color.black();
        this.borderColor = Color.lightGray();
        this.borderWeight = StiLineWeight.Thin;
        this.allowNull = false;
        this._dateTimeValueIsNotNull = true;
        this._dateTimeFormat = "m/d/yy h:MM tt";
        this._dateFomat = "dd.mm.yyyy";
        this._timeFormat = "HH:MM:ss";
        this._initialState = StiDateTimeBoxValueChoise.Custom;
        this._dateValue = new Date();
        this._valueType = StiDateTimeBoxType.Date;
        this._dateValueString = "";
        this.form = form;
        this.geometry = new StiRectangleGeometry(150, 40);
        this.label = new StiLabelItem(this, StiLocalizationService.get("Dialogs", "StiDateTimePickerControl"));
        this.updateDateValueString();
    }
    updateToViewerMode() {
        this.initialState = StiDateTimeBoxValueChoise.Custom;
    }
    get dateValueString() {
        return this._dateValueString;
    }
    get dateTimeValueIsNotNull() {
        return this._dateTimeValueIsNotNull;
    }
    set dateTimeValueIsNotNull(value) {
        this._dateTimeValueIsNotNull = value;
        this.updateDateValueString();
    }
    set form(value) {
        this._form = value;
        this.updateDateValueString();
    }
    get form() {
        return this._form;
    }
    get initialState() {
        return this._initialState;
    }
    get dateValue() {
        return this._dateValue;
    }
    set dateValue(value) {
        this._dateValue = value;
        this._initialState = StiDateTimeBoxValueChoise.Custom;
        this.updateDateValueString();
    }
    updateDateValueString() {
        if ((this.allowNull && !this.dateTimeValueIsNotNull) || (this.dateValue == null)) {
            this._dateValueString = "";
        }
        else {
            this._dateValueString = StiStaticHelperService.formatDate(this.dateValue, this.format, this.form?.settings.culture);
        }
    }
    set initialState(value) {
        if (value != this._initialState && value == StiDateTimeBoxValueChoise.Current) {
            this._dateValue = new Date();
        }
        this._initialState = value;
        this.updateDateValueString();
    }
    setDefaultValues() {
        this.initialState = StiDateTimeBoxValueChoise.Current;
    }
    get valueType() {
        return this._valueType;
    }
    set valueType(value) {
        this._valueType = value;
        this.updateDateValueString();
    }
    get value() {
        return this.dateValue;
    }
    get format() {
        let format = this.valueType == StiDateTimeBoxType.Date ? this._dateFomat :
            (this.valueType == StiDateTimeBoxType.DateTime ? this._dateTimeFormat : this._timeFormat);
        return format ?? "";
    }
    get dateTimeFormat() {
        return this._dateTimeFormat;
    }
    set dateTimeFormat(value) {
        this._dateTimeFormat = value;
        this.updateDateValueString();
    }
    get dateFormat() {
        return this._dateFomat;
    }
    set dateFormat(value) {
        this._dateFomat = value;
        this.updateDateValueString();
    }
    get timeFormat() {
        return this._timeFormat;
    }
    set timeFormat(value) {
        this._timeFormat = value;
        this.updateDateValueString();
    }
    setTabIndex(value) {
        this.tabIndex = value++;
        return value;
    }
    getUsedFonts(usedFonts) {
        super.getUsedFonts(usedFonts);
        usedFonts.add(this.font.family, this.arrayOfStylesToString(this.font.style), this.font.size);
        usedFonts.add(this.description.font.family, this.arrayOfStylesToString(this.description.font.style), this.description.font.size);
    }
    getDefaultEditableContent() {
        let textItem = new StiInputTextItem("", this.font, this.textColor, this.backgroundColor, this.contentAlignment);
        textItem.element = this;
        return textItem;
    }
    htmlStyle() {
        return `${this.font.htmlStyle()};color:${this.textColor.toHtmlColor()};text-align:${Enum.getName(StiHorizontalAlignment, this.contentAlignment)}`;
    }
    saveToJsonObject(options) {
        let jObject = super.saveToJsonObject(options);
        jObject.addPropertyJObject("Label", this.label.saveToJsonObject(options));
        jObject.addPropertyJObject("Description", this.description.saveToJsonObject(options));
        jObject.addPropertyJObject("Font", this.font.saveToJsonObject(options));
        jObject.addPropertyColor("BackgroundColor", this.backgroundColor);
        jObject.addPropertyColor("TextColor", this.textColor);
        jObject.addPropertyString("Tooltip", this.tooltip);
        jObject.addPropertyString("DateTimeFormat", this.dateTimeFormat);
        jObject.addPropertyString("DateFormat", this.dateFormat);
        jObject.addPropertyString("TimeFormat", this.timeFormat);
        jObject.addPropertyBool("Required", this.required);
        jObject.addPropertyColor("BorderColor", this.borderColor);
        jObject.addPropertyEnum("BorderWeight", StiLineWeight, this.borderWeight);
        jObject.addPropertyEnum("HorizontalAlignment", StiHorizontalAlignment, this.contentAlignment);
        jObject.addPropertyEnum("ValueType", StiDateTimeBoxType, this.valueType);
        jObject.addPropertyEnum("InitialState", StiDateTimeBoxValueChoise, this.initialState);
        jObject.addPropertyBool("AllowNull", this.allowNull);
        jObject.addPropertyBool("DateTimeValueIsNotNull", this.dateTimeValueIsNotNull);
        jObject.addPropertyString("DateValue", this.dateValue?.toJSON(), "");
        jObject.addPropertyString("DateValueString", this._dateValueString, "");
        return jObject;
    }
    loadFromJsonObject(json) {
        super.loadFromJsonObject(json);
        let dateValue;
        for (let property of json.properties()) {
            switch (property.name) {
                case "BackgroundColor":
                    this.backgroundColor = new Color();
                    this.backgroundColor.deserializeFromString(property.value);
                    break;
                case "TextColor":
                    this.textColor = new Color();
                    this.textColor.deserializeFromString(property.value);
                    break;
                case "Label":
                    this.label = new StiLabelItem(this);
                    this.label.loadFromJsonObject(property.value);
                    break;
                case "Font":
                    this.font = new Font();
                    this.font.loadFromJsonObject(property.value);
                    break;
                case "Description":
                    this.description = new StiTextItem();
                    this.description.loadFromJsonObject(property.value);
                    break;
                case "Tooltip":
                    this.tooltip = property.value;
                    break;
                case "DateTimeFormat":
                    this.dateTimeFormat = property.value;
                    break;
                case "DateFormat":
                    this.dateFormat = property.value;
                    break;
                case "TimeFormat":
                    this.timeFormat = property.value;
                    break;
                case "Required":
                    this.required = property.value;
                    break;
                case "BorderColor":
                    this.borderColor = new Color();
                    this.borderColor.deserializeFromString(property.value);
                    break;
                case "HorizontalAlignment":
                    this.contentAlignment = Enum.parse(StiHorizontalAlignment, property.value);
                    break;
                case "BorderWeight":
                    this.borderWeight = Enum.parse(StiLineWeight, property.value);
                    break;
                case "ValueType":
                    this.valueType = Enum.parse(StiDateTimeBoxType, property.value);
                    break;
                case "InitialState":
                    this.initialState = Enum.parse(StiDateTimeBoxValueChoise, property.value);
                    break;
                case "AllowNull":
                    this.allowNull = property.value;
                    break;
                case "DateTimeValueIsNotNull":
                    this.dateTimeValueIsNotNull = property.value;
                    break;
                case "DateValue":
                    if (property.value?.length > 0) {
                        dateValue = new Date(property.value);
                    }
                    break;
                case "DateValueString":
                    //this.dateValueString = property.value;
                    break;
            }
        }
        this._dateValue = this.initialState == StiDateTimeBoxValueChoise.Current || dateValue == null ? new Date() : dateValue;
        this.updateDateValueString();
    }
}
//# sourceMappingURL=data:application/json;base64,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