import { Injectable } from '@angular/core';
import { Subject } from 'rxjs';
import * as i0 from "@angular/core";
import * as i1 from "../forms/form.service";
import * as i2 from "./helper.service";
import * as i3 from "./model.service";
import * as i4 from "./collections.service";
import * as i5 from "./controller.service";
export class ExportService {
    constructor(formService, helper, model, collections, controller) {
        this.formService = formService;
        this.helper = helper;
        this.model = model;
        this.collections = collections;
        this.controller = controller;
        this.exportSettings = {};
        this.sendMail = false;
        this.subject = new Subject();
        controller.getActionMessage().subscribe((message) => {
            switch (message.action) {
                case 'Export':
                    this.export(message.data.format);
                    break;
                case 'SendEmail':
                    this.export(message.data.format, false, true);
                    break;
            }
        });
    }
    getMessage() {
        return this.subject.asObservable();
    }
    export(format, update = false, sendMail = false) {
        if (!this.helper.checkCloudAuthorization('export')) {
            return;
        }
        this.sendMail = sendMail;
        this.format = format;
        this.exportSettings = this.getDefaultExportSettings(this.format);
        if ((this.model.options.exports.showExportDialog && !sendMail) || (this.model.options.email.showExportDialog && sendMail)) {
            this.exportFormSettings = this.getExportSetings(update);
            if (!update) {
                this.formService.closeForm('exportForm');
                setTimeout(() => {
                    this.formService.showForm('exportForm');
                });
            }
        }
        else {
            this.postExport();
        }
    }
    postExport() {
        if (!this.sendMail) {
            this.controller.postExport(this.format, this.exportSettings);
        }
        else {
            this.subject.next('postMail');
        }
    }
    getExportSetings(update) {
        let result;
        this.getDefaultSettings();
        switch (this.format) {
            case 'Document':
                result = { components: this.getComponents(['SaveReportMdc', 'SaveReportMdz', 'SaveReportMdx', 'SaveReportPassword']) };
                break;
            case 'Pdf':
                result = {
                    components: this.getComponents(['PageRangeGroup', 'PageRangeAll', 'PageRangeCurrentPage', 'PageRangePages', 'PageRangePagesText', 'SettingsGroup', 'ImageResolution', 'ImageCompressionMethod',
                        'AllowEditable', 'ImageQuality', /*'StandardPdfFonts',*/ 'EmbeddedFonts', /*'UseUnicode', 'Compressed',*/ 'ExportRtfTextAsImage', 'PdfACompliance', 'UseDigitalSignature', 'DocumentSecurityButton', 'DigitalSignatureButton',
                        'OpenAfterExport', 'PasswordInputUser', 'PasswordInputOwner', 'PrintDocument', 'ModifyContents', 'CopyTextAndGraphics',
                        'AddOrModifyTextAnnotations', 'KeyLength', 'GetCertificateFromCryptoUI', 'SubjectNameString']),
                    openAfterExport: true
                };
                break;
            case 'Xps':
                result = {
                    components: this.getComponents(['PageRangeGroup', 'PageRangeAll', 'PageRangeCurrentPage', 'PageRangePages', 'PageRangePagesText', 'SettingsGroup', 'ImageResolution', 'ImageQuality', 'OpenAfterExport',
                        'ExportRtfTextAsImage']),
                    openAfterExport: true
                };
                break;
            case 'Ppt2007':
                result = { components: this.getComponents(['PageRangeGroup', 'PageRangeAll', 'PageRangeCurrentPage', 'PageRangePages', 'PageRangePagesText', 'SettingsGroup', 'ImageResolution', 'ImageQuality']) };
                break;
            case 'Html':
                result = {
                    components: this.getComponents(['HtmlType', 'Zoom', 'ImageFormatForHtml', 'ExportMode', 'UseEmbeddedImages', 'AddPageBreaks']),
                    openAfterExport: true
                };
                break;
            case 'Html5':
                result = {
                    components: this.getComponents(['PageRangeGroup', 'PageRangeAll', 'PageRangeCurrentPage', 'PageRangePages', 'PageRangePagesText', 'SettingsGroup', 'HtmlType', 'ImageFormatForHtml', 'ImageResolution',
                        'ImageQuality', 'ContinuousPages', 'OpenAfterExport']),
                    openAfterExport: true
                };
                break;
            case 'Mht':
                result = {
                    components: this.getComponents(['PageRangeGroup', 'PageRangeAll', 'PageRangeCurrentPage', 'PageRangePages', 'PageRangePagesText', 'SettingsGroup', 'HtmlType', 'Zoom', 'ImageFormatForHtml',
                        'ExportMode', 'AddPageBreaks'])
                };
                break;
            case 'Text':
                result = {
                    components: this.getComponents(['PageRangeGroup', 'PageRangeAll', 'PageRangeCurrentPage', 'PageRangePages', 'PageRangePagesText', 'SettingsGroup', 'KillSpaceLines',
                        'PutFeedPageCode', 'DrawBorder', 'CutLongLines', 'BorderType', 'ZoomX', 'ZoomY', 'EncodingTextOrCsvFile'])
                };
                break;
            case 'Rtf':
                result = {
                    components: this.getComponents(['PageRangeGroup', 'PageRangeAll', 'PageRangeCurrentPage', 'PageRangePages', 'PageRangePagesText', 'SettingsGroup', 'ImageResolution',
                        'ImageQuality', 'ExportModeRtf', 'UsePageHeadersAndFooters', 'RemoveEmptySpaceAtBottom'])
                };
                break;
            case 'Word2007':
                result = {
                    components: this.getComponents(['PageRangeGroup', 'PageRangeAll', 'PageRangeCurrentPage', 'PageRangePages', 'PageRangePagesText', 'SettingsGroup', 'ImageResolution',
                        'ImageQuality', 'UsePageHeadersAndFooters', 'RemoveEmptySpaceAtBottom'])
                };
                break;
            case 'Odt':
                result = {
                    components: this.getComponents(['PageRangeGroup', 'PageRangeAll', 'PageRangeCurrentPage', 'PageRangePages', 'PageRangePagesText', 'SettingsGroup', 'ImageResolution',
                        'ImageQuality', 'RemoveEmptySpaceAtBottom'])
                };
                break;
            case 'Excel':
            case 'ExcelBinary':
                result = {
                    components: this.getComponents(['PageRangeGroup', 'PageRangeAll', 'PageRangeCurrentPage', 'PageRangePages', 'PageRangePagesText', 'SettingsGroup', 'ExcelType', 'ImageResolution',
                        'ImageQuality', 'DataExportMode', 'ExportObjectFormatting', 'UseOnePageHeaderAndFooter', 'ExportEachPageToSheet', 'ExportPageBreaks']),
                    openAfterExport: true
                };
                break;
            case 'ExcelXml':
                result = {
                    components: this.getComponents(['PageRangeGroup', 'PageRangeAll', 'PageRangeCurrentPage', 'PageRangePages', 'PageRangePagesText', 'SettingsGroup', 'ExcelType'])
                };
                break;
            case 'Excel2007':
                result = {
                    components: this.getComponents(['PageRangeGroup', 'PageRangeAll', 'PageRangeCurrentPage', 'PageRangePages', 'PageRangePagesText', 'SettingsGroup', 'ExcelType', 'ImageResolution',
                        'ImageQuality', 'DataExportMode', 'ExportObjectFormatting', 'UseOnePageHeaderAndFooter', 'ExportEachPageToSheet', 'ExportPageBreaks'])
                };
                break;
            case 'Ods':
                result = {
                    components: this.getComponents(['PageRangeGroup', 'PageRangeAll', 'PageRangeCurrentPage', 'PageRangePages', 'PageRangePagesText', 'SettingsGroup', 'ImageResolution',
                        'ImageQuality'])
                };
                break;
            case 'Csv':
                result = {
                    components: this.getComponents(['PageRangeGroup', 'PageRangeAll', 'PageRangeCurrentPage', 'PageRangePages', 'PageRangePagesText', 'SettingsGroup', 'DataType', 'EncodingTextOrCsvFile',
                        'Separator', 'DataExportMode', 'SkipColumnHeaders'])
                };
                break;
            case 'Dbf':
                result = {
                    components: this.getComponents(['PageRangeGroup', 'PageRangeAll', 'PageRangeCurrentPage', 'PageRangePages', 'PageRangePagesText', 'SettingsGroup', 'DataType', 'EncodingDbfFile'])
                };
                break;
            case 'Dif':
                result = {
                    components: this.getComponents(['PageRangeGroup', 'PageRangeAll', 'PageRangeCurrentPage', 'PageRangePages', 'PageRangePagesText', 'SettingsGroup', 'DataType', 'ExportDataOnly',
                        'UseDefaultSystemEncoding', 'EncodingDifFile'])
                };
                break;
            case 'Sylk':
                result = {
                    components: this.getComponents(['PageRangeGroup', 'PageRangeAll', 'PageRangeCurrentPage', 'PageRangePages', 'PageRangePagesText', 'SettingsGroup', 'DataType', 'ExportDataOnly',
                        'UseDefaultSystemEncoding', 'EncodingDifFile'])
                };
                break;
            case 'Json':
                result = {
                    components: this.getComponents(['PageRangeGroup', 'PageRangeAll', 'PageRangeCurrentPage', 'PageRangePages', 'PageRangePagesText', 'SettingsGroup', 'DataType', 'DataExportMode'])
                };
                break;
            case 'Xml':
                result = {
                    components: this.getComponents(['PageRangeGroup', 'PageRangeAll', 'PageRangeCurrentPage', 'PageRangePages', 'PageRangePagesText', 'SettingsGroup', 'DataType'])
                };
                break;
            case 'ImageBmp':
            case 'ImageGif':
            case 'ImageJpeg':
            case 'ImagePcx':
            case 'ImagePng':
            case 'ImageTiff':
            case 'ImageEmf':
            case 'ImageSvg':
            case 'ImageSvgz':
                result = {
                    components: this.getComponents(['PageRangeGroup', 'PageRangeAll', 'PageRangeCurrentPage', 'PageRangePages', 'PageRangePagesText', 'SettingsGroup', 'ImageType', 'ImageZoom', 'ImageResolution',
                        'ImageFormat', 'DitheringType', 'TiffCompressionScheme', 'CompressToArchive', 'CutEdges'])
                };
                break;
        }
        result.groups = this.getGroups(this.format);
        result.update = update;
        return result;
    }
    getComponents(names) {
        const mrgn = '8px';
        const components = [];
        names.forEach(name => {
            switch (name) {
                case 'ImageType':
                    components.push({
                        name, type: 'DropDownListForExportForm', label: this.model.loc('Type'), width: 160, tooltip: this.model.loc('TypeTooltip'),
                        margin: '2px ' + mrgn + ' 2px ' + mrgn, items: this.collections.getImageTypesItems()
                    });
                    break;
                case 'DataType':
                    components.push({
                        name, type: 'DropDownListForExportForm', label: this.model.loc('Type'), width: 160, tooltip: this.model.loc('TypeTooltip'),
                        margin: '2px ' + mrgn + ' 2px ' + mrgn, items: this.collections.getDataTypesItems()
                    });
                    break;
                case 'ExcelType':
                    components.push({
                        name, type: 'DropDownListForExportForm', label: this.model.loc('Type'), width: 160, tooltip: this.model.loc('TypeTooltip'),
                        margin: '2px ' + mrgn + ' 2px ' + mrgn, items: this.collections.getExcelTypesItems()
                    });
                    break;
                case 'HtmlType':
                    components.push({
                        name, type: 'DropDownListForExportForm', label: this.model.loc('Type'), width: 160, tooltip: this.model.loc('TypeTooltip'),
                        margin: '2px ' + mrgn + ' 2px ' + mrgn, items: this.collections.getHtmlTypesItems()
                    });
                    break;
                case 'Zoom':
                    components.push({
                        name, type: 'DropDownListForExportForm', label: this.model.loc('ZoomHtml'), width: 160, tooltip: this.model.loc('ZoomHtmlTooltip'),
                        margin: '2px ' + mrgn + ' 2px ' + mrgn, items: this.collections.getZoomItems()
                    });
                    break;
                case 'ImageZoom':
                    components.push({
                        name, type: 'DropDownListForExportForm', label: this.model.loc('ZoomHtml'), width: 160, tooltip: this.model.loc('ZoomHtmlTooltip'),
                        margin: '2px ' + mrgn + ' 2px ' + mrgn, items: this.collections.getZoomItems()
                    });
                    break;
                case 'ImageFormatForHtml':
                    components.push({
                        name, type: 'DropDownListForExportForm', label: this.model.loc('ImageFormatForHtml'), width: 160, tooltip: this.model.loc('ImageFormatForHtmlTooltip'),
                        margin: '2px ' + mrgn + ' 2px ' + mrgn, items: this.collections.getImageFormatForHtmlItems()
                    });
                    break;
                case 'ExportMode':
                    components.push({
                        name, type: 'DropDownListForExportForm', label: this.model.loc('ExportMode'), width: 160, tooltip: this.model.loc('ExportModeTooltip'),
                        margin: '2px ' + mrgn + ' 2px ' + mrgn, items: this.collections.getExportModeItems()
                    });
                    break;
                case 'CompressToArchive':
                    components.push({
                        name, type: 'CheckBox', caption: this.model.loc('CompressToArchive'), tooltip: this.model.loc('CompressToArchiveTooltip'),
                        margin: '4px ' + mrgn + ' 4px ' + mrgn
                    });
                    break;
                case 'UseEmbeddedImages':
                    components.push({
                        name, type: 'CheckBox', caption: this.model.loc('EmbeddedImageData'), tooltip: this.model.loc('EmbeddedImageDataTooltip'),
                        margin: '4px ' + mrgn + ' 4px ' + mrgn
                    });
                    break;
                case 'AddPageBreaks':
                    components.push({
                        name, type: 'CheckBox', caption: this.model.loc('AddPageBreaks'), tooltip: this.model.loc('AddPageBreaksTooltip'),
                        margin: '4px ' + mrgn + ' 4px ' + mrgn
                    });
                    break;
                case 'ImageResolution':
                    components.push({
                        name, type: 'DropDownListForExportForm', label: this.model.loc('ImageResolution'), width: 160, tooltip: this.model.loc('ImageResolutionTooltip'),
                        margin: '2px 4px 2px ' + mrgn, items: this.collections.getImageResolutionItems()
                    });
                    break;
                case 'ImageCompressionMethod':
                    components.push({
                        name, type: 'DropDownListForExportForm', label: this.model.loc('ImageCompressionMethod'), width: 160, tooltip: this.model.loc('ImageCompressionMethodTooltip'),
                        margin: '2px ' + mrgn + ' 2px ' + mrgn, items: this.collections.getImageCompressionMethodItems()
                    });
                    break;
                case 'AllowEditable':
                    components.push({
                        name, type: 'DropDownListForExportForm', label: this.model.loc('AllowEditable'), width: 160, tooltip: this.model.loc('AllowEditableTooltip'),
                        margin: '2px ' + mrgn + ' 2px ' + mrgn, items: this.collections.getAllowEditableItems()
                    });
                    break;
                case 'ImageQuality':
                    components.push({
                        name, type: 'DropDownListForExportForm', label: this.model.loc('ImageQuality'), width: 160, tooltip: this.model.loc('ImageQualityTooltip'),
                        margin: '2px 4px 2px ' + mrgn, items: this.collections.getImageQualityItems()
                    });
                    break;
                case 'ContinuousPages':
                    components.push({
                        name, type: 'CheckBox', caption: this.model.loc('ContinuousPages'), tooltip: this.model.loc('ContinuousPagesTooltip'),
                        margin: '4px ' + mrgn + ' 4px ' + mrgn
                    });
                    break;
                case 'StandardPdfFonts':
                    components.push({
                        name, type: 'CheckBox', caption: this.model.loc('StandardPDFFonts'), tooltip: this.model.loc('StandardPDFFontsTooltip'),
                        margin: '4px ' + mrgn + ' 4px ' + mrgn
                    });
                    break;
                case 'EmbeddedFonts':
                    components.push({
                        name, type: 'CheckBox', caption: this.model.loc('EmbeddedFonts'), tooltip: this.model.loc('EmbeddedFontsTooltip'),
                        margin: '4px ' + mrgn + ' 4px ' + mrgn
                    });
                    break;
                case 'UseUnicode':
                    components.push({
                        name, type: 'CheckBox', caption: this.model.loc('UseUnicode'), tooltip: this.model.loc('UseUnicodeTooltip'),
                        margin: '4px ' + mrgn + ' 4px ' + mrgn
                    });
                    break;
                case 'Compressed':
                    components.push({
                        name, type: 'CheckBox', caption: this.model.loc('Compressed'), tooltip: this.model.loc('CompressedTooltip'),
                        margin: '4px ' + mrgn + ' 4px ' + mrgn
                    });
                    break;
                case 'ExportRtfTextAsImage':
                    components.push({
                        name, type: 'CheckBox', caption: this.model.loc('ExportRtfTextAsImage'), tooltip: this.model.loc('ExportRtfTextAsImageTooltip'),
                        margin: '4px ' + mrgn + ' 4px ' + mrgn
                    });
                    break;
                case 'PdfACompliance':
                    components.push({
                        name, type: 'CheckBox', caption: this.model.loc('PdfACompliance'), tooltip: this.model.loc('PdfAComplianceTooltip'),
                        margin: '4px ' + mrgn + ' 4px ' + mrgn
                    });
                    break;
                case 'KillSpaceLines':
                    components.push({
                        name, type: 'CheckBox', caption: this.model.loc('KillSpaceLines'), tooltip: this.model.loc('KillSpaceLinesTooltip'),
                        margin: '4px ' + mrgn + ' 4px ' + mrgn
                    });
                    break;
                case 'PutFeedPageCode':
                    components.push({
                        name, type: 'CheckBox', caption: this.model.loc('PutFeedPageCode'), tooltip: this.model.loc('PutFeedPageCodeTooltip'),
                        margin: '4px ' + mrgn + ' 4px ' + mrgn
                    });
                    break;
                case 'DrawBorder':
                    components.push({
                        name, type: 'CheckBox', caption: this.model.loc('DrawBorder'), tooltip: this.model.loc('DrawBorderTooltip'),
                        margin: '4px ' + mrgn + ' 4px ' + mrgn
                    });
                    break;
                case 'CutLongLines':
                    components.push({
                        name, type: 'CheckBox', caption: this.model.loc('CutLongLines'), tooltip: this.model.loc('CutLongLinesTooltip'),
                        margin: '4px ' + mrgn + ' 4px ' + mrgn
                    });
                    break;
                case 'BorderType':
                    components.push({
                        name, type: 'DropDownListForExportForm', label: this.model.loc('BorderType'), width: 160, tooltip: this.model.loc('BorderTypeTooltip'),
                        margin: '2px ' + mrgn + ' 2px ' + mrgn, items: this.collections.getBorderTypeItems()
                    });
                    break;
                case 'ZoomX':
                    components.push({
                        name, type: 'DropDownListForExportForm', label: this.model.loc('ZoomXY').replace(':', '') + ' X ', width: 160, tooltip: this.model.loc('ZoomXYTooltip'),
                        margin: '2px ' + mrgn + ' 2px ' + mrgn, items: this.collections.getZoomItems()
                    });
                    break;
                case 'ZoomY':
                    components.push({
                        name, type: 'DropDownListForExportForm', label: this.model.loc('ZoomXY').replace(':', '') + ' Y ', width: 160, tooltip: this.model.loc('ZoomXYTooltip'),
                        margin: '2px ' + mrgn + ' 2px ' + mrgn, items: this.collections.getZoomItems()
                    });
                    break;
                case 'EncodingTextOrCsvFile':
                    components.push({
                        name, type: 'DropDownListForExportForm', label: this.model.loc('EncodingData'), width: 160, tooltip: this.model.loc('EncodingDataTooltip'),
                        margin: '2px ' + mrgn + ' 2px ' + mrgn, items: this.collections.getEncodingDataItems()
                    });
                    break;
                case 'ImageFormat':
                    components.push({
                        name, type: 'DropDownListForExportForm', label: this.model.loc('ImageFormat'), width: 160, tooltip: this.model.loc('ImageFormatTooltip'),
                        margin: '2px ' + mrgn + ' 2px ' + mrgn, items: this.collections.getImageFormatItems()
                    });
                    break;
                case 'DitheringType':
                    components.push({
                        name, type: 'DropDownListForExportForm', label: this.model.loc('MonochromeDitheringType'), width: 160, tooltip: this.model.loc('MonochromeDitheringTypeTooltip'),
                        margin: '2px ' + mrgn + ' 2px ' + mrgn, items: this.collections.getMonochromeDitheringTypeItems()
                    });
                    break;
                case 'TiffCompressionScheme':
                    components.push({
                        name, type: 'DropDownListForExportForm', label: this.model.loc('TiffCompressionScheme'), width: 160, tooltip: this.model.loc('TiffCompressionSchemeTooltip'),
                        margin: '2px ' + mrgn + ' 2px ' + mrgn, items: this.collections.getTiffCompressionSchemeItems()
                    });
                    break;
                case 'CutEdges':
                    components.push({
                        name, type: 'CheckBox', caption: this.model.loc('CutEdges'), tooltip: this.model.loc('CutEdgesTooltip'),
                        margin: '4px ' + mrgn + ' 4px ' + mrgn
                    });
                    break;
                case 'MultipleFiles':
                    components.push({
                        name, type: 'CheckBox', caption: this.model.loc('MultipleFiles'), tooltip: this.model.loc('MultipleFilesTooltip'),
                        margin: '4px ' + mrgn + ' 4px ' + mrgn
                    });
                    break;
                case 'ExportDataOnly':
                    components.push({
                        name, type: 'CheckBox', caption: this.model.loc('ExportDataOnly'), tooltip: this.model.loc('ExportDataOnlyTooltip'),
                        margin: '4px ' + mrgn + ' 4px ' + mrgn
                    });
                    break;
                case 'UseDefaultSystemEncoding':
                    components.push({
                        name, type: 'CheckBox', caption: this.model.loc('UseDefaultSystemEncoding'), tooltip: this.model.loc('UseDefaultSystemEncodingTooltip'),
                        margin: '4px ' + mrgn + ' 4px ' + mrgn
                    });
                    break;
                case 'EncodingDifFile':
                    components.push({
                        name, type: 'DropDownListForExportForm', label: this.model.loc('EncodingDifFile'), width: 160, tooltip: this.model.loc('EncodingDifFileTooltip'),
                        margin: '2px ' + mrgn + ' 2px ' + mrgn, items: this.collections.getEncodingDifFileItems()
                    });
                    break;
                case 'ExportModeRtf':
                    components.push({
                        name, type: 'DropDownListForExportForm', label: this.model.loc('ExportModeRtf'), width: 160, tooltip: this.model.loc('ExportModeRtfTooltip'),
                        margin: '2px ' + mrgn + ' 2px ' + mrgn, items: this.collections.getExportModeRtfItems()
                    });
                    break;
                case 'UsePageHeadersAndFooters':
                    components.push({
                        name, type: 'CheckBox', caption: this.model.loc('UsePageHeadersFooters'), tooltip: this.model.loc('UsePageHeadersFootersTooltip'),
                        margin: '4px ' + mrgn + ' 4px ' + mrgn
                    });
                    break;
                case 'RemoveEmptySpaceAtBottom':
                    components.push({
                        name, type: 'CheckBox', caption: this.model.loc('RemoveEmptySpace'), tooltip: this.model.loc('RemoveEmptySpaceTooltip'),
                        margin: '4px ' + mrgn + ' 4px ' + mrgn
                    });
                    break;
                case 'Separator':
                    components.push({
                        name, type: 'TextBox', label: this.model.loc('Separator'), tooltip: this.model.loc('SeparatorTooltip'), width: 160,
                        margin: '2px ' + mrgn + ' 2px ' + mrgn
                    });
                    break;
                case 'DataExportMode':
                    components.push({
                        name, type: 'DropDownListForExportForm', label: this.model.loc('BandsFilter'), width: 160, tooltip: this.model.loc('BandsFilterTooltip'),
                        margin: '2px ' + mrgn + ' 2px ' + mrgn, items: this.collections.getDataExportModeItems()
                    });
                    break;
                case 'SkipColumnHeaders':
                    components.push({
                        name, type: 'CheckBox', caption: this.model.loc('SkipColumnHeaders'), tooltip: this.model.loc('SkipColumnHeadersTooltip'),
                        margin: '4px ' + mrgn + ' 4px ' + mrgn
                    });
                    break;
                case 'ExportObjectFormatting':
                    components.push({
                        name, type: 'CheckBox', caption: this.model.loc('ExportObjectFormatting'), tooltip: this.model.loc('ExportObjectFormattingTooltip'),
                        margin: '4px ' + mrgn + ' 4px ' + mrgn
                    });
                    break;
                case 'UseOnePageHeaderAndFooter':
                    components.push({
                        name, type: 'CheckBox', caption: this.model.loc('UseOnePageHeaderFooter'), tooltip: this.model.loc('UseOnePageHeaderFooterTooltip'),
                        margin: '4px ' + mrgn + ' 4px ' + mrgn
                    });
                    break;
                case 'ExportEachPageToSheet':
                    components.push({
                        name, type: 'CheckBox', caption: this.model.loc('ExportEachPageToSheet'), tooltip: this.model.loc('ExportEachPageToSheetTooltip'),
                        margin: '4px ' + mrgn + ' 4px ' + mrgn
                    });
                    break;
                case 'ExportPageBreaks':
                    components.push({
                        name, type: 'CheckBox', caption: this.model.loc('ExportPageBreaks'), tooltip: this.model.loc('ExportPageBreaksTooltip'),
                        margin: '4px ' + mrgn + ' 4px ' + mrgn
                    });
                    break;
                case 'EncodingDbfFile':
                    components.push({
                        name, type: 'DropDownListForExportForm', label: this.model.loc('EncodingDbfFile'), width: 160, tooltip: this.model.loc('EncodingDbfFileTooltip'),
                        margin: '2px ' + mrgn + ' 2px ' + mrgn, items: this.collections.getEncodingDbfFileItems()
                    });
                    break;
                case 'DocumentSecurityButton':
                    components.push({
                        name, type: 'SmallButton', caption: this.model.loc('DocumentSecurityButton'), width: 160,
                        margin: '2px ' + mrgn + ' 2px ' + mrgn, items: this.collections.getEncodingDbfFileItems()
                    });
                    break;
                case 'UseDigitalSignature':
                    components.push({
                        name, caption: this.model.loc('DigitalSignatureButton'), type: 'DropDownListForExportForm', tooltip: this.model.loc('UseDigitalSignatureTooltip'), width: 160,
                        items: this.collections.getPdfSecurityCertificatesItems(), margin: '4px ' + mrgn + ' 4px ' + mrgn
                    });
                    break;
                default:
                    break;
            }
        });
        return components;
    }
    getGroups(format) {
        const openingGroupsStr = this.helper.getCookie('StimulsoftWebViewerExportSettingsOpeningGroups');
        const openingGroups = openingGroupsStr ? JSON.parse(openingGroupsStr) : null;
        const pageRangeAllIsDisabled = format.indexOf('Image') === 0 && format !== 'ImageTiff';
        return {
            savingReportGroup: { visible: format === 'Document', opened: openingGroups ? openingGroups.SavingReportGroup : true },
            pageRangeGroup: { visible: format !== 'Document', opened: openingGroups ? openingGroups.PageRangeGroup : true, pageRangeAllIsDisabled },
            settingsGroup: { visible: format !== 'Document', opened: openingGroups ? openingGroups.SettingsGroup : false }
        };
    }
    getDefaultSettings() {
        let settings;
        if (this.model.options.exports.storeExportSettings && this.helper.getCookie('StimulsoftWebViewerExportSettings' + this.getCommonExportFormat())) {
            const exportSettingsStr = this.helper.getCookie('StimulsoftWebViewerExportSettings' + this.getCommonExportFormat());
            const exportSettings = JSON.parse(exportSettingsStr);
            let exportFormat = exportSettings.ImageType || exportSettings.DataType || exportSettings.ExcelType || exportSettings.HtmlType;
            if (exportFormat === 'ExcelBinary') {
                exportFormat = 'Excel';
            }
            this.format = exportFormat ? (exportSettings.ImageType ? 'Image' + exportFormat : exportFormat) : this.format;
            const defSettings = this.getDefaultExportSettings(this.format);
            const resultSettings = {};
            Object.keys(defSettings).forEach(key => resultSettings[key] = exportSettings[key] || defSettings[key]);
            Object.keys(exportSettings).forEach(key => resultSettings[key] = exportSettings[key] || defSettings[key]);
            if (this.model.options.exports.showOpenAfterExport === false) {
                resultSettings.OpenAfterExport = true;
            }
            else if (!(this.format === 'Pdf' || this.format === 'Xps' || this.format === 'Html' || this.format === 'Excel' || this.format === 'ExcelBinary')) {
                resultSettings.OpenAfterExport = false;
            }
            if (this.model.options.exports.openAfterExport === true || this.model.options.exports.openAfterExport === false) {
                resultSettings.OpenAfterExport = this.model.options.exports.openAfterExport;
            }
            settings = resultSettings;
        }
        else {
            settings = this.getDefaultExportSettings(this.format);
        }
        this.updateTypes(settings);
        this.exportSettings = this.helper.copyObject(settings);
    }
    updateTypes(defaultExportSettings) {
        const types = ['ImageType', 'DataType', 'ExcelType', 'HtmlType'];
        types.forEach((propertyName) => {
            if (defaultExportSettings[propertyName]) {
                switch (propertyName) {
                    case 'ImageType':
                        if (!this.model.options.exports.showExportToImageBmp && defaultExportSettings[propertyName] === 'Bmp') {
                            defaultExportSettings[propertyName] = 'Gif';
                        }
                        if (!this.model.options.exports.showExportToImageGif && defaultExportSettings[propertyName] === 'Gif') {
                            defaultExportSettings[propertyName] = 'Jpeg';
                        }
                        if (!this.model.options.exports.showExportToImageJpeg && defaultExportSettings[propertyName] === 'Jpeg') {
                            defaultExportSettings[propertyName] = 'Pcx';
                        }
                        if (!this.model.options.exports.showExportToImagePcx && defaultExportSettings[propertyName] === 'Pcx') {
                            defaultExportSettings[propertyName] = 'Png';
                        }
                        if (!this.model.options.exports.showExportToImagePng && defaultExportSettings[propertyName] === 'Png') {
                            defaultExportSettings[propertyName] = 'Tiff';
                        }
                        if (!this.model.options.exports.showExportToImageTiff && defaultExportSettings[propertyName] === 'Tiff') {
                            defaultExportSettings[propertyName] = 'Emf';
                        }
                        if (!this.model.options.exports.showExportToImageMetafile && defaultExportSettings[propertyName] === 'Emf') {
                            defaultExportSettings[propertyName] = 'Svg';
                        }
                        if (!this.model.options.exports.showExportToImageSvg && defaultExportSettings[propertyName] === 'Svg') {
                            defaultExportSettings[propertyName] = 'Svgz';
                        }
                        if (!this.model.options.exports.showExportToImageSvgz && defaultExportSettings[propertyName] === 'Svgz') {
                            defaultExportSettings[propertyName] = 'Bmp';
                        }
                        break;
                    case 'DataType':
                        if (!this.model.options.exports.showExportToCsv && defaultExportSettings[propertyName] === 'Csv') {
                            defaultExportSettings[propertyName] = 'Dbf';
                        }
                        if (!this.model.options.exports.showExportToDbf && defaultExportSettings[propertyName] === 'Dbf') {
                            defaultExportSettings[propertyName] = 'Xml';
                        }
                        if (!this.model.options.exports.showExportToXml && defaultExportSettings[propertyName] === 'Xml') {
                            defaultExportSettings[propertyName] = 'Dif';
                        }
                        if (!this.model.options.exports.showExportToDif && defaultExportSettings[propertyName] === 'Dif') {
                            defaultExportSettings[propertyName] = 'Sylk';
                        }
                        if (!this.model.options.exports.showExportToSylk && defaultExportSettings[propertyName] === 'Sylk') {
                            defaultExportSettings[propertyName] = 'Csv';
                        }
                        if (!this.model.options.exports.showExportToJson && defaultExportSettings[propertyName] === 'Json') {
                            defaultExportSettings[propertyName] = 'Json';
                        }
                        break;
                    case 'ExcelType':
                        if (!this.model.options.exports.showExportToExcel2007 && defaultExportSettings[propertyName] === 'Excel2007') {
                            defaultExportSettings[propertyName] = 'ExcelBinary';
                        }
                        if (!this.model.options.exports.showExportToExcel && defaultExportSettings[propertyName] === 'ExcelBinary') {
                            defaultExportSettings[propertyName] = 'ExcelXml';
                        }
                        if (!this.model.options.exports.showExportToExcelXml && defaultExportSettings[propertyName] === 'ExcelXml') {
                            defaultExportSettings[propertyName] = 'Excel2007';
                        }
                        break;
                    case 'HtmlType':
                        if (!this.model.options.exports.showExportToHtml && defaultExportSettings[propertyName] === 'Html') {
                            defaultExportSettings[propertyName] = 'Html5';
                        }
                        if (!this.model.options.exports.showExportToHtml5 && defaultExportSettings[propertyName] === 'Html5') {
                            defaultExportSettings[propertyName] = 'Mht';
                        }
                        if (!this.model.options.exports.showExportToMht && defaultExportSettings[propertyName] === 'Mht') {
                            defaultExportSettings[propertyName] = 'Html';
                        }
                        break;
                }
            }
        });
    }
    getCommonExportFormat() {
        if (this.format === 'Html' || this.format === 'Html5' || this.format === 'Mht') {
            return 'Html';
        }
        if (this.format === 'Excel' || this.format === 'Excel2007' || this.format === 'ExcelXml') {
            return 'Excel';
        }
        if (this.format === 'Csv' || this.format === 'Dbf' || this.format === 'Xml' || this.format === 'Dif' || this.format === 'Sylk') {
            return 'Data';
        }
        if (this.format === 'ImageBmp' || this.format === 'ImageGif' || this.format === 'ImageJpeg' || this.format === 'ImagePcx' || this.format === 'ImagePng' ||
            this.format === 'ImageTiff' || this.format === 'ImageEmf' || this.format === 'ImageSvg' || this.format === 'ImageSvgz') {
            return 'Image';
        }
        return this.format;
    }
    getDefaultExportSettings(format, isDashboardExport = false) {
        let exportSettings = null;
        if (isDashboardExport) {
            return this.model.options.exports.defaultSettings['Dashboard' + format];
        }
        switch (format) {
            case 'Document':
                exportSettings = { Format: 'Mdc' };
                break;
            case 'Pdf':
                exportSettings = this.model.options.exports.defaultSettings['StiPdfExportSettings'];
                break;
            case 'Xps':
                exportSettings = this.model.options.exports.defaultSettings['StiXpsExportSettings'];
                break;
            case 'Ppt2007':
                exportSettings = this.model.options.exports.defaultSettings['StiPpt2007ExportSettings'];
                break;
            case 'Html':
                exportSettings = this.model.options.exports.defaultSettings['StiHtmlExportSettings'];
                exportSettings.HtmlType = 'Html';
                break;
            case 'Html5':
                exportSettings = this.model.options.exports.defaultSettings['StiHtmlExportSettings'];
                exportSettings.HtmlType = 'Html5';
                break;
            case 'Mht':
                exportSettings = this.model.options.exports.defaultSettings['StiHtmlExportSettings'];
                exportSettings.HtmlType = 'Mht';
                break;
            case 'Text':
                exportSettings = this.model.options.exports.defaultSettings['StiTxtExportSettings'];
                break;
            case 'Rtf':
                exportSettings = this.model.options.exports.defaultSettings['StiRtfExportSettings'];
                break;
            case 'Word2007':
                exportSettings = this.model.options.exports.defaultSettings['StiWord2007ExportSettings'];
                break;
            case 'Odt':
                exportSettings = this.model.options.exports.defaultSettings['StiOdtExportSettings'];
                break;
            case 'Excel':
                exportSettings = this.model.options.exports.defaultSettings['StiExcelExportSettings'];
                exportSettings.ExcelType = 'ExcelBinary';
                break;
            case 'ExcelXml':
                exportSettings = this.model.options.exports.defaultSettings['StiExcelExportSettings'];
                exportSettings.ExcelType = 'ExcelXml';
                break;
            case 'Excel2007':
                exportSettings = this.model.options.exports.defaultSettings['StiExcelExportSettings'];
                exportSettings.ExcelType = 'Excel2007';
                break;
            case 'Ods':
                exportSettings = this.model.options.exports.defaultSettings['StiOdsExportSettings'];
                break;
            case 'ImageBmp':
                exportSettings = this.model.options.exports.defaultSettings['StiImageExportSettings'];
                exportSettings.ImageType = 'Bmp';
                break;
            case 'ImageGif':
                exportSettings = this.model.options.exports.defaultSettings['StiImageExportSettings'];
                exportSettings.ImageType = 'Gif';
                break;
            case 'ImageJpeg':
                exportSettings = this.model.options.exports.defaultSettings['StiImageExportSettings'];
                exportSettings.ImageType = 'Jpeg';
                break;
            case 'ImagePcx':
                exportSettings = this.model.options.exports.defaultSettings['StiImageExportSettings'];
                exportSettings.ImageType = 'Pcx';
                break;
            case 'ImagePng':
                exportSettings = this.model.options.exports.defaultSettings['StiImageExportSettings'];
                exportSettings.ImageType = 'Png';
                break;
            case 'ImageTiff':
                exportSettings = this.model.options.exports.defaultSettings['StiImageExportSettings'];
                exportSettings.ImageType = 'Tiff';
                break;
            case 'ImageSvg':
                exportSettings = this.model.options.exports.defaultSettings['StiImageExportSettings'];
                exportSettings.ImageType = 'Svg';
                break;
            case 'ImageSvgz':
                exportSettings = this.model.options.exports.defaultSettings['StiImageExportSettings'];
                exportSettings.ImageType = 'Svgz';
                break;
            case 'ImageEmf':
                exportSettings = this.model.options.exports.defaultSettings['StiImageExportSettings'];
                exportSettings.ImageType = 'Emf';
                break;
            case 'Xml':
                exportSettings = this.model.options.exports.defaultSettings['StiDataExportSettings'];
                exportSettings.DataType = 'Xml';
                break;
            case 'Csv':
                exportSettings = this.model.options.exports.defaultSettings['StiDataExportSettings'];
                exportSettings.DataType = 'Csv';
                break;
            case 'Dbf':
                exportSettings = this.model.options.exports.defaultSettings['StiDataExportSettings'];
                exportSettings.DataType = 'Dbf';
                break;
            case 'Dif':
                exportSettings = this.model.options.exports.defaultSettings['StiDataExportSettings'];
                exportSettings.DataType = 'Dif';
                break;
            case 'Sylk':
                exportSettings = this.model.options.exports.defaultSettings['StiDataExportSettings'];
                exportSettings.DataType = 'Sylk';
                break;
        }
        exportSettings.OpenAfterExport = !(this.model.options.exports.showOpenAfterExport === false);
        if (!(format === 'Pdf' || format === 'Xps' || format === 'Html' || format === 'Excel' || format === 'ExcelBinary')) {
            exportSettings.OpenAfterExport = false;
        }
        return exportSettings;
    }
}
ExportService.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "14.3.0", ngImport: i0, type: ExportService, deps: [{ token: i1.FormService }, { token: i2.HelperService }, { token: i3.ModelService }, { token: i4.CollectionsService }, { token: i5.ControllerService }], target: i0.ɵɵFactoryTarget.Injectable });
ExportService.ɵprov = i0.ɵɵngDeclareInjectable({ minVersion: "12.0.0", version: "14.3.0", ngImport: i0, type: ExportService });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "14.3.0", ngImport: i0, type: ExportService, decorators: [{
            type: Injectable
        }], ctorParameters: function () { return [{ type: i1.FormService }, { type: i2.HelperService }, { type: i3.ModelService }, { type: i4.CollectionsService }, { type: i5.ControllerService }]; } });
//# sourceMappingURL=data:application/json;base64,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