import { Char } from "./Char";
export class TimeSpan {
    constructor(param1 = 0, minutes = null, seconds = null, milliseconds = null) {
        this._minus = false;
        if (minutes == null) {
            this.ticks = param1;
        }
        else {
            let hours1 = param1;
            if (milliseconds == null)
                milliseconds = 0;
            this.ticks = TimeSpan.timeToTicks(0, hours1, minutes, seconds, milliseconds);
        }
    }
    get zero() {
        return new TimeSpan();
    }
    static getNetTypeName() {
        return "System.TimeSpan";
    }
    static minValue() {
        if (this._minValue == null)
            this._minValue = new TimeSpan(0, 0, Number.MIN_VALUE);
        return this._minValue;
    }
    static maxValue() {
        if (this._maxValue == null)
            this._maxValue = new TimeSpan(0, 0, Number.MAX_VALUE);
        return this._maxValue;
    }
    get days() {
        return Math.floor(this.ticks / TimeSpan.ticksPerDay);
    }
    get hours() {
        return Math.floor(this.ticks / TimeSpan.ticksPerHour) % 24;
    }
    get milliseconds() {
        return Math.floor(this.ticks / TimeSpan.ticksPerMillisecond) % 1000;
    }
    get minutes() {
        return Math.floor(this.ticks / TimeSpan.ticksPerMinute) % 60;
    }
    get seconds() {
        return Math.floor(this.ticks / TimeSpan.ticksPerSecond) % 60;
    }
    get totalDays() {
        return this.ticks * TimeSpan.daysPerTick;
    }
    get totalHours() {
        return this.ticks * TimeSpan.hoursPerTick;
    }
    get totalMilliseconds() {
        let temp = this.ticks * TimeSpan.millisecondsPerTick;
        if (temp > TimeSpan.maxMilliSeconds)
            return TimeSpan.maxMilliSeconds;
        if (temp < TimeSpan.minMilliSeconds)
            return TimeSpan.minMilliSeconds;
        return temp;
    }
    get totalMinutes() {
        return this.ticks * TimeSpan.minutesPerTick;
    }
    get totalSeconds() {
        return this.ticks * TimeSpan.secondsPerTick;
    }
    static fromString(value, format = null) {
        if (value == null)
            return null;
        let isDefaultFormat = false;
        if (format == null || format.length == 0) {
            format = "hh:mm:ss";
            isDefaultFormat = true;
        }
        let days = 0;
        let hours = 0;
        let minutes = 0;
        let seconds = 0;
        let milliseconds = 0;
        let char;
        let pos = 0;
        let values = [];
        let stringValues = [];
        let minus = false;
        // Parse time into separate numeric values
        while (pos < value.length) {
            char = value.charAt(pos);
            if (Char.isDigit(char)) {
                stringValues.push(char);
                pos++;
                while (pos < value.length && Char.isDigit(char = value.charAt(pos))) {
                    stringValues[stringValues.length - 1] += value.charAt(pos);
                    pos++;
                }
                values.push(+(stringValues[stringValues.length - 1]));
            }
            else if (char == "-")
                minus = true;
            if (isDefaultFormat && char === ".") {
                if (values.length === 1)
                    format = "d.hh:mm:ss";
                else
                    format += ".f";
            }
            pos++;
        }
        pos = 0;
        let index = -1;
        // Parsing format and replacement of appropriate values of time
        while (pos < format.length && index + 1 < values.length) {
            char = format.charAt(pos);
            if (char === "d" || char === "h" || char === "H" || char === "m" || char === "s" || char === "f" || char === "F") {
                index++;
                while (pos < format.length && format.charAt(pos) === char)
                    pos++;
            }
            switch (char) {
                case "d": // days
                    days = values[index];
                    break;
                case "h":
                case "H": // hours
                    hours = values[index];
                    break;
                case "m": // minutes
                    minutes = values[index];
                    break;
                case "s": // seconds
                    seconds = values[index];
                    break;
                case "f":
                case "F": // seconds fraction
                    milliseconds = values[index];
                    break;
                default:
                    pos++;
                    break;
            }
        }
        let timeSpan = TimeSpan.create(days, hours, minutes, seconds, milliseconds);
        timeSpan._minus = minus;
        return timeSpan;
    }
    static interval(value, scale) {
        let tmp = value * scale;
        let millis = tmp + (value >= 0 ? 0.5 : -0.5);
        return new TimeSpan(millis * TimeSpan.ticksPerMillisecond);
    }
    /* static fromTicks(value: number): TimeSpan {
       return new TimeSpan(value);
     }
  
     static fromSeconds(value: number): TimeSpan {
       return this.interval(value, this.millisPerSecond);
     }
  
     static fromMilliseconds(value: number): TimeSpan {
       return this.interval(value, 1);
     }
  
     add(value: number): TimeSpan {
       return new TimeSpan(this.ticks + value);
     }
  
     add2(value: TimeSpan): TimeSpan {
       return new TimeSpan(this.ticks + value.ticks);
     }*/
    toString(format) {
        if (format == null || format.length == 0) {
            format = "hh:mm:ss";
            if (this.days != 0) {
                format = "d.hh:mm:ss";
            }
        }
        /*switch (format) {
          case "t":
            format = CultureInfo.currentCulture.dateTimeFormat.shortTimePattern;
            break;
          case "T":
            format = CultureInfo.currentCulture.dateTimeFormat.longTimePattern;
            break;
        }
    
        format = format.replace(/H/gm, "h");
    */
        let result = "";
        let pos = 0;
        let isDays = false;
        /*  while (pos < format.length) {
            let f = format.charAt(pos);
    
            if (!Char.isLetter(f)) {
              result += f;
              pos++;
              continue;
            }
    
            while (pos + 1 < format.length && Char.isLetter(format.charAt(pos + 1))) {
              pos++;
              f += format.charAt(pos);
            }
    
            switch (f[0]) {
              case "d":
                isDays = true;
                result += Math.abs(this.days).toString().ss.padLeft(f.length, "0");
                break;
              case "h":
                result += Math.abs(isDays ? this.hours : StiMath.trunc(StiMath.round2(this.totalHours, 2))).toString().ss.padLeft(f.length, "0");
                break;
              case "m":
                result += Math.abs(this.minutes).toString().ss.padLeft(f.length, "0");
                break;
              case "s":
                result += Math.abs(this.seconds).toString().ss.padLeft(f.length, "0");
                break;
              case "f":
                result += Math.abs(this.milliseconds).toString().ss.padLeft(f.length, "0");
                break
              case "F":
                if (this.milliseconds != 0)
                  result += Math.abs(this.milliseconds).toString().ss.padLeft(f.length, "0");
                break;
            }
            pos++;
          }
    
          if (this._minus) return "-" + result;*/
        return result;
    }
    negate() {
        if (this.ticks == TimeSpan.minValue().ticks)
            throw new Error("Overflow_NegateTwosCompNum");
        return new TimeSpan(-this.ticks);
    }
    static timeToTicks(day, hour, minute, second, milliseconds) {
        let totalSeconds = day * 86400 + hour * 3600 + minute * 60 + second;
        return totalSeconds * TimeSpan.ticksPerSecond;
    }
    static create(days = 0, hours = 0, minutes = 0, seconds = 0, milliseconds = 0) {
        return null; // new TimeSpan(TimeSpan.timeToTicks(days, hours, minutes, seconds, milliseconds));
    }
}
TimeSpan.ticksPerMillisecond = 1;
TimeSpan.millisecondsPerTick = 1 / TimeSpan.ticksPerMillisecond;
TimeSpan.ticksPerSecond = TimeSpan.ticksPerMillisecond * 1000;
TimeSpan.secondsPerTick = 1 / TimeSpan.ticksPerSecond;
TimeSpan.ticksPerMinute = TimeSpan.ticksPerSecond * 60;
TimeSpan.minutesPerTick = 1 / TimeSpan.ticksPerMinute;
TimeSpan.ticksPerHour = TimeSpan.ticksPerMinute * 60;
TimeSpan.hoursPerTick = 1 / TimeSpan.ticksPerHour;
TimeSpan.ticksPerDay = TimeSpan.ticksPerHour * 24;
TimeSpan.daysPerTick = 1 / TimeSpan.ticksPerDay;
TimeSpan.millisPerSecond = 1000;
TimeSpan.millisPerMinute = TimeSpan.millisPerSecond * 60;
TimeSpan.millisPerHour = TimeSpan.millisPerMinute * 60;
TimeSpan.millisPerDay = TimeSpan.millisPerHour * 24;
TimeSpan.maxSeconds = Number.MAX_VALUE / TimeSpan.ticksPerSecond;
TimeSpan.minSeconds = Number.MIN_VALUE / TimeSpan.ticksPerSecond;
TimeSpan.maxMilliSeconds = Number.MAX_VALUE / TimeSpan.ticksPerMillisecond;
TimeSpan.minMilliSeconds = Number.MIN_VALUE / TimeSpan.ticksPerMillisecond;
TimeSpan.ticksPerTenthSecond = TimeSpan.ticksPerMillisecond * 100;
TimeSpan._minValue = null;
TimeSpan._maxValue = null;
//# sourceMappingURL=data:application/json;base64,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