import { DateTime } from "./DateTime";
import { Enum } from "./Enum";
import List from "./List";
import { StiString } from "./StiObject";
export default class StiJson {
    constructor(name = null, value = null, isProperty = false, options = {}) {
        this.isProperty = false;
        this.isArray = false;
        this.name = name;
        if (value != null && typeof value == "string")
            value = value.replace(/#!rn#/gi, "\r\n").replace(/#!q#/gi, '"').replace(/#!t#/gi, "\t").replace(/#!!#/gi, "\\");
        this.value = value;
        this.isProperty = isProperty;
        this.options = options;
    }
    static dateToJsonDate(date) {
        let offset = date.date.getTimezoneOffset() * -1;
        let hoursOffset = Math.abs(Math.floor(offset / 60)).toString();
        let minutesOffset = Math.abs(offset % 60).toString();
        if (hoursOffset.length == 1)
            hoursOffset = "0" + hoursOffset;
        if (minutesOffset.length == 1)
            minutesOffset = "0" + minutesOffset;
        return (
        /* "/Date(" +*/ Date.parse(date.date.toString()).toString() +
            (offset >= 0 ? "+" : "-") +
            hoursOffset +
            minutesOffset /*+ ")/"*/);
    }
    static jsonDateFormatToDate(jsonDate) {
        let dateStr = jsonDate.split("/Date(").join("").split(")/").join("");
        let date = new DateTime(new Date(parseInt(dateStr)));
        let startOffsetIndex = dateStr.indexOf("+");
        if (startOffsetIndex == -1)
            startOffsetIndex = dateStr.indexOf("-");
        let localOffset = date.date.getTimezoneOffset();
        if (startOffsetIndex != -1)
            localOffset = 0;
        let localHoursOffset = Math.floor(localOffset / 60);
        let localMinutesOffset = localOffset % 60;
        let localHoursOffsetString = (Math.abs(localHoursOffset) < 10 ? "0" : "") + Math.abs(localHoursOffset).toString();
        let localMinutesOffsetString = (Math.abs(localMinutesOffset) < 10 ? "0" : "") + Math.abs(localMinutesOffset).toString();
        let offset = (localOffset >= 0 ? "+" : "-") + localHoursOffsetString + localMinutesOffsetString;
        let multiple = 1;
        if (startOffsetIndex != -1)
            offset = dateStr.substring(startOffsetIndex);
        if (offset.substr(0, 1) == "-")
            multiple = -1;
        date.addHours(localHoursOffset + parseInt(offset.substr(1, 2)) * multiple);
        date.addMinutes(localMinutesOffset + parseInt(offset.substr(3, 2)) * multiple);
        return date;
    }
    properties() {
        if (!this.isProperty) {
            if (this.value instanceof List)
                return this.value;
            else
                return this.value.value;
        }
        return new List();
    }
    removeProperty(propertyName) {
        if (propertyName != null || propertyName != "") {
            let elements = this.value.value.filter((x) => x.name == propertyName);
            if (elements.length > 0) {
                this.value.value.remove(elements[0]);
            }
        }
    }
    addPropertyNumber(propertyName, value) {
        if (this.value == null)
            this.value = new StiJson("", new List());
        this.removeProperty(propertyName);
        this.value.value.push(new StiJson(propertyName, value, true));
    }
    addPropertyNumberNotNull(propertyName, value) {
        if (value) {
            this.addPropertyNumber(propertyName, value);
        }
    }
    addPropertyNumberNoDefaultValue(propertyName, value) {
        if (this.value == null)
            this.value = new StiJson("", new List());
        this.removeProperty(propertyName);
        this.value.value.push(new StiJson(propertyName, value, true));
    }
    addPropertyJObject(propertyName, value) {
        if (this.value == null)
            this.value = new StiJson("", new List());
        this.removeProperty(propertyName);
        if (value == null || value.count == 0)
            return;
        this.value.value.push(new StiJson(propertyName, value));
    }
    addPropertyArray(propertyName, elements) {
        if (this.value == null)
            this.value = new StiJson("", new List());
        this.removeProperty(propertyName);
        if (elements == null)
            return;
        let values = [];
        elements.forEach((e) => values.push(e?.saveToJsonObject != null
            ? e.saveToJsonObject(this.options)
            : typeof e == "string"
                ? `"${e
                    .replace(/[\\]/g, "\\\\")
                    .replace(/[\"]/g, '\\"')
                    // .replace(/[\/]/g, '\\/')
                    .replace(/[\b]/g, "\\b")
                    .replace(/[\f]/g, "\\f")
                    .replace(/\r\n/g, "\n")
                    .replace(/[\r]/g, "\n")
                    .replace(/[\r\n]/g, "\\r\\n")
                    .replace(/[\t]/g, "\\t")
                    // eslint-disable-next-line no-control-regex
                    .replace(/[\u0001-\u001F]/g, (chr) => "\\u" + ("0000" + chr.charCodeAt(0).toString(16)).substr(-4))}"`
                : e));
        let json = new StiJson(propertyName, values);
        json.isArray = true;
        this.value.value.push(json);
    }
    addPropertyIdent(propertyName, value) {
        if (this.value == null)
            this.value = new StiJson("", new List());
        this.value.value.push(new StiJson(propertyName, value, true));
    }
    addPropertyBool(propertyName, value) {
        if (this.value == null)
            this.value = new StiJson("", new List());
        this.removeProperty(propertyName);
        this.value.value.push(new StiJson(propertyName, value, true));
    }
    /*addPropertyDateTime(propertyName: string, value: DateTime) {
        if (this.value == null) this.value = new StiJson("", new List());
        this.removeProperty(propertyName);
  
        this.value.value.push(new StiJson(propertyName, StiJson.dateToJsonDate(value), true));
    }*/
    addPropertyEnum(propertyName, enumType, value, defaultValue = null) {
        if (this.value == null)
            this.value = new StiJson("", new List());
        this.removeProperty(propertyName);
        if (typeof defaultValue != "undefined" && value == defaultValue)
            return;
        this.value.value.push(new StiJson(propertyName, Enum.getName(enumType, value), true));
    }
    addPropertyColor(propertyName, color, defColor = null) {
        if (color == null || color.equals(defColor))
            return;
        let value = color.isNamedColor ? color.name.replace(" ", "") : `${color.a},${color.r},${color.g},${color.b}`;
        this.addPropertyString(propertyName, value);
    }
    addPropertyString(propertyName, value, defaultValue) {
        if (this.value == null)
            this.value = new StiJson("", new List());
        this.removeProperty(propertyName);
        if (value == defaultValue)
            return;
        this.value.value.push(new StiJson(propertyName, value, true));
    }
    addPropertyStringNullOrEmpty(propertyName, value) {
        if (this.value == null)
            this.value = new StiJson("", new List());
        this.removeProperty(propertyName);
        if (StiString.isNullOrEmpty(value))
            return;
        this.value.value.push(new StiJson(propertyName, value, true));
    }
    get count() {
        let count = 0;
        if (!this.isProperty && this.value != null) {
            // eslint-disable-next-line @typescript-eslint/no-unused-vars
            for (let propertyName of this.value.value) {
                count++;
            }
        }
        return count;
    }
    getArray() {
        let result = [];
        this.value.forEach((e) => result.push(e.value));
        return result;
    }
    serialize(indent = 0) {
        let jsonString = "";
        let tabs = StiJson.prettyPrint ? Array(indent + 1).join(" ") : "";
        for (let property of this.value.properties()) {
            let valueString = "";
            if (property.isProperty) {
                if (property.value != null && typeof property.value == "string")
                    valueString =
                        '"' +
                            property.value
                                .replace(/[\\]/g, "\\\\")
                                .replace(/[\"]/g, '\\"')
                                // .replace(/[\/]/g, '\\/')
                                .replace(/[\b]/g, "\\b")
                                .replace(/[\f]/g, "\\f")
                                .replace(/\r\n/g, "\n")
                                .replace(/[\r]/g, "\n")
                                .replace(/[\r\n]/g, "\\r\\n")
                                .replace(/[\t]/g, "\\t")
                                // eslint-disable-next-line no-control-regex
                                .replace(/[\u0001-\u001F]/g, (chr) => "\\u" + ("0000" + chr.charCodeAt(0).toString(16)).substr(-4)) +
                            '"';
                // "\"" + property.value.replace(/\"/gi, "\\\"").replace(/\r\n/gi, "\n").replace(/\r/gi, "\n").replace(/\n/gi, "\\r\\n").replace(/\\/gi, "\\\\") + "\"";
                else
                    valueString = property.value;
            }
            else if (property.isArray) {
                if (property.value.length == 0) {
                    valueString += "[";
                }
                for (let element of property.value) {
                    if (valueString)
                        valueString += ",";
                    else
                        valueString += "[";
                    valueString += element?.serialize != null ? element.serialize(indent + 4) : element;
                }
                valueString += "]";
            }
            else {
                valueString = property.value.serialize(indent + 2);
            }
            if (jsonString)
                jsonString += ",";
            if (StiJson.prettyPrint)
                jsonString += "\n" + tabs + "  ";
            jsonString += '"' + property.name + '":';
            if (StiJson.prettyPrint)
                jsonString += " ";
            jsonString += valueString;
        }
        if (!jsonString)
            return "";
        if (StiJson.prettyPrint)
            return "{" + jsonString + "\n" + tabs + "}";
        return "{" + jsonString + "}";
    }
    deserialize(text) {
        if (typeof text == "string") {
            let pos = 0;
            let start = -1;
            while (pos < text.length) {
                if (text[pos] == '"' &&
                    (pos > 0 && text[pos - 1] != "\\"
                        ? true
                        : pos > 1 && text[pos - 2] == "\\" && pos > 2 && text[pos - 3] != "\\")) {
                    if (start > -1) {
                        let subText = text.substring(start, pos);
                        let modText = subText
                            .replace(/\\\\/gi, "#!!#")
                            .replace(/\\t/gi, "#!t#")
                            .replace(/\\\"/gi, "#!q#")
                            .replace(/\\r\\n/gi, "#!rn#")
                            .replace(/\\r/gi, "#!rn#")
                            .replace(/\\n/gi, "#!rn#");
                        if (modText.length != subText.length) {
                            // eslint-disable-next-line @typescript-eslint/restrict-plus-operands
                            text = text.substring(0, start) + modText + text.substring(pos);
                            pos += modText.length - subText.length;
                        }
                        start = -1;
                    }
                    else
                        start = pos + 1;
                }
                pos++;
            }
        }
        let reportObject = typeof text == "object" ? text : JSON.parse(text);
        this.deserializeFromObject(reportObject);
        this.value = this.value.value;
    }
    deserializeFromObject(child) {
        for (let propertyName in child) {
            if (typeof child[propertyName] == "object") {
                let jObject = new StiJson(propertyName);
                if (this.value == null)
                    this.value = new StiJson(null, new List());
                this.value.value.push(jObject);
                jObject.deserializeFromObject(child[propertyName]);
            }
            else if (typeof child[propertyName] != "function") {
                let jObject = new StiJson(propertyName, child[propertyName], true);
                if (this.value == null)
                    this.value = new StiJson(null, new List());
                this.value.value.push(jObject);
            }
        }
        if (child && Array.isArray(child) && child.length == 0) {
            this.value = new StiJson(null, new List());
        }
    }
    toString() {
        return this.name;
    }
}
StiJson.prettyPrint = true;
//# sourceMappingURL=data:application/json;base64,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