import StiJson from "./StiJson";
export class Font {
    constructor(family, size, style = [], changesInterceptor) {
        this.changesInterceptor = changesInterceptor;
        this.fontSizeScale = 0.75;
        this.isChanged = true;
        this._family = family;
        this._size = size;
        this._style = style;
    }
    setProperty(propName, value) {
        switch (propName) {
            case "family":
                this.family = value;
                break;
            case "style":
                this.style = value;
                break;
            case "size":
                this.size = value;
                break;
        }
    }
    set style(value) {
        let this_ = this;
        let font = this.changesInterceptor ? this_.clone() : null;
        this._style = new Proxy(value, {
            deleteProperty: function (target, property) {
                let font = this.changesInterceptor ? this_.clone() : null;
                setTimeout(() => {
                    if (this.changesInterceptor) {
                        this.changesInterceptor.propertyChanged("font.style", this_.cloneStyle, font);
                    }
                    this_.isChanged = true;
                }, 0);
                return true;
            },
            set: function (target, property, value, receiver) {
                let font = this.changesInterceptor ? this_.clone() : null;
                target[property] = value;
                if (this.changesInterceptor) {
                    this.changesInterceptor.propertyChanged("font.style", this_.cloneStyle, font);
                }
                this_.isChanged = true;
                return true;
            },
        });
        if (this.changesInterceptor) {
            this.changesInterceptor.propertyChanged("font.style", this_.cloneStyle, font);
        }
        this_.isChanged = true;
    }
    get cloneStyle() {
        return this.style.map((s) => s);
    }
    get style() {
        return this._style;
    }
    resetChanged() {
        this.isChanged = false;
    }
    get family() {
        return this._family;
    }
    set family(value) {
        let font = this.changesInterceptor ? this.clone() : null;
        let oldValue = this._family;
        this._family = value;
        this.isChanged = true;
        if (this.changesInterceptor && oldValue != this._family) {
            this.changesInterceptor.propertyChanged("font.family", value, font);
        }
    }
    get size() {
        return this._size;
    }
    set size(value) {
        let font = this.changesInterceptor ? this.clone() : null;
        let oldValue = this._size;
        this._size = value;
        this.isChanged = true;
        if (this.changesInterceptor && oldValue != this._size) {
            this.changesInterceptor.propertyChanged("font.size", value, font);
        }
    }
    saveToJsonObject(options) {
        let json = new StiJson();
        json.addPropertyStringNullOrEmpty("Family", this.family);
        json.addPropertyNumber("Size", this.size);
        let style = this.style.filter((s) => s && s != "").join(",");
        json.addPropertyString("Style", style.length != 0 ? style : "Regular");
        return json;
    }
    loadFromJsonObject(json) {
        for (let property of json.properties()) {
            switch (property.name) {
                case "Family":
                    this._family = property.value;
                    break;
                case "Size":
                    this._size = property.value;
                    break;
                case "Style":
                    this._style = property.value.replace("Regular", "").split(",");
                    break;
            }
        }
    }
    isBold() {
        return this.style.indexOf("Bold") >= 0;
    }
    isItalic() {
        return this.style.indexOf("Italic") >= 0;
    }
    isUnderline() {
        return this.style.indexOf("Underline") >= 0;
    }
    htmlStyle() {
        return `font-weight:${this.isBold() ? "bold" : "normal"};font-style:${this.isItalic() ? "italic" : "normal"}; text-decoration:${this.isUnderline() ? "underline" : ""};font-family:"${this.family}";font-size:${this.size}pt`;
    }
    equals(font) {
        return this.htmlStyle() == font?.htmlStyle();
    }
    clone() {
        return new Font(this._family, this.size, this.cloneStyle);
    }
}
//# sourceMappingURL=data:application/json;base64,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