import StiPadding from "../../properties/StiPadding";
import { Enum } from "../../system/Enum";
import { StiDisplayCurrencyAs, StiThousandSeparator, StiUnitAlignment } from "../../system/enums";
import StiJson from "../../system/StiJson";
import { StiDescriptionSettings } from "./StiDescriptionSettings";
import { StiInnerTextSettings } from "./StiInnerTextSettings";
import { StiLabelSettings } from "./StiLabelSettings";
export class StiFormSettings {
    constructor(form) {
        this.form = form;
        this._label = new StiLabelSettings(this);
        this._description = new StiDescriptionSettings(this);
        this._innerText = new StiInnerTextSettings(this);
        this._dateTimeFormat = "m/d/yy h:MM tt";
        this._dateFomat = "dd.mm.yyyy";
        this._timeFormat = "HH:MM:ss";
        this._currency = "USD";
        this._currencyFormat = StiDisplayCurrencyAs.Code;
        this._unitAlignment = StiUnitAlignment.Prefix;
        this._thousandSeparator = StiThousandSeparator.Comma;
        this._decimalDigits = 2;
        this._padding = new StiPadding(8, 8, 8, 8, this);
        this._gridColumns = 12;
        this._gridVerticalStep = 5;
    }
    get reportImage() {
        return this.form.reportImage;
    }
    set reportImage(value) {
        let oldValue = this.reportImage;
        this.form.reportImage = value;
        if (oldValue != value) {
            this.propertyChanged("reportImage", value, {});
        }
    }
    get formName() {
        return this.form.formName;
    }
    set formName(value) {
        let oldValue = this.form.formName;
        this.form.formName = value;
        if (oldValue != value) {
            this.propertyChanged("formName", value, {});
        }
    }
    get gridColumns() {
        return this._gridColumns;
    }
    set gridColumns(value) {
        let oldValue = this._gridColumns;
        if (typeof value != "number" ? Number(value) < 12 : value < 12) {
            this._gridColumns = 12;
        }
        else {
            this._gridColumns = value;
        }
        if (oldValue != value) {
            this.propertyChanged("gridColumns", value, {});
        }
    }
    get gridVerticalStep() {
        return this._gridVerticalStep;
    }
    set gridVerticalStep(value) {
        let oldValue = this._gridVerticalStep;
        if (value <= 1) {
            this._gridVerticalStep = 1;
        }
        else if (value > this.form.pages[this.form.pages.length - 1].contentAreaHeight) {
            this._gridVerticalStep = this.form.pages[this.form.pages.length - 1].contentAreaHeight;
        }
        else {
            this._gridVerticalStep = value;
        }
        if (oldValue != value) {
            this.propertyChanged("gridVerticalStep", value, {});
        }
    }
    get padding() {
        return this._padding;
    }
    set padding(value) {
        let oldObject = this._padding.clone();
        this._padding = value;
        this._padding.changesInterceptor = this;
        this.propertyChanged("padding", value, oldObject);
    }
    get label() {
        return this._label;
    }
    set label(value) {
        this._label = value;
    }
    get description() {
        return this._description;
    }
    set description(value) {
        this._description = value;
    }
    get innerText() {
        return this._innerText;
    }
    set innerText(value) {
        this._innerText = value;
    }
    get dateTimeFormat() {
        return this._dateTimeFormat;
    }
    set dateTimeFormat(value) {
        let oldValue = this._dateTimeFormat;
        this._dateTimeFormat = value;
        if (oldValue != value) {
            this.propertyChanged("dateTimeFormat", value, oldValue);
        }
    }
    get dateFormat() {
        return this._dateFomat;
    }
    set dateFormat(value) {
        let oldValue = this._dateFomat;
        this._dateFomat = value;
        if (oldValue != value) {
            this.propertyChanged("dateFormat", value, oldValue);
        }
    }
    get timeFormat() {
        return this._timeFormat;
    }
    set timeFormat(value) {
        let oldValue = this._timeFormat;
        this._timeFormat = value;
        if (oldValue != value) {
            this.propertyChanged("timeFormat", value, oldValue);
        }
    }
    get currency() {
        return this._currency;
    }
    set currency(value) {
        let oldValue = this._currency;
        this._currency = value;
        if (oldValue != value) {
            this.propertyChanged("currency", value, oldValue);
        }
    }
    get currencyFormat() {
        return this._currencyFormat;
    }
    set currencyFormat(value) {
        let oldValue = this._currencyFormat;
        this._currencyFormat = value;
        if (oldValue != value) {
            this.propertyChanged("currencyFormat", value, oldValue);
        }
    }
    get unitAlignment() {
        return this._unitAlignment;
    }
    set unitAlignment(value) {
        let oldValue = this._unitAlignment;
        this._unitAlignment = value;
        if (oldValue != value) {
            this.propertyChanged("unitAlignment", value, oldValue);
        }
    }
    get thousandSeparator() {
        return this._thousandSeparator;
    }
    set thousandSeparator(value) {
        let oldValue = this._thousandSeparator;
        this._thousandSeparator = value;
        if (oldValue != value) {
            this.propertyChanged("thousandSeparator", value, oldValue);
        }
    }
    get decimalDigits() {
        return this._decimalDigits;
    }
    set decimalDigits(value) {
        let oldValue = this._decimalDigits;
        if (value <= 0) {
            this._decimalDigits = 0;
        }
        else {
            this._decimalDigits = value;
        }
        if (oldValue != value) {
            this.propertyChanged("decimalDigits", value, oldValue);
        }
    }
    propertyChanged(propName, value, oldObject) {
        this.form.pages.forEach((p) => p.allElements.forEach((el) => {
            el.setProperty(propName, value["clone"] ? value.clone() : value, oldObject);
        }));
        if (propName.toLowerCase().indexOf("color") == -1) {
            this.form.formEvents.next({ name: "UpdateAllSizes" });
        }
        if (propName == "thousandSeparator" || propName == "decimalDigits") {
            this.form.formEvents.next({ name: "UpdateForm" });
        }
    }
    saveToJsonObject(options) {
        let json = new StiJson();
        json.addPropertyJObject("Label", this.label.saveToJsonObject(options));
        json.addPropertyJObject("Description", this.description.saveToJsonObject(options));
        json.addPropertyJObject("InnerText", this.innerText.saveToJsonObject(options));
        json.addPropertyString("TimeFormat", this.timeFormat);
        json.addPropertyString("DateFormat", this.dateFormat);
        json.addPropertyString("DateTimeFormat", this.dateTimeFormat);
        json.addPropertyNumber("DecimalDigits", this.decimalDigits);
        json.addPropertyNumber("GridColumns", this.gridColumns);
        json.addPropertyNumber("GridVerticalStep", this.gridVerticalStep);
        json.addPropertyString("Currency", this.currency);
        json.addPropertyEnum("CurrencyFormat", StiDisplayCurrencyAs, this.currencyFormat);
        json.addPropertyEnum("UnitAlignment", StiUnitAlignment, this.unitAlignment);
        json.addPropertyEnum("ThousandSeparator", StiThousandSeparator, this.thousandSeparator);
        json.addPropertyJObject("Padding", this.padding.saveToJsonObject(options));
        return json;
    }
    loadFormJsonString(text) {
        let jObject = new StiJson();
        jObject.deserialize(text);
        this.loadFromJsonObject(jObject);
    }
    loadFromJsonObject(json) {
        for (let property of json.properties()) {
            switch (property.name) {
                case "Label":
                    this._label = new StiLabelSettings(this);
                    this._label.loadFromJsonObject(property.value);
                    break;
                case "Description":
                    this._description = new StiDescriptionSettings(this);
                    this._description.loadFromJsonObject(property.value);
                    break;
                case "InnerText":
                    this._innerText = new StiInnerTextSettings(this);
                    this._innerText.loadFromJsonObject(property.value);
                    break;
                case "TimeFormat":
                    this.timeFormat = property.value;
                    break;
                case "DateFormat":
                    this.dateFormat = property.value;
                    break;
                case "DateTimeFormat":
                    this.dateTimeFormat = property.value;
                    break;
                case "DecimalDigits":
                    this.decimalDigits = property.value;
                    break;
                case "GridColumns":
                    this.gridColumns = property.value < 12 ? 12 : property.value;
                    break;
                case "GridVerticalStep":
                    this.gridVerticalStep = property.value;
                    break;
                case "Currency":
                    this._currency = property.value;
                    break;
                case "CurrencyFormat":
                    this._currencyFormat = Enum.parse(StiDisplayCurrencyAs, property.value);
                    break;
                case "UnitAlignment":
                    this._unitAlignment = Enum.parse(StiUnitAlignment, property.value);
                    break;
                case "Separator":
                case "ThousandSeparator":
                    this._thousandSeparator = Enum.parse(StiThousandSeparator, property.value);
                    break;
                case "Padding":
                    this._padding = new StiPadding();
                    this._padding.loadFromJsonObject(property.value);
                    break;
            }
        }
    }
    applySettings(element) {
        element.setProperty("label.font.family", this.label.font.family, null);
        element.setProperty("label.font.style", this.label.font.cloneStyle, null);
        element.setProperty("label.font.size", this.label.font.size, null);
        element.setProperty("label.color", this.label.color, null);
        element.setProperty("label.alignment", this.label.alignment, null);
        element.setProperty("label.width", this.label.width, null);
        element.setProperty("padding", this.padding, null);
        element.setProperty("description.font.family", this.description.font.family, null);
        element.setProperty("description.font.style", this.description.font.cloneStyle, null);
        element.setProperty("description.font.size", this.description.font.size, null);
        element.setProperty("description.color", this.description.color.clone(), null);
        element.setProperty("innerText.font.family", this.innerText.font.family, null);
        element.setProperty("innerText.font.style", this.innerText.font.cloneStyle, null);
        element.setProperty("innerText.font.size", this.innerText.font.size, null);
        element.setProperty("innerText.color", this.innerText.color.clone(), null);
        element.setProperty("innerText.backgroundColor", this.innerText.backgroundColor.clone(), null);
        element.setProperty("innerText.borderColor", this.innerText.borderColor.clone(), null);
        element.setProperty("innerText.borderWeight", this.innerText.borderWeight, null);
        element.setProperty("innerText.checkColor", this.innerText.checkColor.clone(), null);
        element.setProperty("innerText.contentAlignment", this.innerText.contentAlignment, null);
        element.setProperty("timeFormat", this.timeFormat, null);
        element.setProperty("dateFormat", this.dateFormat, null);
        element.setProperty("dateTimeFormat", this.dateTimeFormat, null);
        element.setProperty("currency", this.currency, null);
        element.setProperty("currencyFormat", this.currencyFormat, null);
        element.setProperty("unitAlignment", this.unitAlignment, null);
        element.setProperty("thousandSeparator", this.thousandSeparator, null);
        element.setProperty("decimalDigits", this.decimalDigits, null);
    }
}
//# sourceMappingURL=data:application/json;base64,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