import StiPadding from "../../properties/StiPadding";
import StiRectangleGeometry from "../../properties/StiRectangleGeometry";
import StiFormObject from "./StiFormObject";
import { StiLabelItem } from "../text/StiLabelItem";
import { StiLabelAlignment, StiVerticalAlignment } from "../../system/enums";
export class StiFormElement extends StiFormObject {
    constructor() {
        super(...arguments);
        this._visible = true;
        this._geometry = new StiRectangleGeometry(0, 0, this);
        this._padding = new StiPadding(8, 8, 8, 8, this);
        this._label = new StiLabelItem(this, "");
        this._readOnly = false;
        this._isChanged = true;
        this.render = true;
        this.position = "absolute";
        // public label: StiLabelItem = new StiLabelItem(this, "");
        this.hasGeometry = false;
        this.isAdded = false;
    }
    get maxWidth() {
        if (this.label.alignment == StiLabelAlignment.Top || this.label.eValue.length == 0) {
            return this.geometry.width - this.padding.left - this.padding.right;
        }
        else {
            return this.geometry.width - this.label.width - this.padding.left - this.padding.right;
        }
    }
    setDefaultValues() { }
    isChanged() {
        return this._isChanged || this._geometry.isChanged() || this._padding.isChanged();
    }
    getDefaultEditableContent() {
        return null;
    }
    canUpdateProperty(propName) {
        return true;
    }
    isLabelTopAlignment() {
        return this.label.alignment == StiLabelAlignment.Top ? true : false;
    }
    getVerticalAlignment() {
        if (this.label.verticalAlignment == StiVerticalAlignment.Top) {
            return "start";
        }
        else if (this.label.verticalAlignment == StiVerticalAlignment.Center) {
            return "center";
        }
        else {
            return "flex-end";
        }
    }
    resetChanged() {
        this._isChanged = false;
        this._geometry.resetChanged();
        this._padding.resetChanged();
    }
    get visible() {
        return this._visible;
    }
    set visible(value) {
        this._visible = value;
        this._isChanged = true;
    }
    get label() {
        return this._label;
    }
    set label(value) {
        let oldValue = this._label;
        if (oldValue != value) {
            this._label = value;
            this._isChanged = true;
        }
    }
    get geometry() {
        return this._geometry;
    }
    set geometry(value) {
        if (!this.geometry || !value || !this.geometry.equals(value)) {
            this._isChanged = true;
        }
        else {
            value.resetChanged();
        }
        this._geometry = value;
        this._geometry.changesInterceptor = this;
    }
    get padding() {
        return this._padding;
    }
    set padding(value) {
        let oldValue = this._padding;
        if (oldValue != value) {
            this._padding = value;
            this._padding.changesInterceptor = this;
            this._isChanged = true;
            this.propertyChanged("padding");
        }
    }
    get name() {
        return this._name;
    }
    set name(value) {
        this._name = value;
        this._isChanged = true;
    }
    get readOnly() {
        return this._readOnly;
    }
    set readOnly(value) {
        this._readOnly = value;
        this._isChanged = true;
    }
    get computedHeight() {
        return Number(this.height) + Number(this.padding.top) + Number(this.padding.bottom);
    }
    get computedWidth() {
        return Number(this.width) - Number(this.padding.left) - Number(this.padding.right);
    }
    get height() {
        return this.geometry.height;
    }
    set height(value) {
        this.geometry.height = value;
    }
    get width() {
        return this.geometry.width;
    }
    set width(value) {
        this.geometry.width = value;
    }
    get x() {
        return this.geometry.x;
    }
    set x(value) {
        this.geometry.x = value;
    }
    get y() {
        return this.geometry.y;
    }
    set y(value) {
        this.geometry.y = value;
    }
    hasText() {
        return false;
    }
    propertyChanged(propName) { }
    showLabelPlaceholder(isSelected) {
        return this.label.alignment == StiLabelAlignment.Top || this.label.width < 15 || !isSelected ? true : false;
    }
    getContentAlign() {
        return this.label.alignment == StiLabelAlignment.Right ? "end" : "start";
    }
    getLabelWidth() {
        return this.label.alignment == StiLabelAlignment.Top ? "100%" : this.label.width + "px";
    }
    getTopMargin() {
        return this.label.alignment == StiLabelAlignment.Top && this.label.value ? "8" : "0";
    }
    setProperty(propName, value, oldObject) {
        if (!this.canUpdateProperty(propName)) {
            return;
        }
        if (propName.startsWith("label") && "label" in this) {
            this.label.setProperty(propName.replace("label.", ""), value.clone ? value.clone() : value, oldObject);
        }
        else if (propName.startsWith("description") && "description" in this) {
            this.description.setProperty(propName.replace("description.", ""), value.clone ? value.clone() : value, oldObject);
        }
        else if (this[propName] &&
            (oldObject === null ||
                (propName == "padding" && this[propName].equals(oldObject)) ||
                this[propName] == oldObject)) {
            this[propName] = value.clone ? value.clone() : value;
        }
        else if (propName == "innerText.color" &&
            this["textColor"] &&
            (oldObject === null || oldObject.equals(this["textColor"]))) {
            this["textColor"] = value.clone();
        }
        else if (propName == "innerText.backgroundColor" &&
            this["backgroundColor"] &&
            (oldObject === null || oldObject.equals(this["backgroundColor"]))) {
            this["backgroundColor"] = value.clone();
        }
        else if (propName == "innerText.borderColor" &&
            this["borderColor"] &&
            (oldObject === null || oldObject.equals(this["borderColor"]))) {
            this["borderColor"] = value.clone();
        }
        else if (propName == "innerText.checkColor" &&
            this["checkColor"] &&
            (oldObject === null || oldObject.equals(this["checkColor"]))) {
            this["checkColor"] = value.clone();
        }
        else if (propName == "innerText.borderWeight" &&
            this["borderWeight"] &&
            (oldObject === null || this["borderWeight"] == oldObject)) {
            this["borderWeight"] = value;
        }
        else if (propName == "innerText.contentAlignment" &&
            this["contentAlignment"] &&
            (oldObject === null || this["contentAlignment"] == oldObject)) {
            this["contentAlignment"] = value;
        }
        else if (propName.startsWith("innerText.font") &&
            this["font"] &&
            (oldObject === null || oldObject.equals(this["font"]))) {
            this["font"].setProperty(propName.replace("innerText.font.", ""), value);
        }
        else if (propName.startsWith("padding.") &&
            this["padding"] &&
            (oldObject === null || oldObject.equals(this["padding"]))) {
            let opt = propName.replace("padding.", "");
            this["padding"][opt] = value;
        }
    }
    arrayOfStylesToString(array) {
        let styles = [];
        if (array.length > 0) {
            array.forEach((item) => {
                if (item != "Underline") {
                    styles.push(item);
                }
            });
        }
        return styles.length > 0 ? styles.join(" ").toLocaleLowerCase() : "";
    }
    getUsedFonts(usedFonts) {
        usedFonts.add(this._label.font.family, this.arrayOfStylesToString(this._label.font.style), this._label.font.size);
    }
    getParent(form) {
        let container;
        form.pages.forEach((page) => {
            if (page.elements.indexOf(this) >= 0) {
                container = page;
            }
            else {
                page.allElements.forEach((e) => {
                    let elements = e.elements;
                    if (elements && elements.indexOf(this) >= 0) {
                        container = e;
                    }
                });
            }
        });
        return container;
    }
    getOverPageGeometry(form) {
        let geometry = this.geometry.clone();
        form.pages.forEach((page) => {
            let allElements = page.allElements;
            let child = this;
            let find = true;
            while (find) {
                find = false;
                allElements.forEach((e) => {
                    let elements = e.elements;
                    if (elements && elements.indexOf(child) >= 0) {
                        geometry.x += e.geometry.x + e.padding.left;
                        geometry.y += e.geometry.y + e.padding.top;
                        child = e;
                        find = true;
                    }
                });
            }
        });
        return geometry;
    }
    saveToJsonObject(options) {
        let jObject = super.saveToJsonObject(options);
        jObject.addPropertyBool("Visible", this.visible);
        jObject.addPropertyBool("ReadOnly", this.readOnly);
        jObject.addPropertyJObject("Padding", this.padding.saveToJsonObject(options));
        jObject.addPropertyJObject("Geometry", this.geometry.saveToJsonObject(options));
        jObject.addPropertyStringNullOrEmpty("Name", this.name);
        return jObject;
    }
    loadFromJsonObject(json) {
        this.isAdded = true;
        super.loadFromJsonObject(json);
        for (let property of json.properties()) {
            switch (property.name) {
                case "Visible":
                    this.visible = property.value;
                    break;
                case "Enabled":
                case "ReadOnly":
                    if (property.name == "Enabled") {
                        this.readOnly = !property.value;
                    }
                    else {
                        this.readOnly = property.value;
                    }
                    break;
                case "Padding":
                    this.padding = new StiPadding();
                    this.padding.loadFromJsonObject(property.value);
                    break;
                case "Geometry":
                    this.geometry = new StiRectangleGeometry();
                    this.geometry.loadFromJsonObject(property.value);
                    break;
                case "Name":
                    this.name = property.value;
                    break;
            }
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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