import { StiInputTextItem } from "../helper/StiInputTextItem";
import StiRectangleGeometry from "../properties/StiRectangleGeometry";
import { StiLocalizationService } from "../services/sti-localization.service";
import Color from "../system/Color";
import { Enum } from "../system/Enum";
import { StiHorizontalAlignment, StiTextRangeFormat, StiLineWeight } from "../system/enums";
import { Font } from "../system/Font";
import { StiFormElement } from "./base/StiFormElement";
import { StiLabelItem } from "./text/StiLabelItem";
import { StiTextItem } from "./text/StiTextItem";
export class StiTextBoxElement extends StiFormElement {
    constructor() {
        super();
        this._linesCount = 1;
        this._maximum = 255;
        this.type = "Text_BoxElement";
        this.tooltip = "";
        this.required = false;
        this.description = new StiTextItem();
        this.font = new Font("Segoe UI", 10);
        this.text = "";
        this.backgroundColor = Color.white();
        this.textColor = Color.black();
        this.borderColor = Color.lightGray();
        this.contentAlignment = StiHorizontalAlignment.Left;
        this.isMultiline = false;
        this.password = "";
        this.isPassword = false;
        this.useRange = false;
        this.minimum = 0;
        this.rangeFormat = StiTextRangeFormat.Characters;
        this.borderWeight = StiLineWeight.Thin;
        this.geometry = new StiRectangleGeometry(150, 40);
        this.label = new StiLabelItem(this, StiLocalizationService.get("Dialogs", "StiTextBoxControl"));
    }
    setDefaultValues() {
        this.value = "";
    }
    setTabIndex(value) {
        this.tabIndex = value++;
        return value;
    }
    getDefaultEditableContent() {
        let textItem = new StiInputTextItem(this.text, this.font, this.textColor, this.backgroundColor, this.contentAlignment);
        textItem.element = this;
        return textItem;
    }
    get maximum() {
        return this._maximum;
    }
    set maximum(value) {
        this._maximum = Math.max(1, value);
    }
    get linesCount() {
        return this._linesCount;
    }
    set linesCount(value) {
        if (value <= 1) {
            this._linesCount = 1;
        }
        else if (value >= 10) {
            this._linesCount = 10;
        }
        else {
            this._linesCount = value;
        }
    }
    get value() {
        return this.text;
    }
    set value(value) {
        this.text = value;
    }
    get eValue() {
        return this.value;
    }
    set eValue(value) {
        this.value = value;
    }
    get eFont() {
        return this.font;
    }
    set eFont(value) {
        this.font = value;
    }
    get eColor() {
        return this.textColor;
    }
    set eColor(value) {
        this.textColor = value;
    }
    get eBackgroundColor() {
        return this.backgroundColor;
    }
    set eBackgroundColor(value) {
        this.backgroundColor = value;
    }
    get eContentAlignment() {
        return this.contentAlignment;
    }
    set eContentAlignment(value) {
        this.contentAlignment = value;
    }
    haveContentAlignment() {
        return true;
    }
    haveVerticalAlignment() {
        return false;
    }
    getUsedFonts(usedFonts) {
        super.getUsedFonts(usedFonts);
        usedFonts.add(this.font.family, this.arrayOfStylesToString(this.font.style), this.font.size);
        usedFonts.add(this.description.font.family, this.arrayOfStylesToString(this.description.font.style), this.description.font.size);
    }
    haveBackgoundColor() {
        return true;
    }
    htmlStyle() {
        return `${this.font.htmlStyle()};color:${this.eColor.toHtmlColor()};text-align:${Enum.getName(StiHorizontalAlignment, this.contentAlignment)}`;
    }
    saveToJsonObject(options) {
        if (this.element?.nativeElement) {
            this.heightForExport = this.element.nativeElement.offsetHeight;
        }
        let jObject = super.saveToJsonObject(options);
        jObject.addPropertyJObject("Label", this.label.saveToJsonObject(options));
        jObject.addPropertyJObject("Description", this.description.saveToJsonObject(options));
        jObject.addPropertyJObject("Font", this.font.saveToJsonObject(options));
        jObject.addPropertyString("Text", this.text);
        jObject.addPropertyColor("BackgroundColor", this.backgroundColor);
        jObject.addPropertyColor("TextColor", this.textColor);
        jObject.addPropertyEnum("ContentAlignment", StiHorizontalAlignment, this.contentAlignment);
        jObject.addPropertyString("Tooltip", this.tooltip);
        jObject.addPropertyBool("Required", this.required);
        jObject.addPropertyBool("IsPassword", this.isPassword);
        jObject.addPropertyBool("IsMultiline", this.isMultiline);
        jObject.addPropertyNumber("LinesCount", this.linesCount);
        jObject.addPropertyBool("UseRange", this.useRange);
        jObject.addPropertyNumber("Minimum", this.minimum);
        jObject.addPropertyNumber("Maximum", this.maximum);
        jObject.addPropertyEnum("RangeFormat", StiTextRangeFormat, this.rangeFormat);
        jObject.addPropertyColor("BorderColor", this.borderColor);
        jObject.addPropertyEnum("BorderWeight", StiLineWeight, this.borderWeight);
        if (this.element?.nativeElement) {
            jObject.addPropertyNumber("HeightForExport", this.heightForExport);
        }
        return jObject;
    }
    loadFromJsonObject(json) {
        super.loadFromJsonObject(json);
        for (let property of json.properties()) {
            switch (property.name) {
                case "ContentAlignment":
                    this.contentAlignment = Enum.parse(StiHorizontalAlignment, property.value);
                    break;
                case "BackgroundColor":
                    this.backgroundColor = new Color();
                    this.backgroundColor.deserializeFromString(property.value);
                    break;
                case "TextColor":
                    this.textColor = new Color();
                    this.textColor.deserializeFromString(property.value);
                    break;
                case "Text":
                    this.text = property.value;
                    break;
                case "Label":
                    this.label = new StiLabelItem(this);
                    this.label.loadFromJsonObject(property.value);
                    break;
                case "Font":
                    this.font = new Font();
                    this.font.loadFromJsonObject(property.value);
                    break;
                case "Description":
                    this.description = new StiTextItem();
                    this.description.loadFromJsonObject(property.value);
                    break;
                case "Tooltip":
                    this.tooltip = property.value;
                    break;
                case "Required":
                    this.required = property.value;
                    break;
                case "IsPassword":
                    this.isPassword = property.value;
                    break;
                case "IsMultiline":
                    this.isMultiline = property.value;
                    break;
                case "LinesCount":
                    this.linesCount = property.value;
                    break;
                case "UseRange":
                    this.useRange = property.value;
                    break;
                case "Minimum":
                    this.minimum = property.value;
                    break;
                case "Maximum":
                    this.maximum = property.value;
                    break;
                case "RangeFormat":
                    this.rangeFormat = Enum.parse(StiTextRangeFormat, property.value);
                    break;
                case "BorderColor":
                    this.borderColor = new Color();
                    this.borderColor.deserializeFromString(property.value);
                    break;
                case "BorderWeight":
                    this.borderWeight = Enum.parse(StiLineWeight, property.value);
                    break;
            }
        }
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiU3RpVGV4dEJveEVsZW1lbnQuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi8uLi8uLi9wcm9qZWN0cy9zdGltdWxzb2Z0LWZvcm1zL3NyYy9saWIvZWxlbWVudHMvU3RpVGV4dEJveEVsZW1lbnQudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQUEsT0FBTyxFQUFFLGdCQUFnQixFQUFFLE1BQU0sNEJBQTRCLENBQUM7QUFDOUQsT0FBTyxvQkFBb0IsTUFBTSxvQ0FBb0MsQ0FBQztBQUV0RSxPQUFPLEVBQUUsc0JBQXNCLEVBQUUsTUFBTSxzQ0FBc0MsQ0FBQztBQUM5RSxPQUFPLEtBQUssTUFBTSxpQkFBaUIsQ0FBQztBQUNwQyxPQUFPLEVBQUUsSUFBSSxFQUFFLE1BQU0sZ0JBQWdCLENBQUM7QUFDdEMsT0FBTyxFQUFFLHNCQUFzQixFQUFFLGtCQUFrQixFQUFFLGFBQWEsRUFBcUIsTUFBTSxpQkFBaUIsQ0FBQztBQUMvRyxPQUFPLEVBQUUsSUFBSSxFQUFFLE1BQU0sZ0JBQWdCLENBQUM7QUFHdEMsT0FBTyxFQUFFLGNBQWMsRUFBRSxNQUFNLHVCQUF1QixDQUFDO0FBQ3ZELE9BQU8sRUFBRSxZQUFZLEVBQUUsTUFBTSxxQkFBcUIsQ0FBQztBQUNuRCxPQUFPLEVBQUUsV0FBVyxFQUFFLE1BQU0sb0JBQW9CLENBQUM7QUFFakQsTUFBTSxPQUFPLGlCQUNYLFNBQVEsY0FBYztJQXlCdEI7UUFDRSxLQUFLLEVBQUUsQ0FBQztRQXZCRixnQkFBVyxHQUFXLENBQUMsQ0FBQztRQUN4QixhQUFRLEdBQVcsR0FBRyxDQUFDO1FBQ3hCLFNBQUksR0FBVyxpQkFBaUIsQ0FBQztRQUNqQyxZQUFPLEdBQUcsRUFBRSxDQUFDO1FBQ2IsYUFBUSxHQUFHLEtBQUssQ0FBQztRQUNqQixnQkFBVyxHQUFnQixJQUFJLFdBQVcsRUFBRSxDQUFDO1FBQzdDLFNBQUksR0FBUyxJQUFJLElBQUksQ0FBQyxVQUFVLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFDdEMsU0FBSSxHQUFHLEVBQUUsQ0FBQztRQUNWLG9CQUFlLEdBQVUsS0FBSyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBQ3ZDLGNBQVMsR0FBVSxLQUFLLENBQUMsS0FBSyxFQUFFLENBQUM7UUFDakMsZ0JBQVcsR0FBVSxLQUFLLENBQUMsU0FBUyxFQUFFLENBQUM7UUFDdkMscUJBQWdCLEdBQTJCLHNCQUFzQixDQUFDLElBQUksQ0FBQztRQUN2RSxnQkFBVyxHQUFHLEtBQUssQ0FBQztRQUNwQixhQUFRLEdBQVcsRUFBRSxDQUFDO1FBQ3RCLGVBQVUsR0FBRyxLQUFLLENBQUM7UUFDbkIsYUFBUSxHQUFHLEtBQUssQ0FBQztRQUNqQixZQUFPLEdBQVcsQ0FBQyxDQUFDO1FBQ3BCLGdCQUFXLEdBQXVCLGtCQUFrQixDQUFDLFVBQVUsQ0FBQztRQUNoRSxpQkFBWSxHQUFrQixhQUFhLENBQUMsSUFBSSxDQUFDO1FBTXRELElBQUksQ0FBQyxRQUFRLEdBQUcsSUFBSSxvQkFBb0IsQ0FBQyxHQUFHLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFDbEQsSUFBSSxDQUFDLEtBQUssR0FBRyxJQUFJLFlBQVksQ0FBQyxJQUFJLEVBQUUsc0JBQXNCLENBQUMsR0FBRyxDQUFDLFNBQVMsRUFBRSxtQkFBbUIsQ0FBQyxDQUFDLENBQUM7SUFDbEcsQ0FBQztJQUNELGdCQUFnQjtRQUNkLElBQUksQ0FBQyxLQUFLLEdBQUcsRUFBRSxDQUFDO0lBQ2xCLENBQUM7SUFFRCxXQUFXLENBQUMsS0FBYTtRQUN2QixJQUFJLENBQUMsUUFBUSxHQUFHLEtBQUssRUFBRSxDQUFDO1FBQ3hCLE9BQU8sS0FBSyxDQUFDO0lBQ2YsQ0FBQztJQUVNLHlCQUF5QjtRQUM5QixJQUFJLFFBQVEsR0FBRyxJQUFJLGdCQUFnQixDQUNqQyxJQUFJLENBQUMsSUFBSSxFQUNULElBQUksQ0FBQyxJQUFJLEVBQ1QsSUFBSSxDQUFDLFNBQVMsRUFDZCxJQUFJLENBQUMsZUFBZSxFQUNwQixJQUFJLENBQUMsZ0JBQWdCLENBQ3RCLENBQUM7UUFDRixRQUFRLENBQUMsT0FBTyxHQUFHLElBQUksQ0FBQztRQUN4QixPQUFPLFFBQVEsQ0FBQztJQUNsQixDQUFDO0lBQ0QsSUFBSSxPQUFPO1FBQ1QsT0FBTyxJQUFJLENBQUMsUUFBUSxDQUFDO0lBQ3ZCLENBQUM7SUFDRCxJQUFJLE9BQU8sQ0FBQyxLQUFhO1FBQ3ZCLElBQUksQ0FBQyxRQUFRLEdBQUcsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDckMsQ0FBQztJQUNELElBQVcsVUFBVTtRQUNuQixPQUFPLElBQUksQ0FBQyxXQUFXLENBQUM7SUFDMUIsQ0FBQztJQUNELElBQVcsVUFBVSxDQUFDLEtBQWE7UUFDakMsSUFBSSxLQUFLLElBQUksQ0FBQyxFQUFFO1lBQ2QsSUFBSSxDQUFDLFdBQVcsR0FBRyxDQUFDLENBQUM7U0FDdEI7YUFBTSxJQUFJLEtBQUssSUFBSSxFQUFFLEVBQUU7WUFDdEIsSUFBSSxDQUFDLFdBQVcsR0FBRyxFQUFFLENBQUM7U0FDdkI7YUFBTTtZQUNMLElBQUksQ0FBQyxXQUFXLEdBQUcsS0FBSyxDQUFDO1NBQzFCO0lBQ0gsQ0FBQztJQUNELElBQUksS0FBSztRQUNQLE9BQU8sSUFBSSxDQUFDLElBQUksQ0FBQztJQUNuQixDQUFDO0lBRUQsSUFBSSxLQUFLLENBQUMsS0FBYTtRQUNyQixJQUFJLENBQUMsSUFBSSxHQUFHLEtBQUssQ0FBQztJQUNwQixDQUFDO0lBRUQsSUFBSSxNQUFNO1FBQ1IsT0FBTyxJQUFJLENBQUMsS0FBSyxDQUFDO0lBQ3BCLENBQUM7SUFFRCxJQUFJLE1BQU0sQ0FBQyxLQUFhO1FBQ3RCLElBQUksQ0FBQyxLQUFLLEdBQUcsS0FBSyxDQUFDO0lBQ3JCLENBQUM7SUFFRCxJQUFJLEtBQUs7UUFDUCxPQUFPLElBQUksQ0FBQyxJQUFJLENBQUM7SUFDbkIsQ0FBQztJQUVELElBQUksS0FBSyxDQUFDLEtBQVc7UUFDbkIsSUFBSSxDQUFDLElBQUksR0FBRyxLQUFLLENBQUM7SUFDcEIsQ0FBQztJQUVELElBQUksTUFBTTtRQUNSLE9BQU8sSUFBSSxDQUFDLFNBQVMsQ0FBQztJQUN4QixDQUFDO0lBRUQsSUFBSSxNQUFNLENBQUMsS0FBWTtRQUNyQixJQUFJLENBQUMsU0FBUyxHQUFHLEtBQUssQ0FBQztJQUN6QixDQUFDO0lBRUQsSUFBSSxnQkFBZ0I7UUFDbEIsT0FBTyxJQUFJLENBQUMsZUFBZSxDQUFDO0lBQzlCLENBQUM7SUFFRCxJQUFJLGdCQUFnQixDQUFDLEtBQVk7UUFDL0IsSUFBSSxDQUFDLGVBQWUsR0FBRyxLQUFLLENBQUM7SUFDL0IsQ0FBQztJQUNELElBQUksaUJBQWlCO1FBQ25CLE9BQU8sSUFBSSxDQUFDLGdCQUFnQixDQUFDO0lBQy9CLENBQUM7SUFDRCxJQUFJLGlCQUFpQixDQUFDLEtBQTZCO1FBQ2pELElBQUksQ0FBQyxnQkFBZ0IsR0FBRyxLQUFLLENBQUM7SUFDaEMsQ0FBQztJQUNELG9CQUFvQjtRQUNsQixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFDRCxxQkFBcUI7UUFDbkIsT0FBTyxLQUFLLENBQUM7SUFDZixDQUFDO0lBQ00sWUFBWSxDQUFDLFNBQTJCO1FBQzdDLEtBQUssQ0FBQyxZQUFZLENBQUMsU0FBUyxDQUFDLENBQUM7UUFDOUIsU0FBUyxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLE1BQU0sRUFBRSxJQUFJLENBQUMscUJBQXFCLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsRUFBRSxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQzdGLFNBQVMsQ0FBQyxHQUFHLENBQ1gsSUFBSSxDQUFDLFdBQVcsQ0FBQyxJQUFJLENBQUMsTUFBTSxFQUM1QixJQUFJLENBQUMscUJBQXFCLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLEVBQ3ZELElBQUksQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDLElBQUksQ0FDM0IsQ0FBQztJQUNKLENBQUM7SUFDRCxrQkFBa0I7UUFDaEIsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQsU0FBUztRQUNQLE9BQU8sR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxVQUFVLElBQUksQ0FBQyxNQUFNLENBQUMsV0FBVyxFQUFFLGVBQWUsSUFBSSxDQUFDLE9BQU8sQ0FDM0Ysc0JBQXNCLEVBQ3RCLElBQUksQ0FBQyxnQkFBZ0IsQ0FDdEIsRUFBRSxDQUFDO0lBQ04sQ0FBQztJQUVELGdCQUFnQixDQUFDLE9BQVk7UUFDM0IsSUFBSSxJQUFJLENBQUMsT0FBTyxFQUFFLGFBQWEsRUFBRTtZQUMvQixJQUFJLENBQUMsZUFBZSxHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsYUFBYSxDQUFDLFlBQVksQ0FBQztTQUNoRTtRQUNELElBQUksT0FBTyxHQUFHLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUM5QyxPQUFPLENBQUMsa0JBQWtCLENBQUMsT0FBTyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsZ0JBQWdCLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQztRQUMxRSxPQUFPLENBQUMsa0JBQWtCLENBQUMsYUFBYSxFQUFFLElBQUksQ0FBQyxXQUFXLENBQUMsZ0JBQWdCLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQztRQUN0RixPQUFPLENBQUMsa0JBQWtCLENBQUMsTUFBTSxFQUFFLElBQUksQ0FBQyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQztRQUN4RSxPQUFPLENBQUMsaUJBQWlCLENBQUMsTUFBTSxFQUFFLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUM3QyxPQUFPLENBQUMsZ0JBQWdCLENBQUMsaUJBQWlCLEVBQUUsSUFBSSxDQUFDLGVBQWUsQ0FBQyxDQUFDO1FBQ2xFLE9BQU8sQ0FBQyxnQkFBZ0IsQ0FBQyxXQUFXLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxDQUFDO1FBQ3RELE9BQU8sQ0FBQyxlQUFlLENBQUMsa0JBQWtCLEVBQUUsc0JBQXNCLEVBQUUsSUFBSSxDQUFDLGdCQUFnQixDQUFDLENBQUM7UUFDM0YsT0FBTyxDQUFDLGlCQUFpQixDQUFDLFNBQVMsRUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDLENBQUM7UUFDbkQsT0FBTyxDQUFDLGVBQWUsQ0FBQyxVQUFVLEVBQUUsSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBQ25ELE9BQU8sQ0FBQyxlQUFlLENBQUMsWUFBWSxFQUFFLElBQUksQ0FBQyxVQUFVLENBQUMsQ0FBQztRQUN2RCxPQUFPLENBQUMsZUFBZSxDQUFDLGFBQWEsRUFBRSxJQUFJLENBQUMsV0FBVyxDQUFDLENBQUM7UUFDekQsT0FBTyxDQUFDLGlCQUFpQixDQUFDLFlBQVksRUFBRSxJQUFJLENBQUMsVUFBVSxDQUFDLENBQUM7UUFDekQsT0FBTyxDQUFDLGVBQWUsQ0FBQyxVQUFVLEVBQUUsSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBQ25ELE9BQU8sQ0FBQyxpQkFBaUIsQ0FBQyxTQUFTLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBQ25ELE9BQU8sQ0FBQyxpQkFBaUIsQ0FBQyxTQUFTLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBQ25ELE9BQU8sQ0FBQyxlQUFlLENBQUMsYUFBYSxFQUFFLGtCQUFrQixFQUFFLElBQUksQ0FBQyxXQUFXLENBQUMsQ0FBQztRQUM3RSxPQUFPLENBQUMsZ0JBQWdCLENBQUMsYUFBYSxFQUFFLElBQUksQ0FBQyxXQUFXLENBQUMsQ0FBQztRQUMxRCxPQUFPLENBQUMsZUFBZSxDQUFDLGNBQWMsRUFBRSxhQUFhLEVBQUUsSUFBSSxDQUFDLFlBQVksQ0FBQyxDQUFDO1FBQzFFLElBQUksSUFBSSxDQUFDLE9BQU8sRUFBRSxhQUFhLEVBQUU7WUFDL0IsT0FBTyxDQUFDLGlCQUFpQixDQUFDLGlCQUFpQixFQUFFLElBQUksQ0FBQyxlQUFlLENBQUMsQ0FBQztTQUNwRTtRQUNELE9BQU8sT0FBTyxDQUFDO0lBQ2pCLENBQUM7SUFFTSxrQkFBa0IsQ0FBQyxJQUFhO1FBQ3JDLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUMvQixLQUFLLElBQUksUUFBUSxJQUFJLElBQUksQ0FBQyxVQUFVLEVBQUUsRUFBRTtZQUN0QyxRQUFRLFFBQVEsQ0FBQyxJQUFJLEVBQUU7Z0JBQ3JCLEtBQUssa0JBQWtCO29CQUNyQixJQUFJLENBQUMsZ0JBQWdCLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxzQkFBc0IsRUFBRSxRQUFRLENBQUMsS0FBSyxDQUFDLENBQUM7b0JBQzNFLE1BQU07Z0JBQ1IsS0FBSyxpQkFBaUI7b0JBQ3BCLElBQUksQ0FBQyxlQUFlLEdBQUcsSUFBSSxLQUFLLEVBQUUsQ0FBQztvQkFDbkMsSUFBSSxDQUFDLGVBQWUsQ0FBQyxxQkFBcUIsQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLENBQUM7b0JBQzNELE1BQU07Z0JBQ1IsS0FBSyxXQUFXO29CQUNkLElBQUksQ0FBQyxTQUFTLEdBQUcsSUFBSSxLQUFLLEVBQUUsQ0FBQztvQkFDN0IsSUFBSSxDQUFDLFNBQVMsQ0FBQyxxQkFBcUIsQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLENBQUM7b0JBQ3JELE1BQU07Z0JBQ1IsS0FBSyxNQUFNO29CQUNULElBQUksQ0FBQyxJQUFJLEdBQUcsUUFBUSxDQUFDLEtBQUssQ0FBQztvQkFDM0IsTUFBTTtnQkFDUixLQUFLLE9BQU87b0JBQ1YsSUFBSSxDQUFDLEtBQUssR0FBRyxJQUFJLFlBQVksQ0FBQyxJQUFJLENBQUMsQ0FBQztvQkFDcEMsSUFBSSxDQUFDLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLENBQUM7b0JBQzlDLE1BQU07Z0JBQ1IsS0FBSyxNQUFNO29CQUNULElBQUksQ0FBQyxJQUFJLEdBQUcsSUFBSSxJQUFJLEVBQUUsQ0FBQztvQkFDdkIsSUFBSSxDQUFDLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLENBQUM7b0JBQzdDLE1BQU07Z0JBQ1IsS0FBSyxhQUFhO29CQUNoQixJQUFJLENBQUMsV0FBVyxHQUFHLElBQUksV0FBVyxFQUFFLENBQUM7b0JBQ3JDLElBQUksQ0FBQyxXQUFXLENBQUMsa0JBQWtCLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQyxDQUFDO29CQUNwRCxNQUFNO2dCQUNSLEtBQUssU0FBUztvQkFDWixJQUFJLENBQUMsT0FBTyxHQUFHLFFBQVEsQ0FBQyxLQUFLLENBQUM7b0JBQzlCLE1BQU07Z0JBQ1IsS0FBSyxVQUFVO29CQUNiLElBQUksQ0FBQyxRQUFRLEdBQUcsUUFBUSxDQUFDLEtBQUssQ0FBQztvQkFDL0IsTUFBTTtnQkFDUixLQUFLLFlBQVk7b0JBQ2YsSUFBSSxDQUFDLFVBQVUsR0FBRyxRQUFRLENBQUMsS0FBSyxDQUFDO29CQUNqQyxNQUFNO2dCQUNSLEtBQUssYUFBYTtvQkFDaEIsSUFBSSxDQUFDLFdBQVcsR0FBRyxRQUFRLENBQUMsS0FBSyxDQUFDO29CQUNsQyxNQUFNO2dCQUNSLEtBQUssWUFBWTtvQkFDZixJQUFJLENBQUMsVUFBVSxHQUFHLFFBQVEsQ0FBQyxLQUFLLENBQUM7b0JBQ2pDLE1BQU07Z0JBQ1IsS0FBSyxVQUFVO29CQUNiLElBQUksQ0FBQyxRQUFRLEdBQUcsUUFBUSxDQUFDLEtBQUssQ0FBQztvQkFDL0IsTUFBTTtnQkFDUixLQUFLLFNBQVM7b0JBQ1osSUFBSSxDQUFDLE9BQU8sR0FBRyxRQUFRLENBQUMsS0FBSyxDQUFDO29CQUM5QixNQUFNO2dCQUNSLEtBQUssU0FBUztvQkFDWixJQUFJLENBQUMsT0FBTyxHQUFHLFFBQVEsQ0FBQyxLQUFLLENBQUM7b0JBQzlCLE1BQU07Z0JBQ1IsS0FBSyxhQUFhO29CQUNoQixJQUFJLENBQUMsV0FBVyxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsa0JBQWtCLEVBQUUsUUFBUSxDQUFDLEtBQUssQ0FBQyxDQUFDO29CQUNsRSxNQUFNO2dCQUNSLEtBQUssYUFBYTtvQkFDaEIsSUFBSSxDQUFDLFdBQVcsR0FBRyxJQUFJLEtBQUssRUFBRSxDQUFDO29CQUMvQixJQUFJLENBQUMsV0FBVyxDQUFDLHFCQUFxQixDQUFDLFFBQVEsQ0FBQyxLQUFLLENBQUMsQ0FBQztvQkFDdkQsTUFBTTtnQkFDUixLQUFLLGNBQWM7b0JBQ2pCLElBQUksQ0FBQyxZQUFZLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxhQUFhLEVBQUUsUUFBUSxDQUFDLEtBQUssQ0FBQyxDQUFDO29CQUM5RCxNQUFNO2FBQ1Q7U0FDRjtJQUNILENBQUM7Q0FDRiIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IFN0aUlucHV0VGV4dEl0ZW0gfSBmcm9tIFwiLi4vaGVscGVyL1N0aUlucHV0VGV4dEl0ZW1cIjtcclxuaW1wb3J0IFN0aVJlY3RhbmdsZUdlb21ldHJ5IGZyb20gXCIuLi9wcm9wZXJ0aWVzL1N0aVJlY3RhbmdsZUdlb21ldHJ5XCI7XHJcbmltcG9ydCB7IFN0aVVzZWRGb250c0xpc3QgfSBmcm9tIFwiLi4vc2VydmljZXMvb2JqZWN0c1wiO1xyXG5pbXBvcnQgeyBTdGlMb2NhbGl6YXRpb25TZXJ2aWNlIH0gZnJvbSBcIi4uL3NlcnZpY2VzL3N0aS1sb2NhbGl6YXRpb24uc2VydmljZVwiO1xyXG5pbXBvcnQgQ29sb3IgZnJvbSBcIi4uL3N5c3RlbS9Db2xvclwiO1xyXG5pbXBvcnQgeyBFbnVtIH0gZnJvbSBcIi4uL3N5c3RlbS9FbnVtXCI7XHJcbmltcG9ydCB7IFN0aUhvcml6b250YWxBbGlnbm1lbnQsIFN0aVRleHRSYW5nZUZvcm1hdCwgU3RpTGluZVdlaWdodCwgU3RpTGFiZWxBbGlnbm1lbnQgfSBmcm9tIFwiLi4vc3lzdGVtL2VudW1zXCI7XHJcbmltcG9ydCB7IEZvbnQgfSBmcm9tIFwiLi4vc3lzdGVtL0ZvbnRcIjtcclxuaW1wb3J0IHsgU3RpTGFiZWwsIFN0aURlc2NyaXB0aW9uLCBTdGlFZGl0YWJsZUNvbnRlbnQsIFN0aVZhbHVlRWxlbWVudCwgU3RpVGFiSW5kZXggfSBmcm9tIFwiLi4vc3lzdGVtL2ludGVyZmFjZXNcIjtcclxuaW1wb3J0IFN0aUpzb24gZnJvbSBcIi4uL3N5c3RlbS9TdGlKc29uXCI7XHJcbmltcG9ydCB7IFN0aUZvcm1FbGVtZW50IH0gZnJvbSBcIi4vYmFzZS9TdGlGb3JtRWxlbWVudFwiO1xyXG5pbXBvcnQgeyBTdGlMYWJlbEl0ZW0gfSBmcm9tIFwiLi90ZXh0L1N0aUxhYmVsSXRlbVwiO1xyXG5pbXBvcnQgeyBTdGlUZXh0SXRlbSB9IGZyb20gXCIuL3RleHQvU3RpVGV4dEl0ZW1cIjtcclxuXHJcbmV4cG9ydCBjbGFzcyBTdGlUZXh0Qm94RWxlbWVudFxyXG4gIGV4dGVuZHMgU3RpRm9ybUVsZW1lbnRcclxuICBpbXBsZW1lbnRzIFN0aUxhYmVsLCBTdGlEZXNjcmlwdGlvbiwgU3RpRWRpdGFibGVDb250ZW50LCBTdGlWYWx1ZUVsZW1lbnQsIFN0aVRhYkluZGV4XHJcbntcclxuICBwcml2YXRlIF9saW5lc0NvdW50OiBudW1iZXIgPSAxO1xyXG4gIHByaXZhdGUgX21heGltdW06IG51bWJlciA9IDI1NTtcclxuICBwdWJsaWMgdHlwZTogc3RyaW5nID0gXCJUZXh0X0JveEVsZW1lbnRcIjtcclxuICBwdWJsaWMgdG9vbHRpcCA9IFwiXCI7XHJcbiAgcHVibGljIHJlcXVpcmVkID0gZmFsc2U7XHJcbiAgcHVibGljIGRlc2NyaXB0aW9uOiBTdGlUZXh0SXRlbSA9IG5ldyBTdGlUZXh0SXRlbSgpO1xyXG4gIHB1YmxpYyBmb250OiBGb250ID0gbmV3IEZvbnQoXCJTZWdvZSBVSVwiLCAxMCk7XHJcbiAgcHVibGljIHRleHQgPSBcIlwiO1xyXG4gIHB1YmxpYyBiYWNrZ3JvdW5kQ29sb3I6IENvbG9yID0gQ29sb3Iud2hpdGUoKTtcclxuICBwdWJsaWMgdGV4dENvbG9yOiBDb2xvciA9IENvbG9yLmJsYWNrKCk7XHJcbiAgcHVibGljIGJvcmRlckNvbG9yOiBDb2xvciA9IENvbG9yLmxpZ2h0R3JheSgpO1xyXG4gIHB1YmxpYyBjb250ZW50QWxpZ25tZW50OiBTdGlIb3Jpem9udGFsQWxpZ25tZW50ID0gU3RpSG9yaXpvbnRhbEFsaWdubWVudC5MZWZ0O1xyXG4gIHB1YmxpYyBpc011bHRpbGluZSA9IGZhbHNlO1xyXG4gIHB1YmxpYyBwYXNzd29yZDogc3RyaW5nID0gXCJcIjtcclxuICBwdWJsaWMgaXNQYXNzd29yZCA9IGZhbHNlO1xyXG4gIHB1YmxpYyB1c2VSYW5nZSA9IGZhbHNlO1xyXG4gIHB1YmxpYyBtaW5pbXVtOiBudW1iZXIgPSAwO1xyXG4gIHB1YmxpYyByYW5nZUZvcm1hdDogU3RpVGV4dFJhbmdlRm9ybWF0ID0gU3RpVGV4dFJhbmdlRm9ybWF0LkNoYXJhY3RlcnM7XHJcbiAgcHVibGljIGJvcmRlcldlaWdodDogU3RpTGluZVdlaWdodCA9IFN0aUxpbmVXZWlnaHQuVGhpbjtcclxuICBwdWJsaWMgaGVpZ2h0Rm9yRXhwb3J0OiBudW1iZXI7XHJcbiAgcHVibGljIHRhYkluZGV4OiBudW1iZXI7XHJcblxyXG4gIGNvbnN0cnVjdG9yKCkge1xyXG4gICAgc3VwZXIoKTtcclxuICAgIHRoaXMuZ2VvbWV0cnkgPSBuZXcgU3RpUmVjdGFuZ2xlR2VvbWV0cnkoMTUwLCA0MCk7XHJcbiAgICB0aGlzLmxhYmVsID0gbmV3IFN0aUxhYmVsSXRlbSh0aGlzLCBTdGlMb2NhbGl6YXRpb25TZXJ2aWNlLmdldChcIkRpYWxvZ3NcIiwgXCJTdGlUZXh0Qm94Q29udHJvbFwiKSk7XHJcbiAgfVxyXG4gIHNldERlZmF1bHRWYWx1ZXMoKSB7XHJcbiAgICB0aGlzLnZhbHVlID0gXCJcIjtcclxuICB9XHJcblxyXG4gIHNldFRhYkluZGV4KHZhbHVlOiBudW1iZXIpOiBudW1iZXIge1xyXG4gICAgdGhpcy50YWJJbmRleCA9IHZhbHVlKys7XHJcbiAgICByZXR1cm4gdmFsdWU7XHJcbiAgfVxyXG5cclxuICBwdWJsaWMgZ2V0RGVmYXVsdEVkaXRhYmxlQ29udGVudCgpOiBTdGlFZGl0YWJsZUNvbnRlbnQge1xyXG4gICAgbGV0IHRleHRJdGVtID0gbmV3IFN0aUlucHV0VGV4dEl0ZW0oXHJcbiAgICAgIHRoaXMudGV4dCxcclxuICAgICAgdGhpcy5mb250LFxyXG4gICAgICB0aGlzLnRleHRDb2xvcixcclxuICAgICAgdGhpcy5iYWNrZ3JvdW5kQ29sb3IsXHJcbiAgICAgIHRoaXMuY29udGVudEFsaWdubWVudFxyXG4gICAgKTtcclxuICAgIHRleHRJdGVtLmVsZW1lbnQgPSB0aGlzO1xyXG4gICAgcmV0dXJuIHRleHRJdGVtO1xyXG4gIH1cclxuICBnZXQgbWF4aW11bSgpIHtcclxuICAgIHJldHVybiB0aGlzLl9tYXhpbXVtO1xyXG4gIH1cclxuICBzZXQgbWF4aW11bSh2YWx1ZTogbnVtYmVyKSB7XHJcbiAgICB0aGlzLl9tYXhpbXVtID0gTWF0aC5tYXgoMSwgdmFsdWUpO1xyXG4gIH1cclxuICBwdWJsaWMgZ2V0IGxpbmVzQ291bnQoKSB7XHJcbiAgICByZXR1cm4gdGhpcy5fbGluZXNDb3VudDtcclxuICB9XHJcbiAgcHVibGljIHNldCBsaW5lc0NvdW50KHZhbHVlOiBudW1iZXIpIHtcclxuICAgIGlmICh2YWx1ZSA8PSAxKSB7XHJcbiAgICAgIHRoaXMuX2xpbmVzQ291bnQgPSAxO1xyXG4gICAgfSBlbHNlIGlmICh2YWx1ZSA+PSAxMCkge1xyXG4gICAgICB0aGlzLl9saW5lc0NvdW50ID0gMTA7XHJcbiAgICB9IGVsc2Uge1xyXG4gICAgICB0aGlzLl9saW5lc0NvdW50ID0gdmFsdWU7XHJcbiAgICB9XHJcbiAgfVxyXG4gIGdldCB2YWx1ZSgpOiBzdHJpbmcge1xyXG4gICAgcmV0dXJuIHRoaXMudGV4dDtcclxuICB9XHJcblxyXG4gIHNldCB2YWx1ZSh2YWx1ZTogc3RyaW5nKSB7XHJcbiAgICB0aGlzLnRleHQgPSB2YWx1ZTtcclxuICB9XHJcblxyXG4gIGdldCBlVmFsdWUoKTogc3RyaW5nIHtcclxuICAgIHJldHVybiB0aGlzLnZhbHVlO1xyXG4gIH1cclxuXHJcbiAgc2V0IGVWYWx1ZSh2YWx1ZTogc3RyaW5nKSB7XHJcbiAgICB0aGlzLnZhbHVlID0gdmFsdWU7XHJcbiAgfVxyXG5cclxuICBnZXQgZUZvbnQoKTogRm9udCB7XHJcbiAgICByZXR1cm4gdGhpcy5mb250O1xyXG4gIH1cclxuXHJcbiAgc2V0IGVGb250KHZhbHVlOiBGb250KSB7XHJcbiAgICB0aGlzLmZvbnQgPSB2YWx1ZTtcclxuICB9XHJcblxyXG4gIGdldCBlQ29sb3IoKTogQ29sb3Ige1xyXG4gICAgcmV0dXJuIHRoaXMudGV4dENvbG9yO1xyXG4gIH1cclxuXHJcbiAgc2V0IGVDb2xvcih2YWx1ZTogQ29sb3IpIHtcclxuICAgIHRoaXMudGV4dENvbG9yID0gdmFsdWU7XHJcbiAgfVxyXG5cclxuICBnZXQgZUJhY2tncm91bmRDb2xvcigpOiBDb2xvciB7XHJcbiAgICByZXR1cm4gdGhpcy5iYWNrZ3JvdW5kQ29sb3I7XHJcbiAgfVxyXG5cclxuICBzZXQgZUJhY2tncm91bmRDb2xvcih2YWx1ZTogQ29sb3IpIHtcclxuICAgIHRoaXMuYmFja2dyb3VuZENvbG9yID0gdmFsdWU7XHJcbiAgfVxyXG4gIGdldCBlQ29udGVudEFsaWdubWVudCgpOiBTdGlIb3Jpem9udGFsQWxpZ25tZW50IHtcclxuICAgIHJldHVybiB0aGlzLmNvbnRlbnRBbGlnbm1lbnQ7XHJcbiAgfVxyXG4gIHNldCBlQ29udGVudEFsaWdubWVudCh2YWx1ZTogU3RpSG9yaXpvbnRhbEFsaWdubWVudCkge1xyXG4gICAgdGhpcy5jb250ZW50QWxpZ25tZW50ID0gdmFsdWU7XHJcbiAgfVxyXG4gIGhhdmVDb250ZW50QWxpZ25tZW50KCk6IGJvb2xlYW4ge1xyXG4gICAgcmV0dXJuIHRydWU7XHJcbiAgfVxyXG4gIGhhdmVWZXJ0aWNhbEFsaWdubWVudCgpOiBib29sZWFuIHtcclxuICAgIHJldHVybiBmYWxzZTtcclxuICB9XHJcbiAgcHVibGljIGdldFVzZWRGb250cyh1c2VkRm9udHM6IFN0aVVzZWRGb250c0xpc3QpIHtcclxuICAgIHN1cGVyLmdldFVzZWRGb250cyh1c2VkRm9udHMpO1xyXG4gICAgdXNlZEZvbnRzLmFkZCh0aGlzLmZvbnQuZmFtaWx5LCB0aGlzLmFycmF5T2ZTdHlsZXNUb1N0cmluZyh0aGlzLmZvbnQuc3R5bGUpLCB0aGlzLmZvbnQuc2l6ZSk7XHJcbiAgICB1c2VkRm9udHMuYWRkKFxyXG4gICAgICB0aGlzLmRlc2NyaXB0aW9uLmZvbnQuZmFtaWx5LFxyXG4gICAgICB0aGlzLmFycmF5T2ZTdHlsZXNUb1N0cmluZyh0aGlzLmRlc2NyaXB0aW9uLmZvbnQuc3R5bGUpLFxyXG4gICAgICB0aGlzLmRlc2NyaXB0aW9uLmZvbnQuc2l6ZVxyXG4gICAgKTtcclxuICB9XHJcbiAgaGF2ZUJhY2tnb3VuZENvbG9yKCk6IGJvb2xlYW4ge1xyXG4gICAgcmV0dXJuIHRydWU7XHJcbiAgfVxyXG5cclxuICBodG1sU3R5bGUoKTogc3RyaW5nIHtcclxuICAgIHJldHVybiBgJHt0aGlzLmZvbnQuaHRtbFN0eWxlKCl9O2NvbG9yOiR7dGhpcy5lQ29sb3IudG9IdG1sQ29sb3IoKX07dGV4dC1hbGlnbjoke0VudW0uZ2V0TmFtZShcclxuICAgICAgU3RpSG9yaXpvbnRhbEFsaWdubWVudCxcclxuICAgICAgdGhpcy5jb250ZW50QWxpZ25tZW50XHJcbiAgICApfWA7XHJcbiAgfVxyXG5cclxuICBzYXZlVG9Kc29uT2JqZWN0KG9wdGlvbnM6IGFueSk6IFN0aUpzb24ge1xyXG4gICAgaWYgKHRoaXMuZWxlbWVudD8ubmF0aXZlRWxlbWVudCkge1xyXG4gICAgICB0aGlzLmhlaWdodEZvckV4cG9ydCA9IHRoaXMuZWxlbWVudC5uYXRpdmVFbGVtZW50Lm9mZnNldEhlaWdodDtcclxuICAgIH1cclxuICAgIGxldCBqT2JqZWN0ID0gc3VwZXIuc2F2ZVRvSnNvbk9iamVjdChvcHRpb25zKTtcclxuICAgIGpPYmplY3QuYWRkUHJvcGVydHlKT2JqZWN0KFwiTGFiZWxcIiwgdGhpcy5sYWJlbC5zYXZlVG9Kc29uT2JqZWN0KG9wdGlvbnMpKTtcclxuICAgIGpPYmplY3QuYWRkUHJvcGVydHlKT2JqZWN0KFwiRGVzY3JpcHRpb25cIiwgdGhpcy5kZXNjcmlwdGlvbi5zYXZlVG9Kc29uT2JqZWN0KG9wdGlvbnMpKTtcclxuICAgIGpPYmplY3QuYWRkUHJvcGVydHlKT2JqZWN0KFwiRm9udFwiLCB0aGlzLmZvbnQuc2F2ZVRvSnNvbk9iamVjdChvcHRpb25zKSk7XHJcbiAgICBqT2JqZWN0LmFkZFByb3BlcnR5U3RyaW5nKFwiVGV4dFwiLCB0aGlzLnRleHQpO1xyXG4gICAgak9iamVjdC5hZGRQcm9wZXJ0eUNvbG9yKFwiQmFja2dyb3VuZENvbG9yXCIsIHRoaXMuYmFja2dyb3VuZENvbG9yKTtcclxuICAgIGpPYmplY3QuYWRkUHJvcGVydHlDb2xvcihcIlRleHRDb2xvclwiLCB0aGlzLnRleHRDb2xvcik7XHJcbiAgICBqT2JqZWN0LmFkZFByb3BlcnR5RW51bShcIkNvbnRlbnRBbGlnbm1lbnRcIiwgU3RpSG9yaXpvbnRhbEFsaWdubWVudCwgdGhpcy5jb250ZW50QWxpZ25tZW50KTtcclxuICAgIGpPYmplY3QuYWRkUHJvcGVydHlTdHJpbmcoXCJUb29sdGlwXCIsIHRoaXMudG9vbHRpcCk7XHJcbiAgICBqT2JqZWN0LmFkZFByb3BlcnR5Qm9vbChcIlJlcXVpcmVkXCIsIHRoaXMucmVxdWlyZWQpO1xyXG4gICAgak9iamVjdC5hZGRQcm9wZXJ0eUJvb2woXCJJc1Bhc3N3b3JkXCIsIHRoaXMuaXNQYXNzd29yZCk7XHJcbiAgICBqT2JqZWN0LmFkZFByb3BlcnR5Qm9vbChcIklzTXVsdGlsaW5lXCIsIHRoaXMuaXNNdWx0aWxpbmUpO1xyXG4gICAgak9iamVjdC5hZGRQcm9wZXJ0eU51bWJlcihcIkxpbmVzQ291bnRcIiwgdGhpcy5saW5lc0NvdW50KTtcclxuICAgIGpPYmplY3QuYWRkUHJvcGVydHlCb29sKFwiVXNlUmFuZ2VcIiwgdGhpcy51c2VSYW5nZSk7XHJcbiAgICBqT2JqZWN0LmFkZFByb3BlcnR5TnVtYmVyKFwiTWluaW11bVwiLCB0aGlzLm1pbmltdW0pO1xyXG4gICAgak9iamVjdC5hZGRQcm9wZXJ0eU51bWJlcihcIk1heGltdW1cIiwgdGhpcy5tYXhpbXVtKTtcclxuICAgIGpPYmplY3QuYWRkUHJvcGVydHlFbnVtKFwiUmFuZ2VGb3JtYXRcIiwgU3RpVGV4dFJhbmdlRm9ybWF0LCB0aGlzLnJhbmdlRm9ybWF0KTtcclxuICAgIGpPYmplY3QuYWRkUHJvcGVydHlDb2xvcihcIkJvcmRlckNvbG9yXCIsIHRoaXMuYm9yZGVyQ29sb3IpO1xyXG4gICAgak9iamVjdC5hZGRQcm9wZXJ0eUVudW0oXCJCb3JkZXJXZWlnaHRcIiwgU3RpTGluZVdlaWdodCwgdGhpcy5ib3JkZXJXZWlnaHQpO1xyXG4gICAgaWYgKHRoaXMuZWxlbWVudD8ubmF0aXZlRWxlbWVudCkge1xyXG4gICAgICBqT2JqZWN0LmFkZFByb3BlcnR5TnVtYmVyKFwiSGVpZ2h0Rm9yRXhwb3J0XCIsIHRoaXMuaGVpZ2h0Rm9yRXhwb3J0KTtcclxuICAgIH1cclxuICAgIHJldHVybiBqT2JqZWN0O1xyXG4gIH1cclxuXHJcbiAgcHVibGljIGxvYWRGcm9tSnNvbk9iamVjdChqc29uOiBTdGlKc29uKSB7XHJcbiAgICBzdXBlci5sb2FkRnJvbUpzb25PYmplY3QoanNvbik7XHJcbiAgICBmb3IgKGxldCBwcm9wZXJ0eSBvZiBqc29uLnByb3BlcnRpZXMoKSkge1xyXG4gICAgICBzd2l0Y2ggKHByb3BlcnR5Lm5hbWUpIHtcclxuICAgICAgICBjYXNlIFwiQ29udGVudEFsaWdubWVudFwiOlxyXG4gICAgICAgICAgdGhpcy5jb250ZW50QWxpZ25tZW50ID0gRW51bS5wYXJzZShTdGlIb3Jpem9udGFsQWxpZ25tZW50LCBwcm9wZXJ0eS52YWx1ZSk7XHJcbiAgICAgICAgICBicmVhaztcclxuICAgICAgICBjYXNlIFwiQmFja2dyb3VuZENvbG9yXCI6XHJcbiAgICAgICAgICB0aGlzLmJhY2tncm91bmRDb2xvciA9IG5ldyBDb2xvcigpO1xyXG4gICAgICAgICAgdGhpcy5iYWNrZ3JvdW5kQ29sb3IuZGVzZXJpYWxpemVGcm9tU3RyaW5nKHByb3BlcnR5LnZhbHVlKTtcclxuICAgICAgICAgIGJyZWFrO1xyXG4gICAgICAgIGNhc2UgXCJUZXh0Q29sb3JcIjpcclxuICAgICAgICAgIHRoaXMudGV4dENvbG9yID0gbmV3IENvbG9yKCk7XHJcbiAgICAgICAgICB0aGlzLnRleHRDb2xvci5kZXNlcmlhbGl6ZUZyb21TdHJpbmcocHJvcGVydHkudmFsdWUpO1xyXG4gICAgICAgICAgYnJlYWs7XHJcbiAgICAgICAgY2FzZSBcIlRleHRcIjpcclxuICAgICAgICAgIHRoaXMudGV4dCA9IHByb3BlcnR5LnZhbHVlO1xyXG4gICAgICAgICAgYnJlYWs7XHJcbiAgICAgICAgY2FzZSBcIkxhYmVsXCI6XHJcbiAgICAgICAgICB0aGlzLmxhYmVsID0gbmV3IFN0aUxhYmVsSXRlbSh0aGlzKTtcclxuICAgICAgICAgIHRoaXMubGFiZWwubG9hZEZyb21Kc29uT2JqZWN0KHByb3BlcnR5LnZhbHVlKTtcclxuICAgICAgICAgIGJyZWFrO1xyXG4gICAgICAgIGNhc2UgXCJGb250XCI6XHJcbiAgICAgICAgICB0aGlzLmZvbnQgPSBuZXcgRm9udCgpO1xyXG4gICAgICAgICAgdGhpcy5mb250LmxvYWRGcm9tSnNvbk9iamVjdChwcm9wZXJ0eS52YWx1ZSk7XHJcbiAgICAgICAgICBicmVhaztcclxuICAgICAgICBjYXNlIFwiRGVzY3JpcHRpb25cIjpcclxuICAgICAgICAgIHRoaXMuZGVzY3JpcHRpb24gPSBuZXcgU3RpVGV4dEl0ZW0oKTtcclxuICAgICAgICAgIHRoaXMuZGVzY3JpcHRpb24ubG9hZEZyb21Kc29uT2JqZWN0KHByb3BlcnR5LnZhbHVlKTtcclxuICAgICAgICAgIGJyZWFrO1xyXG4gICAgICAgIGNhc2UgXCJUb29sdGlwXCI6XHJcbiAgICAgICAgICB0aGlzLnRvb2x0aXAgPSBwcm9wZXJ0eS52YWx1ZTtcclxuICAgICAgICAgIGJyZWFrO1xyXG4gICAgICAgIGNhc2UgXCJSZXF1aXJlZFwiOlxyXG4gICAgICAgICAgdGhpcy5yZXF1aXJlZCA9IHByb3BlcnR5LnZhbHVlO1xyXG4gICAgICAgICAgYnJlYWs7XHJcbiAgICAgICAgY2FzZSBcIklzUGFzc3dvcmRcIjpcclxuICAgICAgICAgIHRoaXMuaXNQYXNzd29yZCA9IHByb3BlcnR5LnZhbHVlO1xyXG4gICAgICAgICAgYnJlYWs7XHJcbiAgICAgICAgY2FzZSBcIklzTXVsdGlsaW5lXCI6XHJcbiAgICAgICAgICB0aGlzLmlzTXVsdGlsaW5lID0gcHJvcGVydHkudmFsdWU7XHJcbiAgICAgICAgICBicmVhaztcclxuICAgICAgICBjYXNlIFwiTGluZXNDb3VudFwiOlxyXG4gICAgICAgICAgdGhpcy5saW5lc0NvdW50ID0gcHJvcGVydHkudmFsdWU7XHJcbiAgICAgICAgICBicmVhaztcclxuICAgICAgICBjYXNlIFwiVXNlUmFuZ2VcIjpcclxuICAgICAgICAgIHRoaXMudXNlUmFuZ2UgPSBwcm9wZXJ0eS52YWx1ZTtcclxuICAgICAgICAgIGJyZWFrO1xyXG4gICAgICAgIGNhc2UgXCJNaW5pbXVtXCI6XHJcbiAgICAgICAgICB0aGlzLm1pbmltdW0gPSBwcm9wZXJ0eS52YWx1ZTtcclxuICAgICAgICAgIGJyZWFrO1xyXG4gICAgICAgIGNhc2UgXCJNYXhpbXVtXCI6XHJcbiAgICAgICAgICB0aGlzLm1heGltdW0gPSBwcm9wZXJ0eS52YWx1ZTtcclxuICAgICAgICAgIGJyZWFrO1xyXG4gICAgICAgIGNhc2UgXCJSYW5nZUZvcm1hdFwiOlxyXG4gICAgICAgICAgdGhpcy5yYW5nZUZvcm1hdCA9IEVudW0ucGFyc2UoU3RpVGV4dFJhbmdlRm9ybWF0LCBwcm9wZXJ0eS52YWx1ZSk7XHJcbiAgICAgICAgICBicmVhaztcclxuICAgICAgICBjYXNlIFwiQm9yZGVyQ29sb3JcIjpcclxuICAgICAgICAgIHRoaXMuYm9yZGVyQ29sb3IgPSBuZXcgQ29sb3IoKTtcclxuICAgICAgICAgIHRoaXMuYm9yZGVyQ29sb3IuZGVzZXJpYWxpemVGcm9tU3RyaW5nKHByb3BlcnR5LnZhbHVlKTtcclxuICAgICAgICAgIGJyZWFrO1xyXG4gICAgICAgIGNhc2UgXCJCb3JkZXJXZWlnaHRcIjpcclxuICAgICAgICAgIHRoaXMuYm9yZGVyV2VpZ2h0ID0gRW51bS5wYXJzZShTdGlMaW5lV2VpZ2h0LCBwcm9wZXJ0eS52YWx1ZSk7XHJcbiAgICAgICAgICBicmVhaztcclxuICAgICAgfVxyXG4gICAgfVxyXG4gIH1cclxufVxyXG4iXX0=