import { StiInputTextItem } from "../helper/StiInputTextItem";
import StiRectangleGeometry from "../properties/StiRectangleGeometry";
import { StiLocalizationService } from "../services/sti-localization.service";
import Color from "../system/Color";
import { Font } from "../system/Font";
import { StiFormElement } from "./base/StiFormElement";
import { StiLabelItem } from "./text/StiLabelItem";
import { StiOptionItem } from "./text/StiOptionItem";
import { StiTextItem } from "./text/StiTextItem";
export class StiSingleSelectionElement extends StiFormElement {
    constructor() {
        super();
        this.type = "Single_SelectionElement";
        this.tooltip = "";
        this.required = false;
        this.description = new StiTextItem();
        this.font = new Font("Segoe UI", 10);
        this._columns = 1;
        this.backgroundColor = Color.white();
        this.textColor = Color.black();
        this.checkColor = Color.black();
        this.borderColor = Color.lightGray();
        this.prefix = "";
        this.suffix = "";
        this.optionsContentEditable = [];
        this.useCustomOption = false;
        this.customOptionText = "";
        this.selectedOption = "";
        this.geometry = new StiRectangleGeometry(150, 80);
        this.label = new StiLabelItem(this, StiLocalizationService.get("Components", "StiSingleSelection"));
        this.options = [];
        this.options.push(StiLocalizationService.get("PropertyMain", "First"));
        this.options.push(StiLocalizationService.get("PropertyMain", "Second"));
        this.options.push(StiLocalizationService.get("PropertyMain", "Third"));
    }
    setDefaultValues() {
        this.selectedOption = "";
    }
    setTabIndex(value) {
        this.tabIndex = value;
        return value + (this.options?.length ?? 0);
    }
    get columns() {
        return this._columns;
    }
    set columns(value) {
        if (value <= 1) {
            this._columns = 1;
        }
        else {
            this._columns = value;
        }
    }
    set options(value) {
        this.nativeOptions = value;
        let this_ = this;
        this._options = new Proxy(value, {
            deleteProperty: function (target, property) {
                setTimeout(() => {
                    this_.updateOptionsContentEditable();
                }, 0);
                return true;
            },
            set: function (target, property, value, receiver) {
                target[property] = value;
                this_.updateOptionsContentEditable();
                return true;
            },
        });
        this.updateOptionsContentEditable();
    }
    getUsedFonts(usedFonts) {
        super.getUsedFonts(usedFonts);
        usedFonts.add(this.font.family, this.arrayOfStylesToString(this.font.style), this.font.size);
        usedFonts.add(this.description.font.family, this.arrayOfStylesToString(this.description.font.style), this.description.font.size);
    }
    updateOptionsContentEditable() {
        this.optionsContentEditable = [];
        this.options.forEach((l, index) => this.optionsContentEditable.push(new StiOptionItem(l, this, index)));
    }
    getDefaultEditableContent() {
        let textItem = new StiInputTextItem("", this.font, this.textColor);
        textItem.element = this;
        return textItem;
    }
    get value() {
        return this.selectedOption;
    }
    get options() {
        return this._options;
    }
    htmlStyle() {
        return `${this.font.htmlStyle()};color:${this.textColor.toHtmlColor()};`;
    }
    saveToJsonObject(options) {
        let jObject = super.saveToJsonObject(options);
        jObject.addPropertyJObject("Label", this.label.saveToJsonObject(options));
        jObject.addPropertyJObject("Description", this.description.saveToJsonObject(options));
        jObject.addPropertyString("Tooltip", this.tooltip);
        jObject.addPropertyBool("Required", this.required);
        jObject.addPropertyJObject("Font", this.font.saveToJsonObject(options));
        jObject.addPropertyNumber("Columns", this.columns);
        jObject.addPropertyColor("BackgroundColor", this.backgroundColor);
        jObject.addPropertyColor("TextColor", this.textColor);
        jObject.addPropertyColor("CheckColor", this.checkColor);
        jObject.addPropertyColor("BorderColor", this.borderColor);
        jObject.addPropertyArray("Options", this.options);
        jObject.addPropertyBool("UseCustomOption", this.useCustomOption);
        jObject.addPropertyString("CustomOptionText", this.customOptionText);
        jObject.addPropertyString("SelectedOption", this.selectedOption);
        jObject.addPropertyString("Prefix", this.prefix);
        jObject.addPropertyString("Suffix", this.suffix);
        return jObject;
    }
    loadFromJsonObject(json) {
        super.loadFromJsonObject(json);
        for (let property of json.properties()) {
            switch (property.name) {
                case "BackgroundColor":
                    this.backgroundColor = new Color();
                    this.backgroundColor.deserializeFromString(property.value);
                    break;
                case "TextColor":
                    this.textColor = new Color();
                    this.textColor.deserializeFromString(property.value);
                    break;
                case "Label":
                    this.label = new StiLabelItem(this);
                    this.label.loadFromJsonObject(property.value);
                    break;
                case "Font":
                    this.font = new Font();
                    this.font.loadFromJsonObject(property.value);
                    break;
                case "Description":
                    this.description = new StiTextItem();
                    this.description.loadFromJsonObject(property.value);
                    break;
                case "Tooltip":
                    this.tooltip = property.value;
                    break;
                case "Required":
                    this.required = property.value;
                    break;
                case "Columns":
                    this.columns = property.value;
                    break;
                case "BorderColor":
                    this.borderColor = new Color();
                    this.borderColor.deserializeFromString(property.value);
                    break;
                case "CheckColor":
                    this.checkColor = new Color();
                    this.checkColor.deserializeFromString(property.value);
                    break;
                case "CustomOptionText":
                    this.customOptionText = property.value;
                    break;
                case "UseCustomOption":
                    this.useCustomOption = property.value;
                    break;
                case "SelectedOption":
                    this.selectedOption = property.value;
                    break;
                case "Prefix":
                    this.prefix = property.value;
                    break;
                case "Suffix":
                    this.suffix = property.value;
                    break;
                case "Options":
                    this.options = [];
                    property.value.properties().forEach((p) => this.options.push(p.value));
                    break;
            }
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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