import StiRectangleGeometry from "../properties/StiRectangleGeometry";
import { StiLocalizationService } from "../services/sti-localization.service";
import Color from "../system/Color";
import { StiHorizontalAlignment } from "../system/enums";
import { StiFormElement } from "./base/StiFormElement";
import { StiImageEditorResult } from "./signature/StiImageEditorResult";
import { StiLabelItem } from "./text/StiLabelItem";
import { StiTextItem } from "./text/StiTextItem";
export class StiSignatureElement extends StiFormElement {
    constructor() {
        super();
        this.type = "SignatureElement";
        this.required = false;
        this.description = new StiTextItem();
        this.signaturePanelWidth = 185;
        this.signaturePanelHeight = 100;
        this.insertedImage = new StiImageEditorResult();
        this.insertedText = new StiTextItem();
        this.penColor = Color.black();
        this.fullName = "";
        this.initials = "";
        this.specialFont = "Teddy Bear";
        this.currentSignatureMode = "Type";
        this.canvasHeight = 120;
        this.isSigned = false;
        this.geometry = new StiRectangleGeometry(180, 145);
        this.label = new StiLabelItem(this, StiLocalizationService.get("Components", "StiElectronicSignature"));
        this.description.value = "Signature";
        this.insertedText.font.size = 20;
        this.insertedText.horizontalAlignment = StiHorizontalAlignment.Center;
    }
    get value() {
        return this.backgroundImage;
    }
    saveToJsonObject(options) {
        if (this.element) {
            if (this.element.nativeElement?.offsetWidth > 0 && this.element.nativeElement?.offsetHeight > 0) {
                this.signaturePanelWidth = this.element.nativeElement?.offsetWidth;
                this.signaturePanelHeight = this.element.nativeElement?.offsetHeight;
            }
        }
        let jObject = super.saveToJsonObject(options);
        jObject.addPropertyJObject("Label", this.label.saveToJsonObject(options));
        jObject.addPropertyJObject("Description", this.description.saveToJsonObject(options));
        jObject.addPropertyNumber("SignaturePanelWidth", this.signaturePanelWidth);
        jObject.addPropertyNumber("SignaturePanelHeight", this.signaturePanelHeight);
        jObject.addPropertyColor("PenColor", this.penColor);
        jObject.addPropertyJObject("InsertedImage", this.insertedImage?.saveToJsonObject(options));
        jObject.addPropertyJObject("InsertedText", this.insertedText?.saveToJsonObject(options));
        if (!(options && options["notSerializeImages"])) {
            jObject.addPropertyStringNullOrEmpty("DrawedImage", this.drawedImage?.substring(this.drawedImage.indexOf("base64,") + 7));
        }
        jObject.addPropertyStringNullOrEmpty("FullName", this.fullName);
        jObject.addPropertyStringNullOrEmpty("Initials", this.initials);
        jObject.addPropertyStringNullOrEmpty("SpecialFont", this.specialFont);
        jObject.addPropertyStringNullOrEmpty("CurrentSignatureMode", this.currentSignatureMode);
        if (!(options && options["notSerializeImages"])) {
            jObject.addPropertyStringNullOrEmpty("BackgroundImage", this.backgroundImage?.substring(this.backgroundImage.indexOf("base64,") + 7));
        }
        return jObject;
    }
    loadFromJsonObject(json) {
        super.loadFromJsonObject(json);
        for (let property of json.properties()) {
            switch (property.name) {
                case "Label":
                    this.label = new StiLabelItem(this);
                    this.label.loadFromJsonObject(property.value);
                    break;
                case "Description":
                    this.description = new StiTextItem();
                    this.description.loadFromJsonObject(property.value);
                    break;
                case "SignaturePanelWidth":
                    this.signaturePanelWidth = property.value;
                    break;
                case "PenColor":
                    this.penColor = new Color();
                    this.penColor.deserializeFromString(property.value);
                    break;
                case "SignaturePanelHeight":
                    this.signaturePanelHeight = property.value;
                    break;
                case "InsertedImage":
                    this.insertedImage = new StiImageEditorResult();
                    this.insertedImage.loadFromJsonObject(property.value);
                    break;
                case "InsertedText":
                    this.insertedText = new StiTextItem();
                    this.insertedText.loadFromJsonObject(property.value);
                    break;
                case "DrawedImage":
                    this.drawedImage = `data:image/png;base64,${property.value}`;
                    break;
                case "FullName":
                    this.fullName = property.value;
                    break;
                case "Initials":
                    this.initials = property.value;
                    break;
                case "SpecialFont":
                    this.specialFont = property.value;
                    break;
                case "CurrentSignatureMode":
                    this.currentSignatureMode = property.value;
                    break;
                case "BackgroundImage":
                    this.backgroundImage = `data:image/png;base64,${property.value}`;
                    break;
            }
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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