import { StiInputTextItem } from "../helper/StiInputTextItem";
import StiRectangleGeometry from "../properties/StiRectangleGeometry";
import { StiLocalizationService } from "../services/sti-localization.service";
import Color from "../system/Color";
import { Enum } from "../system/Enum";
import { StiHorizontalAlignment, StiNumberType, StiThousandSeparator, StiNumberRangeFormat, StiUnitAlignment, StiDisplayCurrencyAs, StiLineWeight, } from "../system/enums";
import { Font } from "../system/Font";
import { StiFormElement } from "./base/StiFormElement";
import { StiLabelItem } from "./text/StiLabelItem";
import { StiTextItem } from "./text/StiTextItem";
export class StiNumberBoxElement extends StiFormElement {
    constructor() {
        super();
        this._decimalDigits = 2;
        this._maximum = 15;
        this.type = "Number_BoxElement";
        this.tooltip = "";
        this.required = false;
        this.description = new StiTextItem();
        this.font = new Font("Segoe UI", 10);
        this.contentAlignment = StiHorizontalAlignment.Right;
        this.backgroundColor = Color.white();
        this.textColor = Color.black();
        this.value = 0;
        this.allowNegative = true;
        this.numberType = StiNumberType.Decimal;
        this.thousandSeparator = StiThousandSeparator.None;
        this.useRange = false;
        this._minimum = 0;
        this.rangeFormat = StiNumberRangeFormat.Digits;
        this.unitLabel = "";
        this.unitAlignment = StiUnitAlignment.Suffix;
        this.unitForeground = Color.black();
        this.currencyLabelFormat = StiDisplayCurrencyAs.Code;
        this.currency = "USD";
        this.borderColor = Color.lightGray();
        this.borderWeight = StiLineWeight.Thin;
        this.geometry = new StiRectangleGeometry(150, 40);
        this.label = new StiLabelItem(this, StiLocalizationService.get("Components", "StiNumberBox"));
    }
    setDefaultValues() {
        this.value = 0;
    }
    setTabIndex(value) {
        this.tabIndex = value++;
        return value;
    }
    getDefaultEditableContent() {
        let textItem = new StiInputTextItem(this.value?.toString(), this.font, this.textColor, this.backgroundColor, this.contentAlignment);
        textItem.element = this;
        return textItem;
    }
    get isCurrency() {
        return this.numberType == StiNumberType.Currency ? true : false;
    }
    get eValue() {
        return this.value.toString();
    }
    get decimalDigits() {
        return this._decimalDigits;
    }
    set decimalDigits(value) {
        this._decimalDigits = Math.max(0, Math.min(10, value));
    }
    get maximum() {
        return this._maximum;
    }
    set maximum(value) {
        this._maximum = value <= this.minimum ? this.minimum : value;
    }
    get minimum() {
        return this._minimum;
    }
    set minimum(value) {
        this._minimum = value >= this.maximum ? this.maximum : value;
    }
    set eValue(value) {
        try {
            this.value = parseFloat(value);
        }
        catch {
            this.value = 0;
        }
    }
    get eFont() {
        return this.font;
    }
    set eFont(value) {
        this.font = value;
    }
    get eBackgroundColor() {
        return this.backgroundColor;
    }
    set eBackgroundColor(value) {
        this.backgroundColor = value;
    }
    get eContentAlignment() {
        return this.contentAlignment;
    }
    set eContentAlignment(value) {
        this.contentAlignment = value;
    }
    haveContentAlignment() {
        return true;
    }
    haveVerticalAlignment() {
        return false;
    }
    getUsedFonts(usedFonts) {
        super.getUsedFonts(usedFonts);
        usedFonts.add(this.font.family, this.arrayOfStylesToString(this.font.style), this.font.size);
        usedFonts.add(this.description.font.family, this.arrayOfStylesToString(this.description.font.style), this.description.font.size);
    }
    haveBackgoundColor() {
        return true;
    }
    get eColor() {
        return this.textColor;
    }
    set eColor(value) {
        this.textColor = value;
    }
    htmlStyle() {
        return `${this.font.htmlStyle()};color:${this.textColor.toHtmlColor()};text-align:${Enum.getName(StiHorizontalAlignment, this.contentAlignment)}`;
    }
    setProperty(propName, value, oldObject) {
        super.setProperty(propName, value, oldObject);
        if (propName == "thousandSeparator" && (oldObject === null || oldObject == this.thousandSeparator)) {
            this.thousandSeparator = value;
        }
        else if (propName == "currencyFormat" && (oldObject === null || oldObject == this.currencyLabelFormat)) {
            this.currencyLabelFormat = value;
        }
    }
    saveToJsonObject(options) {
        let jObject = super.saveToJsonObject(options);
        jObject.addPropertyJObject("Label", this.label.saveToJsonObject(options));
        jObject.addPropertyJObject("Description", this.description.saveToJsonObject(options));
        jObject.addPropertyString("Tooltip", this.tooltip);
        jObject.addPropertyBool("Required", this.required);
        jObject.addPropertyJObject("Font", this.font.saveToJsonObject(options));
        jObject.addPropertyNumber("Value", this.value);
        jObject.addPropertyBool("AllowNegative", this.allowNegative);
        jObject.addPropertyEnum("NumberType", StiNumberType, this.numberType);
        jObject.addPropertyEnum("ThousandSeparator", StiThousandSeparator, this.thousandSeparator);
        jObject.addPropertyNumber("DecimalDigits", this.decimalDigits);
        jObject.addPropertyEnum("ContentAlignment", StiHorizontalAlignment, this.contentAlignment);
        jObject.addPropertyBool("UseRange", this.useRange);
        jObject.addPropertyNumber("Minimum", this.minimum);
        jObject.addPropertyNumber("Maximum", this.maximum);
        jObject.addPropertyEnum("RangeFormat", StiNumberRangeFormat, this.rangeFormat);
        jObject.addPropertyString("UnitLabel", this.unitLabel);
        jObject.addPropertyEnum("UnitAlignment", StiUnitAlignment, this.unitAlignment);
        jObject.addPropertyColor("UnitForeground", this.unitForeground);
        jObject.addPropertyEnum("CurrencyLabelFormat", StiDisplayCurrencyAs, this.currencyLabelFormat);
        jObject.addPropertyString("Currency", this.currency);
        jObject.addPropertyColor("BackgroundColor", this.backgroundColor);
        jObject.addPropertyColor("TextColor", this.textColor);
        jObject.addPropertyColor("BorderColor", this.borderColor);
        jObject.addPropertyEnum("BorderWeight", StiLineWeight, this.borderWeight);
        return jObject;
    }
    loadFromJsonObject(json) {
        super.loadFromJsonObject(json);
        for (let property of json.properties()) {
            switch (property.name) {
                case "Label":
                    this.label = new StiLabelItem(this);
                    this.label.loadFromJsonObject(property.value);
                    break;
                case "Description":
                    this.description = new StiTextItem();
                    this.description.loadFromJsonObject(property.value);
                    break;
                case "Tooltip":
                    this.tooltip = property.value;
                    break;
                case "Required":
                    this.required = property.value;
                    break;
                case "Font":
                    this.font = new Font();
                    this.font.loadFromJsonObject(property.value);
                    break;
                case "Value":
                    this.value = property.value;
                    break;
                case "AllowNegative":
                    this.allowNegative = property.value;
                    break;
                case "NumberType":
                    this.numberType = Enum.parse(StiNumberType, property.value);
                    break;
                case "ThousandSeparator":
                    this.thousandSeparator = Enum.parse(StiThousandSeparator, property.value);
                    break;
                case "DecimalDigits":
                    this.decimalDigits = property.value;
                    break;
                case "ContentAlignment":
                    this.contentAlignment = Enum.parse(StiHorizontalAlignment, property.value);
                    break;
                case "UseRange":
                    this.useRange = property.value;
                    break;
                case "Minimum":
                    this.minimum = property.value;
                    break;
                case "Maximum":
                    this.maximum = property.value;
                    break;
                case "RangeFormat":
                    this.rangeFormat = Enum.parse(StiNumberRangeFormat, property.value);
                    break;
                case "UnitLabel":
                    this.unitLabel = property.value;
                    break;
                case "UnitAlignment":
                    this.unitAlignment = Enum.parse(StiUnitAlignment, property.value);
                    break;
                case "UnitForeground":
                    this.unitForeground = new Color();
                    this.unitForeground.deserializeFromString(property.value);
                    break;
                case "CurrencyLabelFormat":
                    this.currencyLabelFormat = Enum.parse(StiDisplayCurrencyAs, property.value);
                    break;
                case "Currency":
                    this.currency = property.value;
                    break;
                case "BackgroundColor":
                    this.backgroundColor = new Color();
                    this.backgroundColor.deserializeFromString(property.value);
                    break;
                case "TextColor":
                    this.textColor = new Color();
                    this.textColor.deserializeFromString(property.value);
                    break;
                case "BorderColor":
                    this.borderColor = new Color();
                    this.borderColor.deserializeFromString(property.value);
                    break;
                case "BorderWeight":
                    this.borderWeight = Enum.parse(StiLineWeight, property.value);
                    break;
            }
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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