import { StiInputTextItem } from "../helper/StiInputTextItem";
import StiRectangleGeometry from "../properties/StiRectangleGeometry";
import { StiLocalizationService } from "../services/sti-localization.service";
import Color from "../system/Color";
import { Font } from "../system/Font";
import { StiFormElement } from "./base/StiFormElement";
import { StiLabelItem } from "./text/StiLabelItem";
import { StiOptionItem } from "./text/StiOptionItem";
import { StiTextItem } from "./text/StiTextItem";
export class StiMultipleSelectionElement extends StiFormElement {
    constructor() {
        super();
        this.type = "Multiple_SelectionElement";
        this.tooltip = "";
        this.required = false;
        this.description = new StiTextItem();
        this.font = new Font("Segoe UI", 10);
        this._columns = 1;
        this.backgroundColor = Color.white();
        this.textColor = Color.black();
        this.checkColor = Color.black();
        this.borderColor = Color.lightGray();
        this.prefix = "";
        this.suffix = "";
        this.optionsContentEditable = [];
        this.useCustomOption = false;
        this.customOptionText = "";
        this.selectedOptions = [];
        this.geometry = new StiRectangleGeometry(150, 80);
        this.label = new StiLabelItem(this, StiLocalizationService.get("Components", "StiMultipleSelection"));
        this.options = [];
        this.options.push(StiLocalizationService.get("PropertyMain", "First"));
        this.options.push(StiLocalizationService.get("PropertyMain", "Second"));
        this.options.push(StiLocalizationService.get("PropertyMain", "Third"));
    }
    setDefaultValues() {
        this.selectedOptions = [];
    }
    setTabIndex(value) {
        this.tabIndex = value++;
        return value;
    }
    set options(value) {
        this.nativeOptions = value;
        let this_ = this;
        this._options = new Proxy(value, {
            deleteProperty: function (target, property) {
                setTimeout(() => {
                    this_.updateOptionsContentEditable();
                }, 0);
                return true;
            },
            set: function (target, property, value, receiver) {
                target[property] = value;
                this_.updateOptionsContentEditable();
                return true;
            },
        });
        this.updateOptionsContentEditable();
    }
    updateOptionsContentEditable() {
        this.optionsContentEditable = [];
        this.options.forEach((l, index) => this.optionsContentEditable.push(new StiOptionItem(l, this, index)));
    }
    getDefaultEditableContent() {
        let textItem = new StiInputTextItem("", this.font, this.textColor);
        textItem.element = this;
        return textItem;
    }
    get options() {
        return this._options;
    }
    get value() {
        if (this.selectedOptions) {
            return this.selectedOptions.length == 1 ? this.selectedOptions[0] : this.selectedOptions;
        }
        return null;
    }
    get columns() {
        return this._columns;
    }
    set columns(value) {
        if (value <= 1) {
            this._columns = 1;
        }
        else {
            this._columns = value;
        }
    }
    getUsedFonts(usedFonts) {
        super.getUsedFonts(usedFonts);
        usedFonts.add(this.font.family, this.arrayOfStylesToString(this.font.style), this.font.size);
        usedFonts.add(this.description.font.family, this.arrayOfStylesToString(this.description.font.style), this.description.font.size);
    }
    htmlStyle() {
        return `${this.font.htmlStyle()};color:${this.textColor.toHtmlColor()};`;
    }
    saveToJsonObject(options) {
        let jObject = super.saveToJsonObject(options);
        jObject.addPropertyJObject("Label", this.label.saveToJsonObject(options));
        jObject.addPropertyJObject("Description", this.description.saveToJsonObject(options));
        jObject.addPropertyString("Tooltip", this.tooltip);
        jObject.addPropertyBool("Required", this.required);
        jObject.addPropertyJObject("Font", this.font.saveToJsonObject(options));
        jObject.addPropertyNumber("Columns", this.columns);
        jObject.addPropertyColor("BackgroundColor", this.backgroundColor);
        jObject.addPropertyColor("TextColor", this.textColor);
        jObject.addPropertyColor("CheckColor", this.checkColor);
        jObject.addPropertyColor("BorderColor", this.borderColor);
        jObject.addPropertyArray("Options", this.options);
        jObject.addPropertyBool("UseCustomOption", this.useCustomOption);
        jObject.addPropertyString("CustomOptionText", this.customOptionText);
        jObject.addPropertyArray("SelectedOptions", this.selectedOptions);
        jObject.addPropertyString("Prefix", this.prefix);
        jObject.addPropertyString("Suffix", this.suffix);
        return jObject;
    }
    loadFromJsonObject(json) {
        super.loadFromJsonObject(json);
        for (let property of json.properties()) {
            switch (property.name) {
                case "BackgroundColor":
                    this.backgroundColor = new Color();
                    this.backgroundColor.deserializeFromString(property.value);
                    break;
                case "TextColor":
                    this.textColor = new Color();
                    this.textColor.deserializeFromString(property.value);
                    break;
                case "Label":
                    this.label = new StiLabelItem(this);
                    this.label.loadFromJsonObject(property.value);
                    break;
                case "Font":
                    this.font = new Font();
                    this.font.loadFromJsonObject(property.value);
                    break;
                case "Description":
                    this.description = new StiTextItem();
                    this.description.loadFromJsonObject(property.value);
                    break;
                case "Tooltip":
                    this.tooltip = property.value;
                    break;
                case "Required":
                    this.required = property.value;
                    break;
                case "Columns":
                    this.columns = property.value;
                    break;
                case "BorderColor":
                    this.borderColor = new Color();
                    this.borderColor.deserializeFromString(property.value);
                    break;
                case "CheckColor":
                    this.checkColor = new Color();
                    this.checkColor.deserializeFromString(property.value);
                    break;
                case "CustomOptionText":
                    this.customOptionText = property.value;
                    break;
                case "UseCustomOption":
                    this.useCustomOption = property.value;
                    break;
                case "Prefix":
                    this.prefix = property.value;
                    break;
                case "Suffix":
                    this.suffix = property.value;
                    break;
                case "SelectedOptions":
                    this.selectedOptions = [];
                    property.value.properties().forEach((p) => this.selectedOptions.push(p.value));
                    break;
                case "Options":
                    this.options = [];
                    property.value.properties().forEach((p) => this.options.push(p.value));
                    break;
            }
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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