import { StiInputTextItem } from "../helper/StiInputTextItem";
import StiRectangleGeometry from "../properties/StiRectangleGeometry";
import { StiLocalizationService } from "../services/sti-localization.service";
import Color from "../system/Color";
import { Enum } from "../system/Enum";
import { StiHorizontalAlignment, StiLineWeight, StiDateTimeBoxType, StiDateTimeBoxValueChoise } from "../system/enums";
import { Font } from "../system/Font";
import { StiFormElement } from "./base/StiFormElement";
import { StiLabelItem } from "./text/StiLabelItem";
import { StiTextItem } from "./text/StiTextItem";
export class StiDateTimeBoxElement extends StiFormElement {
    constructor() {
        super();
        this._dateTimeFormat = "m/d/yy h:MM tt";
        this._dateFomat = "dd.mm.yyyy";
        this._timeFormat = "HH:MM:ss";
        this.type = "DateTime_BoxElement";
        this.tooltip = "";
        this.required = false;
        this.description = new StiTextItem();
        this.font = new Font("Segoe UI", 10);
        this.contentAlignment = StiHorizontalAlignment.Left;
        this.backgroundColor = Color.white();
        this.textColor = Color.black();
        this.borderColor = Color.lightGray();
        this.borderWeight = StiLineWeight.Thin;
        this.valueType = StiDateTimeBoxType.Date;
        this.initialState = StiDateTimeBoxValueChoise.Custom;
        this.allowNull = false;
        this.dateTimeValueIsNotNull = true;
        this.dateValue = "";
        this.geometry = new StiRectangleGeometry(150, 40);
        this.label = new StiLabelItem(this, StiLocalizationService.get("Dialogs", "StiDateTimePickerControl"));
    }
    setDefaultValues() {
        this.initialState = StiDateTimeBoxValueChoise.Current;
    }
    get value() {
        return this.dateValue;
    }
    get dateTimeFormat() {
        return this._dateTimeFormat;
    }
    set dateTimeFormat(value) {
        this._dateTimeFormat = value;
    }
    get dateFormat() {
        return this._dateFomat;
    }
    set dateFormat(value) {
        this._dateFomat = value;
    }
    get timeFormat() {
        return this._timeFormat;
    }
    set timeFormat(value) {
        this._timeFormat = value;
    }
    setTabIndex(value) {
        this.tabIndex = value++;
        return value;
    }
    getUsedFonts(usedFonts) {
        super.getUsedFonts(usedFonts);
        usedFonts.add(this.font.family, this.arrayOfStylesToString(this.font.style), this.font.size);
        usedFonts.add(this.description.font.family, this.arrayOfStylesToString(this.description.font.style), this.description.font.size);
    }
    getDefaultEditableContent() {
        let textItem = new StiInputTextItem("", this.font, this.textColor, this.backgroundColor, this.contentAlignment);
        textItem.element = this;
        return textItem;
    }
    htmlStyle() {
        return `${this.font.htmlStyle()};color:${this.textColor.toHtmlColor()};text-align:${Enum.getName(StiHorizontalAlignment, this.contentAlignment)}`;
    }
    saveToJsonObject(options) {
        let jObject = super.saveToJsonObject(options);
        jObject.addPropertyJObject("Label", this.label.saveToJsonObject(options));
        jObject.addPropertyJObject("Description", this.description.saveToJsonObject(options));
        jObject.addPropertyJObject("Font", this.font.saveToJsonObject(options));
        jObject.addPropertyColor("BackgroundColor", this.backgroundColor);
        jObject.addPropertyColor("TextColor", this.textColor);
        jObject.addPropertyString("Tooltip", this.tooltip);
        jObject.addPropertyString("DateTimeFormat", this.dateTimeFormat);
        jObject.addPropertyString("DateFormat", this.dateFormat);
        jObject.addPropertyString("TimeFormat", this.timeFormat);
        jObject.addPropertyBool("Required", this.required);
        jObject.addPropertyColor("BorderColor", this.borderColor);
        jObject.addPropertyEnum("BorderWeight", StiLineWeight, this.borderWeight);
        jObject.addPropertyEnum("HorizontalAlignment", StiHorizontalAlignment, this.contentAlignment);
        jObject.addPropertyEnum("ValueType", StiDateTimeBoxType, this.valueType);
        jObject.addPropertyEnum("InitialState", StiDateTimeBoxValueChoise, this.initialState);
        jObject.addPropertyBool("AllowNull", this.allowNull);
        jObject.addPropertyBool("DateTimeValueIsNotNull", this.dateTimeValueIsNotNull);
        if (this.allowNull && !this.dateTimeValueIsNotNull) {
            jObject.addPropertyString("DateValue", null, "");
            jObject.addPropertyString("DateValueString", null, "");
        }
        else {
            jObject.addPropertyString("DateValue", new Date(this.customDate).toJSON());
            jObject.addPropertyString("DateValueString", this.dateValue);
        }
        return jObject;
    }
    loadFromJsonObject(json) {
        super.loadFromJsonObject(json);
        for (let property of json.properties()) {
            switch (property.name) {
                case "BackgroundColor":
                    this.backgroundColor = new Color();
                    this.backgroundColor.deserializeFromString(property.value);
                    break;
                case "TextColor":
                    this.textColor = new Color();
                    this.textColor.deserializeFromString(property.value);
                    break;
                case "Label":
                    this.label = new StiLabelItem(this);
                    this.label.loadFromJsonObject(property.value);
                    break;
                case "Font":
                    this.font = new Font();
                    this.font.loadFromJsonObject(property.value);
                    break;
                case "Description":
                    this.description = new StiTextItem();
                    this.description.loadFromJsonObject(property.value);
                    break;
                case "Tooltip":
                    this.tooltip = property.value;
                    break;
                case "DateTimeFormat":
                    this.dateTimeFormat = property.value;
                    break;
                case "DateFormat":
                    this.dateFormat = property.value;
                    break;
                case "TimeFormat":
                    this.timeFormat = property.value;
                    break;
                case "Required":
                    this.required = property.value;
                    break;
                case "BorderColor":
                    this.borderColor = new Color();
                    this.borderColor.deserializeFromString(property.value);
                    break;
                case "HorizontalAlignment":
                    this.contentAlignment = Enum.parse(StiHorizontalAlignment, property.value);
                    break;
                case "BorderWeight":
                    this.borderWeight = Enum.parse(StiLineWeight, property.value);
                    break;
                case "ValueType":
                    this.valueType = Enum.parse(StiDateTimeBoxType, property.value);
                    break;
                case "InitialState":
                    this.initialState = Enum.parse(StiDateTimeBoxValueChoise, property.value);
                    break;
                case "AllowNull":
                    this.allowNull = property.value;
                    break;
                case "DateTimeValueIsNotNull":
                    this.dateTimeValueIsNotNull = property.value;
                    break;
                case "DateValue":
                    if (property.value.length > 0) {
                        this.customDate = property.value;
                    }
                    else {
                        this.customDate = new Date();
                    }
                    break;
            }
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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