import { StiInputTextItem } from "../helper/StiInputTextItem";
import StiRectangleGeometry from "../properties/StiRectangleGeometry";
import { StiLocalizationService } from "../services/sti-localization.service";
import Color from "../system/Color";
import { Enum } from "../system/Enum";
import { StiLineWeight } from "../system/enums";
import { Font } from "../system/Font";
import { StiFormElement } from "./base/StiFormElement";
import { StiLabelItem } from "./text/StiLabelItem";
import { StiTextItem } from "./text/StiTextItem";
export class StiComboBoxElement extends StiFormElement {
    constructor() {
        super();
        this.type = "Combo_BoxElement";
        this.tooltip = "";
        this.required = false;
        this.description = new StiTextItem();
        this.font = new Font("Segoe UI", 10);
        this.backgroundColor = Color.white();
        this.textColor = Color.black();
        this.borderColor = Color.lightGray();
        this.borderWeight = StiLineWeight.Thin;
        this.editable = false;
        this.options = [];
        this.selectedOption = "";
        this.geometry = new StiRectangleGeometry(150, 40);
        this.label = new StiLabelItem(this, StiLocalizationService.get("Components", "StiComboBox"));
        this.options.push(StiLocalizationService.get("PropertyMain", "First"));
        this.options.push(StiLocalizationService.get("PropertyMain", "Second"));
        this.options.push(StiLocalizationService.get("PropertyMain", "Third"));
    }
    setDefaultValues() {
        this.selectedOption = "";
    }
    setTabIndex(value) {
        this.tabIndex = value++;
        return value;
    }
    getUsedFonts(usedFonts) {
        super.getUsedFonts(usedFonts);
        usedFonts.add(this.font.family, this.arrayOfStylesToString(this.font.style), this.font.size);
        usedFonts.add(this.description.font.family, this.arrayOfStylesToString(this.description.font.style), this.description.font.size);
    }
    getDefaultEditableContent() {
        let textItem = new StiInputTextItem(this.selectedOption, this.font, this.textColor, this.backgroundColor);
        textItem.element = this;
        return textItem;
    }
    get eFont() {
        return this.font;
    }
    set eFont(value) {
        this.font = value;
    }
    get eBackgroundColor() {
        return this.backgroundColor;
    }
    set eBackgroundColor(value) {
        this.backgroundColor = value;
    }
    get eColor() {
        return this.textColor;
    }
    set eColor(value) {
        this.textColor = value;
    }
    get value() {
        return this.selectedOption;
    }
    haveBackgoundColor() {
        return true;
    }
    haveContentAlignment() {
        return false;
    }
    haveVerticalAlignment() {
        return false;
    }
    htmlStyle() {
        return `${this.font.htmlStyle()};color:${this.eColor.toHtmlColor()}}`;
    }
    saveToJsonObject(options) {
        let jObject = super.saveToJsonObject(options);
        jObject.addPropertyJObject("Label", this.label.saveToJsonObject(options));
        jObject.addPropertyJObject("Description", this.description.saveToJsonObject(options));
        jObject.addPropertyJObject("Font", this.font.saveToJsonObject(options));
        jObject.addPropertyColor("BackgroundColor", this.backgroundColor);
        jObject.addPropertyColor("TextColor", this.textColor);
        jObject.addPropertyString("Tooltip", this.tooltip);
        jObject.addPropertyBool("Required", this.required);
        jObject.addPropertyColor("BorderColor", this.borderColor);
        jObject.addPropertyEnum("BorderWeight", StiLineWeight, this.borderWeight);
        jObject.addPropertyBool("Editable", this.editable);
        jObject.addPropertyString("SelectedOption", this.selectedOption);
        jObject.addPropertyArray("Options", this.options);
        return jObject;
    }
    loadFromJsonObject(json) {
        super.loadFromJsonObject(json);
        for (let property of json.properties()) {
            switch (property.name) {
                case "BackgroundColor":
                    this.backgroundColor = new Color();
                    this.backgroundColor.deserializeFromString(property.value);
                    break;
                case "TextColor":
                    this.textColor = new Color();
                    this.textColor.deserializeFromString(property.value);
                    break;
                case "Label":
                    this.label = new StiLabelItem(this);
                    this.label.loadFromJsonObject(property.value);
                    break;
                case "Font":
                    this.font = new Font();
                    this.font.loadFromJsonObject(property.value);
                    break;
                case "Description":
                    this.description = new StiTextItem();
                    this.description.loadFromJsonObject(property.value);
                    break;
                case "Tooltip":
                    this.tooltip = property.value;
                    break;
                case "Required":
                    this.required = property.value;
                    break;
                case "BorderColor":
                    this.borderColor = new Color();
                    this.borderColor.deserializeFromString(property.value);
                    break;
                case "BorderWeight":
                    this.borderWeight = Enum.parse(StiLineWeight, property.value);
                    break;
                case "Editable":
                    this.editable = property.value;
                    break;
                case "SelectedOption":
                    this.selectedOption = property.value;
                    break;
                case "Options":
                    this.options = [];
                    property.value.properties().forEach((p) => this.options.push(p.value));
                    break;
            }
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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