import { StiInputTextItem } from "../helper/StiInputTextItem";
import StiRectangleGeometry from "../properties/StiRectangleGeometry";
import { StiLocalizationService } from "../services/sti-localization.service";
import Color from "../system/Color";
import { Font } from "../system/Font";
import { StiFormElement } from "./base/StiFormElement";
import { StiLabelItem } from "./text/StiLabelItem";
import { StiTextItem } from "./text/StiTextItem";
export class StiCheckBoxElement extends StiFormElement {
    constructor() {
        super();
        this.type = "Check_BoxElement";
        this.tooltip = "";
        this.required = false;
        this.description = new StiTextItem();
        this.text = new StiTextItem(StiLocalizationService.get("Components", "StiCheckBox"), new Font("Segoe UI", 9));
        this.isChecked = false;
        this.backgroundColor = Color.white();
        this.checkColor = Color.black();
        this.borderColor = Color.lightGray();
        this.initialState = false;
        this.geometry = new StiRectangleGeometry(100, 40);
        this.label = new StiLabelItem(this, StiLocalizationService.get("Components", "StiCheckBox"));
    }
    setDefaultValues() {
        this.isChecked = false;
    }
    setTabIndex(value) {
        this.tabIndex = value++;
        return value;
    }
    getUsedFonts(usedFonts) {
        super.getUsedFonts(usedFonts);
        usedFonts.add(this.text.font.family, this.arrayOfStylesToString(this.text.font.style), this.text.font.size);
        usedFonts.add(this.description.font.family, this.arrayOfStylesToString(this.description.font.style), this.description.font.size);
    }
    get value() {
        return this.isChecked;
    }
    getDefaultEditableContent() {
        let textItem = new StiInputTextItem("", this.text.font, this.text.color);
        textItem.element = this;
        return textItem;
    }
    setProperty(propName, value, oldObject) {
        super.setProperty(propName, value, oldObject);
        if (propName == "innerText.backgroundColor" &&
            (oldObject === null || this.text.backgroundColor.equals(oldObject))) {
            this.text.backgroundColor = value.clone();
        }
        else if (propName == "innerText.borderColor" && (oldObject === null || this.borderColor.equals(oldObject))) {
            this.borderColor = value.clone();
        }
        else if (propName.startsWith("innerText.font") || propName == "innerText.color") {
            this.text.setProperty(propName.replace("innerText.", ""), value.clone ? value.clone() : value, oldObject);
        }
    }
    saveToJsonObject(options) {
        let jObject = super.saveToJsonObject(options);
        jObject.addPropertyJObject("Label", this.label.saveToJsonObject(options));
        jObject.addPropertyJObject("Description", this.description.saveToJsonObject(options));
        jObject.addPropertyString("Tooltip", this.tooltip);
        jObject.addPropertyBool("Required", this.required);
        jObject.addPropertyJObject("Text", this.text.saveToJsonObject(options));
        jObject.addPropertyBool("IsChecked", this.isChecked);
        jObject.addPropertyColor("BackgroundColor", this.backgroundColor);
        jObject.addPropertyColor("CheckColor", this.checkColor);
        jObject.addPropertyColor("BorderColor", this.borderColor);
        return jObject;
    }
    loadFromJsonObject(json) {
        super.loadFromJsonObject(json);
        for (let property of json.properties()) {
            switch (property.name) {
                case "BackgroundColor":
                    this.backgroundColor = new Color();
                    this.backgroundColor.deserializeFromString(property.value);
                    break;
                case "Label":
                    this.label = new StiLabelItem(this);
                    this.label.loadFromJsonObject(property.value);
                    break;
                case "Text":
                    this.text = new StiTextItem();
                    this.text.loadFromJsonObject(property.value);
                    break;
                case "Description":
                    this.description = new StiTextItem();
                    this.description.loadFromJsonObject(property.value);
                    break;
                case "Tooltip":
                    this.tooltip = property.value;
                    break;
                case "Required":
                    this.required = property.value;
                    break;
                case "IsChecked":
                    this.isChecked = property.value;
                    break;
                case "BorderColor":
                    this.borderColor = new Color();
                    this.borderColor.deserializeFromString(property.value);
                    break;
                case "CheckColor":
                    this.checkColor = new Color();
                    this.checkColor.deserializeFromString(property.value);
                    break;
            }
        }
    }
    htmlStyle() {
        return `${this.text.htmlStyle()};
    )}`;
    }
}
//# sourceMappingURL=data:application/json;base64,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