import { StiInputTextItem } from "../helper/StiInputTextItem";
import StiRectangleGeometry from "../properties/StiRectangleGeometry";
import { StiLocalizationService } from "../services/sti-localization.service";
import Color from "../system/Color";
import { Enum } from "../system/Enum";
import { StiHorizontalAlignment, StiLineWeight } from "../system/enums";
import { Font } from "../system/Font";
import { StiFormElement } from "./base/StiFormElement";
import { StiAddressLabelItem } from "./text/StiAddressLabelItem";
import { StiLabelItem } from "./text/StiLabelItem";
export class StiAddressElement extends StiFormElement {
    constructor() {
        super();
        this.type = "AddressElement";
        this.font = new Font("Segoe UI", 10);
        this.descriptionFont = new Font("Segoe UI", 9);
        this.descriptionColor = Color.black();
        this.backgroundColor = Color.white();
        this.textColor = Color.black();
        this.borderColor = Color.lightGray();
        this.borderWeight = StiLineWeight.Thin;
        this.contentAlignment = StiHorizontalAlignment.Left;
        this.streetAddress = "";
        this.streetAddress2 = "";
        this.city = "";
        this.state = "";
        this.zipCode = "";
        this.country = "";
        this.streetAddressLabel = StiLocalizationService.get("PropertyMain", "StreetAddress");
        this.streetAddress2Label = StiLocalizationService.get("PropertyMain", "StreetAddress") + " 2";
        this.cityLabel = StiLocalizationService.get("PropertyMain", "City");
        this.stateLabel = StiLocalizationService.get("PropertyMain", "State");
        this.zipCodeLabel = StiLocalizationService.get("Components", "StiZipCode");
        this.countryLabel = StiLocalizationService.get("PropertyMain", "Country");
        this.streetAddressLabelItem = new StiAddressLabelItem("streetAddressLabel", this);
        this.streetAddress2LabelItem = new StiAddressLabelItem("streetAddress2Label", this);
        this.cityLabelItem = new StiAddressLabelItem("cityLabel", this);
        this.stateLabelItem = new StiAddressLabelItem("stateLabel", this);
        this.zipCodeLabelItem = new StiAddressLabelItem("zipCodeLabel", this);
        this.countryLabelItem = new StiAddressLabelItem("countryLabel", this);
        this.useStreetAddress = true;
        this.useStreetAddress2 = true;
        this.useCity = true;
        this.useState = true;
        this.useZipCode = true;
        this.useCountry = true;
        this.maximum = 255;
        this.geometry = new StiRectangleGeometry(150, 200);
        this.label = new StiLabelItem(this, StiLocalizationService.get("Report", "Address"));
        this.geometry.minWidth = 150;
        this.geometry.minHeight = 50;
    }
    setDefaultValues() {
        this.streetAddress = "";
        this.streetAddress2 = "";
        this.city = "";
        this.state = "";
        this.zipCode = "";
        this.country = "";
    }
    setTabIndex(value) {
        this.tabIndex = value;
        return value + 6;
    }
    get contentAreaWidth() {
        let margin = this.padding;
        return this.width - margin.left - margin.right;
    }
    get contentAreaHeight() {
        let margin = this.padding;
        return this.height - margin.top - margin.bottom - 35;
    }
    getUsedFonts(usedFonts) {
        super.getUsedFonts(usedFonts);
        usedFonts.add(this.font.family, this.arrayOfStylesToString(this.font.style), this.font.size);
        usedFonts.add(this.descriptionFont.family, this.arrayOfStylesToString(this.descriptionFont.style), this.descriptionFont.size);
    }
    getDefaultEditableContent() {
        let textItem = new StiInputTextItem("", this.font, this.textColor, this.backgroundColor, this.contentAlignment);
        textItem.element = this;
        return textItem;
    }
    setProperty(propName, value, oldObject) {
        super.setProperty(propName, value, oldObject);
        if (propName.startsWith("description.font") &&
            (oldObject === null || oldObject.equals(this.descriptionFont))) {
            this.descriptionFont.setProperty(propName.replace("description.font.", ""), value);
        }
        else if (propName == "description.color" &&
            (oldObject === null || oldObject.equals(this.descriptionColor))) {
            this.descriptionColor = value.clone();
        }
    }
    saveToJsonObject(options) {
        let jObject = super.saveToJsonObject(options);
        jObject.addPropertyJObject("Label", this.label.saveToJsonObject(options));
        jObject.addPropertyColor("BackgroundColor", this.backgroundColor);
        jObject.addPropertyColor("TextColor", this.textColor);
        jObject.addPropertyJObject("Font", this.font.saveToJsonObject(options));
        jObject.addPropertyJObject("DescriptionFont", this.descriptionFont.saveToJsonObject(options));
        jObject.addPropertyColor("DescriptionColor", this.descriptionColor);
        jObject.addPropertyColor("BorderColor", this.borderColor);
        jObject.addPropertyString("StreetAddress", this.streetAddress);
        jObject.addPropertyString("StreetAddress2", this.streetAddress2);
        jObject.addPropertyString("City", this.city);
        jObject.addPropertyString("State", this.state);
        jObject.addPropertyString("ZipCode", this.zipCode);
        jObject.addPropertyString("Country", this.country);
        jObject.addPropertyString("StreetAddressLabel", this.streetAddressLabel);
        jObject.addPropertyString("StreetAddress2Label", this.streetAddress2Label);
        jObject.addPropertyString("CityLabel", this.cityLabel);
        jObject.addPropertyString("StateLabel", this.stateLabel);
        jObject.addPropertyString("ZipCodeLabel", this.zipCodeLabel);
        jObject.addPropertyString("CountryLabel", this.countryLabel);
        jObject.addPropertyEnum("BorderWeight", StiLineWeight, this.borderWeight);
        jObject.addPropertyEnum("ContentAlignment", StiHorizontalAlignment, this.contentAlignment);
        jObject.addPropertyBool("UseCity", this.useCity);
        jObject.addPropertyBool("UseCountry", this.useCountry);
        jObject.addPropertyBool("UseState", this.useState);
        jObject.addPropertyBool("UseStreetAddress", this.useStreetAddress);
        jObject.addPropertyBool("UseStreetAddress2", this.useStreetAddress2);
        jObject.addPropertyBool("UseZipCode", this.useZipCode);
        return jObject;
    }
    loadFromJsonObject(json) {
        super.loadFromJsonObject(json);
        for (let property of json.properties()) {
            switch (property.name) {
                case "Label":
                    this.label = new StiLabelItem(this);
                    this.label.loadFromJsonObject(property.value);
                    break;
                case "BackgroundColor":
                    this.backgroundColor = new Color();
                    this.backgroundColor.deserializeFromString(property.value);
                    break;
                case "DescriptionColor":
                    this.descriptionColor = new Color();
                    this.descriptionColor.deserializeFromString(property.value);
                    break;
                case "TextColor":
                    this.textColor = new Color();
                    this.textColor.deserializeFromString(property.value);
                    break;
                case "BorderColor":
                    this.borderColor = new Color();
                    this.borderColor.deserializeFromString(property.value);
                    break;
                case "Font":
                    this.font = new Font();
                    this.font.loadFromJsonObject(property.value);
                    break;
                case "DescriptionFont":
                    this.descriptionFont = new Font();
                    this.descriptionFont.loadFromJsonObject(property.value);
                    break;
                case "StreetAddress":
                    this.streetAddress = property.value;
                    break;
                case "StreetAddress2":
                    this.streetAddress2 = property.value;
                    break;
                case "City":
                    this.city = property.value;
                    break;
                case "State":
                    this.state = property.value;
                    break;
                case "ZipCode":
                    this.zipCode = property.value;
                    break;
                case "Country":
                    this.country = property.value;
                    break;
                case "StreetAddressLabel":
                    this.streetAddressLabel = property.value;
                    break;
                case "StreetAddress2Label":
                    this.streetAddress2Label = property.value;
                    break;
                case "CityLabel":
                    this.cityLabel = property.value;
                    break;
                case "StateLabel":
                    this.stateLabel = property.value;
                    break;
                case "ZipCodeLabel":
                    this.zipCodeLabel = property.value;
                    break;
                case "CountryLabel":
                    this.countryLabel = property.value;
                    break;
                case "BorderWeight":
                    this.borderWeight = Enum.parse(StiLineWeight, property.value);
                    break;
                case "ContentAlignment":
                    this.contentAlignment = Enum.parse(StiHorizontalAlignment, property.value);
                    break;
                case "UseCity":
                    this.useCity = property.value;
                    break;
                case "UseCountry":
                    this.useCountry = property.value;
                    break;
                case "UseState":
                    this.useState = property.value;
                    break;
                case "UseStreetAddress":
                    this.useStreetAddress = property.value;
                    break;
                case "UseStreetAddress2":
                    this.useStreetAddress2 = property.value;
                    break;
                case "UseZipCode":
                    this.useZipCode = property.value;
                    break;
            }
        }
    }
    htmlStyle() {
        return `${this.font.htmlStyle()};color:${this.textColor.toHtmlColor()};text-align:${Enum.getName(StiHorizontalAlignment, this.contentAlignment)}`;
    }
    comboBoxHtmlStyle() {
        return `${this.font.htmlStyle()};color:${this.textColor.toHtmlColor()}}`;
    }
}
//# sourceMappingURL=data:application/json;base64,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