import { Component, ViewChild } from "@angular/core";
import Color from "../../system/Color";
import { StiTextItem } from "../../elements/text/StiTextItem";
import { StiHorizontalAlignment } from "../../system/enums";
import { Enum } from "../../system/Enum";
import { StiImageEditorResult } from "../../elements/signature/StiImageEditorResult";
import * as i0 from "@angular/core";
import * as i1 from "../../services/model.service";
import * as i2 from "../../services/sti-localization.service";
import * as i3 from "../../services/mouse.service";
import * as i4 from "../../services/sti-signature.service";
import * as i5 from "../../services/sti-draggable-modal-window.service";
import * as i6 from "../../services/sti-color-dialog.service";
import * as i7 from "@angular/common";
import * as i8 from "@angular/forms";
import * as i9 from "../sti-tooltip/sti-tooltip.directive";
export class StiElectronicSignatureEditorComponent {
    constructor(model, loc, mouseService, signatureService, modalWindowService, colorService) {
        this.model = model;
        this.loc = loc;
        this.mouseService = mouseService;
        this.signatureService = signatureService;
        this.modalWindowService = modalWindowService;
        this.colorService = colorService;
        this.drawingMode = false;
        this.isDrawing = false;
        this.defaultPreviewText = "FullName FN";
    }
    get brushColor() {
        return this.signatureService?.currentSignatureElement?.penColor
            ? this.signatureService?.currentSignatureElement?.penColor
            : Color.black();
    }
    get hasTemplates() {
        return this.signatureService.savedTemplates.length > 0;
    }
    get insertedText() {
        return this.signatureService.currentEditableText;
    }
    get insertedImage() {
        return this.signatureService.currentEditableImage;
    }
    get fullName() {
        return this.signatureService.currentEditableFullName;
    }
    set fullName(value) {
        this.signatureService.currentEditableFullName = value;
    }
    get initials() {
        return this.signatureService.currentEditableInitials;
    }
    set initials(value) {
        this.signatureService.currentEditableInitials = value;
    }
    isEmptyContent() {
        if (this.signatureService.selectedMode == "Draw") {
            return (!this.signatureService.currentEditableImage?.loadedImage &&
                !this.signatureService.currentEditableText?.value &&
                !this.signatureService.currentCanvasContent);
        }
        else {
            return !this.signatureService.currentEditableFullName && !this.signatureService.currentEditableInitials;
        }
    }
    saveTemplate() {
        if (this.signatureService.selectedMode == "Draw") {
            let imageObj = {
                type: "Image",
                image: this.signatureService.currentEditableImage,
                text: this.signatureService.currentEditableText,
                drawedImage: this.signatureService.currentCanvasContent,
            };
            if (!this.signatureService.savedTemplates.find((item) => item == imageObj)) {
                this.signatureService.savedTemplates.push(imageObj);
            }
        }
        else {
            let textObj = {
                type: "Text",
                fullName: this.signatureService.currentEditableFullName,
                initials: this.signatureService.currentEditableInitials,
                specialFont: this.signatureService.currentPreviewFont,
            };
            if (!this.signatureService.savedTemplates.find((item) => item == textObj)) {
                this.signatureService.savedTemplates.push(textObj);
            }
        }
        localStorage.removeItem("StiSignatureTemplates");
        localStorage.setItem("StiSignatureTemplates", JSON.stringify(this.signatureService.savedTemplates));
    }
    showSavedTemplatesPreview(button) {
        let buttonRect = this.model.getPositionRelativeToForm(button);
        this.signatureService.templatesPreviewX = buttonRect.right - 335;
        this.signatureService.templatesPreviewY = buttonRect.bottom + 1;
        this.signatureService.showSavedTemplates = !this.signatureService.showSavedTemplates;
    }
    ngOnDestroy() {
        this.signatureService.currentEditableText = new StiTextItem();
        this.signatureService.currentCanvasContent = null;
        this.signatureService.currentEditableImage = new StiImageEditorResult();
        this.signatureService.currentSignatureElement = null;
    }
    isCanvasBlank(canvas) {
        const context = canvas.getContext("2d");
        const pixelBuffer = new Uint32Array(context.getImageData(0, 0, canvas.width, canvas.height).data.buffer);
        return !pixelBuffer.some((color) => color !== 0);
    }
    ngAfterViewInit() {
        const onResize = (event) => {
            this.signatureService.editorComponentX = (window.innerWidth - this.signatureService.editorComponentWidth) / 2;
            this.signatureService.editorComponentY = (window.innerHeight - this.signatureService.editorComponentHeight) / 2;
        };
        window.addEventListener("resize", onResize);
        this.fillDrawingCanvas();
    }
    get image() {
        return "url(" + this.insertedImage?.loadedImage + ")";
    }
    getHorizontalAlignment() {
        if (this.insertedImage?.horizontalAlignment == StiHorizontalAlignment.Center) {
            return "center";
        }
        else if (this.insertedImage?.horizontalAlignment == StiHorizontalAlignment.Left) {
            return "start";
        }
        else {
            return "end";
        }
    }
    onMouseDown(event, canvasElement) {
        if (this.drawingMode) {
            event.preventDefault();
            event.stopPropagation();
            this.isDrawing = true;
            this.context = canvasElement?.getContext("2d");
            this.context.strokeStyle = this.brushColor.toHtmlColor();
            const coords = this.relativeCoords(event, canvasElement);
            this.context.moveTo(coords.x, coords.y);
        }
    }
    relativeCoords(event, canvasElement) {
        const bounds = this.model.getPositionRelativeToForm(canvasElement);
        const x = (event.clientX - bounds.left) / this.model.zoom;
        const y = (event.clientY - bounds.top) / this.model.zoom;
        return { x: x, y: y };
    }
    onCanvasMove(event, canvasElement) {
        if (this.isDrawing) {
            const coords = this.relativeCoords(event, canvasElement);
            this.context.lineTo(coords.x, coords.y);
            this.context.stroke();
        }
    }
    getBackgroundSize(container) {
        if (!this.insertedImage?.loadedImage) {
            return;
        }
        if (this.insertedImage?.aspectRatio && this.insertedImage?.stretch) {
            return "contain";
        }
        if (this.insertedImage?.stretch) {
            return `${container.offsetWidth}px ${container.offsetHeight}px`;
        }
        else {
            return `${this.insertedImage?.naturalWidth}px ${this.insertedImage?.naturalHeight}px`;
        }
    }
    getBackgroundPosition() {
        if (this.insertedImage?.stretch && !this.insertedImage?.aspectRatio) {
            return `center`;
        }
        else {
            return `${this.signatureService.getVerticalPosition(this.insertedImage?.verticalAlignment)} ${this.signatureService.getHorizontalPosition(this.insertedImage?.horizontalAlignment)}`;
        }
    }
    get previewText() {
        return (this.fullName + " " + this.initials).length > 1
            ? this.fullName + " " + this.initials
            : this.defaultPreviewText;
    }
    get alignment() {
        if (this.insertedText.horizontalAlignment == StiHorizontalAlignment.Center) {
            return "center";
        }
        else if (this.insertedText.horizontalAlignment == StiHorizontalAlignment.Right) {
            return "end";
        }
        else {
            return "start";
        }
    }
    clear() {
        this.signatureService.currentEditableText.value = "";
        if (this.signatureService?.currentEditableImage?.loadedImage) {
            this.signatureService.currentEditableImage.loadedImage = "";
        }
        if (!this.context) {
            this.context = this.canvas?.nativeElement?.getContext("2d");
        }
        this.signatureService.currentCanvasContent = "";
        this.context?.clearRect(0, 0, this.canvas?.nativeElement?.width, this.canvas?.nativeElement?.height);
        this.context?.beginPath();
    }
    showEditor(editorType) {
        this.model.showDraggableModalWindow = true;
        if (editorType == "Text") {
            this.modalWindowService.isTextEditor = true;
            this.signatureService.currentEditableText = this.insertedText;
        }
        else {
            this.modalWindowService.isImageEditor = true;
            this.signatureService.currentEditableImage = this.insertedImage;
        }
    }
    showColorPicker(element, object, isRichText = false) {
        this.colorService.openDialog(element, object, isRichText);
    }
    fillDrawingCanvas() {
        let ctx = this.canvas?.nativeElement?.getContext("2d");
        if (ctx && this.signatureService.currentCanvasContent) {
            ctx.clearRect(0, 0, this.canvas?.nativeElement?.width, this.canvas?.nativeElement?.height);
            let image = new Image();
            image.onload = () => {
                ctx.drawImage(image, 0, 0, this.canvas?.nativeElement?.width, this.canvas?.nativeElement?.height);
            };
            image.src = this.signatureService.currentCanvasContent;
        }
    }
    showFontsPreview(element) {
        this.signatureService.showFontsPreview = !this.signatureService.showFontsPreview;
        let rect = this.model.getPositionRelativeToForm(element);
        this.signatureService.fontsPreviewX = rect.right - 200;
        this.signatureService.fontsPreviewY = rect.bottom + 2;
    }
    acceptChanges() {
        this.signatureService.currentActiveLoadingImages = 0;
        this.signatureService.completedLoadingImages = 0;
        this.signatureService.currentSignatureElement.fullName = this.signatureService.currentEditableFullName;
        this.signatureService.currentSignatureElement.initials = this.signatureService.currentEditableInitials;
        this.signatureService.currentSignatureElement.specialFont = this.signatureService.currentPreviewFont;
        let canvasRect = this.model.getPositionRelativeToForm(this.canvas.nativeElement);
        if (canvasRect && this.signatureService.selectedMode == "Draw") {
            let resultCanvas = document.createElement("canvas");
            this.canvasWidth = canvasRect.width;
            this.canvasHeight = canvasRect.height;
            resultCanvas.width = this.canvasWidth;
            resultCanvas.height = this.canvasHeight;
            let ctx = resultCanvas.getContext("2d");
            this.resultCanvasContext = ctx;
            this.insertedImageLayer = this.createLayer(resultCanvas);
            this.insertedTextLayer = this.createLayer(resultCanvas);
            this.drawedLayer = this.createLayer(resultCanvas);
            if (this.insertedImage?.loadedImage) {
                this.fillLayer(this.insertedImageLayer, this.insertedImage);
            }
            if (this.insertedText?.value) {
                this.fillLayer(this.insertedTextLayer, this.insertedText);
            }
            if (!this.isCanvasBlank(this.canvas?.nativeElement)) {
                this.fillLayer(this.drawedLayer, this.canvas?.nativeElement?.toDataURL("image/png"));
            }
            else {
                this.signatureService.currentCanvasContent = "";
            }
            this.signatureService.checkLoading();
        }
        else {
            this.updateResultCanvas("Text");
        }
        if (this.signatureService.selectedMode == "Draw") {
            this.signatureService.currentSignatureElement.currentSignatureMode = "Draw";
        }
        else {
            this.signatureService.currentSignatureElement.currentSignatureMode = "Type";
        }
    }
    createLayer(canvas) {
        let layer = document.createElement("canvas");
        layer.width = canvas.width;
        layer.height = canvas.height;
        return layer.getContext("2d");
    }
    drawLayer(context, layer) {
        if (layer) {
            context.drawImage(layer.canvas, 0, 0);
        }
    }
    fillLayer(layer, content) {
        if (content instanceof StiTextItem) {
            if (content.value) {
                layer.font = `${content.font.isItalic() ? "italic " : ""}${content.font.isBold() ? "bold " : ""}${content.font.size + "px "}${content.font.family}`;
                layer.textBaseline = "center";
                layer.textAlign = Enum.getName(StiHorizontalAlignment, content.horizontalAlignment).toLocaleLowerCase();
                layer.fillText(content.value, Enum.getName(StiHorizontalAlignment, content.horizontalAlignment).toLocaleLowerCase() == "left"
                    ? 0
                    : Enum.getName(StiHorizontalAlignment, content.horizontalAlignment).toLocaleLowerCase() == "center"
                        ? this.canvasWidth / 2
                        : this.canvasWidth - layer.measureText(content.value).width, this.canvasHeight / 2);
            }
        }
        else if (content?.loadedImage) {
            this.signatureService.currentActiveLoadingImages += 1;
            let image = new Image();
            image.onload = () => {
                if (content?.aspectRatio && content?.stretch) {
                    let hRatio = this.canvasWidth / image.naturalWidth;
                    let vRatio = this.canvasHeight / image.naturalHeight;
                    let ratio = Math.min(hRatio, vRatio);
                    let centerShiftX = (this.canvasWidth - image.naturalWidth * ratio) / 2;
                    let centerShiftY = (this.canvasHeight - image.naturalHeight * ratio) / 2;
                    layer.drawImage(image, 0, 0, image.naturalWidth, image.naturalHeight, centerShiftX, centerShiftY, image.naturalWidth * ratio, image.naturalHeight * ratio);
                    this.signatureService.completedLoadingImages += 1;
                }
                else {
                    layer.drawImage(image, 0, 0, content.stretch ? this.canvasWidth : image.naturalWidth, content.stretch ? this.canvasHeight : image.naturalHeight);
                    this.signatureService.completedLoadingImages += 1;
                }
            };
            image.src = content.loadedImage;
        }
        else {
            if (typeof content === "string") {
                this.signatureService.currentActiveLoadingImages += 1;
                let image = new Image();
                image.onload = () => {
                    layer.drawImage(image, 0, 0, this.canvasWidth, this.canvasHeight);
                    this.signatureService.completedLoadingImages += 1;
                };
                image.src = content;
            }
        }
    }
    changeMode(mode) {
        this.signatureService.selectedMode = mode;
        if (mode == "Draw") {
            setTimeout(() => {
                this.fillDrawingCanvas();
            });
        }
    }
    updateResultCanvas(type = "Draw") {
        if (type == "Draw") {
            this.drawLayer(this.resultCanvasContext, this.insertedImageLayer);
            this.drawLayer(this.resultCanvasContext, this.insertedTextLayer);
            this.drawLayer(this.resultCanvasContext, this.drawedLayer);
            // console.log(this.resultCanvasContext.canvas.toDataURL("image/png"));
            if (this.resultCanvasContext?.canvas?.toDataURL("image/png")) {
                this.signatureService.currentSignatureElement.backgroundImage =
                    this.resultCanvasContext.canvas.toDataURL("image/png");
            }
            this.signatureService.currentSignatureElement.insertedImage = this.signatureService.currentEditableImage;
            this.signatureService.currentSignatureElement.insertedText = this.signatureService.currentEditableText;
            if (this.signatureService.currentCanvasContent) {
                this.signatureService.currentSignatureElement.drawedImage = this.signatureService.currentCanvasContent;
            }
            else {
                this.signatureService.currentSignatureElement.drawedImage = "";
            }
        }
        this.signatureService.currentSignatureElement.isSigned = true;
        this.model.showSignatureEditor = false;
        // this.signatureService.currentSignatureElement = null;
    }
    onMouseMove(event) {
        if (this.isDragging) {
            let deltaX = event.clientX - this.startX;
            let deltaY = event.clientY - this.startY;
            if (this.startComponentX + deltaX < 0) {
                this.signatureService.editorComponentX = 0;
            }
            else if (this.startComponentX + deltaX > window.innerWidth - this.signatureService.editorComponentWidth) {
                this.signatureService.editorComponentX = window.innerWidth - this.signatureService.editorComponentWidth;
            }
            else {
                this.signatureService.editorComponentX = this.startComponentX + deltaX;
            }
            if (this.startComponentY + deltaY < 0) {
                this.signatureService.editorComponentY = 0;
            }
            else if (this.startComponentY + deltaY > window.innerHeight - this.signatureService.editorComponentHeight) {
                this.signatureService.editorComponentY = window.innerHeight - this.signatureService.editorComponentHeight;
            }
            else {
                this.signatureService.editorComponentY = this.startComponentY + deltaY;
            }
        }
    }
    startDrag(event) {
        this.startX = event.clientX;
        this.startY = event.clientY;
        this.startComponentX = this.signatureService.editorComponentX;
        this.startComponentY = this.signatureService.editorComponentY;
        this.isDragging = true;
    }
    ngOnInit() {
        if (localStorage.getItem("StiSignatureTemplates")) {
            this.signatureService.savedTemplates = JSON.parse(localStorage.getItem("StiSignatureTemplates"));
        }
        this.signatureService.editorComponentX = (window.innerWidth - this.signatureService.editorComponentWidth) / 2;
        this.signatureService.editorComponentY = (window.innerHeight - this.signatureService.editorComponentHeight) / 2;
        this.updateSignatureSubscription = this.signatureService.updateSignature.subscribe((event) => {
            if (event.type == "Text") {
                this.signatureService.currentEditableText = event.value;
            }
            else if (event.type == "Image") {
                this.signatureService.currentEditableImage = event.value;
            }
        });
        this.updateResultCanvasSubscription = this.signatureService.completedLoading.subscribe(() => {
            this.updateResultCanvas();
        });
        this.mouseUpSubscription = this.mouseService.getDocumentMouseUp().subscribe(() => {
            if (this.isDrawing && !this.isCanvasBlank(this.canvas?.nativeElement)) {
                this.signatureService.currentCanvasContent = this.canvas?.nativeElement?.toDataURL("image/png");
            }
            this.isDragging = false;
            this.isDrawing = false;
        });
        this.mouseMoveSubscription = this.mouseService.getDocumentMouseMove().subscribe((event) => {
            this.onMouseMove(event);
        });
    }
}
StiElectronicSignatureEditorComponent.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "14.3.0", ngImport: i0, type: StiElectronicSignatureEditorComponent, deps: [{ token: i1.StiModelService }, { token: i2.StiLocalizationService }, { token: i3.StiMouseService }, { token: i4.SignatureService }, { token: i5.StiDraggableModalWindowService }, { token: i6.StiColorDialogService }], target: i0.ɵɵFactoryTarget.Component });
StiElectronicSignatureEditorComponent.ɵcmp = i0.ɵɵngDeclareComponent({ minVersion: "14.0.0", version: "14.3.0", type: StiElectronicSignatureEditorComponent, selector: "sti-electronic-signature-editor", viewQueries: [{ propertyName: "canvas", first: true, predicate: ["canvas"], descendants: true }], ngImport: i0, template: `<div
      class="editor-container"
      [style.left.px]="signatureService.editorComponentX"
      [style.top.px]="signatureService.editorComponentY"
    >
      <div class="editor-navbar" (mousedown)="startDrag($event)">
        <span [style.flex]="1">{{ loc.get2("Components", "StiSignature") }}</span>
        <div class="close-button" (click)="model.showSignatureEditor = false">
          <img [src]="model.getImage('images.Other.CloseForm.png')" />
        </div>
      </div>
      <div class="content-container">
        <div class="content-navigation">
          <div
            class="navigation-button {{ signatureService.selectedMode == 'Type' ? 'current' : '' }}"
            (click)="changeMode('Type')"
          >
            <div class="navigation-label">
              <span>{{ loc.get2("PropertyEnum", "StiSignatureTypeType") }}</span>
            </div>
            <div class="indicator {{ signatureService.selectedMode == 'Type' ? 'current-indicator' : '' }}"></div>
          </div>
          <div
            class="navigation-button {{ signatureService.selectedMode == 'Draw' ? 'current' : '' }}"
            (click)="changeMode('Draw')"
          >
            <div class="navigation-label">
              <span>{{ loc.get2("PropertyEnum", "StiSignatureTypeDraw") }}</span>
            </div>
            <div class="indicator {{ signatureService.selectedMode == 'Draw' ? 'current-indicator' : '' }}"></div>
          </div>
          <div class="templates-navigation-container">
            <div (click)="saveTemplate()" class="template-navigation-button {{ isEmptyContent() ? 'disabled' : '' }}">
              <img [src]="model.getImage('signature.StiSignatureSave.png')" />
            </div>
            <div
              #button
              (click)="showSavedTemplatesPreview(button)"
              class="template-navigation-button show-button"
              [style.width.px]="38"
            >
              <img [src]="model.getImage('signature.StiSignatureOpen.png')" /><i class="pi pi-chevron-down"></i>
            </div>
          </div>
        </div>
        <div class="bottom-divider"></div>
        <div class="type-container" *ngIf="signatureService.selectedMode == 'Type'">
          <div class="input-field">
            <span>{{ loc.getP("FullName") }}</span
            ><input type="text" [(ngModel)]="fullName" />
          </div>
          <div class="input-field">
            <span>{{ loc.getP("Initials") }}</span
            ><input type="text" [(ngModel)]="initials" />
          </div>
          <div class="button-panel">
            <div #selectButton class="select-button" (click)="showFontsPreview(selectButton)">
              <span class="button-label">{{ loc.get2("ChartRibbon", "Style") }}</span
              ><i class="pi pi-chevron-down"></i>
            </div>
          </div>
          <div class="preview-panel" [style.font-family]="signatureService.currentPreviewFont">
            <span>{{ previewText }}</span>
          </div>
        </div>
        <div class="draw-container" *ngIf="signatureService.selectedMode == 'Draw'">
          <div class="draw-buttons-panel">
            <div
              class="button-container {{ drawingMode ? 'active' : '' }}"
              [stiTooltip]="loc.getP('ForeColor')"
              tooltipPosition="bottom"
              showDelay="500"
              (mousedown)="$event.preventDefault()"
              #fontColorEditor
            >
              <div class="preview-container" (click)="drawingMode = !drawingMode">
                <div class="foreground" [style]="{ backgroundPosition: 'center', backgroundRepeat: 'no-repeat' }">
                  <img [src]="model.getImage('images.Brush.png')" alt="" />
                </div>
                <span
                  class="color-preview"
                  [style.background-color]="brushColor != null ? brushColor?.toHtmlColor() : 'rgb(182, 181, 180)'"
                ></span>
              </div>
              <div
                (click)="showColorPicker(fontColorEditor, brushColor)"
                [style]="{ position: 'relative', display: 'flex', alignItems: 'center' }"
              >
                <i class="pi pi-chevron-down"></i>
              </div>
            </div>
            <div class="draw-panel-button" (click)="showEditor('Text')">
              <img [src]="model.getImage('signature.Text.png')" />
            </div>
            <div class="draw-panel-button" (click)="showEditor('Image')">
              <img [src]="model.getImage('images.Toolbox.StiImageElement.png')" />
            </div>
            <div class="draw-panel-button" (click)="clear()">
              <img [src]="model.getImage('images.Page.ClearContent.png')" />
            </div>
          </div>
          <div class="draw-panel" #previewContainer>
            <div class="canvas-container">
              <canvas
                [width]="previewContainer.clientWidth"
                [height]="previewContainer.clientHeight"
                (mousedown)="onMouseDown($event, canvas)"
                (mousemove)="onCanvasMove($event, canvas)"
                #canvas
              >
              </canvas>
            </div>
            <div class="preview-inserted-text" [style]="insertedText.htmlStyle()" [style.justify-content]="alignment">
              <span>{{ insertedText.value }}</span>
            </div>
            <div class="inserted-image-container" [style.justify-content]="getHorizontalAlignment()">
              <div
                class="preview-inserted-image"
                [style.width.px]="
                  insertedImage?.aspectRatio && insertedImage?.stretch
                    ? previewContainer.clientHeight
                    : previewContainer.clientWidth
                "
                [style.height.px]="previewContainer.clientHeight"
                [style.background-image]="image"
                [style.background-position]="getBackgroundPosition()"
                [style.background-size]="getBackgroundSize(previewContainer)"
                [style.background-repeat]="image ? 'no-repeat' : ''"
              ></div>
            </div>
            <div class="signature-description">
              <span>{{ "Signature" }}</span>
            </div>
          </div>
        </div>
      </div>
      <div class="setup-accept">
        <div class="accept button" (click)="acceptChanges()">
          <span [style.margin-left.px]="0">{{
            model.isPreviewMode
              ? loc.get2("Buttons", "Sign")
              : loc.get2("Buttons", "Save").replace("&amp", "").replace(";", "")
          }}</span>
        </div>
        <div class="cancel button" (click)="model.showSignatureEditor = false">
          <span [style.margin-left.px]="0">{{ loc.getP("Cancel") }}</span>
        </div>
      </div>
    </div>
    <div
      class="fonts-preview-container"
      *ngIf="signatureService.showFontsPreview"
      [style.left.px]="signatureService.fontsPreviewX"
      [style.top.px]="signatureService.fontsPreviewY"
    >
      <div
        *ngFor="let font of signatureService.previewFonts"
        class="font-preview"
        [style.font-family]="font"
        (click)="signatureService.currentPreviewFont = font; signatureService.showFontsPreview = false"
      >
        <div class="vertical-line {{ signatureService.currentPreviewFont == font ? 'current-indicator' : '' }}"></div>
        <span> {{ previewText }}</span>
      </div>
    </div>
    <div
      *ngIf="signatureService.showSavedTemplates"
      class="saved-templates-container"
      [style.left.px]="signatureService.templatesPreviewX"
      [style.top.px]="signatureService.templatesPreviewY"
      [style.overflow-y]="hasTemplates ? 'scroll' : 'hidden'"
    >
      <div class="empty-container" *ngIf="!hasTemplates">
        <span>{{ "No saved items" }}</span>
      </div>
      <div
        class="template-item"
        *ngFor="let temp of signatureService.savedTemplates; let i = index"
        [style.margin-top.px]="i == 0 ? 6 : 0"
        (click)="signatureService.setTemplate(temp)"
      >
        <div class="delete-button" (click)="signatureService.deleteTemplate(i)">
          <img [src]="model.getImage('images.Other.CloseForm.png')" />
        </div>
        <div
          class="image-template"
          *ngIf="temp.type == 'Image'"
          #parentContainer
          [style.justify-content]="signatureService.getTemplateHorizontalAlignment(temp)"
        >
          <div
            class="loadedImage image-content"
            [style.background-image]="signatureService.getTemplateImage(temp)"
            [style.background-position]="signatureService.getTemplateBackgroundPosition(temp)"
            [style.background-size]="signatureService.getTemplateBackgroundSize(temp, parentContainer)"
            [style.background-repeat]="'no-repeat'"
          ></div>
          <div class="insertedText image-content">
            <span [style]="temp.text.htmlStyle()">{{ temp?.text?.value }}</span>
          </div>
          <div
            class="drawedImage image-content"
            [style.background-size]="'contain'"
            [style.background-image]="signatureService.getTemplateDrawedImage(temp)"
          ></div>
        </div>
        <div class="text-template" *ngIf="temp.type == 'Text'">
          <span [style.font-family]="temp.specialFont" [style.font-size.pt]="16">{{
            temp.fullName + " " + temp.initials
          }}</span>
        </div>
      </div>
    </div>`, isInline: true, styles: ["@font-face{font-family:Teddy Bear;src:url(projects/stimulsoft-forms/src/lib/assets/TeddyBear.ttf) format(\"truetype\")}@font-face{font-family:Denistina;src:url(projects/stimulsoft-forms/src/lib/assets/Denistina.ttf) format(\"truetype\")}@font-face{font-family:MADE Likes Script;src:url(projects/stimulsoft-forms/src/lib/assets/MADELikesScript.otf) format(\"opentype\")}.image-content{height:100%;width:100%;top:0;left:0;position:absolute}.drawedImage{z-index:3}.loadedImage{z-index:1}.insertedText{z-index:2;display:flex;align-items:center;justify-content:center}.text-template{display:flex;align-items:center;justify-content:center;height:50px}.image-template{position:relative;height:130px}.template-item{width:calc(100% - 12px);height:-moz-fit-content;height:fit-content;padding:2px;margin:0 6px 6px;border:1px solid lightgray;border-radius:3px;position:relative}.delete-button{position:absolute;z-index:999;right:0;top:0;width:20px;height:20px;display:flex;align-items:center;justify-content:center;border-radius:3px}.delete-button img{height:16px;width:16px}.delete-button:hover img{filter:saturate(500%) contrast(100%) brightness(900%) invert(900%) sepia(50%)}.template-item:hover{background-color:#d7d7d7;cursor:pointer}.empty-container{height:100%;width:100%;background-color:#fafafa;display:flex;align-items:center;justify-content:center;color:#454545;font-size:13px;font-family:Arial}.saved-templates-container{position:absolute;overflow-x:hidden;overflow-y:scroll;width:335px;height:270px;border-radius:3px;box-shadow:0 0 7px #0000004d;border:1px solid rgb(25,71,138);display:flex;flex-direction:column;background-color:#fff;z-index:99999999999999999}.active{background-color:#d3d3d3;outline:1px solid rgb(61,107,186)}.canvas-container{position:absolute;left:0;top:0;height:100%;width:100%;z-index:99999}.preview-inserted-image{position:absolute}.inserted-image-container{position:absolute;left:0;top:0;display:flex;height:100%;width:100%}.preview-inserted-text{position:absolute;height:100%;width:100%;display:flex;align-items:center;justify-content:center;z-index:999}.signature-content{z-index:9999;height:100%;width:100%}.signature-description{width:calc(100% - 10px);height:50px;margin:0 5px;padding:0 5px;border-top:1px solid lightgray;display:flex;align-items:center;overflow:hidden;z-index:999}.disabled{pointer-events:none;opacity:.5}.preview-container{display:flex;width:100%;position:relative;flex-direction:column}.preview-container img{height:16px;width:16px}.button-container{display:flex;height:26px;width:40px;position:relative;margin-right:5px;border-radius:3px}.button-container:hover{background-color:#d3d3d3}.button-container:active{background-color:var(--columnDeleteHover-color)}.color-preview{width:20px;height:5px;margin-left:4px}i{height:12px;width:12px;right:3px;top:50%;vertical-align:middle;color:var(--text-color)}.button-container:hover i{color:#fff}.foreground{margin-left:8px;margin-top:2px;width:16px;height:16px}.template-navigation-button{height:26px;width:26px;margin-right:2px;border-radius:3px;display:flex;align-items:center;justify-content:center}.template-navigation-button:hover{background-color:#d3d3d3}.template-navigation-button img{height:16px;width:16px}.templates-navigation-container{width:100%;display:flex;align-items:center;justify-content:end}.vertical-line{width:3px;height:calc(100% - 14px);margin:7px 10px 7px 5px}.font-preview{font-size:16px;height:42px;display:flex;align-items:center;padding-left:5px;border-radius:3px;margin:1px}.font-preview:hover{background-color:#dadada}.fonts-preview-container{position:absolute;width:200px;height:134px;z-index:9999999999;padding:1px;border-radius:3px;border:1px solid #c6c6c6;background:#ffffff;color:#444;box-shadow:0 0 7px #0000004d;cursor:default}.draw-buttons-panel{height:26px;margin-bottom:5px;margin-top:10px;display:flex;padding-left:24px}.draw-panel-button:hover{background-color:#d3d3d3}.draw-panel-button{height:26px;width:26px;margin-right:5px;border-radius:3px;display:flex;align-items:center;justify-content:center}.draw-panel-button img{height:16px;width:16px}.draw-panel{border:1px dashed #ababab;border-radius:3px;margin:0 24px;width:calc(100% - 48px);height:186px;display:flex;flex-direction:column-reverse;position:relative}i{margin-left:5px;font-size:7px;padding-top:4px}.select-button{border:1px solid #dddddd;border-radius:3px;min-width:30px;padding-left:10px;padding-right:6px;display:flex;color:#444;align-items:center;margin-right:24px;margin-top:24px;height:24px}.select-button:hover{background-color:#d3d3d3}.button-panel{width:100%;display:flex;justify-content:end}.preview-panel{height:80px;width:calc(100% - 48px);border:1px solid #ababab;border-radius:3px;font-size:16pt;margin:10px 24px;padding-left:12px;display:flex;align-items:center;overflow:hidden}.input-field{width:calc(100% - 24px);display:flex;margin:12px 12px 6px;padding:0 12px;align-items:center}.input-field span{min-width:100px;font-size:12px}.input-field input{border-radius:3px;padding-left:5px;height:24px;outline:none;border:1px solid lightgray;width:100%}.input-field input:hover{border-color:#454545}.current-indicator{background-color:#19478a}.other-indicator{background-color:transparent}.current{background-color:#e5e5e5}.navigation-button:hover{background-color:#dadada}.navigation-button{border-radius:3px;height:26px;min-width:50px;display:flex;flex-direction:column;color:#444;font-size:12px;font-family:Arial;margin-right:2px}.navigation-label{height:27px;width:100%;height:18px;padding:0 10px;display:flex;align-items:center;justify-content:center;white-space:nowrap}.navigation-label span{margin-top:7px}.indicator{height:3px;margin:1px 9px}.bottom-divider{height:1px;width:calc(100% - 24px);background-color:#d3d3d3;margin:5px 12px}.content-navigation{display:flex;padding:12px 12px 0}.content-container{height:100%;width:100%}.editor-navbar{display:flex;height:30px;border-top-left-radius:3px;border-top-right-radius:3px;background-color:#515c6a;-webkit-user-select:none;color:#fff;align-items:center}.editor-navbar span{font-size:13px;margin-left:10px}.close-button{height:30px;width:30px;border-radius:3px;display:flex;align-items:center;justify-content:center}.close-button:hover{background-color:#dadada}.close-button img{height:16px;width:16px;filter:brightness(0) invert(1)}.editor-container{background-color:var(--back-color);position:absolute;left:calc(50% - 182px);top:calc(50% - 175px);z-index:999;width:365px;height:350px;overflow:hidden;display:flex;flex-direction:column;box-shadow:0 0 7px #0000004d;border:1px solid rgb(25,71,138);color:var(--text-color);font-size:12px;border-radius:5px;font-family:Arial;-webkit-user-select:none}.button{height:24px;width:80px;border-radius:3px;margin:3px 5px;display:flex;align-items:center;justify-content:center;border:1px solid transparent}.accept:active{background-color:#355da0;border-color:#264c7d}.cancel:active{background-color:#e2e2e2;border-color:#838383}.accept{background-color:#19478a;color:#fff}.accept:hover{background-color:#3d6298}.cancel{background-color:#ddd;color:#444}.cancel:hover{background-color:#d5d5d5}.setup-accept{padding:8px;display:flex;align-items:flex-end;justify-content:flex-end}\n"], dependencies: [{ kind: "directive", type: i7.NgForOf, selector: "[ngFor][ngForOf]", inputs: ["ngForOf", "ngForTrackBy", "ngForTemplate"] }, { kind: "directive", type: i7.NgIf, selector: "[ngIf]", inputs: ["ngIf", "ngIfThen", "ngIfElse"] }, { kind: "directive", type: i8.DefaultValueAccessor, selector: "input:not([type=checkbox])[formControlName],textarea[formControlName],input:not([type=checkbox])[formControl],textarea[formControl],input:not([type=checkbox])[ngModel],textarea[ngModel],[ngDefaultControl]" }, { kind: "directive", type: i8.NgControlStatus, selector: "[formControlName],[ngModel],[formControl]" }, { kind: "directive", type: i8.NgModel, selector: "[ngModel]:not([formControlName]):not([formControl])", inputs: ["name", "disabled", "ngModel", "ngModelOptions"], outputs: ["ngModelChange"], exportAs: ["ngModel"] }, { kind: "directive", type: i9.StiTooltipDirective, selector: "[stiTooltip]", inputs: ["tooltipPosition", "stiTooltip", "showDelay"] }] });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "14.3.0", ngImport: i0, type: StiElectronicSignatureEditorComponent, decorators: [{
            type: Component,
            args: [{ selector: "sti-electronic-signature-editor", template: `<div
      class="editor-container"
      [style.left.px]="signatureService.editorComponentX"
      [style.top.px]="signatureService.editorComponentY"
    >
      <div class="editor-navbar" (mousedown)="startDrag($event)">
        <span [style.flex]="1">{{ loc.get2("Components", "StiSignature") }}</span>
        <div class="close-button" (click)="model.showSignatureEditor = false">
          <img [src]="model.getImage('images.Other.CloseForm.png')" />
        </div>
      </div>
      <div class="content-container">
        <div class="content-navigation">
          <div
            class="navigation-button {{ signatureService.selectedMode == 'Type' ? 'current' : '' }}"
            (click)="changeMode('Type')"
          >
            <div class="navigation-label">
              <span>{{ loc.get2("PropertyEnum", "StiSignatureTypeType") }}</span>
            </div>
            <div class="indicator {{ signatureService.selectedMode == 'Type' ? 'current-indicator' : '' }}"></div>
          </div>
          <div
            class="navigation-button {{ signatureService.selectedMode == 'Draw' ? 'current' : '' }}"
            (click)="changeMode('Draw')"
          >
            <div class="navigation-label">
              <span>{{ loc.get2("PropertyEnum", "StiSignatureTypeDraw") }}</span>
            </div>
            <div class="indicator {{ signatureService.selectedMode == 'Draw' ? 'current-indicator' : '' }}"></div>
          </div>
          <div class="templates-navigation-container">
            <div (click)="saveTemplate()" class="template-navigation-button {{ isEmptyContent() ? 'disabled' : '' }}">
              <img [src]="model.getImage('signature.StiSignatureSave.png')" />
            </div>
            <div
              #button
              (click)="showSavedTemplatesPreview(button)"
              class="template-navigation-button show-button"
              [style.width.px]="38"
            >
              <img [src]="model.getImage('signature.StiSignatureOpen.png')" /><i class="pi pi-chevron-down"></i>
            </div>
          </div>
        </div>
        <div class="bottom-divider"></div>
        <div class="type-container" *ngIf="signatureService.selectedMode == 'Type'">
          <div class="input-field">
            <span>{{ loc.getP("FullName") }}</span
            ><input type="text" [(ngModel)]="fullName" />
          </div>
          <div class="input-field">
            <span>{{ loc.getP("Initials") }}</span
            ><input type="text" [(ngModel)]="initials" />
          </div>
          <div class="button-panel">
            <div #selectButton class="select-button" (click)="showFontsPreview(selectButton)">
              <span class="button-label">{{ loc.get2("ChartRibbon", "Style") }}</span
              ><i class="pi pi-chevron-down"></i>
            </div>
          </div>
          <div class="preview-panel" [style.font-family]="signatureService.currentPreviewFont">
            <span>{{ previewText }}</span>
          </div>
        </div>
        <div class="draw-container" *ngIf="signatureService.selectedMode == 'Draw'">
          <div class="draw-buttons-panel">
            <div
              class="button-container {{ drawingMode ? 'active' : '' }}"
              [stiTooltip]="loc.getP('ForeColor')"
              tooltipPosition="bottom"
              showDelay="500"
              (mousedown)="$event.preventDefault()"
              #fontColorEditor
            >
              <div class="preview-container" (click)="drawingMode = !drawingMode">
                <div class="foreground" [style]="{ backgroundPosition: 'center', backgroundRepeat: 'no-repeat' }">
                  <img [src]="model.getImage('images.Brush.png')" alt="" />
                </div>
                <span
                  class="color-preview"
                  [style.background-color]="brushColor != null ? brushColor?.toHtmlColor() : 'rgb(182, 181, 180)'"
                ></span>
              </div>
              <div
                (click)="showColorPicker(fontColorEditor, brushColor)"
                [style]="{ position: 'relative', display: 'flex', alignItems: 'center' }"
              >
                <i class="pi pi-chevron-down"></i>
              </div>
            </div>
            <div class="draw-panel-button" (click)="showEditor('Text')">
              <img [src]="model.getImage('signature.Text.png')" />
            </div>
            <div class="draw-panel-button" (click)="showEditor('Image')">
              <img [src]="model.getImage('images.Toolbox.StiImageElement.png')" />
            </div>
            <div class="draw-panel-button" (click)="clear()">
              <img [src]="model.getImage('images.Page.ClearContent.png')" />
            </div>
          </div>
          <div class="draw-panel" #previewContainer>
            <div class="canvas-container">
              <canvas
                [width]="previewContainer.clientWidth"
                [height]="previewContainer.clientHeight"
                (mousedown)="onMouseDown($event, canvas)"
                (mousemove)="onCanvasMove($event, canvas)"
                #canvas
              >
              </canvas>
            </div>
            <div class="preview-inserted-text" [style]="insertedText.htmlStyle()" [style.justify-content]="alignment">
              <span>{{ insertedText.value }}</span>
            </div>
            <div class="inserted-image-container" [style.justify-content]="getHorizontalAlignment()">
              <div
                class="preview-inserted-image"
                [style.width.px]="
                  insertedImage?.aspectRatio && insertedImage?.stretch
                    ? previewContainer.clientHeight
                    : previewContainer.clientWidth
                "
                [style.height.px]="previewContainer.clientHeight"
                [style.background-image]="image"
                [style.background-position]="getBackgroundPosition()"
                [style.background-size]="getBackgroundSize(previewContainer)"
                [style.background-repeat]="image ? 'no-repeat' : ''"
              ></div>
            </div>
            <div class="signature-description">
              <span>{{ "Signature" }}</span>
            </div>
          </div>
        </div>
      </div>
      <div class="setup-accept">
        <div class="accept button" (click)="acceptChanges()">
          <span [style.margin-left.px]="0">{{
            model.isPreviewMode
              ? loc.get2("Buttons", "Sign")
              : loc.get2("Buttons", "Save").replace("&amp", "").replace(";", "")
          }}</span>
        </div>
        <div class="cancel button" (click)="model.showSignatureEditor = false">
          <span [style.margin-left.px]="0">{{ loc.getP("Cancel") }}</span>
        </div>
      </div>
    </div>
    <div
      class="fonts-preview-container"
      *ngIf="signatureService.showFontsPreview"
      [style.left.px]="signatureService.fontsPreviewX"
      [style.top.px]="signatureService.fontsPreviewY"
    >
      <div
        *ngFor="let font of signatureService.previewFonts"
        class="font-preview"
        [style.font-family]="font"
        (click)="signatureService.currentPreviewFont = font; signatureService.showFontsPreview = false"
      >
        <div class="vertical-line {{ signatureService.currentPreviewFont == font ? 'current-indicator' : '' }}"></div>
        <span> {{ previewText }}</span>
      </div>
    </div>
    <div
      *ngIf="signatureService.showSavedTemplates"
      class="saved-templates-container"
      [style.left.px]="signatureService.templatesPreviewX"
      [style.top.px]="signatureService.templatesPreviewY"
      [style.overflow-y]="hasTemplates ? 'scroll' : 'hidden'"
    >
      <div class="empty-container" *ngIf="!hasTemplates">
        <span>{{ "No saved items" }}</span>
      </div>
      <div
        class="template-item"
        *ngFor="let temp of signatureService.savedTemplates; let i = index"
        [style.margin-top.px]="i == 0 ? 6 : 0"
        (click)="signatureService.setTemplate(temp)"
      >
        <div class="delete-button" (click)="signatureService.deleteTemplate(i)">
          <img [src]="model.getImage('images.Other.CloseForm.png')" />
        </div>
        <div
          class="image-template"
          *ngIf="temp.type == 'Image'"
          #parentContainer
          [style.justify-content]="signatureService.getTemplateHorizontalAlignment(temp)"
        >
          <div
            class="loadedImage image-content"
            [style.background-image]="signatureService.getTemplateImage(temp)"
            [style.background-position]="signatureService.getTemplateBackgroundPosition(temp)"
            [style.background-size]="signatureService.getTemplateBackgroundSize(temp, parentContainer)"
            [style.background-repeat]="'no-repeat'"
          ></div>
          <div class="insertedText image-content">
            <span [style]="temp.text.htmlStyle()">{{ temp?.text?.value }}</span>
          </div>
          <div
            class="drawedImage image-content"
            [style.background-size]="'contain'"
            [style.background-image]="signatureService.getTemplateDrawedImage(temp)"
          ></div>
        </div>
        <div class="text-template" *ngIf="temp.type == 'Text'">
          <span [style.font-family]="temp.specialFont" [style.font-size.pt]="16">{{
            temp.fullName + " " + temp.initials
          }}</span>
        </div>
      </div>
    </div>`, styles: ["@font-face{font-family:Teddy Bear;src:url(projects/stimulsoft-forms/src/lib/assets/TeddyBear.ttf) format(\"truetype\")}@font-face{font-family:Denistina;src:url(projects/stimulsoft-forms/src/lib/assets/Denistina.ttf) format(\"truetype\")}@font-face{font-family:MADE Likes Script;src:url(projects/stimulsoft-forms/src/lib/assets/MADELikesScript.otf) format(\"opentype\")}.image-content{height:100%;width:100%;top:0;left:0;position:absolute}.drawedImage{z-index:3}.loadedImage{z-index:1}.insertedText{z-index:2;display:flex;align-items:center;justify-content:center}.text-template{display:flex;align-items:center;justify-content:center;height:50px}.image-template{position:relative;height:130px}.template-item{width:calc(100% - 12px);height:-moz-fit-content;height:fit-content;padding:2px;margin:0 6px 6px;border:1px solid lightgray;border-radius:3px;position:relative}.delete-button{position:absolute;z-index:999;right:0;top:0;width:20px;height:20px;display:flex;align-items:center;justify-content:center;border-radius:3px}.delete-button img{height:16px;width:16px}.delete-button:hover img{filter:saturate(500%) contrast(100%) brightness(900%) invert(900%) sepia(50%)}.template-item:hover{background-color:#d7d7d7;cursor:pointer}.empty-container{height:100%;width:100%;background-color:#fafafa;display:flex;align-items:center;justify-content:center;color:#454545;font-size:13px;font-family:Arial}.saved-templates-container{position:absolute;overflow-x:hidden;overflow-y:scroll;width:335px;height:270px;border-radius:3px;box-shadow:0 0 7px #0000004d;border:1px solid rgb(25,71,138);display:flex;flex-direction:column;background-color:#fff;z-index:99999999999999999}.active{background-color:#d3d3d3;outline:1px solid rgb(61,107,186)}.canvas-container{position:absolute;left:0;top:0;height:100%;width:100%;z-index:99999}.preview-inserted-image{position:absolute}.inserted-image-container{position:absolute;left:0;top:0;display:flex;height:100%;width:100%}.preview-inserted-text{position:absolute;height:100%;width:100%;display:flex;align-items:center;justify-content:center;z-index:999}.signature-content{z-index:9999;height:100%;width:100%}.signature-description{width:calc(100% - 10px);height:50px;margin:0 5px;padding:0 5px;border-top:1px solid lightgray;display:flex;align-items:center;overflow:hidden;z-index:999}.disabled{pointer-events:none;opacity:.5}.preview-container{display:flex;width:100%;position:relative;flex-direction:column}.preview-container img{height:16px;width:16px}.button-container{display:flex;height:26px;width:40px;position:relative;margin-right:5px;border-radius:3px}.button-container:hover{background-color:#d3d3d3}.button-container:active{background-color:var(--columnDeleteHover-color)}.color-preview{width:20px;height:5px;margin-left:4px}i{height:12px;width:12px;right:3px;top:50%;vertical-align:middle;color:var(--text-color)}.button-container:hover i{color:#fff}.foreground{margin-left:8px;margin-top:2px;width:16px;height:16px}.template-navigation-button{height:26px;width:26px;margin-right:2px;border-radius:3px;display:flex;align-items:center;justify-content:center}.template-navigation-button:hover{background-color:#d3d3d3}.template-navigation-button img{height:16px;width:16px}.templates-navigation-container{width:100%;display:flex;align-items:center;justify-content:end}.vertical-line{width:3px;height:calc(100% - 14px);margin:7px 10px 7px 5px}.font-preview{font-size:16px;height:42px;display:flex;align-items:center;padding-left:5px;border-radius:3px;margin:1px}.font-preview:hover{background-color:#dadada}.fonts-preview-container{position:absolute;width:200px;height:134px;z-index:9999999999;padding:1px;border-radius:3px;border:1px solid #c6c6c6;background:#ffffff;color:#444;box-shadow:0 0 7px #0000004d;cursor:default}.draw-buttons-panel{height:26px;margin-bottom:5px;margin-top:10px;display:flex;padding-left:24px}.draw-panel-button:hover{background-color:#d3d3d3}.draw-panel-button{height:26px;width:26px;margin-right:5px;border-radius:3px;display:flex;align-items:center;justify-content:center}.draw-panel-button img{height:16px;width:16px}.draw-panel{border:1px dashed #ababab;border-radius:3px;margin:0 24px;width:calc(100% - 48px);height:186px;display:flex;flex-direction:column-reverse;position:relative}i{margin-left:5px;font-size:7px;padding-top:4px}.select-button{border:1px solid #dddddd;border-radius:3px;min-width:30px;padding-left:10px;padding-right:6px;display:flex;color:#444;align-items:center;margin-right:24px;margin-top:24px;height:24px}.select-button:hover{background-color:#d3d3d3}.button-panel{width:100%;display:flex;justify-content:end}.preview-panel{height:80px;width:calc(100% - 48px);border:1px solid #ababab;border-radius:3px;font-size:16pt;margin:10px 24px;padding-left:12px;display:flex;align-items:center;overflow:hidden}.input-field{width:calc(100% - 24px);display:flex;margin:12px 12px 6px;padding:0 12px;align-items:center}.input-field span{min-width:100px;font-size:12px}.input-field input{border-radius:3px;padding-left:5px;height:24px;outline:none;border:1px solid lightgray;width:100%}.input-field input:hover{border-color:#454545}.current-indicator{background-color:#19478a}.other-indicator{background-color:transparent}.current{background-color:#e5e5e5}.navigation-button:hover{background-color:#dadada}.navigation-button{border-radius:3px;height:26px;min-width:50px;display:flex;flex-direction:column;color:#444;font-size:12px;font-family:Arial;margin-right:2px}.navigation-label{height:27px;width:100%;height:18px;padding:0 10px;display:flex;align-items:center;justify-content:center;white-space:nowrap}.navigation-label span{margin-top:7px}.indicator{height:3px;margin:1px 9px}.bottom-divider{height:1px;width:calc(100% - 24px);background-color:#d3d3d3;margin:5px 12px}.content-navigation{display:flex;padding:12px 12px 0}.content-container{height:100%;width:100%}.editor-navbar{display:flex;height:30px;border-top-left-radius:3px;border-top-right-radius:3px;background-color:#515c6a;-webkit-user-select:none;color:#fff;align-items:center}.editor-navbar span{font-size:13px;margin-left:10px}.close-button{height:30px;width:30px;border-radius:3px;display:flex;align-items:center;justify-content:center}.close-button:hover{background-color:#dadada}.close-button img{height:16px;width:16px;filter:brightness(0) invert(1)}.editor-container{background-color:var(--back-color);position:absolute;left:calc(50% - 182px);top:calc(50% - 175px);z-index:999;width:365px;height:350px;overflow:hidden;display:flex;flex-direction:column;box-shadow:0 0 7px #0000004d;border:1px solid rgb(25,71,138);color:var(--text-color);font-size:12px;border-radius:5px;font-family:Arial;-webkit-user-select:none}.button{height:24px;width:80px;border-radius:3px;margin:3px 5px;display:flex;align-items:center;justify-content:center;border:1px solid transparent}.accept:active{background-color:#355da0;border-color:#264c7d}.cancel:active{background-color:#e2e2e2;border-color:#838383}.accept{background-color:#19478a;color:#fff}.accept:hover{background-color:#3d6298}.cancel{background-color:#ddd;color:#444}.cancel:hover{background-color:#d5d5d5}.setup-accept{padding:8px;display:flex;align-items:flex-end;justify-content:flex-end}\n"] }]
        }], ctorParameters: function () { return [{ type: i1.StiModelService }, { type: i2.StiLocalizationService }, { type: i3.StiMouseService }, { type: i4.SignatureService }, { type: i5.StiDraggableModalWindowService }, { type: i6.StiColorDialogService }]; }, propDecorators: { canvas: [{
                type: ViewChild,
                args: ["canvas"]
            }] } });
//# sourceMappingURL=data:application/json;base64,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